// -*- indent-tabs-mode: 1; tab-width: 4; c-basic-offset: 4 -*-
//
// Gelato RIB Reader main
//
/////////////////////////////////////////////////////////////////////////////
// Copyright 2004 NVIDIA Corporation.  All Rights Reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
// * Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// * Neither the name of NVIDIA nor the names of its contributors
//   may be used to endorse or promote products derived from this software
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// (This is the Modified BSD License)
/////////////////////////////////////////////////////////////////////////////
//
// $Revision: $    $Date:  $
//
// "RIB" is a trademark of Pixar, Inc.
//

#include <stdio.h>

#include "gelatoapi.h"
using namespace Gelato;
#include "imageio.h"

#include "Ribelato.h"

#define VERSION "1.0.0.24"

class RibGenerator: public GelatoAPI::Generator, public Ribelato {
  public:
	RibGenerator(): Ribelato(NULL) { _abort = false; }
	virtual ~RibGenerator() {}
	virtual void run(GelatoAPI *renderer, const char *filename);

	// Override this to route error msgs appropriately.
	virtual void handleError(int severity, const char *msg);

	const char *_file;
	bool _abort;
};

// print error, or pop a window, put in status line, open file and highlight offending line, etc.

const char *severityStrs[] = { "INFO", "WARNING", "ERROR", "SEVERE" };

void RibGenerator::handleError(int severity, const char* msg)
{
	if (_errorHandler == ribErrIgnore)
		return;
	fprintf(stderr, "%s: %s, line %d: %s\n", severityStrs[severity], _file, _line, msg);
	if (severity == ribErrSevere || (severity == ribErrError && _errorHandler == ribErrAbort))
		_abort = true;
}

static int getChar(void *fp) { return getc((FILE *)fp); }

void RibGenerator::run(GelatoAPI *renderer, const char *filename)
{
	static bool printed = false;
	int verbosity = 1;
	renderer->GetAttribute("int verbosity", verbosity);
	if ((verbosity == 1 && !printed) || verbosity >= 2) {
		printf("# rib.generator.so v%s: reading '%s'.\n", VERSION, filename);
		printed = true;
	}

	if (FILE *fp = fopen(filename, "rb")) {
		this->_file = filename;
		this->_api = renderer;
		this->_getc = getChar;
		this->_cbData = fp;

		bool eof = false;
		do {
			ribErr err = doCmd(eof);
			// ToDo: progress would be good here
		} while(!eof && !_abort);

		fclose(fp);
	} else
		fprintf(stderr, "rib.generator.so: Can't open \"%s\".\n", filename);
}

extern "C" {

EXPORT int generator_version = GelatoAPI::API_VERSION;

EXPORT GelatoAPI::Generator *rib_generator_create() { return new RibGenerator(); }

};
