// -*- indent-tabs-mode: 1; tab-width: 4; c-basic-offset: 4 -*-
/////////////////////////////////////////////////////////////////////////////
// Copyright 2004 NVIDIA Corporation.  All Rights Reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
// * Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// * Neither the name of NVIDIA nor the names of its contributors
//   may be used to endorse or promote products derived from this software
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// (This is the Modified BSD License)
/////////////////////////////////////////////////////////////////////////////
//
// $Revision: $    $Date:  $
//
// "RIB" is a trademark of Pixar, Inc.
//

#define _USE_MATH_DEFINES // for M_PI
#include <math.h>
#include <stdio.h>
#include <string.h>
#ifdef _WIN32
#include <io.h>
#include <fcntl.h>
#else
#include <unistd.h>
#endif

// Turn off "conversion from double to float" warnings
#ifdef _WIN32
#pragma warning(disable : 4244)
#pragma warning(disable : 4305)
#define huge hugef // fmath.h's inline huge() doesn't compile under msvc++
#endif

#include "gelatoapi.h"
#include "vecmat.h"

#include "Ribelato.h"

using namespace Gelato;

// some Ri tokens
static const char
	*RI_LINEAR = "linear",
	*RI_BILINEAR = "bilinear",
	*RI_PERIODIC = "periodic",
	*RI_NONPERIODIC = "nonperiodic",
	*RI_BSPLINE = "b-spline",
	*RI_S = "s",
	*RI_T = "t",
	*RI_ST = "st";

// and some Gelato tokens
static const char
	*RIGEL_BEZIER = "bezier",
	*RIGEL_BORDER = "constant int border",
	*RIGEL_BSPLINE = "bspline", // no dash?
	*RIGEL_CATMULLROM = "catmull-rom",
	*RIGEL_LINEAR_S = "linear float s",
	*RIGEL_LINEAR_T = "linear float t",
	*RIGEL_PERPIECE = "perpiece",
	*RIGEL_VERTEX = "vertex",
	*RIGEL_VERTEX_S = "vertex float s",
	*RIGEL_VERTEX_T = "vertex float t";

Ribelato::texCoords Ribelato::_defaultTexCoords = {
	//s1 s2 s3 s4
	{ 0, 1, 0, 1 },
	{ 0, 0, 1, 1 }
	//t1 t2 t3 t4
};

const float
	BezierBasis[4][4] = {
		-1,  3, -3,  1,
		 3, -6,  3,  0,
		-3,  3,  0,  0,
		 1,  0,  0,  0
	},
	BSplineBasis[4][4] = {
		-1./6,  3./6, -3./6,  1./6,
		 3./6, -6./6,  3./6,  0,
		-3./6,  0,     3./6,  0,
		 1./6,  4./6,  1./6,  0
	},
	CatmullRomBasis[4][4] = {
		-1./2,  3./2, -3./2,  1./2,
		 2./2, -5./2,  4./2, -1./2,
		-1./2,  0,     1./2,  0,
		 0,     2./2,  0,     0		
	},
	HermiteBasis[4][4] = {
		 2,  1, -2,  1,
		-3, -2,  3, -1,
		 0,  1,  0,  0,
		 1,  0,  0,  0
	},
	PowerBasis[4][4] = {
		 1,  0,  0,  0,
		 0,  1,  0,  0,
		 0,  0,  1,  0,
		 0,  0,  0,  1
	};

Ribelato::basis Ribelato::_defaultBasis = {
	RIGEL_BEZIER, RIGEL_BEZIER, 3, 3
};

// Append a C string, including the null, to a vector<char>.

inline void append(const char *s, vector<char> &v) {
	do v.push_back(*s); while(*s++);
}


int intersectlines2D(float *p1, float *t1,
					 float *p2, float *t2,
					 float *p)
{
  float ua, ub, den, nomua, nomub;

  den = t2[1]*t1[0] - t2[0]*t1[1];

  if(fabs(den) < 1.0e-6)
    return 0;

  nomua = (t2[0]*(p1[1]-p2[1]) - t2[1]*(p1[0]-p2[0]));
  nomub = (t1[0]*(p1[1]-p2[1]) - t1[1]*(p1[0]-p2[0]));

  if((fabs(den) < 1.0e-6) &&
     (fabs(nomua) < 1.0e-6) &&
     (fabs(nomub) < 1.0e-6))
    return 0;

  ua = nomua/den;
  ub = nomub/den;

  p[0] = p1[0] + ua*t1[0];
  p[1] = p1[1] + ua*t1[1];

 return 1;
} // intersectlines2D


int circlearc(float r, float ths, float the,
			  int *length, float **knotv, float **controlv)
{
	float theta = 0.0, dtheta = 0.0, angle = 0.0, w1 = 0.0;
	float *Pw = NULL, *U = NULL;
	int i, j, index = 0, narcs = 0, n = 0;
	float P0[4] = {0}, P1[4] = {0}, P2[4] = {0};
	float T0[2] = {0}, T2[2] = {0};

	if(the < ths)
		the = 2.0*M_PI - the;

	theta = the - ths;

	if(theta <= M_PI/2.0)
		narcs = 1;
	else /* XXXX this +1.0e-6 is necessary due to roundoff err,
		  * do we need it in other places as well? Is it harmless?
		  */
		if(theta <= M_PI+1.0e-6)
			narcs = 2;
		else
			if(theta <= M_PI+M_PI/2.0)
				narcs = 3;
			else
				narcs = 4;

	dtheta = theta/narcs;
	n = 2 * narcs; /* n+1 control points */
	w1 = cos(dtheta/2.0); /* dtheta/2 == base angle */

	/* alloc mem for Pw and U */
	if(!(Pw = (float*)calloc((n+1)*4, sizeof(float))))
		return 0;
	if(knotv)
		if(!(U = (float*)calloc((n+1)+4, sizeof(float))))
			{ free(Pw); return 0; }

	P0[0] = r * cos(ths);
	P0[1] = r * sin(ths);
	P0[3] = 1.0;
	T0[0] = -sin(ths);
	T0[1] = cos(ths);
	Pw[0] = P0[0];
	Pw[1] = P0[1];
	Pw[3] = 1.0;
	index = 0; angle = ths;
	for(i = 1; i <= narcs; i++)
		{
			angle += dtheta;

			P2[0] = r * cos(angle);
			P2[1] = r * sin(angle);
			P2[3] = 1.0;
			T2[0] = -sin(angle);
			T2[1] = cos(angle);
			//memset(P1,0,4*sizeof(float));
			intersectlines2D(P0, T0, P2, T2, P1);

			Pw[(index+1)*4]     = w1*P1[0];
			Pw[((index+1)*4)+1] = w1*P1[1];
			Pw[((index+1)*4)+3] = w1;

			memcpy(&Pw[(index+2)*4], P2, 4*sizeof(float));

			index += 2;
			if(i < index)
				{
					memcpy(P0, P2, 4*sizeof(float));
					memcpy(T0, T2, 2*sizeof(float));
				}
		} // for

	if(knotv)
		{
			j = 2*narcs+1;
			for(i = 0; i < 3; i++)
				{
					U[i] = 0.0;
					U[i+j] = 1.0;
				} // for

			switch(narcs)
				{
				case 1:
					break;
				case 2:
					U[3] = 0.5;
					U[4] = 0.5;
					break;
				case 3:
					U[3] = 1.0/3.0;
					U[4] = 1.0/3.0;
					U[5] = 2.0/3.0;
					U[6] = 2.0/3.0;
					break;
				case 4:
					U[3] = 0.25;
					U[4] = 0.25;
					U[5] = 0.5;
					U[6] = 0.5;
					U[7] = 0.75;
					U[8] = 0.75;
					break;
				} // switch

			*knotv = U;
		} // if

	*controlv = Pw;
	*length = n+1;

	return 1;
} // circlearc



// Create.

Ribelato::Ribelato(GelatoAPI *api, int (*getc)(void *cbData), void *cbData)
	: RIBParser(getc, cbData)
{
	_api = api;
	_worldBegun = false;
	_gain = 1;
	_gamma = 1;
	_texCoords = _defaultTexCoords;
	append("gaussian", _filter);
	_filterWidth[0] = 2;
	_filterWidth[1] = 2;
	_basis = _defaultBasis;
	_errorHandler = ribErrPrint;
}

// Destroy.

Ribelato::~Ribelato()
{
}

static bool isBasis(const float b1[4][4], const float b2[4][4]) {
	if (b1 == b2)
		return true;
	for(int i = 0; i < 4; i++)
		for(int j = 0; j < 4; j++)
			if (fabs(b1[i][j] - b2[i][j]) > 1.0e-6)
				return false;
	return true;
}

// Handle a command.

RIBParser::ribErr Ribelato::handleCmd(ribCode code)
{
	ribErr err = reNoErr;

	// Execute the command.
	switch(code) {
		#define I(n)     _tokens[n].value.i
		#define F(n)     _tokens[n].value.f
		#define S(n)     _tokens[n].value.s
		#define AF(n)    _tokens[n].value.af
		#define AFlen(n) _tokens[n].value.af.len
		#define AI(n)    _tokens[n].value.ai
		#define AIlen(n) _tokens[n].value.ai.len
		#define AS(n)    _tokens[n].value.as
		#define ASlen(n) _tokens[n].value.as.len

		// bracketing
		case riWorldBegin:
			_api->World();
			_worldBegun = true;
			break;
		case riWorldEnd:
			_api->Render("camera");
			break;
		case riMotionBegin:
			_api->Motion(AFlen(0), AF(0));
			break;

		// options
		case riFormat: {
			int v[2] = { I(0), I(1) };
			_api->Attribute("int[2] resolution", v);
			float aspect = F(2);
			if (aspect != 1 && aspect != -1)
				_api->Attribute("float pixelaspect", aspect);
			break;
		}
		case riScreenWindow:
			_api->Attribute("float[4] screen", AF(0));
			break;
		case riCropWindow:
			_api->Attribute("float[4] crop", AF(0));
			break;
		case riProjection: {
			_api->Attribute ("string projection", S(0));
			for(int i = 0; i < _nParams; i++)
				if (strcmp(_names[i], "fov") == 0)
					_api->Attribute("float fov", *(float *)_parms[i]);
			break;
		}
		case riClipping:
			_api->Attribute("float near", F(0));
			_api->Attribute("float far", F(1));
			break;
		case riShutter: {
			float v[2] = { F(0), F(1) };
			_api->Attribute("float[2] shutter", v);
			break;
		}
		case riPixelSamples: {
			int v[2] = { (int)F(0), (int)F(1) };
			_api->Attribute("int[2] spatialquality", v);
			break;
		}
		case riPixelFilter: {
			_filter.clear();
			append(S(0), _filter);
			_filterWidth[0] = F(1);
			_filterWidth[1] = F(2);
			break;
		}
		case riExposure:
			_gain = F(0);
			_gamma = F(1);
			break;
		case riQuantize: {
			quantize q;
			if (strlen(S(0)) < QNAMELEN)
				strcpy(q.name, S(0));
			else {
				strncpy(q.name, S(0), QNAMELEN - 1); // silently truncate and hope for the best :-(
				q.name[QNAMELEN - 1] = '\0';
			}
			q.one = I(1), q.min = I(2), q.max = I(3);
			q.dither = F(4);
			_quants.push_back(q);
			break;
		}
		case riDisplay:
			Display(S(0), S(1), S(2));
			break;
	    case riDisplayChannel:
			// No current equivalent
			break;
		case riDepthOfField:
			_api->Attribute("float fstop", F(0));
			_api->Attribute("float focallength", F(1));
			_api->Attribute("float focaldistance", F(2));
			break;
		case riOption:
			Option(S(0));
			break;

		// atributes
		case riAttributeBegin:
			_api->PushAttributes();
			_tcStack.push_back(_texCoords);
			_basisStack.push_back(_basis);
			break;
		case riAttributeEnd:
			_api->PopAttributes();
			_texCoords = _tcStack.back();
			_tcStack.pop_back();
			_basis = _basisStack.back();
			_basisStack.pop_back();
			break;
		case riColor:
			_api->Attribute("color C", AF(0));
			break;
		case riOpacity:
			_api->Attribute("color opacity", AF(0));
			break;
		case riTextureCoordinates:
			// We both translate and stack these: translate, for shaders; and stack, to paste onto
			// patches w/o using GetAttribute(), which is not supported by libpygout.
			_api->Attribute("float[8] user:textureCoordinates", AF(0));

			_texCoords.s[0] = AF(0)[0]; _texCoords.t[0] = AF(0)[1];
			_texCoords.s[1] = AF(0)[2]; _texCoords.t[1] = AF(0)[3];
			_texCoords.s[2] = AF(0)[4]; _texCoords.t[2] = AF(0)[5];
			_texCoords.s[3] = AF(0)[6]; _texCoords.t[3] = AF(0)[7];
			break;
		case riDisplacement:
		case riSurface:
		case riAtmosphere:
			mapTokens(true/*noClass*/);
			if (_nParams)
				sendParms(_nParams, &_names2[0], &_parms[0]);
			_api->Shader(code == riDisplacement ? "displacement" : code == riSurface ? "surface"
				: "atmosphere", S(0));
			break;
		case riShadingRate:
			_api->Attribute("float shadingquality", sqrt(1.0f / F(0)));
			break;
		case riMatte:
			_api->Attribute("int holdout", I(0));
			break;
		case riGeometricApproximation:
			if (strcmp(S(0), "motionfactor") == 0)
				_api->Attribute("dice:motionfactor", F(1));
			break;
		case riOrientation:
			_api->Attribute("string orientation", S(0));
			break;
		case riReverseOrientation:
			_api->Attribute("string orientation", "reverse");
			break;
		case riSides:
			_api->Attribute("int twosided", I(0) - 1);
			break;
		case riIdentity:
			_api->SetTransform(_worldBegun ? "world" : "camera");
			break;
		case riTransform:
			_api->SetTransform(AF(0));
			break;
		case riConcatTransform:
			_api->AppendTransform(AF(0));
			break;
		case riPerspective: {
			float s = atanf(M_PI/180.0*(F(0)/2.0f));
			Matrix4 m(1, 0,  0, 0,
            		  0, 1,  0, 0,
            		  0, 0,  s, s,
            		  0, 0, -1, 0);
			_api->AppendTransform((const float *)&m);
			break;
		}
		case riTranslate:
			_api->Translate(F(0), F(1), F(2));
			break;
		case riRotate:
			_api->Rotate(F(0), F(1), F(2), F(3));
			break;
		case riScale:
			_api->Scale(F(0), F(1), F(2));
			break;
		case riSkew: {
			// per Goldman, Graphics Gems II, p338, w/Q=0.
			float angle = AF(0)[0], v1 = AF(0)[1], v2 = AF(0)[2], v3 = AF(0)[3],
				w1 = AF(0)[4], w2 = AF(0)[5], w3 = AF(0)[6];
			float t = tan(angle * (M_PI / 180));
			float tv1 = t*v1, tv2 = t*v2, tv3 = t*v3;
			Matrix4 m(1+tv1*w1,   tv1*w2,   tv1*w3, 0,
			    		tv2*w1, 1+tv2*w2,   tv2*w3, 0,
						tv3*w1,   tv3*w2, 1+tv3*w3, 0,
				    		 0,        0,        0, 1);
			_api->AppendTransform((const float *)&m);
			break;
		}
		case riCoordinateSystem:
			_api->SaveAttributes(S(0), "transform");
			break;
		case riCoordSysTransform:
			_api->SetTransform(S(0));
			break;
		case riTransformBegin:
			_api->PushTransform();
			break;
		case riTransformEnd:
			_api->PopTransform();
			break;
		case riBasis: {
			const char *err = "unimplemented patch basis.";
			if (_tokens[0].type == ribToken::rtString) {
				if (strcmp(S(0), RIGEL_BEZIER) == 0)
					_basis.ubasis = RIGEL_BEZIER;
				else if (strcmp(S(0), RI_BSPLINE) == 0)
					_basis.ubasis = RIGEL_BSPLINE;
				else if (strcmp(S(0), RIGEL_CATMULLROM) == 0)
					_basis.ubasis = RIGEL_CATMULLROM;
				else
					handleError(ribErrWarning, err);
			} else {
				const float (*ubasis)[4] = (const float (*)[4])(const float *)AF(0);
				if (isBasis(ubasis, BezierBasis))
					_basis.ubasis = RIGEL_BEZIER;
				else if (isBasis(ubasis, BSplineBasis))
					_basis.ubasis = RIGEL_BSPLINE;
				else if (isBasis(ubasis, CatmullRomBasis))
					_basis.ubasis = RIGEL_CATMULLROM;
				else
					handleError(ribErrWarning, err);
			}
			_basis.ustep = I(1);
			if (_tokens[2].type == ribToken::rtString) {
				if (strcmp(S(2), RIGEL_BEZIER) == 0)
					_basis.vbasis = RIGEL_BEZIER;
				else if (strcmp(S(2), RI_BSPLINE) == 0)
					_basis.vbasis = RIGEL_BSPLINE;
				else if (strcmp(S(2), RIGEL_CATMULLROM) == 0)
					_basis.vbasis = RIGEL_CATMULLROM;
				else
					handleError(ribErrWarning, err);
			} else {
				const float (*vbasis)[4] = (const float (*)[4])(const float *)AF(2);
				if (isBasis(vbasis, BezierBasis))
					_basis.vbasis = RIGEL_BEZIER;
				else if (isBasis(vbasis, BSplineBasis))
					_basis.vbasis = RIGEL_BSPLINE;
				else if (isBasis(vbasis, CatmullRomBasis))
					_basis.vbasis = RIGEL_CATMULLROM;
				else
					handleError(ribErrWarning, err);
			}
			_basis.vstep = I(3);
			break;
		}
		case riAttribute:
			Option(S(0));
			break;
		case riLightSource:
		case riAreaLightSource: {
			char lightName[64];
			const char* lightNamep = lightName;
			if (_tokens[1].type == ribToken::rtInt)
				sprintf(lightName, "light%d", I(1));
			else
				lightNamep = S(1);
			Light(code == riAreaLightSource/*isAreaLight*/, S(0), lightNamep);
			break;
		}
		case riIlluminate: {
			if (_tokens[0].type == ribToken::rtInt) {
				char lightName[64];
				sprintf(lightName, "light%d", I(0));
				_api->LightSwitch(lightName, I(1) ? true : false);
			} else
				_api->LightSwitch(S(0), I(1) ? true : false);
			break;
		}

		// geometry
		case riCone:
			Cone(AF(0)[0], AF(0)[1], AF(0)[2]);
			break;
		case riCylinder:
			Cylinder(AF(0)[0], AF(0)[1], AF(0)[2], AF(0)[3]);
			break;
		case riDisk:
			Disk(AF(0)[0], AF(0)[1], AF(0)[2]);
			break;
		case riHyperboloid:
			Hyperboloid(AF(0)[0], AF(0)[1], AF(0)[2],
						AF(0)[3], AF(0)[4], AF(0)[5], AF(0)[6]);
			break;
		case riParaboloid:
			Paraboloid(AF(0)[0], AF(0)[1], AF(0)[2], AF(0)[3]);
			break;
		case riTorus:
			Torus(AF(0)[0], AF(0)[1], AF(0)[2], AF(0)[3], AF(0)[4]);
			break;
		case riPolygon:					Polygon(_Plen); break;
		case riGeneralPolygon:			GeneralPolygon(AIlen(0), AI(0)); break;
		case riPointsPolygons:			PointsPolygons(AIlen(0), AI(0), AI(1)); break;
		case riPointsGeneralPolygons:	PointsGeneralPolygons(AIlen(0), AI(0), AI(1), AI(2)); break;
		case riSubdivisionMesh:			SubdivisionMesh(S(0), AIlen(1), AI(1), AI(2), ASlen(3),
											AS(3), AI(4), AI(5), AF(6)); break;
		case riPatch:					Patch(S(0)); break;
		case riPatchMesh:				PatchMesh(S(0), I(1), S(2), I(3), S(4)); break;
		case riNuPatch:					NuPatch(I(0), I(1), AF(2), F(3), F(4), I(5), I(6), AF(7),
											F(8), F(9)); break;
		case riTrimCurve:				TrimCurve(AIlen(0), AI(0), AI(1), AF(2), AF(3), AF(4),
											AI(5), AF(6), AF(7), AF(8)); break;
		case riSphere:					Sphere(AF(0)[0], AF(0)[1], AF(0)[2], AF(0)[3]); break;
		case riCurves:					Curves(S(0), AIlen(1), AI(1), S(2)); break;
		case riPoints:					Points(_Plen); break;
		case riProcedural:				Procedural(S(0), AS(1), AF(2)); break;

		// other
		case riDeclare:					Declare(S(0), S(1)); break;
		case riReadArchive:				_api->Input(S(0)); break;
		case riErrorHandler:			ErrorHandler(S(0)); break;

		#undef I
		#undef F
		#undef S
		#undef AF
		#undef AFlen
		#undef AI
		#undef AIlen
		#undef AS
		#undef ASlen
	}

	return err;
}

// options

void Ribelato::Display(const char* name, const char* type, const char* mode)
{
	mapTokens();
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0]);

	// Add gain & gamma.
	_api->Parameter("float gain", _gain);
	_api->Parameter("float gamma", _gamma);

	// Add quantization & dither.
	float qv[4];
	for(int j = 0; j < _quants.size(); j++) {
		quantize &q = _quants[j];
		if (strcmp(q.name, mode) == 0) {
			qv[0] = 0, qv[1] = q.one, qv[2] = q.min, qv[3] = q.max;
			_api->Parameter("float[4] quantize", qv);
			_api->Parameter("float dither", q.dither);
			break;
		}
	}

	// Add the pixel filter.
	_api->Parameter("string filter", &_filter[0]);
	_api->Parameter("float[2] filterwidth", _filterWidth);

	// strip off leading '+'
	const char *namep = name[0] == '+' ? name+1 : name;

	// convert types
	if (strcmp(type, "framebuffer") == 0)
		type = "iv";
	else if (strcmp(type, "file") == 0)
		type = "tiff";

	_api->Output(namep, type, mode, "camera");
}

// prman -> gelato parameter list translation table.
// If [2] is NULL, it means there's no Gelato equivalent, so ignore.
static char *odict[][3] = {
	"dice", "binary", "int dice:binary",
	"dice", "rasterorient", "int dice:rasterorient",
	"displacementbound", "coordinatesystem", "string displace:maxspace",
	"displacementbound", "sphere", "float displace:maxradius",
	"identifier", "name", "string name",
	"identifier", "shadinggroup", NULL,
	"light", "nsamples", "int light:nsamples",
	"limits", "bucketsize", "int[2] limits:bucketsize",
	"limits", "eyesplits", NULL,
	"limits", "gridsize", "int limits:gridsize",
	"limits", "othreshold", "color limits:opacitythreshold",
	"limits", "texturefiles", "int limits:texturefiles",
	"limits", "texturememory", "int limits:texturememory",
	"render", "max_raylevel", "int ray:maxdepth",
	"render", "tracedisplacements", "int dice:tracedisplacements",
	"searchpath", "archive", "string path:input",
	"searchpath", "display", "string path:imageio",
	"searchpath", "procedural", "string path:generator",
	"searchpath", "shader", "string path:shader",
	"searchpath", "texture", "string path:texture",
	"shadow", "bias", "float shadow:bias",
	"shadow", "bias0", "float shadow:bias",
	"shadow", "bias1", NULL,
	"statistics", "endofframe", "int statistics:level",
	"statistics", "filelog", "string statistics:filename",
	"stitch", "enable", NULL,
	"stitch", "newgroup", NULL,
	"texture", "enable gaussian", NULL,
	"texture", "enable lerp", NULL,
	"trace", "maxdepth", "int ray:maxdepth",
	"trimcurve", "sense", "string trimcurve:sense"
};

void Ribelato::Option(const char* name)
{
	// Translate the option & param list.
	for(int i = 0; i < _nParams; i++) {
		const char *tokname = _names[i];
		if (const char *tp = strrchr(_names[i], ' '))
			tokname = tp+1;

		if (strcmp(name, "user") == 0) {
			if (tokname == _names[i]) {
				char msg[80];
				sprintf(msg, "%s: User params must be fully spec'd (e.g., \"int %s\").", tokname,
					tokname);
				handleError(ribErrWarning, msg);
			} else {
				char s[256];
				int nc = tokname - _names[i];
				strncpy(s, _names[i], nc);
				sprintf(s+nc, "user:%s", tokname);
				_api->Attribute(s, _parms[i]);
			}
		} else if (strcmp(name, "gelato") == 0) {
			_api->Attribute(_names[i], _parms[i]);
		} else if (strcmp(name, "render") == 0) {
			if (strcmp(tokname, "casts_shadows") == 0) { // ToDo: doublecheck that it's a string
				if (strcmp(((const char **)_parms[i])[0], "opaque") == 0)
					_api->Attribute("int ray:opaqueshadows", 1);
			}
		} else if (strcmp(name, "visibility") == 0) {
			const char *geometryset = "string geometryset";
			if (strcmp(tokname, "shadow") == 0)
				_api->Attribute(geometryset, ((int *)_parms[i])[0] ? "+shadow" : "-shadow");
			else if (strcmp(tokname, "reflection") == 0)
				_api->Attribute(geometryset, ((int *)_parms[i])[0] ? "+reflection" : "-reflection");
			else if (strcmp(tokname, "camera") == 0)
				_api->Attribute(geometryset, ((int *)_parms[i])[0] ? "+camera" : "-camera");
			else if (strcmp(tokname, "transmission") == 0) {
				if (strcmp(((const char **)_parms[i])[0], "opaque") == 0)
					_api->Attribute(geometryset, "+raytrace");
				else if (strcmp(((const char **)_parms[i])[0], "transparent") == 0)
					_api->Attribute(geometryset, "-raytrace");
			}
		} else {
			for(int j = 0, nj = sizeof(odict)/sizeof(odict[0]); j < nj; j++) {
				if (strcmp(name, odict[j][0]) == 0 && strcmp(tokname, odict[j][1]) == 0) {
					if (odict[j][2] != NULL)
						_api->Attribute(odict[j][2], _parms[i]);	
					// silently swallow attribs NULL as the Gelato equivalent
					break;
				} else if (j == nj-1) {
					char msg[80];
					sprintf(msg, "Undefined option: %s", tokname);
					handleError(ribErrWarning, msg);
				}
			}
		}
	}
}

// attributes

void Ribelato::Light(bool isAreaLight, const char* shaderName, const char* lightName)
{
	// Name the light and geometry set.
	char lightGeomName[256];
	if (isAreaLight)
		sprintf(lightGeomName, "+%sGeom", lightName);

	// Send the param list, plus "string geometry".
	mapTokens(true/*noClass*/);
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0]);
	if (isAreaLight)
		_api->Parameter("string geometry", lightGeomName + 1); // no '+'

	// Make the new light.
	_api->Light(lightName, shaderName);
	if (isAreaLight)
		_api->Attribute("string geometryset", lightGeomName);
}

// geometry


void Ribelato::Paraboloid(float rmax, float zmin, float zmax, float tmax)
{
	float *arccv = NULL, *uknotv = NULL;
	float vknotv[8] = {0.0f,0.0f,0.0f,0.0f,1.0f,1.0f,1.0f,1.0f};
	float *controlv = NULL;
	float r = 0.0f, z = 0.0f, K, B0, B3, Sx, Sy;
	int w, a, i;

	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");

	sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);

	tmax = tmax * M_PI / 180.0;

	if(zmin > 0.0)
		r = sqrt(zmin/zmax);
	K  = rmax * rmax;
	B0 = zmin  - ((2 * zmin) / K);
	B3 = zmax - (2 * zmax);
	K  = (2 * zmax) / K;
	Sx = (B3 - B0) / (K * (r - rmax));
	Sy = (K * r * Sx + B0);

	if(tmax < 0.0)
		circlearc(rmax, tmax, 0.0f, &w, NULL, &arccv);
	else
		circlearc(rmax, 0.0f, tmax, &w, NULL, &arccv);

	controlv = (float*) calloc(w*4*4, sizeof(float));

	a = w*3*4;
	memcpy(&(controlv[a]), arccv, w*4*sizeof(float));

	if(arccv)
		free(arccv);
	arccv = NULL;
	a += 2;
	for(i = 0; i < w; i++)
		{
			controlv[a] = zmax * controlv[a+1];
			a += 4;
		}

	if(zmin > 0.0)
		{
			if(tmax < 0.0)
				circlearc(r, tmax, 0.0f, &w, NULL, &arccv);
			else
				circlearc(r, 0.0f, tmax, &w, NULL, &arccv);

			memcpy(controlv, arccv, w*4*sizeof(float));

			if(arccv)
				free(arccv);
			arccv = NULL;
			a = 2;
			for(i = 0; i < w; i++)
				{
					controlv[a] = zmin * controlv[a+1];
					a += 4;
				}
		}
	else
		{
			a = 3;
			for(i = 0; i < w; i++)
				{
					controlv[a] = controlv[a+w*3*4];
					a += 4;
				}
		} // if

	r = (r + 2*Sx)/3.0;
	if(tmax < 0.0)
		circlearc(r, tmax, 0.0f, &w, NULL, &arccv);
	else
		circlearc(r, 0.0f, tmax, &w, NULL, &arccv);

	memcpy(&(controlv[w*4]), arccv, w*4*sizeof(float));
	if(arccv)
		free(arccv);
	arccv = NULL;

	z = ((zmin + 2 * Sy) / 3.0);
	a = w * 4 + 2;
	for(i = 0; i < w; i++)
		{
			controlv[a] = z * controlv[a+1];
			a += 4;
		}
	
	r = (rmax + 2*Sx)/3.0;
	if(tmax < 0.0)
		circlearc(r, tmax, 0.0f, &w, &uknotv, &arccv);
	else
		circlearc(r, 0.0f, tmax, &w, &uknotv, &arccv);

	memcpy(&(controlv[2*w*4]), arccv, w*4*sizeof(float));
	if(arccv)
		free(arccv);
	arccv = NULL;
	z = ((zmax + 2*Sy)/3.0);
	a = (2*w*4) + 2;
	for(i = 0; i < w; i++)
		{
			controlv[a] = z * controlv[a+1];
			a += 4;
		}

	_api->Parameter("vertex hpoint Pw", (const float*)controlv);

	_api->Patch(w, 3, uknotv, 0.0f/*uknotv[2]*/, 1.0f/*uknotv[w]*/,
				4, 4, vknotv, 0.0f, 1.0f);

	if(arccv)
		free(arccv);

} // Paraboloid


void Ribelato::Torus(float majrad, float minrad, float phimin, float phimax,
					 float tmax)
{
	float *arccv = NULL, *uknotv = NULL, *vknotv = NULL, *controlv = NULL;
	float *arccv2 = NULL, r;
	int a, b, i, j, w, h;

	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");

	sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);

	phimin = phimin * M_PI / 180.0;
	phimax = phimax * M_PI / 180.0;
	tmax = tmax * M_PI / 180.0;

	circlearc(minrad, phimin, phimax, &h, &vknotv, &arccv);
	a = 0;
	b = 0;
	for(i = 0; i < h; i++)
		{
			r = arccv[a] + majrad*arccv[a+3];

			if(tmax < 0.0f)
				circlearc(r, tmax, 0.0f, &w, uknotv?NULL:&uknotv, &arccv2);
			else
				circlearc(r, 0.0f, tmax, &w, uknotv?NULL:&uknotv, &arccv2);

			if(!controlv)
				controlv = (float*) calloc(h*w*4, sizeof(float));
			if(controlv)
				{
					memcpy(&(controlv[b]), arccv2, w*4*sizeof(float));

					for(j = 0; j < w; j++)
						{
							controlv[b+2] = arccv[a+1]*controlv[b+3];
							controlv[b+3] *= arccv[a+3];
							b += 4;
						}
				} // if

			if(arccv2)
				free(arccv2);
			arccv2 = NULL;

			a += 4;
		} // for

	_api->Parameter("vertex hpoint Pw", (const float*)controlv);

	_api->Patch(w, 3, uknotv, 0.0f/*uknotv[2]*/, 1.0f/*uknotv[w]*/,
				h, 3, vknotv, 0.0f/*vknotv[2]*/, 1.0f/*vknotv[h]*/);

	if(arccv)
		free(arccv);

} // Torus


void Ribelato::Polygon(int nverts)
{
	mapTokens();

	remapTokenTypes(_nParams, scVarying, RIGEL_VERTEX, scUniform, RIGEL_PERPIECE);

	// Extend and fill the iota vector.
	if (_verts.size() < nverts)
		_verts.resize(nverts);
	for(int i = 0; i < nverts; i++)
		_verts[i] = i;

	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, nverts/*n_st*/);
	_api->Mesh(RI_LINEAR, 1, &nverts, &_verts[0]);
}

void Ribelato::GeneralPolygon(int nloops, const int nverts[])
{
//	if (nloops == 1) {
		Polygon(nverts[0]);
//		return;
//	}

    // ToDo
}

void Ribelato::PointsPolygons(int npolys, const int nverts[], const int verts[])
{
	mapTokens();

	remapTokenTypes(_nParams, scVarying, RIGEL_VERTEX, scUniform, RIGEL_PERPIECE);

	// Find the number of vertices (largest indexed, plus one).
	int nv = 0, maxV = 0;
	for(int j = 0; j < npolys; j++)
		nv += nverts[j];
	for(int i = 0; i < nv; i++)
		if (verts[i] > maxV)
			maxV = verts[i];

	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, /*nv*/maxV+1/*n_st*/);
	_api->Mesh(RI_LINEAR, npolys, nverts, verts);
}

void Ribelato::PointsGeneralPolygons(int npolys, const int nloops[], const int nverts[],
	const int verts[])
{
	// Are they really general polygons?
	int maxNLoops = 1;
	for(int i = 0; i < npolys && (maxNLoops = nloops[i]) == 1; i++) ;
	if (maxNLoops == 1 /* && polygons are convex */) {
		PointsPolygons(npolys, nverts, verts);
		return;
	}

	// ToDo: handle these for real

	// For now, render the polygons w/o the holes.
	vector<int> _nverts; // ToDo: make this an instance var
	_verts.empty();
	for(int i = 0, loopIndex = 0, vertIndex = 0; i < npolys; i++)
		for(int loop = 0, nl = nloops[i]; loop < nl; loop++, loopIndex++) {
			_nverts.push_back(nverts[loopIndex]);
			for(int v = 0, nv = nverts[loopIndex]; v < nv; v++, vertIndex++)
				_verts.push_back(verts[vertIndex]);
		}
	PointsPolygons(_nverts.size()/*npolys*/, &_nverts[0], &_verts[0]);
}

template <class T>
inline void copyVaryingParm(const T *parmp, int nf, const int nverts[], const int verts[],
	int psize, vector<T> &out)
{
    const int *vp = verts;
    for(int f = 0; f < nf; f++) {
        for(int nv = nverts[f]; --nv >= 0; ) {
            int v = *vp++;
            const T *p = parmp + psize * v;
            for(int np = psize; --np >= 0; )
                out.push_back(*p++);
        }
    }
}

void Ribelato::SubdivisionMesh(const char* mask, int nf, const int nverts[], const int verts[],
	int ntags, const char* tags[], const int nargs[], const int intargs[], const float floatargs[])
{
	mapTokens();
	copyParms();

	// Gather all the parameter type data.
	vector<pdata> paramData;
	paramData.resize(_nParams);
	for(int p = 0; p < _nParams; p++) {
		pdata &pd = paramData[p];
		getClassAndType(_names2[p], pd);
		pd.siz *= (pd.dt==dtColor || pd.dt==dtPoint || pd.dt==dtVector || pd.dt==dtNormal) ? 3 :
			pd.dt==dtHPoint ? 4 : pd.dt==dtMatrix ? 16 : 1;
	}

	remapTokenTypes(_nParams, scVarying, RI_LINEAR, scUniform, RIGEL_PERPIECE);

	// Find the total number of vertices and max vertex index.
	int nv = 0, maxV = 0;
	for(int j = 0; j < nf; j++)
		nv += nverts[j];
	for(int i = 0; i < nv; i++)
		if (verts[i] > maxV)
			maxV = verts[i];

	// Find the st coordinate count.
	int n_st = nv;
	for(int p = 0; p < _nParams; p++) {
		const char *tok = _names[p];
		if (const char *sp = strrchr(tok, ' '))
			tok = sp+1;
		if (strcmp(tok, RI_ST) == 0) {
		    pdata &pd = paramData[p];
			if (pd.sc == scVertex)
				n_st = maxV+1;
			break;
		}
	}

    // Promote "varying" to "linear" by copying coords.
	_offsets.clear();
	_offsets.resize(_nParams, 0);
	_floats.clear();						// Clear the param temp storage.
	_floats.push_back(0);					// So offsets are > 0.
	_ints.clear();
	_ints.push_back(0);
	_strings.clear();
	_strings.push_back(NULL);
	for(int p = 0; p < _nParams; p++) {
		pdata &pd = paramData[p];
		if (pd.sc == scVarying) {
			if (pd.dt == dtInt) {
				_offsets[p] = _ints.size();
				copyVaryingParm((const int *)_parms[p], nf, nverts, verts, pd.siz, _ints);
			} else if (pd.dt == dtString) {
				_offsets[p] = _strings.size();
				copyVaryingParm((const char **)_parms[p], nf, nverts, verts, pd.siz, _strings);
			} else {
				_offsets[p] = _floats.size();
				copyVaryingParm((const float *)_parms[p], nf, nverts, verts, pd.siz, _floats);
			}
        }
    }

	// Convert the tags.
	const int *iargp = intargs;
	const float *fargp = floatargs;
	for(int t = 0; t < ntags; t++) {
		const char *tag = tags[t];
		int niargs = nargs[t*2], nfargs = nargs[t*2 + 1];
		if (strcmp(tag, "interpolateboundary") == 0) {
			int border = 1;
			if (niargs > 0)
				border = iargp[0];
			_api->Parameter(RIGEL_BORDER, border);
		} else if (strcmp(tag, "hole") == 0) {
			char token[64];
			sprintf(token, "constant int[%d] holes", niargs);
			_api->Parameter(token, iargp);
		}
		iargp += niargs;
		fargp += nfargs;
	}

	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms2[0], use_offsets, n_st);
	_api->Mesh(mask, nf, nverts, verts);
}

void Ribelato::Patch(const char* type)
{
	// Convert the patch bases.
	char basis[64];
	if (strcmp(type, RI_BILINEAR) == 0)
		strcpy(basis, RI_LINEAR);
	else if (_basis.ubasis == _basis.vbasis)
		strcpy(basis, _basis.ubasis);
	else
		sprintf(basis, "%s,%s", _basis.ubasis, _basis.vbasis);

	// Map the tokens.
	mapTokens();
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "linear");

	// Send the parms, breaking "st" into "s" and "t".
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4/*n_st*/, default_st);

	// Transfer the patch data.
	int nuv = strcmp(type, RI_BILINEAR) == 0 ? 2 : 4;
	_api->Patch(basis, nuv, nuv);
}

void Ribelato::PatchMesh(const char* type, int nu, const char* uwrap, int nv, const char* vwrap)
{
	// Convert the patch bases.
	char basis[64];
	if (strcmp(type, RI_BILINEAR) == 0)
		strcpy(basis, RI_LINEAR);
	else if (_basis.ubasis == _basis.vbasis)
		strcpy(basis, _basis.ubasis);
	else
		sprintf(basis, "%s,%s", _basis.ubasis, _basis.vbasis);

	// Convert the tokens and copy the parm pointers.
	mapTokens();
	copyParms();
	stripUniformAndVarying();

	// offsets into _floats for each parm (calls to _floats.push_back() can realloc memory)
	_offsets.clear();
	_offsets.resize(_nParams, 0);
	_floats.clear();
	_floats.push_back(0); // so offsets are > 0

	// Unwrap the patchmesh if necessary.
	bool bilinear = strcmp(type, RI_BILINEAR) == 0,
		uPeriodic = strcmp(uwrap, RI_PERIODIC) == 0,
		vPeriodic = strcmp(vwrap, RI_PERIODIC) == 0;
	int newnu = nu, newnv = nv;
	if (uPeriodic || vPeriodic) {
		// Figure out how many patches and patch corners there are, and will be.
	//	int npu = bilinear ? (uPeriodic ? nu : nu-1) : (uPeriodic ? nu/ustep : (nu-4)/ustep+1), // # of patches
	//		npv = bilinear ? (vPeriodic ? nv : nv-1) : (vPeriodic ? nv/vstep : (nv-4)/vstep+1);
	//	int ncu = npu + (uPeriodic ? 0 : 1), // # of patch corners (no "varying" in Gelato)
	//		ncv = npv + (vPeriodic ? 0 : 1),
	//		newncu = npu + 1, // since newnpu = npu and newuPeriodic is definitely false (the whole point)
	//		newncv = npv + 1;

		// Figure out how many new vertices there'll be.
		if (uPeriodic)
			newnu += bilinear ? 1 : (4 - _basis.ustep);
		if (vPeriodic)
			newnv += bilinear ? 1 : (4 - _basis.vstep);

		for(int p = 0; p < _nParams; p++) {
			// Get the storage class and type of the parameter.
			pdata pd;
			getClassAndType(_names2[p], pd);

			// Classes constant and uniform are unchanged. Leave _parms[p] pointing to input data.
			if (pd.sc == scConstant /*|| sc == scUniform   no "uniform" in Gelato*/)
				continue;

			// Figure out how many items there are from the storage class and type.
			int ncomp = (pd.dt==dtColor || pd.dt==dtPoint || pd.dt==dtVector || pd.dt==dtNormal) ?
				3 : pd.dt==dtHPoint ? 4 : pd.dt==dtMatrix ? 16 : pd.siz;
			int niu = /*sc==scVarying ? ncu :*/ nu,
				niv = /*sc==scVarying ? ncv :*/ nv,
				newniu = /*sc==scVarying ? newncu :*/ newnu,
				newniv = /*sc==scVarying ? newncv :*/ newnv;

			// Copy the items. (Note this only runs for "vertex" params.)
			const float *fpv0 = (const float *)_parms2[p], *fp = fpv0;
			_offsets[p] = _floats.size();
			for(int iv = 0; iv < newniv; iv++) {		// Copy all *output* (v-major) lines ...
				if (iv == niv) fp = fpv0;				// ... rewinding to v0 for the last few.
				const float *fpu0 = fp;					// Save for replicating the first few items.
				int n;
				for(n = niu*ncomp; --n >= 0; )			// Copy the input data.
					_floats.push_back(*fp++);
				for(n = (newniu-niu)*ncomp; --n >= 0; )	// Duplicate the first few vertices or
					_floats.push_back(*fpu0++);			// corners.
			}
		}
	}

	// Send the parms, breaking "st" into "s" and "t" (assuming vertex here; varying elided above).
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms2[0], use_offsets, newnu*newnv/*n_st*/, default_st);

	// Transfer the patch data.
	_api->Patch(basis, newnu, newnv);
}

void Ribelato::NuPatch(int nu, int uorder, const float uknot[], float umin, float umax,
	int nv, int vorder, const float vknot[], float vmin, float vmax)
{
	// Convert the tokens and copy the parm pointers.
	mapTokens();
	copyParms();
	stripUniformAndVarying();

	// Send the parms, breaking "st" into "s" and "t".
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms2[0], no_offsets, nu*nv/*n_st*/, default_st);

	_api->Patch(nu, uorder, uknot, umin, umax, nv, vorder, vknot, vmin, vmax);
}

// Strip out uniform and varying params, which are not supported by Gelato.

void Ribelato::stripUniformAndVarying()
{
	for(int i = 0; i < _nParams; i++) {
		// Get the storage class and type of the parameter.
		const char* token = _names2[i];
		pdata pd;
		getClassAndType(token, pd);
		if (pd.sc == scUniform || pd.sc == scVarying) {
			char errmsg[256];
			sprintf(errmsg, "\"%s\": parameter type %s not supported; ignoring.", token,
				pd.sc == scUniform ? "uniform" : "varying");
			handleError(ribErrWarning, errmsg); // ToDo: correct line numbers when parsing
			_names2.erase(_names2.begin() + i);
			_parms2.erase(_parms2.begin() + i);
			--_nParams;
			--i; // balance increment; hold at current index (ToDo: iterator?)
		}
	}
}

void Ribelato::TrimCurve(int nloops, const int ncurves[], const int order[], const float knot[],
	const float min[], const float max[], const int n[], const float u[], const float v[],
	const float w[])
{
	// Interleave u[], v[], and w[].
	int totalCurves = 0;
	for(int i = 0; i < nloops; i++)
		totalCurves += ncurves[i];
	int totalN = 0;
	for(int j = 0; j < totalCurves; j++)
		totalN += n[j];
	_floats.clear();
	_floats.reserve(totalN*3);
	for(int k = 0; k < totalN; k++) {
		_floats.push_back(u[k]);
		_floats.push_back(v[k]);
		_floats.push_back(w[k]);
	}

	_api->TrimCurve(nloops, ncurves, n, order, knot, min, max, &_floats[0]);
}

void Ribelato::Sphere(float radius, float zmin, float zmax, float tmax)
{
	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);
	_api->Sphere(radius, zmin, zmax, tmax);
}


void Ribelato::Cylinder(float radius, float zmin, float zmax, float tmax)
{
	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");

	sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);
	sweep_line (radius, radius, zmin, zmax, tmax*M_PI/180.0);
}


void Ribelato::Cone(float z, float radius, float tmax)
{
	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");

	sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);
	sweep_line (radius, 0.0f, 0.0f, z, tmax*M_PI/180.0);
}


void Ribelato::Disk(float z, float radius, float tmax)
{
	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");

	sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);
	sweep_line (radius, 0.0f, z, z, tmax*M_PI/180.0);
}


void Ribelato::Hyperboloid(float x0, float y0, float z0,
						   float x1, float y1, float z1, float tmax)
{
	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");

	sendParms(_nParams, &_names2[0], &_parms[0], no_offsets, 4, default_st);
    float r0 = sqrtf(x0*x0 + y0*y0);
    float r1 = sqrtf(x1*x1 + y1*y1);
	sweep_line (r0, r1, z0, z1, tmax*M_PI/180.0, atan2f(y0,x0), atan2f(y1,x1));
}



void
Ribelato::sweep_line (float radius0, float radius1, float zmin, float zmax,
					  float thetamax, float rotate0, float rotate1) const
{
    const float sqrtOneHalf = sqrtf(0.5f);
    static float full_uknot[12] = {0, 0, 0, 0.25, 0.25, 0.5, 0.5, 0.75, 0.75, 1, 1, 1};
	const float *uknot = full_uknot;
    static float vknot[4] = {0, 0, 1, 1};

    int nu = 9, nv = 2;
    // Control points for a NURBS circle in the XY plane.
    static float circle[9][4] = {
		{1, 0, 0, 1 },
		{sqrtOneHalf, sqrtOneHalf, 0, sqrtOneHalf },
		{0, 1, 0, 1 },
		{-sqrtOneHalf, sqrtOneHalf, 0, sqrtOneHalf },
		{-1, 0, 0, 1 },
		{-sqrtOneHalf, -sqrtOneHalf, 0, sqrtOneHalf },
		{0, -1, 0, 1 },
		{sqrtOneHalf, -sqrtOneHalf, 0, sqrtOneHalf },
		{1, 0, 0, 1 } };
    static float reverse_circle[9][4] = {
		{1, 0, 0, 1 },
		{sqrtOneHalf, -sqrtOneHalf, 0, sqrtOneHalf },
		{0, -1, 0, 1 },
		{-sqrtOneHalf, -sqrtOneHalf, 0, sqrtOneHalf },
		{-1, 0, 0, 1 },
		{-sqrtOneHalf, sqrtOneHalf, 0, sqrtOneHalf },
		{0, 1, 0, 1 },
		{sqrtOneHalf, sqrtOneHalf, 0, sqrtOneHalf },
		{1, 0, 0, 1 } };
    float cv[18][4];

    // Copy the circle control points to the start and end of the 
    // sweep in v.
    if (thetamax > 0.0f) {
		memcpy(cv, circle, 36*sizeof(float));
		memcpy(&cv[9][0], circle, 36*sizeof(float));
    } else {
		memcpy(cv, reverse_circle, 36*sizeof(float));
		memcpy(&cv[9][0], reverse_circle, 36*sizeof(float));
    }

    if (rotate0 != 0.0f || rotate1 != 0.0f) {
		// For hyperboloids, we can rotate the cv's about the z axis
		float costheta = cosf(rotate0), sintheta = sinf(rotate0);
		for (int i = 0; i < nu; ++i) {
			float newx = cv[i][0]*costheta - cv[i][1]*sintheta;
			float newy = cv[i][0]*sintheta + cv[i][1]*costheta;
			cv[i][0] = newx;
			cv[i][1] = newy;
		}
		costheta = cosf(rotate1), sintheta = sinf(rotate1);
		for (int i = 0; i < nu; ++i) {
			float newx = cv[nu+i][0]*costheta - cv[nu+i][1]*sintheta;
			float newy = cv[nu+i][0]*sintheta + cv[nu+i][1]*costheta;
			cv[nu+i][0] = newx;
			cv[nu+i][1] = newy;
		}
    }

    // Now fix them up: scale the radii in xy at the start and the end
    // and set up the z values.
    for (int i = 0; i < nu; ++i) {
		cv[i][0] *= radius0;
		cv[i][1] *= radius0;
		cv[i][2] = zmin * circle[i][3];
		cv[9+i][0] *= radius1;
		cv[9+i][1] *= radius1;
		cv[9+i][2] = zmax * circle[i][3];
    }

    float umax = fabsf(thetamax) / (M_PI*2.0);
	if (umax < 0.0f) umax = 0.0f;
	if (umax > 1.0f) umax = 1.0f;
	float newuknot[12];
	if (umax < 0.99999f) { // partial sweep -- adjust u knots
		for (int i = 0;  i < 12;  ++i)
			newuknot[i] = uknot[i] / umax;
		uknot = newuknot;
	}
	_api->Parameter ("vertex hpoint Pw", &(cv[0][0]));
	_api->Patch (nu, 3, uknot, 0.0f, 1.0f, nv, 2, vknot, 0.0f, 1.0f);
}





// Calculate number of curve segments from vertex count nv, linear/periodic, and basis step.

inline int nsegs(int nv, bool linear, bool periodic, int vstep) {
	return linear ? (periodic ? nv : nv - 1) : (periodic ? (nv/vstep) : ((nv-4)/vstep + 1));
}

template <class T>
inline void Ribelato::copyCurveParm(const T *parmp, int ncurves, const int nvertices[],
	int nv, pdata &pd, bool linear, bool periodic, int vstep, vector<T> &out)
{
	// extra verts copied for periodic curves
	int extranv = linear ? 1 : (4 - vstep);

	for(int i = 0; i < ncurves; i++) {	// Find all the curves with this nv.
		if (pd.sc == scUniform) {
			if (nvertices[i] == nv) {
				for(int nf = pd.siz; --nf >= 0; )	// There's only one of whatever it is.
					out.push_back(*parmp++);
			} else
				parmp += pd.siz;
		} else if (pd.sc == scVarying) {
			int nVarying = nsegs(nvertices[i], linear, periodic, vstep)
				+ (periodic ? 0 : 1);
			if (nvertices[i] == nv) {
				for(int nf1 = pd.siz; --nf1 >= 0; )	// Change varying to linear using first ...
					out.push_back(*parmp++);
				parmp += (nVarying-2)*pd.siz;		// ... and last segment values.
				for(int nf2 = pd.siz; --nf2 >= 0; )
					out.push_back(*parmp++);
			} else
				parmp += pd.siz * nVarying;
		} else if (pd.sc == scVertex) {
			if (nvertices[i] == nv) {
				const T *savp = parmp;
				for(int nf = nv*pd.siz; --nf >= 0; )// Copy the base curve.
					out.push_back(*parmp++);
				if (periodic)						// Unwrap periodic curves.
					for(int nf = extranv*pd.siz; --nf >= 0; )
						out.push_back(*savp++);
			} else
				parmp += pd.siz * nvertices[i];
		}
	}
}

void Ribelato::Curves(const char* type, int ncurves, const int nvertices[], const char* wrap)
{
	mapTokens();
	copyParms();

	// Gather all the parameter type data.
	vector<pdata> paramData;
	paramData.resize(_nParams);
	for(int p = 0; p < _nParams; p++) {
		pdata &pd = paramData[p];
		getClassAndType(_names2[p], pd);
		pd.siz *= (pd.dt==dtColor || pd.dt==dtPoint || pd.dt==dtVector || pd.dt==dtNormal) ? 3 :
			pd.dt==dtHPoint ? 4 : pd.dt==dtMatrix ? 16 : 1;
	}

	// Map "uniform" to "perpiece" and "varying" to "linear".
	remapTokenTypes(_nParams, scUniform, RIGEL_PERPIECE, scVarying, RI_LINEAR);

	// Convert the curve basis.
	const char *newType = strcmp(type, RI_LINEAR) == 0 ? RI_LINEAR : _basis.vbasis;

	bool linear = newType == RI_LINEAR;
	bool periodic = strcmp(wrap, RI_PERIODIC) == 0;

	// Make a Curves() call for each unique value in nvertices[].  ToDo: optimize special case
	// (nonperiodic, no varying, nvertices[] all equal, etc) to call Curves() directly.
	vector<char> nvs;
	int nvMax = 0;
	for(int i = 0; i < ncurves; i++)		// Find max number of vertices.
		if (nvertices[i] > nvMax)
			nvMax = nvertices[i];
	nvs.resize(nvMax+1, 0);					// Size (and zero) the hit vector accordingly.
	for(int j = 0; j < ncurves; j++)		// Mark a hit for each nv.
		nvs[nvertices[j]] = 1;
	vector<unsigned> offsets;
	for(int nv = 0; nv < nvs.size(); nv++)	// for each unique # of vertices ...
		if (nvs[nv]) {
			_floats.clear();						// Clear the param temp storage.
			_floats.push_back(0);					// So offsets are > 0.
			_ints.clear();
			_ints.push_back(0);
			_strings.clear();
			_strings.push_back(NULL);
			offsets.clear();						// Zero the offsets.
			offsets.resize(_nParams, 0);

			for(int p = 0; p < _nParams; p++) {			// Format the data for each param.
				pdata &pd = paramData[p];
				if (pd.sc == scConstant)			// No need to copy anything for constants.
					continue;
				else if (pd.dt == dtInt) {			// Copy as ints.
					offsets[p] = _ints.size();
					copyCurveParm<int>((const int *)_parms[p], ncurves, nvertices, nv,
						pd, linear, periodic, _basis.vstep, _ints);
				} else if (pd.dt == dtString) {		// Copy as char*s (could be 64 bits w/some OSs).
					offsets[p] = _strings.size();
					copyCurveParm<const char *>((const char **)_parms[p], ncurves, nvertices, nv,
						pd, linear, periodic, _basis.vstep, _strings);
				} else {							// Everything else can be copied as floats.
					offsets[p] = _floats.size();
					copyCurveParm<float>((const float *)_parms[p], ncurves, nvertices, nv,
						pd, linear, periodic, _basis.vstep, _floats);
				}
			}

			// Do the offset-to-pointer thing.
			for(int p2 = 0; p2 < _nParams; p2++)
				if (unsigned off = offsets[p2]) {
					pdata &pd = paramData[p2];
					if (pd.dt == dtInt)
						_parms2[p2] = &_ints[off];
					else if (pd.dt == dtString)
						_parms2[p2] = &_strings[off];
					else
						_parms2[p2] = &_floats[off];
				}

			// Count the curves with this nv.
			int nc = 0;
			for(int i = 0; i < ncurves; i++)
				if (nvertices[i] == nv)
					++nc;

			// Send the curves.
			if (_nParams)
				sendParms(_nParams, &_names2[0], &_parms2[0]);
			_api->Curves(newType, nc, nv);
		}
}

void Ribelato::Points(int nverts)
{
	mapTokens();

	// Map "uniform" to "constant" and "varying" to "vertex".
	remapTokenTypes(_nParams, scUniform, "constant", scVarying, "vertex");
	if (_nParams)
		sendParms(_nParams, &_names2[0], &_parms[0]);
	_api->Points(nverts);
}

void Ribelato::Procedural(const void* data, const char* tokens[], const float* bound)
{
	if (strcmp((const char *)data, "DelayedReadArchive") == 0) {
		_api->Input(tokens[0], bound);
	} else if (strcmp((const char *)data, "RunProgram") == 0) {
		char cmd[256];
		sprintf(cmd, "rib < %s", tokens[0]);

		// Write the data block to a temp file.
		if (tokens[1] && tokens[1][0]) {
			// Open a temp file for the data block.
			char tempFile[256];
			strcpy(tempFile, "/usr/tmp/RunProgram_XXXXXX");
			#ifdef _WIN32
			mktemp(tempFile);
			int fd = open (tempFile,
                                       _O_CREAT | _O_WRONLY | _O_BINARY);
			#else
			int fd = mkstemp(tempFile);
			#endif

			// Add the temp file name to the command. Even if this fails, mkstemp() should
			// have scribbled over the file name string.
			strcat(cmd, " ");
			strcat(cmd, tempFile);

			// Write the data block to the file.
			if (fd >= 0) {
				write(fd, tokens[1], strlen(tokens[1]));
				close(fd);
			} else {
				char err[256];
				sprintf(err, "Can't open temp file %s for '%s'.", tempFile, cmd);
				handleError(ribErrError, err);
				return;
			}
		}

		// Pass the command to the renderer.  Can't do the popen thing here, as the file may
		// be in another language.
		_api->Input(cmd, bound);
	}
}

// other

void Ribelato::Declare(const char *name, const char *type)
{
	// Concatenate these into the table so type() returns the whole declaration.
	decl d;
	d.typeLoc = _declStrings.size();
	while(*type != '\0') _declStrings.push_back(*type++);
	_declStrings.push_back(' ');
	d.nameLoc = _declStrings.size();
	do _declStrings.push_back(*name); while(*name++ != '\0');
	_decls.push_back(d);
}

void Ribelato::ErrorHandler(const char* handlerName)
{
	if (strcmp(handlerName, "ignore") == 0)
		_errorHandler = ribErrIgnore;
	else if (strcmp(handlerName, "print") == 0)
		_errorHandler = ribErrPrint;
	else if (strcmp(handlerName, "abort") == 0)
		_errorHandler = ribErrAbort;
}

// utils

// prman -> gelato parameter list translation table
static char *pdict[][2] = {
	"amplitude", "float amplitude",
	"background", "color background",
	"beamdistribution", "float beamdistribution",
	"coneangle", "float coneangle",
	"conedeltaangle", "float conedeltaangle",
	"constantwidth", "constant float width",
	"Cs", "varying color C",
	"distance", "float distance",
	"filterwidth", "float[2] filterwidth",
	"from", "point from",
	"intensity", "float intensity",
	"Ka", "float Ka",
	"Kd", "float Kd",
	"Kr", "float Kr",
	"Ks", "float Ks",
	"lightcolor", "color lightcolor",
	"maxdistance", "float maxdistance",
	"mindistance", "float mindistance",
	"Os", "varying color opacity",
	"N", "varying normal N",
	"Np", "uniform normal Np",
	"P", "vertex point P",
	"Pw", "vertex hpoint Pw",
	"Pz", "vertex float Pz",
	"roughness", "float roughness",
	"s", "varying float s",
	"specularcolor", "color specularcolor",
	"t", "varying float t",
	"st", "varying float[2] st",
	"texturename", "string texturename",
	"to", "point to",
	"width", "varying float width",
};

const char* Ribelato::getParamDecl(const char *name)
{
	const char *namep = name;
	if (const char *sp = strrchr(name, ' '))
		namep = sp+1;

	// Check the declarations (most recent first).
	for(int nd = _decls.size(); --nd >= 0; ) {
		decl &d = _decls[nd];
		if (strcmp(d.name(_declStrings), namep) == 0)
			return d.type(_declStrings);
	}

	// Check the hardcoded param table.
	for(int i = 0; i < sizeof(pdict)/sizeof(pdict[0]); i++)
		if (strcmp(namep, pdict[i][0]) == 0)
			return pdict[i][1];

	// Check the options table.
	for(int j = 0, nj = sizeof(odict)/sizeof(odict[0]); j < nj; j++)
		if (strcmp(namep, odict[j][1]) == 0)
			return odict[j][2];

	return NULL;
}

// Remap parameter list tokens.

void Ribelato::mapTokens(bool noClass)
{
	// Clear the token strings for later use in type remapping.
	_tokenStrings.clear();
	_tokenStrings.push_back('\0'); // so offsets are > 0

	_names2.clear();
	for(int i = 0; i < _nParams; i++) {
		if (const char *sp = strrchr(_names[i], ' ')) { // inline decl? (e.g., "float Ka")
			++sp;

			// Special cases: map Cs to C and Os to opacity, preserving storage class.
			if (strcmp(sp, "Cs") == 0) {
				pdata pd;
				getClassAndType(_names[i], pd);
				if (pd.dt == dtColor) {
					switch(pd.sc) {
						case scConstant:	_names2.push_back("constant color C"); break;
						case scUniform:		_names2.push_back("uniform color C"); break;
						case scVarying:		_names2.push_back("varying color C"); break;
						case scFaceVarying:	_names2.push_back("facevarying color C"); break;
						case scVertex:		_names2.push_back("vertex color C"); break;
					}
				} else
					_names2.push_back(_names[i]);
			} else if (strcmp(sp, "Os") == 0) {
				pdata pd;
				getClassAndType(_names[i], pd);
				if (pd.dt == dtColor) {
					switch(pd.sc) {
						case scConstant:	_names2.push_back("constant color opacity"); break;
						case scUniform:		_names2.push_back("uniform color opacity"); break;
						case scVarying:		_names2.push_back("varying color opacity"); break;
						case scFaceVarying:	_names2.push_back("facevarying color opacity"); break;
						case scVertex:		_names2.push_back("vertex color opacity"); break;
					}
				} else
					_names2.push_back(_names[i]);
			} else if (strcmp(sp, "constantwidth") == 0) {
				_names2.push_back(getParamDecl(sp));
			} else
				_names2.push_back(_names[i]);
		} else if (const char *remap = getParamDecl(_names[i]))
			_names2.push_back(remap);
		else
			_names2.push_back(_names[i]); // hand to api and hope for the best

		// Strip off the storage class.
		if (noClass) {
			const char *p = _names2[i];
			if (strncmp(p, "constant", 8) == 0)
				p += 8;
			else if (strncmp(p, "uniform", 7) == 0)
				p += 7;
			else if (strncmp(p, "varying", 7) == 0)
				p += 7;
			else if (strncmp(p, "facevarying", 11) == 0)
				p += 11;
			else if (strncmp(p, "vertex", 6) == 0)
				p += 6;
			while(*p == ' ' || *p == '\t') ++p;
			_names2[i] = p;
		}
	}
}

void Ribelato::copyParms()
{
	_parms2.clear();
	for(int j = 0; j < _nParams; j++)
		_parms2.push_back(_parms[j]);
}

// Send parms, optionally breaking "st" into "s" and "t", and adding default s,t coords.

void Ribelato::sendParms(int n, const char* tokens[], const void* parms[], bool use_offsets,
	int n_st, bool default_st)
{
	// Run through the parms once, splitting "st" into "s" and "t", and checking for any st coords.
	int stParam = -1;
	unsigned sOffset = 0, tOffset = 0;
	bool gotST = false;
	if (n_st) {
		for(int p = 0; p < n && !gotST; p++) {
			const char *tok = tokens[p];
			if (const char *sp = strrchr(tok, ' '))
				tok = sp+1;
			if (strcmp(tok, RI_ST) == 0) {
				if (!use_offsets) { // no_offsets = no floats; init.
					_floats.clear();
					_floats.push_back(0); // so offsets are > 0
				}
				// Make room for duplicated, separated s and t coords.
				_floats.reserve(_floats.size() + n_st*2);
				if (use_offsets)
					if (unsigned off = _offsets[p])
						parms[p] = &_floats[off];
				sOffset = _floats.size();
				const float *st = (const float *)parms[p];	// point to s
				for(int ns = n_st; --ns >= 0; st += 2)
					_floats.push_back(*st);
				tOffset = _floats.size();
				st = (const float *)parms[p] + 1;			// point to t
				for(int nt = n_st; --nt >= 0; st += 2)
					_floats.push_back(*st);
				stParam = p;
				gotST = true;
			} else if (strcmp(tok, RI_S) == 0 || strcmp(tok, RI_T) == 0)
				gotST = true;
		}
	}

	// Now send the parms to the api.
	for(int p = 0; p < n; p++) {
		if (p == stParam) {
			pdata pd;
			getClassAndType(tokens[p], pd);
			ParamType pt(PT_FLOAT, pd.sc == scVertex ? INTERP_VERTEX : INTERP_LINEAR);
			_api->Parameter("s", pt, &_floats[sOffset]);
			_api->Parameter("t", pt, &_floats[tOffset]);
		} else {
			if (use_offsets)
				if (unsigned off = _offsets[p])
					parms[p] = &_floats[off];
			_api->Parameter(tokens[p], parms[p]);
		}
	}

	// Add the TextureCoordinates if there's no "st", "s" or "t" given.
	if (default_st && !gotST && _texCoords != _defaultTexCoords) {
		_api->Parameter(RIGEL_LINEAR_S, _texCoords.s);
		_api->Parameter(RIGEL_LINEAR_T, _texCoords.t);
	}
}

void Ribelato::getClassAndType(const char *decl, pdata &pd)
{
	const char *p = decl;

	// Get the storage class.
	if (strncmp(p, "constant", 8) == 0)
		pd.sc = scConstant, p += 8;
	else if (strncmp(p, "uniform", 7) == 0)
		pd.sc = scUniform, p += 7;
	else if (strncmp(p, "varying", 7) == 0)
		pd.sc = scVarying, p += 7;
	else if (strncmp(p, "facevarying", 11) == 0)
		pd.sc = scFaceVarying, p += 11;
	else if (strncmp(p, "vertex", 6) == 0)
		pd.sc = scVertex, p += 6;
	else
		pd.sc = scConstant;

	// Skip any white space.
	while(*p == ' ' || *p == '\t') ++p;

	// Get the data type.
	if (strncmp(p, "int", 3) == 0)
		pd.dt = dtInt, p += 3;
	else if (strncmp(p, "float", 5) == 0)
		pd.dt = dtFloat, p += 5;
	else if (strncmp(p, "color", 5) == 0)
		pd.dt = dtColor, p += 5;
	else if (strncmp(p, "point", 5) == 0)
		pd.dt = dtPoint, p += 5;
	else if (strncmp(p, "hpoint", 6) == 0)
		pd.dt = dtHPoint, p += 6;
	else if (strncmp(p, "vector", 6) == 0)
		pd.dt = dtVector, p += 6;
	else if (strncmp(p, "normal", 6) == 0)
		pd.dt = dtNormal, p += 6;
	else if (strncmp(p, "matrix", 6) == 0)
		pd.dt = dtMatrix, p += 6;
	else if (strncmp(p, "string", 6) == 0)
		pd.dt = dtString, p += 6;
	else
		pd.dt = dtFloat;

	// Skip any white space.
	while(*p == ' ' || *p == '\t') ++p;

	// Get the size.
	pd.siz = 1;
	if (*p == '[') {
		++p;
		while(*p == ' ' || *p == '\t') ++p;
		pd.siz = atoi(p);
	}
}

// Remap token types. N.B.: This puts strings in _tokenStrings, temporarily using offsets while
// the string is filled.  If you want to add more _tokenStrings after a call to this, mind the
// pointers. scFaceVarying always remaps to linear, for PointsPolygons-to-Mesh() calls.

void Ribelato::remapTokenTypes(int n, storageClass sc1, const char *newsc1, storageClass sc2,
	const char *newsc2)
{
	// Keep offsets to chars in _tokenStrings.
	vector<unsigned> offsets;
	offsets.resize(n, 0);

	for(int i = 0; i < n; i++) {
		// Get the storage class and type of the parameter.
		const char* token = _names2[i];
		pdata pd;
		getClassAndType(token, pd);

		if (pd.sc == sc1 || pd.sc == sc2 || pd.sc == scFaceVarying) {
			const char *newsc = pd.sc==sc1 ? newsc1 : pd.sc==sc2 ? newsc2 : "linear";
 
			// Make up a string with the new storage class.
			if (const char *sp = strchr(token, ' ')) {
				offsets[i] = _tokenStrings.size();
				append(newsc, _tokenStrings);
				_tokenStrings.pop_back(); // pop the null
				append(sp, _tokenStrings);
			}
		}
	}

	// Convert offsets to _tokenStrings ptrs.
	for(int j = 0; j < n; j++)
		if (unsigned off = offsets[j])
			_names2[j] = &_tokenStrings[off];
}
