// -*- indent-tabs-mode: 1; tab-width: 4; c-basic-offset: 4 -*-
//
// RIB parser declarations
//
/////////////////////////////////////////////////////////////////////////////
// Copyright 2004 NVIDIA Corporation.  All Rights Reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
// * Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// * Neither the name of NVIDIA nor the names of its contributors
//   may be used to endorse or promote products derived from this software
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// (This is the Modified BSD License)
/////////////////////////////////////////////////////////////////////////////
//
// $Revision: $    $Date:  $
//
// "RIB" is a trademark of Pixar, Inc.
//

#ifndef _RIBBIT_
#define _RIBBIT_

#include <vector>

using namespace std;

// Parser class

class RIBParser {
  public:
	// Create and destroy.
	RIBParser(int (*getc)(void *cbData) = NULL, void *cbData = NULL);
	virtual ~RIBParser();

	// errors
	enum ribErr {
		reNoErr = 0,		// no error
		reArrayTooBig,		// insufficient memory to construct array
		reBadArgument,		// incorrect parameter value
		reBadArray,			// invalid array specification
		reBadBasis,			// undefined basis matrix name
		reBadColor,			// invalid color specification
		reBadHandle,		// invalid light or object handle
		reBadParamList,		// parameter list type mismatch
		reBadRipCode,		// invalid encoded RIB request code
		reBadStringToken,	// undefined encoded string token
		reBadToken,			// invalid binary token
		reBadVersion,		// protocol version number mismatch
		reLimitCheck,		// overflowing an internal limit
		reOutOfMemory,		// generic instance of insufficient memory
		reProtocolBotch,	// malformed binary encoding
		reStringTooBig,		// insufficient memory to read string
		reSyntaxError,		// general syntactic error
		reUnregistered		// undefined RIB request
	};

	// Parse and execute one command. Returns an error code, and sets eof at end of file.
	ribErr doCmd(bool &eof);

	// Return the line number of the input file, for those handling error messages outside.
	int line() { return _line; }

  protected:
	// RIB/Ri codes
	enum ribCode {
		riAreaLightSource, riAtmosphere, riAttribute, riAttributeBegin, riAttributeEnd, riBasis,
		riBlobby, riBound, riClipping, riClippingPlane, riColor, riColorSamples, riConcatTransform,
		riCone, riCoordSysTransform, riCoordinateSystem, riCropWindow, riCurves, riCylinder,
		riDeclare, riDepthOfField, riDetail, riDetailRange, riDisk, riDisplacement, riDisplay, riDisplayChannel,
		riErrorHandler, riExposure, riExterior, riFormat, riFrameAspectRatio, riFrameBegin,
		riFrameEnd, riGeneralPolygon, riGeometricApproximation, riGeometry, riHider, riHyperboloid,
		riIdentity, riIlluminate, riImager, riInterior, riLightSource, riMakeCubeFaceEnvironment,
		riMakeLatLongEnvironment, riMakeShadow, riMakeTexture, riMatte, riMotionBegin, riMotionEnd,
		riNuPatch, riObjectBegin, riObjectEnd, riObjectInstance, riOpacity, riOption, riOrientation,
		riParaboloid, riPatch, riPatchMesh, riPerspective, riPixelFilter, riPixelSamples,
		riPixelVariance, riPoints, riPointsGeneralPolygons, riPointsPolygons, riPolygon,
		riProcedural, riProjection, riQuantize, riReadArchive, riRelativeDetail,
		riReverseOrientation, riRotate, riScale, riScreenWindow, riShadingInterpolation,
		riShadingRate, riShutter, riSides, riSkew, riSolidBegin, riSolidEnd, riSphere,
		riSubdivisionMesh, riSurface, riTextureCoordinates, riTorus, riTransform, riTransformBegin,
		riTransformEnd, riTranslate, riTrimCurve, riWorldBegin, riWorldEnd, riVersion,
		riUnregistered
	};

	// Implement this to handle commands.
	virtual ribErr handleCmd(ribCode) = 0;

	// Implement this to handle errors.
	enum { ribErrInfo, ribErrWarning, ribErrError, ribErrSevere }; // severities
	virtual void handleError(int severity, const char *msg) = 0;

	// Implement this to return implementation specific param declarations.
	virtual const char *getParamDecl(const char *name) = 0;

	// string location, or resolved direct ptr
	union strloc { unsigned loc; const char *ptr; };

	// Tokens are used both for lexing and for argument storage. Hence, some of the types are
	// terminals, while others (rt{Int,String}Array) are not.
	struct ribToken {
		enum rtType { rtNone = -2, rtEOF = -1, rtName, rtInt, rtFloat, rtString, rtArrayStart,
			rtArrayEnd, rtFloatArray, rtIntArray, rtStringArray };
		rtType type;
		union {
			ribCode n;
			int i;
			float f;
			struct str {	// string; stores offset into growing (possibly realloc'd) vector
				vector<char> *v;	// typically _chars
				unsigned loc;
				operator const char *() { return &(*v)[loc]; }
			} s;
			struct arrf {	// float array; occurs directly in binary, or in args via []
				vector<float> *v;	// typically _floats
				unsigned loc, len;
				operator const float *() { return len ? &(*v)[loc] : NULL; }
			} af;
			struct arri {	// for arguments only; does not occur directly in binary
				vector<int> *v;		// typically _ints
				unsigned loc, len;
				operator const int *() { return len ? &(*v)[loc] : NULL; }
			} ai;
			struct arrs {	// string array; ditto
				vector<strloc> *v;	// typically _strings
				vector<char> *vc;	// typically _chars
				unsigned loc, len;
				void resolve() {	// convert offsets to ptrs
					for(int i = loc, n = len; --n >= 0; i++)
						(*v)[i].ptr = &(*vc)[(*v)[i].loc];
				}
				operator const char **() { return len ? (const char **)&(*v)[loc] : NULL; } // Use resolve()
																			   // first!
			} as;
			// N.B.: The pointers returned by these arrays will be constant during one api call,
			// but may be invalidated over multiple calls to getToken() (and certainly doCmd()).
		} value;
		int line; // line number in file where this was parsed
	};

	// buffered object for RiObjectBegin/End
	struct object {
		ribToken _id;					// object id (could be int or string)
		vector<ribToken> _tokens;		// predigested tokens (simple tokens only)
		vector<char> _chars;			// all string chars stored in here
		vector<float> _floats;			// for binary float arrays
		int _nextToken;					// index of next token, when rereading saved tokens
	};

	// Get the next char, allowing for push back.
	int nextc() { if (_ungottenC) { int c = _ungottenC; _ungottenC = 0; return c; }
				  else return _getc(_cbData); }

	// Get the next non-white, non-comment char.
	int nextRealChar();

	// Get the next token.
	ribToken::rtType getToken(ribToken &, vector<char>&, vector<float>&);
	ribToken::rtType getToken(ribToken &t) { return getToken(t, _chars, _floats); }

	// Get the next item, throwing reSyntaxError on EOF.
	char needc() { int c = nextc(); if (c == EOF) throw reSyntaxError; return char(c); }
	float needf();

	// Look up a name, returning a command code.
	ribCode lookup(const char *);

	// Find the type of a parameter, given the name.
	ribToken::rtType paramType(const char *);

	// Get an array.
	ribToken::rtType getArray(ribToken &t, ribToken::rtType arrayType, bool optional);

	vector<ribToken> _tokens;		// command argument tokens
	int _nParams;					// length of parameter list at end of command
	vector<const char *> _names;	// parameter list names for a single call
	vector<const void *> _parms;	// ... and values
	int _Plen;						// length of "P" param (= nverts, for polygons, etc)

	int (*_getc)(void *cbData);		// get next char callback
	void *_cbData;					// blind data passed to callback
	int _line;						// current line number, for ascii
	int _ungottenC;					// one pushed-back char
	ribToken _ungottenToken;		// one pushed-back token
	object * _currentObject;		// if parsing buffered object tokens

  private:
	vector<unsigned> _stringTab;	// defined strings indexed by token (offsets into next array)
	vector<char> _stringTabStrings;	// defined string storage
	vector<int> _ints;				// accumulating storage vectors
	vector<float> _floats;
	vector<char> _chars;			// all string chars stored in here
	vector<strloc> _strings;		// string offsets for string arrays
	vector<ribCode> _ribCodes;		// defined rib requests
	vector<object*> _objects;		// buffered ObjectBegin/End objects

	struct riCmd {
		const char *name;
		const char *args;
	};
	static riCmd riCmds[];			// table of RIB command names
};

#endif
