// -*- indent-tabs-mode: 1; tab-width: 4; c-basic-offset: 4 -*-
//
// RIB parser implementation
//
/////////////////////////////////////////////////////////////////////////////
// Copyright 2004 NVIDIA Corporation.  All Rights Reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
// * Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// * Neither the name of NVIDIA nor the names of its contributors
//   may be used to endorse or promote products derived from this software
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// (This is the Modified BSD License)
/////////////////////////////////////////////////////////////////////////////
//
// $Revision: $    $Date:  $
//
// "RIB" is a trademark of Pixar, Inc.
//

#include <ctype.h>
#include <stdio.h>

#include "RIBParser.h"

// RIB/Ri commands (indexed by ribCode)

RIBParser::riCmd RIBParser::riCmds[] = {
	"AreaLightSource", "shp", "Atmosphere", "sp", "Attribute", "sp", "AttributeBegin",
	"", "AttributeEnd", "", "Basis", "bibi", "Blobby", "iaiafasp", "Bound", "af", "Clipping", "ff",
	"ClippingPlane", "ffffff", "Color", "af", "ColorSamples", "afaf", "ConcatTransform", "af",
	"Cone", "afp", "CoordSysTransform", "s", "CoordinateSystem", "s", "CropWindow", "af",
	"Curves", "saisp", "Cylinder", "afp", "Declare", "ss", "DepthOfField", "fff", "Detail", "af",
	"DetailRange", "af", "Disk", "afp", "Displacement", "sp",
	"Display", "sssp", "DisplayChannel", "sp",
	"ErrorHandler", "s", "Exposure", "ff", "Exterior", "sp", "Format", "iif",
	"FrameAspectRatio", "f", "FrameBegin", "i", "FrameEnd", "", "GeneralPolygon", "aip",
	"GeometricApproximation", "sf", "Geometry", "sp", "Hider", "sp", "Hyperboloid", "afp",
	"Identity", "", "Illuminate", "hi", "Imager", "sp", "Interior", "sp", "LightSource", "shp",
	"MakeCubeFaceEnvironment", "sssssssfssp", "MakeLatLongEnvironment", "sssffp",
	"MakeShadow", "ssp", "MakeTexture", "sssssffp", "Matte", "i", "MotionBegin", "af",
	"MotionEnd", "", "NuPatch", "iiafffiiafffp", "ObjectBegin", "h", "ObjectEnd", "",
	"ObjectInstance", "h", "Opacity", "af", "Option", "sp", "Orientation", "s", "Paraboloid", "afp",
	"Patch", "sp", "PatchMesh", "sisisp", "Perspective", "f", "PixelFilter", "sff",
	"PixelSamples", "ff", "PixelVariance", "f", "Points", "p", "PointsGeneralPolygons", "aiaiaip",
	"PointsPolygons", "aiaip", "Polygon", "p", "Procedural", "sasaf", "Projection", "sp",
	"Quantize", "siiif", "ReadArchive", "s", "RelativeDetail", "f", "ReverseOrientation", "",
	"Rotate", "ffff", "Scale", "fff", "ScreenWindow", "af", "ShadingInterpolation", "s",
	"ShadingRate", "f", "Shutter", "ff", "Sides", "i", "Skew", "af", "SolidBegin", "s",
	"SolidEnd", "", "Sphere", "afp", "SubdivisionMesh", "saiai[asaiaiaf]p", "Surface", "sp",
	"TextureCoordinates", "af", "Torus", "afp", "Transform", "af", "TransformBegin", "",
	"TransformEnd", "", "Translate", "fff", "TrimCurve", "aiaiafafafaiafafaf", "WorldBegin", "",
	"WorldEnd", "", "version", "f"
};

// error strings

static char *RIBErrorStrings[] = {
	"no error",
	"insufficient memory to construct array",
	"incorrect parameter value",
	"invalid array specification",
	"undefined basis matrix name",
	"invalid color specification",
	"invalid light or object handle",
	"parameter list type mismatch",
	"invalid encoded RIB request code",
	"undefined encoded string token",
	"invalid binary token",
	"protocol version number mismatch",
	"overflowing an internal limit",
	"generic instance of insufficient memory",
	"malformed binary encoding",
	"insufficient memory to read string",
	"general syntactic error",
	"undefined RIB request"
};

// Append a C string, including the null, to a vector<char>.

inline void append(const char *s, vector<char> &v) {
	do v.push_back(*s); while(*s++);
}

// Create.

RIBParser::RIBParser(int (*getc)(void *cbData), void *cbData)
{
	_getc = getc;
	_cbData = cbData;
	_line = 1;
	_ungottenC = 0;
	_ungottenToken.type = ribToken::rtNone;
	_currentObject = NULL;

	// Init the ribCodes table for binary rib.
	_ribCodes.resize(256, riUnregistered);

	// Put a byte into the string table storage, so offsets are always > 0.
	_stringTabStrings.push_back('\0');
}

// Destroy.

RIBParser::~RIBParser()
{
}

// Execute one command.

RIBParser::ribErr RIBParser::doCmd(bool &eof)
{
	ribErr err = reNoErr;
	eof = false;

	// Clear out all the vectors to parse a new command.
	_ints.clear();
	_floats.clear();
	_chars.clear();
	_strings.clear();
	_tokens.clear();
	_nParams = 0;
	_names.clear();
	_parms.clear();

	try {
		// Get the command name. If it's not registered, that gets "caught" below.
		ribToken t;
		ribToken::rtType type;
		while(true)
			if ((type = getToken(t)) == ribToken::rtEOF) {
				eof = true;
				return reNoErr;
			} else if (type == ribToken::rtName)
				break;

		// Parse the arguments.
		ribCode code = t.value.n;
		riCmd *ricmd = &riCmds[(int)code];
		int sequenceNumber = -1;
		bool optional = false;
		for(const char *ap = ricmd->args; *ap; ap++) {
			switch(*ap) {
				case '[':	// begin optional
					optional = true; // only works for arrays at the moment
					break;
				case ']':	// end optional
					optional = false;
					break;
				case 'i':	// int
					if (getToken(t) != ribToken::rtInt)
						throw reSyntaxError;
					_tokens.push_back(t);
					break;
				case 'f':	// float
					if ((type = getToken(t)) == ribToken::rtInt) {
						t.value.f = float(t.value.i);
						t.type = ribToken::rtFloat;
					} else if (type != ribToken::rtFloat)
						throw reSyntaxError;
					_tokens.push_back(t);
					break;
				case 's':	// string
					if (getToken(t) != ribToken::rtString)
						throw reSyntaxError;
					_tokens.push_back(t);
					break;
				case 'h':	// lightsource handle - int or string
					type = getToken(t);
					if (type != ribToken::rtInt && type != ribToken::rtString)
						throw reSyntaxError;
					_tokens.push_back(t);
					break;
				case 'b':	// patch basis - string or matrix
					type = getToken(t);
					if (type == ribToken::rtArrayStart) {
						for(int i = 0; i < 16; i++) {
							type = getToken(t);
							if (type == ribToken::rtEOF)
								throw reBadArray;
							else if (type == ribToken::rtFloat)
								_floats.push_back(t.value.f);
							else if (type == ribToken::rtInt)
								_floats.push_back(float(t.value.i));
						}
						if (getToken(t) != ribToken::rtArrayEnd)
							throw reBadArray;
						t.type = ribToken::rtFloatArray;
						t.value.af.v = &_floats;
						t.value.af.loc = _floats.size() - 16;
						t.value.af.len = 16;
					} else if (type == ribToken::rtFloatArray) {
						// got the whole thing in one token; great.
					} else if (type == ribToken::rtString) {
						#if 0
						const float *bmat = NULL;
						if (strcmp(t.value.s, "bezier") == 0)
							bmat = (const float *)RiBezierBasis;
						else if (strcmp(t.value.s, "b-spline") == 0)
							bmat = (const float *)RiBSplineBasis;
						else if (strcmp(t.value.s, "catmull-rom") == 0)
							bmat = (const float *)RiCatmullRomBasis;
						else if (strcmp(t.value.s, "hermite") == 0)
							bmat = (const float *)RiHermiteBasis;
						else if (strcmp(t.value.s, "power") == 0)
							bmat = (const float *)RiPowerBasis;
						if (bmat == NULL)
							throw reBadBasis;
						for(int i = 0; i < 16; i++)
							_floats.push_back(*bmat++);
						t.type = ribToken::rtFloatArray;
						t.value.af.v = &_floats;
						t.value.af.loc = _floats.size() - 16;
						t.value.af.len = 16;
						#endif
					} else
						throw reSyntaxError;
					_tokens.push_back(t);
					break;
				case 'a': {	// array of ...
					// What type?
					++ap;
					ribToken::rtType arrayType = *ap == 'i' ? ribToken::rtIntArray
						: *ap == 'f' ? ribToken::rtFloatArray : ribToken::rtStringArray;
					ribToken::rtType returnedType = getArray(t, arrayType, optional);
					if (returnedType != arrayType)
						throw reBadArray;
					_tokens.push_back(t);
					break;
				}
				case 'p': {	// parameter list
					while((type = getToken(t)) == ribToken::rtString) {
						// Push the parameter name.
						_tokens.push_back(t);

						// Get the parameter type.
						ribToken::rtType pType = paramType(t.value.s);

						// Get the parameter value.
						type = getArray(t, pType, false/*optional*/);
						_tokens.push_back(t);

						// Chalk up another parameter in the param list.
						++_nParams;
					}
					if (type == ribToken::rtName || type == ribToken::rtEOF)
						_ungottenToken = t;
					else
						throw reSyntaxError;
					break;
				}
			}
		}

		// Resolve the string arrays (converts offsets to ptrs).
		for(int ti = _tokens.size(); --ti >= 0; )
			if (_tokens[ti].type == ribToken::rtStringArray)
				_tokens[ti].value.as.resolve();

		// Rack up param list args acceptable to the "V" routines.
		_Plen = 0;
		for(int p = 0, tokOff = _tokens.size() - 2*_nParams; p < _nParams; p++) {
			const char *name = _tokens[tokOff++].value.s;
			_names.push_back(name);
			ribToken &parmToken = _tokens[tokOff++];
			switch(parmToken.type) {
				case ribToken::rtInt:		  _parms.push_back(&parmToken.value.i);	break;
				case ribToken::rtFloat:		  _parms.push_back(&parmToken.value.f);	break;
				case ribToken::rtString:	  _parms.push_back(parmToken.value.s);	break;
				case ribToken::rtIntArray:	  _parms.push_back(parmToken.value.ai);	break;
				case ribToken::rtFloatArray:  _parms.push_back(parmToken.value.af);	break;
				case ribToken::rtStringArray: _parms.push_back(parmToken.value.as);	break;
			}

			// RiPolygon and RiCurves need the length of "P" (or Pw).
			if (name[0] == 'P')
				if (name[1] == '\0')
					_Plen = parmToken.value.af.len / 3;
				else if (name[1] == 'w' && name[2] == '\0')
					_Plen = parmToken.value.af.len / 4;
		}

		// Execute the command.
		if (code == riObjectBegin) {
			// bracketing
			if (_tokens[0].type != ribToken::rtInt && _tokens[0].type != ribToken::rtString)
				throw reSyntaxError;

			object *obj = new object;
			obj->_id = _tokens[0];
			if (_tokens[0].type == ribToken::rtString) {
				// Move string to obj->_chars, since _tokens disappears after this cmd.
				obj->_id.value.s.v = &obj->_chars;
				obj->_id.value.s.loc = obj->_chars.size();
				append(_tokens[0].value.s, obj->_chars);
			}

			// Buffer up tokens until ObjectEnd.
			while(true)
				if ((type = getToken(t, obj->_chars, obj->_floats)) == ribToken::rtEOF) {
					throw reSyntaxError;
				} else if (type == ribToken::rtName && t.value.n == riObjectEnd) {
					break;
				} else {
					obj->_tokens.push_back(t); // all token data should be in obj
				}

			obj->_nextToken = 0;
			_objects.push_back(obj);
		} else if (code == riObjectInstance) {
			// Find in list
			object *obj = NULL;
			if (_tokens[0].type == ribToken::rtInt) {
				for(int i = 0; i < _objects.size(); i++)
					if (_objects[i]->_id.type == ribToken::rtInt
						&& _objects[i]->_id.value.i == _tokens[0].value.i) {
						obj = _objects[i];
						break;
					}
			} else if (_tokens[0].type == ribToken::rtString) {
				for(int i = 0; i < _objects.size(); i++)
					if (_objects[i]->_id.type == ribToken::rtString
						&& strcmp(_objects[i]->_id.value.s, _tokens[0].value.s) == 0) {
						obj = _objects[i];
						break;
					}
			} else
				throw reSyntaxError;

			if (obj) {
				obj->_nextToken = 0;
				_currentObject = obj;
			} else
				throw reBadHandle;
		} else
			err = handleCmd(code);
	} catch(ribErr err1) {
		err = err1;
		// TODO: use current error handler
		handleError(ribErrError, RIBErrorStrings[(int)err]);
	}

	return err;
}

// Get the type of a parameter, given the name.

RIBParser::ribToken::rtType RIBParser::paramType(const char *name)
{
	const char *type = name;
	if (!strchr(name, ' ')) // no inline declaration? (e.g., "float[4] foopy")
		type = getParamDecl(name);
	if (type == NULL)
		type = "float"; // parse as floats by default, what the hell.

	// Skip the storage class.
	if (strncmp(type, "constant", 8) == 0)
		type += 8;
	else if (strncmp(type, "uniform", 7) == 0)
		type += 7;
	else if (strncmp(type, "varying", 7) == 0)
		type += 7;
	else if (strncmp(type, "facevarying", 11) == 0)
		type += 11;
	else if (strncmp(type, "vertex", 6) == 0)
		type += 6;

	// Skip any white space.
	while(*type == ' ' || *type == '\t') ++type;

	ribToken::rtType t = ribToken::rtFloatArray;
	if (strncmp(type, "int", 3) == 0)
		t = ribToken::rtIntArray;
	else if (strncmp(type, "float", 5) == 0 || strncmp(type, "color", 5) == 0
		|| strncmp(type, "point", 5) == 0 || strncmp(type, "hpoint", 6) == 0
		|| strncmp(type, "vector", 6) == 0 || strncmp(type, "normal", 6) == 0
		|| strncmp(type, "matrix", 6) == 0)
		t = ribToken::rtFloatArray;
	else if (strncmp(type, "string", 6) == 0)
		t = ribToken::rtStringArray;
	else
		throw reSyntaxError; // TODO: more?

	return t;
}

// Get an array of a given type.

RIBParser::ribToken::rtType RIBParser::getArray(RIBParser::ribToken &t, RIBParser::ribToken::rtType arrayType,
	bool optional)
{
	// Get the first token. This will typically be '[', or a binary array, but various commands
	// have lists of float params which are optionally arrays (e.g., Sphere, etc).  (Note 'optional'
	// only works with bona-fide arrays.)
	bool missing = false;
	ribToken::rtType type = getToken(t);
	if (optional && type != ribToken::rtArrayStart && type != ribToken::rtFloatArray) {
		// Fake an empty array.
		type = ribToken::rtArrayStart;
		missing = true;
		_ungottenToken = t;
	}

	unsigned loc = 0;
	switch(type) {
		default:
			throw reBadArray;
			break;
		case ribToken::rtFloatArray:
			// Great, got the whole thing in one token.
			arrayType = ribToken::rtFloatArray;
			break;
		case ribToken::rtArrayStart: {
			// Collect the array values.
			for(int nv = 0; true; nv++) {
				type = missing ? ribToken::rtArrayEnd : getToken(t);
				if (type == ribToken::rtEOF) {
					throw reBadParamList;
				} else if (type == ribToken::rtInt) {
					if (arrayType == ribToken::rtIntArray)
						_ints.push_back(t.value.i);
					else if (arrayType == ribToken::rtFloatArray)
						_floats.push_back(float(t.value.i)); // promote int to float
					else
						throw reBadArray;
				} else if (type == ribToken::rtFloat) {
					if (arrayType == ribToken::rtFloatArray)
						_floats.push_back(t.value.f);
					else if (arrayType == ribToken::rtIntArray) {
						// Convert the ints collected thus far to floats.
						for(int i = _ints.size() - nv; i < _ints.size(); i++)
							_floats.push_back(float(_ints[i]));
						// Peel off the ints.
						_ints.resize(_ints.size() - nv);
						// Add the new float.
						_floats.push_back(t.value.f);
						// Change the array type.
						arrayType = ribToken::rtFloatArray;
					} else
						throw reBadArray;
				} else if (type == ribToken::rtString) {
					if (arrayType != ribToken::rtStringArray && nv > 0)
						throw reBadArray;
					strloc sloc;
					sloc.loc = t.value.s.loc;
					_strings.push_back(sloc);
					arrayType = ribToken::rtStringArray;
				} else if (type == ribToken::rtArrayEnd) {
					// Load up a token of the appropriate type.
					t.type = arrayType;
					if (arrayType == ribToken::rtIntArray) {
						t.value.ai.v = &_ints;
						t.value.ai.loc = _ints.size() - nv;
						t.value.ai.len = nv;
					} else if (arrayType == ribToken::rtFloatArray) {
						t.value.af.v = &_floats;
						t.value.af.loc = _floats.size() - nv;
						t.value.af.len = nv;
					} else if (arrayType == ribToken::rtStringArray) {
						t.value.as.v = &_strings;
						t.value.as.vc = &_chars;
						t.value.as.loc = _strings.size() - nv;
						t.value.as.len = nv;
					}
					break;
				}
			}
			break;
		}
		case ribToken::rtFloat: {
			// Undelimited case: get floats until we hit something else.
			if (arrayType != ribToken::rtFloatArray)
				throw reSyntaxError;
			loc = _floats.size();
			_floats.push_back(t.value.f);
			collectFloats:
			int nf = 1;
			for(; (type = getToken(t)) == ribToken::rtFloat || type == ribToken::rtInt; ++nf)
				_floats.push_back(type == ribToken::rtFloat ? t.value.f : float(t.value.i));
			_ungottenToken = t;
			t.type = arrayType;
			t.value.af.v = &_floats;
			t.value.af.loc = loc;
			t.value.af.len = nf;
			break;
		}
		case ribToken::rtInt: {
			if (arrayType == ribToken::rtIntArray) {
				loc = _ints.size();
				_ints.push_back(t.value.i);
			} else if (arrayType == ribToken::rtFloatArray) {
				loc = _floats.size();
				_floats.push_back(float(t.value.i));
				goto collectFloats;
			} else
				throw reSyntaxError;
			int ni = 1;
			for(; (type = getToken(t)) == ribToken::rtInt; ++ni)
				_ints.push_back(t.value.i);
			_ungottenToken = t;
			t.type = arrayType;
			t.value.ai.v = &_ints;
			t.value.ai.loc = loc;
			t.value.ai.len = ni;
			break;
		}
		case ribToken::rtString: {
			arrayType = ribToken::rtStringArray;
			strloc sloc;
			sloc.loc = t.value.s.loc;
			loc = _strings.size();
			_strings.push_back(sloc);
			int ns = 1;
			#if 0 // only time this happens is strings in param lists with missing brackets
			for(; (type = getToken(t)) == ribToken::rtString; ++ns) {
				sloc.loc = t.value.s.loc;
				_strings.push_back(sloc);
			}
			_ungottenToken = t;
			#endif
			t.type = arrayType;
			t.value.as.v = &_strings;
			t.value.as.vc = &_chars;
			t.value.as.loc = loc;
			t.value.as.len = ns;
			break;
		}
	}

	return arrayType;
}

// Get the next non-white, non-comment char.

int RIBParser::nextRealChar()
{
	int c;
	do {
		c = nextc();
		if (c == '#')		// comment; slurp to end of line
			do {
				c = nextc();
			} while(c != EOF && c != '\n');
		if (c == '\n')		// count lines for ascii mode
			++_line;
	} while(isspace(c));
	return c;
}

// Get the next token.

RIBParser::ribToken::rtType RIBParser::getToken(ribToken &t, vector<char>& chars,
	vector<float>& floats)
{
	// Check for a pushed-back token.
	if (_ungottenToken.type != ribToken::rtNone) {
		t = _ungottenToken;
		_ungottenToken.type = ribToken::rtNone;
		return t.type;
	}

	// Parsing from an object? (due to ObjectInstance)
	if (_currentObject) {
		t = _currentObject->_tokens[_currentObject->_nextToken++];
		if (_currentObject->_nextToken >= _currentObject->_tokens.size())
			_currentObject = NULL;
		return t.type;
	}

	tokenPlease: // yeah, yeah, could be a loop
	int c = nextRealChar();
	t.line = _line;
	switch(c) {
		// end of file
		case EOF:
			t.type = ribToken::rtEOF;
			break;

		// array delimiters
		case '[':
			t.type = ribToken::rtArrayStart;
			break;
		case ']':
			t.type = ribToken::rtArrayEnd;
			break;

		// names and numbers
		default:
			if (isalpha(c)) {
				// Collect the name.
				char name[256];
				name[0] = c;
				for(int i = 1; i < sizeof(name) - 1; ) {
					c = nextc();
					if (isalpha(c)) {
						name[i++] = c;
						if (i == sizeof(name) - 2) {
							name[sizeof(name) - 1] = '\0';
							break;
						}
					} else {
						_ungottenC = c;
						name[i] = '\0';
						break;
					}
				}

				// Look up the name.
				ribCode code = lookup(name);
				if (code == riUnregistered)
					throw reUnregistered;
				t.type = ribToken::rtName;
				t.value.n = code;
			} else if (isdigit(c) || c == '-' || c == '.') { // does anybody start numbers with '+'?
				// Collect the number.
				char num[256];
				num[0] = c;
				for(int i = 1; i < sizeof(num) - 1; ) {
					c = nextc();
					if (isdigit(c) || c == '-' || c == '+' || c == '.' || c == 'e' || c == 'E') {
						num[i++] = c;
						if (i == sizeof(num) - 2) {
							num[sizeof(num) - 1] = '\0';
							break;
						}
					} else {
						_ungottenC = c;
						num[i] = '\0';
						break;
					}
				}

				// Convert to int or float.
				if (strpbrk(num, ".eE")) {
					t.type = ribToken::rtFloat;
					t.value.f = float(atof(num));
				} else {
					t.type = ribToken::rtInt;
					t.value.i = atoi(num);
				}
			} else if (c & 0x80)
				throw reBadToken;
			else
				throw reSyntaxError;
			break;

		// ascii string (of unfortunately unknown length)
		case '"': {
			t.type = ribToken::rtString;
			t.value.s.v = &chars;
			t.value.s.loc = chars.size();
			while(true) {
				// Get the next char.
				int c = nextc();
				if (c == EOF) {
					throw reSyntaxError;
					break;
				} else if (c == '\\') {
					switch(c = nextc()) {
						case EOF:  throw reSyntaxError;   break;
						case 'n':  chars.push_back('\n'); break;
						case 'r':  chars.push_back('\r'); break;
						case 't':  chars.push_back('\t'); break;
						case 'b':  chars.push_back('\b'); break;
						case 'f':  chars.push_back('\f'); break;
						case '\\': chars.push_back('\\'); break;
						case '\n': ++_line;               break; // ignore continuation newline
						default:
							if (c >= '0' && c <= '7') {
								int c2 = nextc();
								if (c2 < '0' || c2 > '7')
									throw reSyntaxError;
								int c3 = nextc();
								if (c3 < '0' || c3 > '7')
									throw reSyntaxError;
								chars.push_back(char(
									((c - '0') << 6) + ((c2 - '0') << 3) + (c3 - '0')));
							} else
								chars.push_back(char(c));
							break;
					}
				} else if (c == '"') {
					chars.push_back('\0');
					break;
				} else
					chars.push_back(char(c));
			}
			break;
		}

		// encoded integers and fixed-point numbers
		case 0200: case 0201: case 0202: case 0203: case 0204: case 0205: case 0206: case 0207:
		case 0210: case 0211: case 0212: case 0213: case 0214: case 0215: case 0216: case 0217: {
			int d = (c - 0200) >> 2;		// number of bytes after decimal point
			int w = c & 3;					// byte length of number (minus 1)
			int n = (int)needc();			// need at least one (sign extended) byte
			while(--w >= 0)					// get remaining bytes
				n = (n << 8) | (unsigned char)(needc());
			if (d == 0) {
				t.type = ribToken::rtInt;
				t.value.i = n;
			} else {
				t.type = ribToken::rtFloat;
				t.value.f = float(n) / (256L << ((d-1)*8));
			}
			break;
		}

		// short binary string
		case 0220: case 0221: case 0222: case 0223: case 0224: case 0225: case 0226: case 0227:
		case 0230: case 0231: case 0232: case 0233: case 0234: case 0235: case 0236: case 0237: {
			t.type = ribToken::rtString;
			t.value.s.v = &chars;
			t.value.s.loc = chars.size();

			unsigned sbsLen = c - 0220;

			// Reserve space for the string.
			try {
				chars.reserve(chars.size() + sbsLen);
			} catch(int) {
				throw reStringTooBig;
			}

			// Now collect the string.
			for(unsigned i = 0; i < sbsLen; i++)
				chars.push_back(needc());
			chars.push_back('\0');

			break;
		}

		// long binary string
		case 0240: case 0241: case 0242: case 0243: {
			t.type = ribToken::rtString;
			t.value.s.v = &chars;
			t.value.s.loc = chars.size();

			// Get the length of the length: 1-4 bytes.
			unsigned lbsLenLen = c - 0240 + 1;

			// Now get the length.
			unsigned lbsLen = 0;
			for(unsigned i = 0; i < lbsLenLen; i++)
				lbsLen = (lbsLen << 8) + (unsigned char)(needc());

			// Reserve space for the string.
			try {
				chars.reserve(chars.size() + lbsLen);
			} catch(int) {
				throw reStringTooBig;
			}

			// Now collect the string.
			for(unsigned j = 0; j < lbsLen; j++)
				chars.push_back(needc());
			chars.push_back('\0');

			break;
		}

		// encoded single precision IEEE floating point value
		case 0244:
			t.type = ribToken::rtFloat;
			t.value.f = needf();
			break;

		// encoded double precision IEEE floating point value
		case 0245: {
			unsigned char c[8];
			for(int i = 0; i < 8; i++) // TODO: platform?
				c[i] = (unsigned char)needc();

			// There are no Ri interface routines that take doubles.
			t.type = ribToken::rtFloat;
			t.value.f = float(*(double *)c);
			break;
		}

		// encoded ri request
		case 0246: {
			unsigned code = (unsigned char)(needc());
			t.type = ribToken::rtName;
			if (_ribCodes.size() <= code || _ribCodes[code] == riUnregistered)
				throw reBadRipCode;
			t.value.n = _ribCodes[code];
			break;
		}

		// encoded single precision array
		case 0310: case 0311: case 0312: case 0313: {
			// Get the array length.
			unsigned arrayLen = (unsigned char)needc();
			for(int arrayLenLen = c - 0310/* minus 1*/; --arrayLenLen >= 0; )
				arrayLen = (arrayLen << 8) + (unsigned char)(needc());

			// Collect the array.
			t.type = ribToken::rtFloatArray;
			t.value.af.v = &floats;
			t.value.af.loc = floats.size();
			t.value.af.len = arrayLen;
			floats.reserve(floats.size() + arrayLen);
			for(int i = arrayLen; --i >= 0; )
				floats.push_back(needf());
			break;
		}

		// define encoded request
		case 0314: {
			// Get the code.
			unsigned code = (unsigned char)(needc());

			// Get the string (always into _chars[], not chars[]).
			ribToken str;
			if (getToken(str) != ribToken::rtString)
				throw reSyntaxError;

			// Look up the string.
			ribCode rc = lookup(str.value.s);
			if (rc == riUnregistered)
				throw reUnregistered;

			// Enter the code,ribCode pair.
			_ribCodes[code] = rc;

			// Go around again!
			goto tokenPlease;
		}

		// define encoded string
		case 0315:
		case 0316: {
			// Get the token.
			unsigned token = (unsigned char)(needc());
			if (c == 0316)
				token = (token << 8) + (unsigned char)(needc());

			// Get the string (always into _chars[], not chars[]).
			ribToken str;
			if (getToken(str) != ribToken::rtString)
				throw reSyntaxError;

			// Put the string in the table.
			if (_stringTab.size() <= token)
				_stringTab.resize(token + 1, 0); // TODO: fails how?
			_stringTab[token] = _stringTabStrings.size();
			append(str.value.s, _stringTabStrings);

			// Go around again!
			goto tokenPlease;
		}

		// interpolate defined string
		case 0317:
		case 0320: {
			// Get the token.
			unsigned token = (unsigned char)(needc());
			if (c == 0320)
				token = (token << 8) + (unsigned char)(needc());

			// Return the associated string.
			if (token >= _stringTab.size() || _stringTab[token] == 0)
				throw reBadStringToken;
			t.type = ribToken::rtString;
			t.value.s.v = &chars;
			t.value.s.loc = chars.size();
			append(&_stringTabStrings[_stringTab[token]], chars);

			//printf("interp def str: \"%s\" (token = %u)\n", (const char *)t.value.s, token);
			break;
		}
	}
	return t.type;
}

// Get the next float, or bust.

float RIBParser::needf()
{
	unsigned n = (unsigned char)(needc());
	n = (n << 8) + (unsigned char)(needc());
	n = (n << 8) + (unsigned char)(needc());
	n = (n << 8) + (unsigned char)(needc());
	return *(float *)&n; // TODO: platform?
}

// Look up a name in the ribCmds.

RIBParser::ribCode RIBParser::lookup(const char *name)
{
	for(int lo = 0, hi = sizeof(riCmds)/sizeof(riCmd) - 1; true; ) {
		int mid = (lo+hi) / 2;
		riCmd *midCmd = &riCmds[mid];
		int cmp = strcmp(name, midCmd->name);
		if (cmp == 0)
			return (ribCode)mid;
		else if (cmp < 0) {
			hi = mid;
			if (lo == hi-1)
				if (strcmp(name, riCmds[lo].name) == 0)
					return (ribCode)lo;
				else
					break;
		} else {
			lo = mid;
			if (lo == hi-1)
				if (strcmp(name, riCmds[hi].name) == 0)
					return (ribCode)hi;
				else
					break;
		}
	}
	return riUnregistered;
}
