// -*- indent-tabs-mode: 1; tab-width: 4; c-basic-offset: 4 -*-
//
// RIB->Gelato declarations
//
/////////////////////////////////////////////////////////////////////////////
// Copyright 2004 NVIDIA Corporation.  All Rights Reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
// * Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// * Neither the name of NVIDIA nor the names of its contributors
//   may be used to endorse or promote products derived from this software
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// (This is the Modified BSD License)
/////////////////////////////////////////////////////////////////////////////
//
// $Revision: $    $Date:  $
//
// "RIB" is a trademark of Pixar, Inc.
//

#ifndef _Ribelato_
#define _Ribelato_

#include <vector>
using namespace std;

#include "RIBParser.h"

class GelatoAPI;

class Ribelato: public RIBParser {
  public:
	// Create and destroy.
	Ribelato(GelatoAPI *, int (*getc)(void* cbData) = NULL, void* cbData = NULL);
	virtual ~Ribelato();

  protected:
	// variable storage classes and types (e.g., "vertex hpoint Pw", or "varying float[2] st")
	enum storageClass { scConstant, scUniform, scVarying, scFaceVarying, scVertex };
	enum dataType { dtInt, dtFloat, dtColor, dtPoint, dtHPoint, dtVector, dtNormal, dtMatrix,
		dtString };
	struct pdata { storageClass sc; dataType dt; int siz; };
	void getClassAndType(const char *decl, pdata &);

	// for saved quantization tuples
	#define QNAMELEN 32
	struct quantize { char name[QNAMELEN]; int one, min, max; float dither; };

	// for saved TextureCoordinates (in gelato-friendly separate "s" and "t" format)
	struct texCoords {
		float s[4], t[4];
		bool operator !=(texCoords &c) {
			return s[0] != c.s[0] || s[1] != c.s[1] || s[2] != c.s[2] || s[3] != c.s[3]
				|| t[0] != c.t[0] || t[1] != c.t[1] || t[2] != c.t[2] || t[3] != c.t[3];
		}
	};
	static texCoords _defaultTexCoords;

	// for saved patch bases
	struct basis {
		const char *ubasis, *vbasis;
		int ustep, vstep;
	};
	static basis _defaultBasis;

	// RIBParser callbacks
	virtual ribErr handleCmd(ribCode);
	virtual const char* getParamDecl(const char *name);

	// Helpers
	void Display(const char* name, const char* type, const char* mode);
	void Option(const char* name);
	void Light(bool isAreaLight, const char* shaderName, const char* lightName);
	void Polygon(int nverts);
	void GeneralPolygon(int nloops, const int nverts[]);
	void PointsPolygons(int npolys, const int nverts[], const int verts[]);
	void PointsGeneralPolygons(int npolys, const int nloops[], const int nverts[],
		const int verts[]);
	void SubdivisionMesh(const char* mask, int nf, const int nverts[], const int verts[], int ntags,
		const char* tags[], const int nargs[], const int intargs[], const float floatargs[]);
	void Patch(const char* type);
	void PatchMesh(const char* type, int nu, const char* uwrap, int nv, const char* vwrap);
	void NuPatch(int nu, int uorder, const float uknot[], float umin, float umax,
		int nv, int vorder, const float vknot[], float vmin, float vmax);
	void TrimCurve(int nloops, const int ncurves[], const int order[], const float knot[],
		const float min[], const float max[], const int n[], const float u[], const float v[],
		const float w[]);
	void Sphere(float radius, float zmin, float zmax, float tmax);
	void Cylinder(float radius, float zmin, float zmax, float tmax);
	void Cone(float z, float radius, float tmax);
	void Disk(float z, float radius, float tmax);
        void Paraboloid(float radius, float zmin, float zmax, float tmax);
	void Hyperboloid(float x0, float y0, float z0,
					 float x1, float y1, float z1, float tmax);
        void Torus (float majorrad, float minorrad, float phimin, float phimax, float tmax);
	void Curves(const char* type, int ncurves, const int nvertices[], const char* wrap);
	void Points(int nverts);
	void Procedural(const void* data, const char* tokens[], const float* bound);
	void Declare(const char *name, const char *declaration);
	void ErrorHandler(const char* handlerName);

	// Curves() helper
	template <class T>
	static inline void copyCurveParm(const T *parmp, int ncurves, const int nvertices[], int nv,
		pdata &, bool linear, bool periodic, int vstep, vector<T> &out);

	// fully-specify param types (e.g., "P" -> "point P"); maps from _names/_parms to
	// _names2, _parms2
	void mapTokens(bool noClass = false);

	// copy paramlist ptrs
	void copyParms();

	// map token types (e.g., "uniform" -> "constant")
	void remapTokenTypes(int n, storageClass sc1, const char *newsc1, storageClass sc2,
		const char *newsc2);

	// Strip out uniform and varying params, which are not supported by Gelato.
	// Uses _names2 and _parms2 arrays, and decrements _nParams.
	void stripUniformAndVarying();

	// Send paramlist to api, optionally breaking "st" into "s" and "t", and adding default s,t
	// coords.
	enum { no_offsets = false, use_offsets = true, default_st = true };
	void sendParms(int n, const char* tokens[], const void* parms[], bool use_offsets = false,
		int n_st = 0, bool default_st = false);

	// helper function: output Patch that is a swept line segment (for
	// cylinder, cone, hyperboloid, etc.
	void sweep_line (float radius0, float radius1, float z0, float z1,
					 float thetamax, float rotate0=0.0f, float rotate1=0.0f) const;

	// RiDeclare items
	struct decl {
		unsigned nameLoc, typeLoc;
		const char *name(vector<char> &v) { return &v[nameLoc]; }
		const char *type(vector<char> &v) { return &v[typeLoc]; }
	};

	// error dispositions
	enum ribErrorHandler { ribErrIgnore, ribErrPrint, ribErrAbort };

	GelatoAPI *_api;				// gelato renderer
	bool _worldBegun;				// got WorldBegin?
	float _gain, _gamma;			// saved exposure params for Display
	vector<quantize> _quants;		// saved quantization tuples for Display
	float _filterWidth[2];			// saved filterwidths for Display
	vector<char> _filter;			// saved filter for Display
	vector<int> _verts;				// iota vector of polygon vertex indices
	vector<const char *> _names2;	// fully-specified param-list tokens
	vector<const void *> _parms2;	// pointers to temp paramlist data
	vector<float> _floats;			// temp float params
	vector<int> _ints;				// temp int param storage
	vector<const char *> _strings;	// and temp strings
	vector<unsigned> _offsets;		// offsets into above arrays (currently only _floats)
	texCoords _texCoords;			// current TextureCoordinates
	vector<texCoords> _tcStack;		// texcoords stack, for Push/PopAttributes
	basis _basis;					// current bases
	vector<basis> _basisStack;		// basis stack, for Push/PopAttributes
	vector<char> _declStrings;		// Declare string storage
	vector<decl> _decls;			// Declare string offsets
	vector<char> _tokenStrings;		// for dynamically-generated tokens
	ribErrorHandler _errorHandler;	// current error handler
};

#endif
