; ============================================================ ;
;    Program: DynamicGrass.Heightmap, version 0.5
;    (realtime vegetation visualisation with dynamic distribution over heightmaps)
;
;    Copyright (C) 2008
;
;    Author: Dmitry Glibitsky
;    Contact information: Email <dima.glib@gmail.com>, ICQ 350900734
;
;    This program is free software: you can redistribute it and/or modify
;    it under the terms of the GNU Lesser General Public License as published by
;    the Free Software Foundation, either version 3 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU Lesser General Public License for more details.
;
;    You should have received a copy of the GNU Lesser General Public License
;    along with this program.  If not, see <http://www.gnu.org/licenses/>,
;    <http://www.opensource.org/licenses/lgpl-3.0.html>.
; ============================================================ ;

; Language: Blitz3D

; Initially the program was simply an attempt to achieve maximum amount of grass
; rendered in realtime. In the current state it is far from being optimal and full-featured
; library for realtime vegetation visualisation. If you use this code, an acknowledgement
; would be highly appreciated, though :-)
; One more note: if you'll make some improvements/optimizations of the algorithm,
; it would be desirable to put it to the same code depository.

; ============================================================ ;

; FEATURE LIST:

; + Feature: Dynamic algorithm that do not require exact placement and state information
;   for each particle.
; = Status: implemented

; + Feature: Possibility to use beforehand assigned placement and state for some particles.
; = Status: not implemented

; + Feature: Dynamically built mesh for simple flat particles.
; = Status: implemented

; + Feature: Depth sort for dynamically built mesh.
; = Status: implemented, but should be improved (using red-black tree + adding option to
;   disable sorting for non-transparent particles or when using irregular Z-buffer)

; + Feature: 'Fading' radial distribution by distance from observer; distribution is 'updated'
;   when observer moves.
; = Status: implemented, but should be improved

; + Feature: Particles maintain their placement and state, until their distance from observer
;   becomes greater than visibility range.
; = Status: implemented

; + Feature: Support for different types of particles (fixed / aligned to heightmap / billboard;
;   3D object / dynamically built flat surface)
; = Status: implemented

; + Feature: Switching between several 3D models and dynamically built 2D particle with
;   changeable 'resolution' depending on distance from observer (i.e. LOD).
; = Status: not implemented

; + Feature: Low consumption of CPU power in the locations where vegetation density = 0.
; = Status: partially implemented, futher improvements may be possible

; + Feature: Density maps for each kind of particles.
; = Status: not implemented

; + Feature: Particle interaction with other in-game objects and events (in response to air
;   movement or rigid bodies movement; changing state of particles; changing heightmap
;   and density map).
; = Status: generally not implemented (only a simple example of global wind influence)

; + Feature: Common depth sort for all transparent particles (it is not necessary if you
;   are using irregular Z-buffer)
; = Status: not implemented

; + Feature: Maximal speed optimization (perhaps only possible from C++)
; = Status: not implemented

; ============================================================ ;

; THEORY:

; There could be different approaches to drawing a terrain dense-filled with grass.

; The brute-force method used in offline rendering creates lots of particles,
;   each of them may look unique. For realtime purposes this could be used
;   to visualise grass only within a short distance around the camera
;   because of great number of triangles to render.

; The next-step approximation I can think of - is to somehow calculate what
;   parts of screen must be filled with grass to satisfy the visual impression.
;   To do this, a program needs to understand where to place grass particles
;   so that they won't be occluded by the the particles. Because visual 'density'
;   of grass is largely determined by grass texture/geometry, it's hard to make
;   an algorithm that could work properly for any grass. Maybe, such algorithm
;   is impossible at all.

; So, for large grass massives we have to use the standard way used in many
;   games: we need to simulate grass by wide flat planes with "grass" texture.
;   I've used the following criteria that the desired algorithm must match:

;   * Little scene loading time. If placement and orientation of each grass particle
;     is determined at the phase of level creation, such information takes
;     lots of disc space and RAM (because there would be MILLIONS of particles
;     in the case of grass density I want). So, the only choice is to always
;     calculate particle position (and some other properties) at the realtime.
;     I should note that designer-placed grass may be preferable in some cases
;     (but they are quite uncommon).

;   * Grass can have different 'distance distribution law' (dependence of particle
;     placement density on the distance from observer). It could vary from
;     even distribution to some 'fading' distribution (particle will be more probably
;     placed near to the observer than far from observer). The latter is usually
;     more useful (the reasons are the same as for LOD). This is only possible
;     when I place particles radially around the observer and then move them
;     some distance away (the distance is determined by the distribution law).
;     So, the option of placing particles in a regular grid pattern doesn't
;     match this criterion. This leads to the following situation: when you visit
;     some game location again and again, each time you'll get different
;     particle distribution (because the algorithm is random by its nature).

;   * 'Boundary conditions satisfaction'. This principle could be formulated
;     as 'observer should not notice randomness of grass'. People don't
;     forget the visual impression of places they saw recently, so, within some
;     range I have to keep 'alive' particles despite their visible/invisible state.
;     For example, if a player is just rotating camera around his/her character,
;     it could be easily noticed if enviroment is changed each time it falls out
;     of scope (perhaps it's even unpleasant to know). The same criterion
;     can be applied to the parts of terrain that should not have grass.

;   * When observer (camera) is moving, I need to 'move' the 'distribution
;     pattern' so that it would correspond to new observer position.
;     In the current version some amount of new particles is created in the
;     direction where the observed has moved (at the edge of visible grass).

;   * Number of grass praticles should not exceed the specified amount
;     (i.e. the max number of particles). When some particles get out
;     of the visibility range (i.e. they are too far), they must be deleted and
;     after that the same amout of new particles can be created.

;   * Use of several overlapping density maps for each kind of grass.
;     For example, I have a 'global' density map for grass and a 'global'
;     density map for flowers (e.g. tulip or camomile). They can represent
;     average density of grass/flowers over the large areas of the map.
;     But when I need to control distribution more precisely in some areas
;     of my map (e.g. near to some house: to forbid grass/flowers to appear
;     inside of the house or within some distance around it), I would prefer
;     a smaller but more spatially detailed distribution map that would
;     'override' the general density determined by 'global' map.
;     I've decided to check the 'active' distribution map by the most great
;     terrainheight value at the point of possible particle creation.
;     Perhaps grassmaps can change density value in different ways:
;     replace, add, subtract, multiply, divide, raise to power...
;     Several density maps can be also useful when dealing with dynamically
;     loaded level (and tile-based terrain, where for each tile a corresponding
;     heightmap and density map exist).

;   * In the areas where no grass can exist (city streets, houses interior)
;     grass generation shouldn't take much of CPU/GPU processing time.

;   * Grass particles should not be static. The most basic exmaple is
;     when grass, flowers or trees are affected by the wind. More complicated
;     examples could feature interaction with in-game objects and changing
;     (and memorizing) state of each particle: characters walking over the field
;     should bend grass/trees; if some local disaster has happened (e.g. fire or
;     explosion), grass must change to burned state or even should disappear.
;     Such situations are the great step to implement, but they increase
;     algorithm's complexity and processor load.
;     Some ideas to complex interaction of particles with game objects/events:
;     * bending due to local air movement (wind waves, helicopter takeoff/landing)
;     * bending due to collision with objects/characters
;     * changing parameters and texture of particles in some area (+disappearing)
;     + Grass emitter should first determine what 'particle affectors' intersect
;        the visible area (make a list of them, sort the list by one of the axis, calculate
;        their general bounding box or bounding circle) -- to later simplify determination
;        of what affector influences on each particle.

;   * Support for common types of particles. These types (as I've classified
;     them) include:
;     1) DynamicLine. Examples: long lines of grass that 'grows vertically'.
;         They are placed, non-uniformly scaled and rotated (over Y-axis).
;         This type of grass is placed on the terrain in a specific way:
;         when particle 'coordinate' is calculated, the particle middle point is
;         placed at this 'coordinate', and particle edge vertices are then pulled
;         up or down to align to the underlying heightmap.
;     2) DynamicBillboard. Examples: flowers or bushes that always face camera.
;         They are placed and non-uniformly scaled.
;         This type of grass behaves like a billboard (or a sprite, in Blitz3D terms).
;         The grass particle always rotates around its Y-axis to face the camera.
;         But if you look at them from the top, you'll obviously see that they are
;         nothing but flat planes that 'look at' camera.
;     3) TemplateFixed. Examples: 3D trees, 3D flowers, 3D bushes.
;         They are placed, uniformly scaled and rotated (over Y-axis) once.
;         After that they can only be animated. This type of grass can participate
;         in collision determination (e.g. objects must collide with trees).
;     4) TemplateLine. Examples: long lines of grass that 'grows along the normal'.
;         They are placed, non-uniformly scaled and rotated (over Y-axis).
;         This type of grass is placed on the terrain in a specific way:
;         when particle 'coordinate' is calculated, the particle middle point is
;         placed at this 'coordinate', and particle is rotated so that its lower corners
;         are positioned on the surface of the underlying heightmap.
;         This type of grass can be 3D and can have animation.
;     5) TemplateBillboard. Examples: flowers or bushes that always face camera.
;         They are placed and non-uniformly scaled.
;         This type of grass behaves like a billboard (or a sprite, in Blitz3D terms).
;         The grass particle always rotates around its Y-axis to face the camera.
;         This type of grass can be 3D and can have animation.
;     + Note: 'dynamic' particles can probably have LOD to look more believable
;         at the close distances. Maybe, to achieve greater LOD, flat dynamic
;         particles should be replaced by 3D objects at close distances.

; ============================================================ ;

; The algorithm I made is 2D and it uses Blitz3D 'terrains' (heightmaps) because
; of the faster height determination (in comparison to line-polygon intersection).
; Maybe it is possible to invent a method that will distribute particles over
; arbitrary polygon surface (which cannot be represented by a heightmap),
; but for me 2D/heightmap is enough :-)

; In the chosen algorithm I have to deal with some issues:
; * Rendering speed. Modern GPUs render mesh with lots of polygons faster
;   than many separate meshes with little amount of polygons. Usually grass
;   particles have the same material/texture for all particles, so, the most
;   efficient way to render them is to clear all polygons from mesh and then
;   dynamically fill it with the visible polygons (as visibility is determined for
;   each particle independently).
; * In the case of grass that has to be rendered with transparency, all polygons
;   must be sorted by distance from camera. Blitz3D automatically does such
;   sorting for separate entities, but to achieve this inside a mesh I have to do the
;   sorting manually.
; * If there are several different 'grass emitters' with transparent particles,
;   each 'grass mesh' will be rendered in a separate render loop, thus leading
;   to the effect when all particles of some kind are rendered on the top
;   of the particles of other kind. To avoid this, I need to place all these different
;   particles inside of a single mesh surface. To make this possible, I need
;   to assemble all related grass textures into the one large texture (or it has to be
;   created manually in off-line image editor) and all grass particles of different
;   kinds will then use different parts of that single texture (by assigning the
;   corresponding vertex texture coordinates).

; In Blitz3D implementation it is impossible to make some optimizations that are
; possible in C++. I've used a simple version of binary tree to sort particles
; by distance from camera, but perhaps this should be rewritten to using a
; red-black tree container.

; Suming up, there are many optimizations and improvements that are still possible
; to apply to this 'grass-building' technique. To get the maximum optimization,
; it should be rewritten in C++, but now I have no possibility (and motivation)
; to do this. My deep respect to the person who would do such thing :-)

; ============================================================ ;

; Grass Types:
; Has template entity. No specific alignment
Const GrassType_Template_Fixed = 0
; Has template entity and stretches along the terrain
Const GrassType_Template_Line = 1
; Has template entity and faces camera
Const GrassType_Template_Billboard = 2
; Dynamically builded and stretched along the terrain
Const GrassType_Dynamic_Line = 3
; Dynamically builded and faces camera
Const GrassType_Dynamic_Billboard = 4

; 'Template' grass types can have animation
; that simulates swinging in the wind.
; Only one animation is supported.

; Grass emitter can check several terrains
; (for the case of dynamically loaded level)

Type TGrassTemplate
	Field GrassType
	Field Width#, Height#, Depth#
	Field dWidth#, dHeight#, dDepth#
	Field TrembleCoef#
	Field WindTau#
	Field WindAlignCoef#
	Field WindParamsDistribution# ; defaults to [0.5, 2]
	Field TemplateEntity
	Field Srf, AlphaSrf
	Field MaxFrames
	; for convenience
	Field bTemplate
	Field bLine
	Field bBillboard
	
	Field MNode.TGrassNode
	Field RNode.TGrassNode
End Type

Type TGrassEmitter
	Field Camera
	Field MaxCount
	Field Near#, Far#
	; subtraction: Rnd(0, CreationTolerance)*GrassEmitter\FadeNear
	Field CreationTolerance# ; measured in % (0.0 - 1.0)
	Field Mesh, AlphaMesh
	
	Field TemplatesCount
	Field Templates.TGrassTemplate[20]
	
	Field HeightmapsCount
	Field Heightmaps[20]
	
	Field Count
	Field VisibleCount
	
	; these are used to create more objects
	; in the direction of lesser concentration
	Field SumX#
	Field SumZ#
	
	Field bPointIsForbidden
	
	; used to determine visibility of particles
	; (intersection of view cone+sphere with particle bounding sphere)
	Field CameraX#
	Field CameraY#
	Field CameraZ#
	Field Camera_AspectRatio#
	Field Camera_Zoom#
	Field Camera_ConeTg#
	Field CameraPiv
	
	; clip planes' normals
	Field clx#, cly#, clz# ; left
	Field crx#, cry#, crz# ; right
	Field ctx#, cty#, ctz# ; top
	Field cbx#, cby#, cbz# ; bottom
	
	Field phase#, phaseCos#
End Type

; type of grass particle objects
Type TGrass
	Field Emitter.TGrassEmitter
	Field Template.TGrassTemplate
	
	Field piv
	Field ent ; 3D object that represents the particle
	Field center ; 'empty' (invisible) object placed at the middle of particle
	Field radius# ; radius of bounding sphere
	Field bVisible
	
	; bVirtual: true if current particle do not exist in 3D world
	; (but I must keep such particles to have average density
	; close to the desired density)
	Field bVirtual
	
	; parameters for 'global wind animation'
	;Field MaxFrames
	Field WindX#, WindZ#
	Field phase#, AnimPhase#
	Field TrembleCoef#
	Field WindTau#, WindTau1#, WindTau2#
	Field WindAlignCoef#
	
	; values for current object
	Field Near#, Far#, FadeSize#, FadeConst#
	
	; values for current object
	Field Width#, Height#, Depth#
	
	; vectors that will define vertex positions
	; in dynamically built particle
	Field vx0#, vy0#, vz0#
	Field vx1#, vy1#, vz1#
	
	; link to the corresponding binary tree node
	Field Node.TGrassNode
End Type

; objects of this type are used to dynamically build grass mesh
Type TGrassNode
	Field x0#, y0#, z0#
	Field x1#, y1#, z1#
	Field dx#, dy#, dz#
	Field alpha#
	Field depth#
	Field LNode.TGrassNode
	Field RNode.TGrassNode
	Field PNode.TGrassNode
	Field PrevNode.TGrassNode
	Field NextNode.TGrassNode
End Type

; global wind speed
Global WindX#, WindZ#

; private
; adds grass particle to the binary tree container (sorted by distance from camera)
Function AddGrassNode(Grass.TGrass)
	Local pNode.TGrassNode = Grass\Template\MNode
	Local xNode.TGrassNode = Grass\Node
	
	xNode\LNode = Null
	xNode\RNode = Null
	xNode\PNode = Null
	xNode\PrevNode = Null
	xNode\NextNode = Null
	
	If pNode = Null
		Grass\Template\MNode = xNode
		Grass\Template\RNode = xNode
		Return
	EndIf
	
	;Stop
	
	Repeat
		If xNode\depth < pNode\depth
			If pNode\LNode = Null
				pNode\LNode = xNode
				xNode\PNode = pNode
				
				xNode\PrevNode = pNode\PrevNode
				If xNode\PrevNode <> Null
					xNode\PrevNode\NextNode = xNode
				EndIf
				xNode\NextNode = pNode
				pNode\PrevNode = xNode
				
				Exit
			Else
				pNode = pNode\LNode
			EndIf
		Else
			If pNode\RNode = Null
				pNode\RNode = xNode
				xNode\PNode = pNode
				
				xNode\NextNode = pNode\NextNode
				If xNode\NextNode <> Null
					xNode\NextNode\PrevNode = xNode
				EndIf
				xNode\PrevNode = pNode
				pNode\NextNode = xNode
				
				If Grass\Template\RNode = pNode
					Grass\Template\RNode = xNode
				EndIf
				
				Exit
			Else
				pNode = pNode\RNode
			EndIf
		EndIf
	Forever
End Function

; private
; builds grass mesh
Function BuildGrassNode(GrassTemplate.TGrassTemplate)
	Local Node.TGrassNode = GrassTemplate\RNode
	Local wSrf = GrassTemplate\AlphaSrf
	Local v0, v1, v2, v3
	
	While Node <> Null
		v0 = AddVertex(wSrf, Node\x0, Node\y0, Node\z0, 0, 1)
		VertexColor wSrf, v0, 255, 255, 255, Node\alpha
		v1 = AddVertex(wSrf, Node\x1, Node\y1, Node\z1, 1, 1)
		VertexColor wSrf, v1, 255, 255, 255, Node\alpha
		v2 = AddVertex(wSrf, Node\x0+Node\dx, Node\y0+Node\dy, Node\z0+Node\dz, 0, 0)
		VertexColor wSrf, v2, 255, 255, 255, Node\alpha
		v3 = AddVertex(wSrf, Node\x1+Node\dx, Node\y1+Node\dy, Node\z1+Node\dz, 1, 0)
		VertexColor wSrf, v3, 255, 255, 255, Node\alpha
		AddTriangle wSrf, v0, v2, v1
		AddTriangle wSrf, v2, v1, v3
		Node = Node\PrevNode
	Wend
End Function

; creates new grass emitter
; MaxCount -- maximum number of particles that grass emitter can create
; Near -- the minimal distance at which any grass particle must be fully opaque
; Far -- the maximal distance at which any grass particle must be fully transparent.
; CreationTolerance -- this parameter randomly decreases Near and Far distance values
;   for each generated particle. CreationTolerance = 1.0 usually gives smoother distribution.
; Note: if distance between the camera and the particle is greater than particle's Far value,
;   particle would be deleted.
Function CreateGrassEmitter.TGrassEmitter(Camera, MaxCount, Near#, Far#, CreationTolerance# = 1.0)
	If Near > Far Then Near = Far
	
	GrassEmitter.TGrassEmitter = New TGrassEmitter
	GrassEmitter\Camera = Camera
	GrassEmitter\MaxCount = MaxCount
	GrassEmitter\Near = Near
	GrassEmitter\Far = Far
	GrassEmitter\CreationTolerance = CreationTolerance
	
	; calculating view cone (faster to check than view frustrum)
	GrassEmitter\Camera_AspectRatio = 0.75
	GrassEmitter\Camera_Zoom = 1
	GrassEmitter\Camera_ConeTg = Sqr(1+GrassEmitter\Camera_AspectRatio*GrassEmitter\Camera_AspectRatio)/GrassEmitter\Camera_Zoom
	GrassEmitter\CameraPiv = CreatePivot()
	
	; create meshes with the corresponding settings
	GrassEmitter\Mesh = CreateMesh()
	; disable backface culling + fullbright
	EntityFX GrassEmitter\Mesh, 16+1
	GrassEmitter\AlphaMesh = CreateMesh()
	; disable backface culling + fullbright + force alpha-blending + use vertex colors instead of brush color
	EntityFX GrassEmitter\AlphaMesh, 16+1+32+2
	
	Return GrassEmitter
End Function

; Attention!
; TemplateEntity must be fit in (111) box
; -1, 1  1, 1
; -1, 0  1, 0
; For Dynamic templates TemplateObj is a texture!
Function AddGrassTemplate.TGrassTemplate(GrassEmitter.TGrassEmitter, GrassType, TemplateObj, w#, h# = 0, d# = 0, dw# = 0, dh# = 0, dd# = 0)
	Local wBrush
	
	If h = 0 Then h = w
	If d = 0 Then d = w
	
	GrassTemplate.TGrassTemplate = New TGrassTemplate
	GrassTemplate\GrassType = GrassType
	GrassTemplate\Width = w
	GrassTemplate\Height = h
	GrassTemplate\Depth = d
	GrassTemplate\dWidth = dw
	GrassTemplate\dHeight = dh
	GrassTemplate\dDepth = dd
	
	; assign some flags to make some checks easier...
	Select GrassTemplate\GrassType
		Case GrassType_Template_Fixed
			GrassTemplate\bTemplate = True
			GrassTemplate\bLine = False
			GrassTemplate\bBillboard = False
		Case GrassType_Template_Line
			GrassTemplate\bTemplate = True
			GrassTemplate\bLine = True
			GrassTemplate\bBillboard = False
		Case GrassType_Template_Billboard
			GrassTemplate\bTemplate = True
			GrassTemplate\bLine = False
			GrassTemplate\bBillboard = True
		Case GrassType_Dynamic_Line
			GrassTemplate\bTemplate = False
			GrassTemplate\bLine = True
			GrassTemplate\bBillboard = False
		Case GrassType_Dynamic_Billboard
			GrassTemplate\bTemplate = False
			GrassTemplate\bLine = False
			GrassTemplate\bBillboard = True
	End Select
	
	If GrassTemplate\bTemplate
		GrassTemplate\TemplateEntity = TemplateObj
		; get animation length
		If AnimSeq(GrassTemplate\TemplateEntity) <> -1
			GrassTemplate\MaxFrames = AnimLength(GrassTemplate\TemplateEntity)-1
		EndIf
	Else
		wBrush = CreateBrush()
		;BrushFX wBrush, 1+2+16
		BrushTexture wBrush, TemplateObj
		; create mesh surface (polygon set) with the corresponding properties
		GrassTemplate\Srf = CreateSurface(GrassEmitter\Mesh)
		PaintSurface GrassTemplate\Srf, wBrush
		; create mesh surface (polygon set) with the corresponding properties
		GrassTemplate\AlphaSrf = CreateSurface(GrassEmitter\AlphaMesh)
		PaintSurface GrassTemplate\AlphaSrf, wBrush
		FreeBrush wBrush
	EndIf
	
	GrassTemplate\TrembleCoef = 0
	GrassTemplate\WindTau = 0
	GrassTemplate\WindAlignCoef = 0
	GrassTemplate\WindParamsDistribution = 2
	
	; add template to the emitter
	GrassEmitter\Templates[GrassEmitter\TemplatesCount] = GrassTemplate
	GrassEmitter\TemplatesCount = GrassEmitter\TemplatesCount + 1
	
	Return GrassTemplate
End Function

; adds grass heightmap to the emitter
; heigtmap here is Blitz3D terrain entity
Function AddGrassHeightmap(GrassEmitter.TGrassEmitter, Heightmap)
	GrassEmitter\Heightmaps[GrassEmitter\HeightmapsCount] = Heightmap
	GrassEmitter\HeightmapsCount = GrassEmitter\HeightmapsCount + 1
End Function

; set some camera parameters for view cone calculation
Function GrassEmitterCameraSetup(GrassEmitter.TGrassEmitter, Camera_Zoom#, Camera_AspectRatio# = 0.75)
	GrassEmitter\Camera_AspectRatio = Camera_AspectRatio
	GrassEmitter\Camera_Zoom = Camera_Zoom
	GrassEmitter\Camera_ConeTg = Sqr(1+GrassEmitter\Camera_AspectRatio*GrassEmitter\Camera_AspectRatio)/GrassEmitter\Camera_Zoom
End Function

; set some global wind parameters
Function GrassTemplateWindSetup(GrassTemplate.TGrassTemplate, WindAlignCoef#, WindTau# = 20, TrembleCoef# = 0, WindParamsDistribution# = 2)
	If WindTau < 0 Then WindTau = 0
	
	GrassTemplate\TrembleCoef = TrembleCoef
	GrassTemplate\WindTau = WindTau
	GrassTemplate\WindAlignCoef = WindAlignCoef
	GrassTemplate\WindParamsDistribution = WindParamsDistribution
End Function

; the main to be called to update all grass
Function UpdateGrass()
	Local GrassEmitter.TGrassEmitter
	Local GrassTemplate.TGrassTemplate
	Local GrassObj.TGrass
	
	Local nSurfs
	Local WindSz#
	
	For GrassEmitter = Each TGrassEmitter
		bDistributeFrom0 = (GrassEmitter\Count = 0)
		
		GrassEmitter\SumX = 0
		GrassEmitter\SumZ = 0
		GrassEmitter\VisibleCount = 0
		
		GrassEmitter\CameraX = EntityX(GrassEmitter\Camera, True)
		GrassEmitter\CameraY = EntityY(GrassEmitter\Camera, True)
		GrassEmitter\CameraZ = EntityZ(GrassEmitter\Camera, True)
		
		; vx = a
		; vy = b
		; =>
		; nx = -b
		; ny = a
		
		; get directions of camera view corners (?)
		TFormNormal 1.0, 0.0, -GrassEmitter\Camera_Zoom, GrassEmitter\Camera, 0
		GrassEmitter\clx = TFormedX()
		GrassEmitter\cly = TFormedY()
		GrassEmitter\clz = TFormedZ()
		TFormNormal -1.0, 0.0, -GrassEmitter\Camera_Zoom, GrassEmitter\Camera, 0
		GrassEmitter\crx = TFormedX()
		GrassEmitter\cry = TFormedY()
		GrassEmitter\crz = TFormedZ()
		TFormNormal 0.0, GrassEmitter\Camera_Zoom, -GrassEmitter\Camera_AspectRatio, GrassEmitter\Camera, 0
		GrassEmitter\ctx = TFormedX()
		GrassEmitter\cty = TFormedY()
		GrassEmitter\ctz = TFormedZ()
		TFormNormal 0.0, GrassEmitter\Camera_Zoom, GrassEmitter\Camera_AspectRatio, GrassEmitter\Camera, 0
		GrassEmitter\cbx = -TFormedX()
		GrassEmitter\cby = -TFormedY()
		GrassEmitter\cbz = -TFormedZ()
		
		; position and rotate invisible entity to match camera's position and rotation to later perform some calculations
		PositionEntity GrassEmitter\CameraPiv, GrassEmitter\CameraX, GrassEmitter\CameraY, GrassEmitter\CameraZ, True
		RotateEntity GrassEmitter\CameraPiv, EntityPitch(GrassEmitter\Camera, True), EntityYaw(GrassEmitter\Camera, True), EntityRoll(GrassEmitter\Camera, True), True
		
		; some wind calculations...
		WindSz = Sqr(WindX*WindX + WindZ*WindZ)
		GrassEmitter\phase = (GrassEmitter\phase + 0.5*WindSz/(1.0+WindSz)) Mod 1.0
		GrassEmitter\phaseCos = Cos(GrassEmitter\phase*360.0)
		
		; clear surfaces...
		nSurfs = CountSurfaces(GrassEmitter\Mesh)
		For i = 1 To nSurfs
			ClearSurface GetSurface(GrassEmitter\Mesh, i)
			ClearSurface GetSurface(GrassEmitter\AlphaMesh, i)
		Next
		
		; generate grass particles if there is such possibility
		If GrassEmitter\TemplatesCount <> 0
			For i = GrassEmitter\Count+1 To GrassEmitter\MaxCount
				UpdateGrass_CreateNewObject(GrassEmitter, bDistributeFrom0)
			Next
		EndIf
	Next
	
	; clear depth-sort trees
	For GrassTemplate = Each TGrassTemplate
		GrassTemplate\MNode = Null
		GrassTemplate\RNode = Null
	Next
	
	; update each grass particle
	For GrassObj = Each TGrass
		UpdateGrass_UpdateObject(GrassObj.TGrass)	
	Next
	
	; build all dynamic grass meshes
	For GrassTemplate = Each TGrassTemplate
		BuildGrassNode(GrassTemplate)
	Next
End Function

; private
; gets height of the point of terrain that lies under the specified point
Function GrassEmitter_GetHeight#(GrassEmitter.TGrassEmitter, x#, y#, z#)
	Local i, max, tx#, ty#, tz#, res#
	Local ty0#, ty1#, ty2#, ty3#
	
	GrassEmitter\bPointIsForbidden = True
	
	max = GrassEmitter\HeightmapsCount-1
	For i = 0 To max
		terr = GrassEmitter\Heightmaps[i]
		tsz = TerrainSize(terr)
		
		TFormPoint x, y, z, 0, terr
		tx = TFormedX()
		tz = TFormedZ()
		
		; I need to increase height only if
		; point is inside of terrain
		If (tx < 0) Or (tx > tsz) Or (tz < 0) Or (tz > tsz)
		Else
			; for Blitz terrains
			ix = Floor(tx) Mod tsz
			iz = Floor(tz) Mod tsz
			ty0 = TerrainHeight(terr, ix, iz)
			If ty0 > 0
				ty1 = TerrainHeight(terr, ix+1, iz)
				If ty1 > 0
					ty2 = TerrainHeight(terr, ix, iz+1)
					If ty2 > 0
						ty3 = TerrainHeight(terr, ix+1, iz+1)
						If ty3 > 0
							tx = tx Mod 1
							tz = tz Mod 1
							ty0 = ty0*(1-tx) + ty1*tx
							ty2 = ty2*(1-tx) + ty3*tx
							ty = ty0*(1-tz) + ty2*tz
							TFormPoint ix, ty, iz, terr, 0
							ty = TFormedY()
							If (ty > res) Or (i = 0)
								GrassEmitter\bPointIsForbidden = False
								res = ty
							EndIf
						EndIf
					EndIf
				EndIf
			EndIf
		EndIf
	Next
	
	Return res
End Function

; private
; bDistributeFrom0 = true when no particle exists. In that case particles are created
; at the distance starting from 0 rather than from particle's Far.
Function UpdateGrass_CreateNewObject(GrassEmitter.TGrassEmitter, bDistributeFrom0)
	Local GrassTemplate.TGrassTemplate
	Local GrassObj.TGrass
	
	Local dx#, dy#, dz#
	Local ga#, gr#, gx#, gz#
	Local rad#
	Local vx0#, vy0#, vz0#
	Local vx1#, vy1#, vz1#
	
	Local wpd#, wpd_inv#, dNear#
	Local w05#, h05#, d05#, bPointIsForbidden
	
	; choose random template...
	GrassTemplate = GrassEmitter\Templates[Rand(0, GrassEmitter\TemplatesCount - 1)]
	
	wpd = GrassTemplate\WindParamsDistribution
	wpd_inv = 1.0/wpd
	;dNear = Rnd(0, GrassEmitter\CreationTolerance)*GrassEmitter\Near
	dNear = Rnd(0, GrassEmitter\CreationTolerance)
;If dNear = 0 Then Stop
	; move probability closer to the observer
	;dNear = (1.0-dNear*dNear)*GrassEmitter\Near
	dNear = Sqr(dNear)*GrassEmitter\Near
	
	; create new object...
	GrassObj = New TGrass
	GrassObj\Emitter = GrassEmitter
	GrassObj\Template = GrassTemplate
	
	; initialize params...
	GrassObj\Width = GrassTemplate\Width + Rnd(-1, 1)*GrassTemplate\dWidth
	GrassObj\Height = GrassTemplate\Height + Rnd(-1, 1)*GrassTemplate\dHeight
	GrassObj\Depth = GrassTemplate\Depth + Rnd(-1, 1)*GrassTemplate\dDepth
	
	GrassObj\Near = GrassEmitter\Near - dNear
	GrassObj\Far = GrassEmitter\Far - dNear
	GrassObj\FadeSize = GrassObj\Far - GrassObj\Near
	GrassObj\FadeConst = 1.0 + GrassObj\Near/GrassObj\FadeSize
	
	; calc some wind parameters...
	GrassObj\TrembleCoef = Rnd(wpd_inv, wpd)*GrassTemplate\TrembleCoef
	GrassObj\WindAlignCoef = Rnd(wpd_inv, wpd)*GrassTemplate\WindAlignCoef
	GrassObj\WindTau = Rnd(wpd_inv, wpd)*GrassTemplate\WindTau
	GrassObj\WindTau1 = 1+GrassObj\WindTau
	GrassObj\WindTau2 = GrassObj\WindTau/GrassObj\WindTau1
	GrassObj\phase = Rnd(0, 1)
	GrassObj\AnimPhase = Rnd(GrassTemplate\MaxFrames)
	
	; calculate position
	gx = Rnd(-1, 1)*GrassEmitter\Far
	gz = Rnd(-1, 1)*GrassEmitter\Far
	If (GrassEmitter\Count <> 0)
		; average of semicircle is ~ 1/2
		gx = gx-4*GrassEmitter\SumX/GrassEmitter\Count
		gz = gz-4*GrassEmitter\SumZ/GrassEmitter\Count
	EndIf
	If bDistributeFrom0
		gr = Rnd(0, GrassObj\Far)
	Else
		gr = GrassObj\Far
	EndIf
	ga = ATan2(gx, gz)
	dx = Sin(ga)*gr
	dz = Cos(ga)*gr
	gx = GrassEmitter\CameraX + dx
	gz = GrassEmitter\CameraZ + dz
	
	; Only here!
	; calc statisics of the average particles position
	GrassEmitter\SumX = GrassEmitter\SumX+dx
	GrassEmitter\SumZ = GrassEmitter\SumZ+dz
	GrassEmitter\Count = GrassEmitter\Count + 1
	
	w05 = GrassObj\Width*0.5
	h05 = GrassObj\Height
	d05 = GrassObj\Depth*0.5
	
	If GrassTemplate\bLine
		ga = Rnd(360)
		GrassObj\vx0 = gx+Sin(ga)*w05
		GrassObj\vz0 = gz+Cos(ga)*w05
		GrassObj\vy0 = GrassEmitter_GetHeight(GrassEmitter, GrassObj\vx0, 0, GrassObj\vz0)
		bPointIsForbidden = bPointIsForbidden Or GrassEmitter\bPointIsForbidden
		
		ga = ga+180
		GrassObj\vx1 = gx+Sin(ga)*w05
		GrassObj\vz1 = gz+Cos(ga)*w05
		GrassObj\vy1 = GrassEmitter_GetHeight(GrassEmitter, GrassObj\vx1, 0, GrassObj\vz1)
		bPointIsForbidden = bPointIsForbidden Or GrassEmitter\bPointIsForbidden
		
		dx = GrassObj\vx1-GrassObj\vx0
		dy = GrassObj\vy1-GrassObj\vy0
		dz = GrassObj\vz1-GrassObj\vz0
		
		GrassObj\radius = Sqr(0.25*(dx*dx+dy*dy+dz*dz) + h05*h05 + d05*d05)
		
		; create center pivot at the middle
		GrassObj\center = CreatePivot()
		PositionEntity GrassObj\center, (GrassObj\vx0+GrassObj\vx1)*0.5, (GrassObj\vy0+GrassObj\vy1)*0.5, (GrassObj\vz0+GrassObj\vz1)*0.5
		AlignToVector GrassObj\center, dx, dy, dz, 1
		TurnEntity GrassObj\center, -EntityPitch(GrassObj\center, False), 0, 0, False
	Else
		GrassObj\vx0 = gx
		GrassObj\vz0 = gz
		GrassObj\vy0 = GrassEmitter_GetHeight(GrassEmitter, GrassObj\vx0, 0, GrassObj\vz0)
		bPointIsForbidden = bPointIsForbidden Or GrassEmitter\bPointIsForbidden
		
		GrassObj\vx1 = GrassObj\vx0
		GrassObj\vz1 = GrassObj\vz0
		GrassObj\vy1 = GrassObj\vy0
		
		GrassObj\radius = Sqr(w05*w05 + h05*h05 + d05*d05)
		
		; create center pivot at the middle
		GrassObj\center = CreatePivot()
		PositionEntity GrassObj\center, GrassObj\vx0, GrassObj\vy0, GrassObj\vz0
	EndIf
	
	If Not bPointIsForbidden
		If GrassTemplate\bTemplate
			GrassObj\piv = CreatePivot(GrassObj\center)
			If GrassTemplate\bLine
				w05 = Sqr(dx*dx+dy*dy+dz*dz)*0.5
				h05 = GrassObj\Height
				d05 = GrassObj\Depth*0.5
			Else
				w05 = GrassObj\Width*0.5
				h05 = GrassObj\Height
				d05 = GrassObj\Depth*0.5
			EndIf
			If (d05 = 0) Then d05 = 1
			ScaleEntity GrassObj\piv, w05, h05, d05
			
			GrassObj\ent = CopyEntity(GrassTemplate\TemplateEntity, GrassObj\piv)
			
			; It is possible that decrease of grass density with distance
			; will not be very noticeable
			;EntityAutoFade GrassObj\ent, GrassObj\Near, GrassObj\Far
		Else
			GrassObj\Node = New TGrassNode
		EndIf
	Else
		GrassObj\bVirtual = True
	EndIf
	GrassObj\bVisible = True
End Function

; private
; process grass particle
Function UpdateGrass_UpdateObject(GrassObj.TGrass)
	Local GrassEmitter.TGrassEmitter
	Local GrassTemplate.TGrassTemplate
	
	Local dx#, dy#, dz#
	Local cdx#, cdy#, cdz#
	Local rad#, gdist#
	Local wpc#, alpha#
	Local WindSz#
	Local v0, v1, v2, v3
	Local wnx#, wny#, wnz#, wnsz#
	
	GrassEmitter = GrassObj\Emitter
	
	cdx = EntityX(GrassObj\center, True) - GrassEmitter\CameraX
	cdz = EntityZ(GrassObj\center, True) - GrassEmitter\CameraZ
	rad = Sqr(cdx*cdx+cdz*cdz)
	
	; distance check
	If rad > GrassObj\Far
		GrassEmitter\Count = GrassEmitter\Count - 1
		FreeEntity GrassObj\center
		If GrassObj\Node <> Null
			Delete GrassObj\Node
		EndIf
		Delete GrassObj
		Return
	EndIf
	
	GrassEmitter\SumX = GrassEmitter\SumX+cdx
	GrassEmitter\SumZ = GrassEmitter\SumZ+cdz
	
	; virtual grass particles are also necessary to keep constant particle distributing
	; when observer moves to the place where particles in some area cannot be created.
	If GrassObj\bVirtual Then Return
	
	GrassTemplate = GrassObj\Template
	
	; visibility check (intersection of view cone+sphere with bounding sphere)
	cdy = EntityY(GrassObj\center, True) - GrassEmitter\CameraY
	
	bVisible = (GrassEmitter\cbx*cdx + GrassEmitter\cby*cdy + GrassEmitter\cbz*cdz) < GrassObj\radius
	If bVisible
		bVisible = (GrassEmitter\clx*cdx + GrassEmitter\cly*cdy + GrassEmitter\clz*cdz) < GrassObj\radius
		If bVisible
			bVisible = (GrassEmitter\crx*cdx + GrassEmitter\cry*cdy + GrassEmitter\crz*cdz) < GrassObj\radius
			If bVisible
				bVisible = (GrassEmitter\ctx*cdx + GrassEmitter\cty*cdy + GrassEmitter\ctz*cdz) < GrassObj\radius
			EndIf
		EndIf
	EndIf
	
	; if visibility state didn't changed from Invisible to Visible, do nothing
	
	If Not bVisible
		If GrassTemplate\bTemplate And GrassObj\bVisible
			GrassObj\bVisible = False
			HideEntity GrassObj\ent
		EndIf
		Return
	EndIf
	
	GrassEmitter\VisibleCount = GrassEmitter\VisibleCount + 1
	
	If GrassTemplate\bTemplate And (Not GrassObj\bVisible)
		GrassObj\bVisible = True
		ShowEntity GrassObj\ent
	EndIf
	
	gdist = rad
	;gdist = EntityDistance(GrassObj\center, GrassEmitter\Camera)
	;TFormPoint 0, 0, 0, GrassObj\center, GrassEmitter\Camera
	;gdist = TFormedZ()
	
	alpha = GrassObj\FadeConst - rad/GrassObj\FadeSize
	
	If GrassTemplate\bTemplate
		wpc = Rnd(-1.0, 1.0)*GrassEmitter\phaseCos*GrassObj\TrembleCoef + 1.0/GrassObj\WindTau1
		GrassObj\WindX = (GrassObj\WindX*GrassObj\WindTau2 + WindX*wpc)*GrassObj\WindAlignCoef
		GrassObj\WindZ = (GrassObj\WindZ*GrassObj\WindTau2 + WindZ*wpc)*GrassObj\WindAlignCoef
		WindSz = Sqr(GrassObj\WindX*GrassObj\WindX + GrassObj\WindZ*GrassObj\WindZ)
		
		EntityAlpha GrassObj\ent, alpha
		If GrassTemplate\MaxFrames
			GrassObj\AnimPhase = (GrassObj\AnimPhase + Sqr(WindSz)*GrassObj\TrembleCoef*0.5) Mod 1
			; ping-pong
			If GrassObj\AnimPhase <= 0.5
				SetAnimTime GrassObj\ent, GrassTemplate\MaxFrames*GrassObj\AnimPhase*2
			Else
				SetAnimTime GrassObj\ent, GrassTemplate\MaxFrames*(1-GrassObj\AnimPhase)*2
			EndIf
		EndIf
	Else
		wpc = Rnd(-1.0, 1.0)*GrassEmitter\phaseCos*GrassObj\TrembleCoef + 1.0/GrassObj\WindTau1
		GrassObj\WindX = (GrassObj\WindX*GrassObj\WindTau2 + WindX*wpc)*GrassObj\WindAlignCoef
		GrassObj\WindZ = (GrassObj\WindZ*GrassObj\WindTau2 + WindZ*wpc)*GrassObj\WindAlignCoef
		WindSz = Sqr(GrassObj\WindX*GrassObj\WindX + 1.0 + GrassObj\WindZ*GrassObj\WindZ)
		
		wnsz = GrassObj\Height/WindSz
		dx = GrassObj\WindX*wnsz
		dy = wnsz
		dz = GrassObj\WindZ*wnsz
	EndIf
	
	Select GrassTemplate\GrassType
		Case GrassType_Template_Fixed
			AlignToVector GrassObj\center, GrassObj\WindX, 1.0/GrassObj\WindAlignCoef, GrassObj\WindZ, 2
		Case GrassType_Template_Line
			TFormNormal GrassObj\WindX, 0, GrassObj\WindZ, 0, GrassObj\center
			TurnEntity GrassObj\center, ATan(TFormedZ()*WindSz)*GrassObj\WindAlignCoef-EntityPitch(GrassObj\center, False), 0, 0, False
		Case GrassType_Template_Billboard
			AlignToVector GrassObj\center, GrassObj\WindX, 1.0/GrassObj\WindAlignCoef, GrassObj\WindZ, 2
			
			TFormPoint GrassEmitter\CameraX, GrassEmitter\CameraY, GrassEmitter\CameraZ, 0, GrassObj\center
			TurnEntity GrassObj\center, 0, -ATan2(TFormedX(), TFormedZ()), 0, False
		Case GrassType_Dynamic_Line
			GrassObj\Node\x0 = GrassObj\vx0
			GrassObj\Node\y0 = GrassObj\vy0
			GrassObj\Node\z0 = GrassObj\vz0
			GrassObj\Node\x1 = GrassObj\vx1
			GrassObj\Node\y1 = GrassObj\vy1
			GrassObj\Node\z1 = GrassObj\vz1
			GrassObj\Node\dx = dx
			GrassObj\Node\dy = dy
			GrassObj\Node\dz = dz
			GrassObj\Node\alpha = alpha
			GrassObj\Node\depth = gdist
			
			AddGrassNode(GrassObj)
		Case GrassType_Dynamic_Billboard
			; vector cross product
			wnx = dy*cdz - dz*cdy
			wny = dz*cdx - dx*cdz
			wnz = dx*cdy - dy*cdx
			wnsz = wnx*wnx+wny*wny+wnz*wnz
			If (wnsz <> 0)
				wnsz = (GrassObj\Width*0.5)/Sqr(wnsz)
				wnx = wnx*wnsz
				wny = wny*wnsz
				wnz = wnz*wnsz
				
				GrassObj\Node\x0 = GrassObj\vx0-wnx
				GrassObj\Node\y0 = GrassObj\vy0-wny
				GrassObj\Node\z0 = GrassObj\vz0-wnz
				GrassObj\Node\x1 = GrassObj\vx0+wnx
				GrassObj\Node\y1 = GrassObj\vy0+wny
				GrassObj\Node\z1 = GrassObj\vz0+wnz
				GrassObj\Node\dx = dx
				GrassObj\Node\dy = dy
				GrassObj\Node\dz = dz
				GrassObj\Node\alpha = alpha
				GrassObj\Node\depth = gdist
				
				AddGrassNode(GrassObj)
			EndIf
	End Select
End Function

;=================================================;

; EXAMPLE

;=================================================;

AppTitle "DynamicGrass.Heightmap v0.5 example"

GfxMode = 1

bUseTemplates = False

input_result$ = Input("Use windowed mode? (Y/N)")
If Lower(input_result) = "y"
	GfxMode = 2
EndIf

Print ""
Print "Use templates instead of "
input_result$ = Input("dynamically built polygons? (Y/N)")
If Lower(input_result) = "y"
	bUseTemplates = True
EndIf

; set graphics mode
Graphics3D 800, 600, 32, GfxMode

; set bigger font
fnt = LoadFont("Courier", 20)
SetFont fnt

; establish collisions between moveable objects and enviroment
CollType_Moveable = 1
CollType_Static = 2
Collisions CollType_Moveable, CollType_Static, 2, 2

cam_range# = 5000

; create directional light
lit = CreateLight(1)
TurnEntity lit, 45, 0, 0

; create floor plane
plane = CreatePlane()
EntityType plane, CollType_Static

; create heightmap
terr = LoadTerrain("Heightmap.jpg")
TerrainDetail terr, 8000
EntityType terr, CollType_Static

; position and scale terrain
tsz# = TerrainSize(terr)
ntsz# = 10000
ntszh# = 200
PositionEntity terr, -ntsz*0.5, -1, -ntsz*0.5
ScaleEntity terr, ntsz/tsz, ntszh, ntsz/tsz
; scale plane too to make texture look on them more similar
ScaleEntity plane, ntsz/tsz, ntszh, ntsz/tsz

; apply texture to plane and terrain
terrain_tex = LoadTexture("GrassyTerrain.jpg")
ScaleTexture terrain_tex, 120*tsz/ntsz, 120*tsz/ntsz
EntityTexture terr, terrain_tex
EntityTexture plane, terrain_tex

; create camera
cam = CreateCamera(CamPivot)
PositionEntity cam, 0, TerrainY(terr, 0, 0, 0)+10, 0
CameraRange cam, 1, cam_range
CameraClsColor cam, 222, 252, 255
CameraFogColor cam, 222, 252, 255
CameraFogMode cam, True
CameraFogRange cam, cam_range*0.5, cam_range
EntityType cam, CollType_Moveable
EntityRadius cam, 2

; Some notes:
; * some problem is with mipmapping -- at high distances grass partcles turn into
;   semi-transparent rectangles, this is not a nice thing to look at.
; * Do not use sprites! (for entity-based particles)
;   If I create 20000 sprites, each of them hidden (except at least one),
;   RenderWorld will fail anyway. There can be no more than 16000
;   sprites simultaneously (or their copies).
;   Meshes do not produce such errors.

; NOTE: visual-appealing grass depends to a large extent on good textures!

Local ParticleTemplate.TGrassTemplate

Grass_Height# = 2

; define width and height of grass particles
grsw# = Grass_Height*4
grsh# = Grass_Height
flww# = Grass_Height/4
flwh# = Grass_Height*1.25

; load flower particle texture with alpha-channel, mipmapping and U,V-clamping
FlowerParticleTex = LoadTexture("FlowerParticle.png", 2+8+32)

FlowerParticleEntity = CreateQuad(-1, 0, 1, 1)
EntityTexture FlowerParticleEntity, FlowerParticleTex
; make entity fullbright (or it will look terrible)
; also disable backface culling (it is important!)
EntityFX FlowerParticleEntity, 1+16

; load grass particle texture with alpha-channel, mipmapping and V-clamping
GrassParticleTex = LoadTexture("GrassParticle.png", 2+8+32)
ScaleTexture GrassParticleTex, 0.25, 1

GrassParticleEntity = CreateQuad(-1, 0, 1, 1)
EntityTexture GrassParticleEntity, GrassParticleTex
; make entity fullbright (or it will look terrible)
; also disable backface culling (it is important!)
EntityFX GrassParticleEntity, 1+16

emitter_range# = 200
; create 'grass emitter' with max num of particles = 5000
GrassEmitter.TGrassEmitter = CreateGrassEmitter(cam, 5000, emitter_range-10, emitter_range, 1.0)
; tell 'grass emitter' to use our heightmap
AddGrassHeightmap(GrassEmitter, terr)
; tell 'grass emitter' to create aligned-to-heightmap dynamically built particles
; with the corresponding texture (GrassParticleTex) and invariable width and height.
If Not bUseTemplates
	ParticleTemplate = AddGrassTemplate(GrassEmitter, GrassType_Dynamic_Line, GrassParticleTex, grsw, grsh, 0, 0, 0, 0)
Else
	ParticleTemplate = AddGrassTemplate(GrassEmitter, GrassType_Template_Line, GrassParticleEntity, grsw, grsh, 0, 0, 0, 0)
EndIf
; set up wind-aligning parameters
GrassTemplateWindSetup(ParticleTemplate, 0.5, 20, 0.05, 2)

emitter_range# = 50
; create secondary 'grass emitter' with lower number of particles and closer range
; to make more dense grass near the observer
GrassEmitter.TGrassEmitter = CreateGrassEmitter(cam, 1000, emitter_range-10, emitter_range, 1.0)
; tell 'grass emitter' to use our heightmap
AddGrassHeightmap(GrassEmitter, terr)
; tell 'grass emitter' to create aligned-to-heightmap dynamically built particles
; with the corresponding texture (GrassParticleTex) and invariable width and height.
If Not bUseTemplates
	ParticleTemplate = AddGrassTemplate(GrassEmitter, GrassType_Dynamic_Line, GrassParticleTex, grsw, grsh, 0, 0, 0, 0)
Else
	ParticleTemplate = AddGrassTemplate(GrassEmitter, GrassType_Template_Line, GrassParticleEntity, grsw, grsh, 0, 0, 0, 0)
EndIf
; set up wind-aligning parameters
GrassTemplateWindSetup(ParticleTemplate, 0.5, 20, 0.05, 2)

emitter_range# = 500
;grass_range# = 2500
; create 'grass emitter' that will generate flowers
FlowerEmitter.TGrassEmitter = CreateGrassEmitter(cam, 1000, emitter_range*0.85, emitter_range, 0.5)
; tell 'grass emitter' to use our heightmap
AddGrassHeightmap(FlowerEmitter, terr)
; tell 'grass emitter' to create billboard-style dynamically built particles
; with the corresponding texture (FlowerParticleTex) and variable width and height.
If Not bUseTemplates
	ParticleTemplate = AddGrassTemplate(FlowerEmitter, GrassType_Dynamic_Billboard, FlowerParticleTex, flww, flwh, 0, flww/25, flwh/20, 0)
Else
	ParticleTemplate = AddGrassTemplate(FlowerEmitter, GrassType_Template_Billboard, FlowerParticleEntity, flww, flwh, 0, flww/25, flwh/20, 0)
EndIf
; set up wind-aligning parameters
GrassTemplateWindSetup(ParticleTemplate, 0.5, 20, 0.05, 2)

WindXRnd.TSmoothRnd = CreateSmoothRnd(1.0, 5.0)
WindZRnd.TSmoothRnd = CreateSmoothRnd(1.0, 5.0)

WindDXRnd.TSmoothRnd = CreateSmoothRnd(0.125, 0.5)
WindDZRnd.TSmoothRnd = CreateSmoothRnd(0.125, 0.5)

WindForce# = 0.3
WindPowerFreq# = 1

bDrawStatistics = True

bWireframe = False

FPS# = 50

HidePointer()

While Not KeyHit(1)
	CycleStartTime = MilliSecs()
	
	If KeyHit(17) Then bWireframe = Not bWireframe ; W
	WireFrame bWireframe
	
	wTime# = MilliSecs()/1000.0
	
	WindForce = WindForce + (KeyDown(199)-KeyDown(207))*0.1 ; Home - End
	If WindForce < 0 Then WindForce = 0
	WindPowerFreq = WindPowerFreq + (KeyDown(201)-KeyDown(209))*0.1 ; PgUp - PgDn
	If WindPowerFreq < 0 Then WindPowerFreq = 0
	
	wrx# = UpdateSmoothRnd(WindXRnd, -1, 1, wTime)
	wrz# = UpdateSmoothRnd(WindZRnd, -1, 1, wTime)
	WindX = (Sgn(wrx)*(Abs(wrx)^WindPowerFreq))*WindForce
	WindZ = (Sgn(wrz)*(Abs(wrz)^WindPowerFreq))*WindForce
	WindX = WindX+UpdateSmoothRnd(WindDXRnd, -1, 1, wTime)*0.25*WindForce
	WindZ = WindZ+UpdateSmoothRnd(WindDZRnd, -1, 1, wTime)*0.25*WindForce
	
	TurnEntity cam, 0, -MouseXSpeed()*0.05, 0, True
	TurnEntity cam, MouseYSpeed()*0.05, 0, 0, False
	MoveMouse GraphicsWidth()/2, GraphicsHeight()/2
	
	spd_mult# = 1
	If KeyDown(157) Or KeyDown(29)
		spd_mult = 10
	EndIf
	
	spdx# = (KeyDown(205)-KeyDown(203))*spd_mult
	spdz# = (KeyDown(200)-KeyDown(208))*spd_mult
	spdz = spdz+(MouseDown(1)-MouseDown(2))*spd_mult
	MoveEntity cam, spdx, 0, spdz
	
	t = MilliSecs()
	UpdateGrass()
	t2 = MilliSecs()-t
	UpdateWorld
	t3 = MilliSecs()
	RenderWorld
	t = MilliSecs()-t
	t3 = MilliSecs()-t3
	
	If KeyHit(59); F1
		bDrawStatistics = Not bDrawStatistics
	EndIf
	
	If bDrawStatistics
		Color 0, 0, 0
		txty = 20
		Text 20, txty, "Tris = "+TrisRendered()+", total time = "+t
		txty = txty + 20
		Text 20, txty, "update time = "+t2+", render time = "+t3
		txty = txty + 20
		Text 20, txty, "WindForce = "+WindForce+", WindPowerFreq = "+WindPowerFreq
		txty = txty + 20
		Text 20, txty, "VideoMemory = "+AvailVidMem()+", Total = "+TotalVidMem()
		txty = txty + 20
		For wGrassEmitter.TGrassEmitter = Each TGrassEmitter
			Text 20, txty, "Emitter: exist = "+wGrassEmitter\Count+", active = "+wGrassEmitter\VisibleCount
			txty = txty + 20
		Next
		Text 20, txty, "Rotate and move camera with mouse."
		txty = txty + 20
		Text 20, txty, "Hold Ctrl to move faster. Use W to switch to wireframe."
		txty = txty + 20
		Text 20, txty, "Use Home/End to change wind force."
		txty = txty + 20
		Text 20, txty, "Use PageUp/PageDown to change wind rotation frequency."
		txty = txty + 20
		Text 20, txty, "Press F1 to turn off help and debug information."
		txty = txty + 20
	EndIf
	
	Flip
	
	While MilliSecs() < (CycleStartTime+Int(1000/FPS))
	Wend
Wend

End

Type TSmoothRnd
	Field x0#, x1#, dx#
	Field Tmin#, Tmax#
	Field t0#, t1#, dt#
End Type

Function CreateSmoothRnd.TSmoothRnd(Tmin#, Tmax#)
	SmoothRnd.TSmoothRnd = New TSmoothRnd
	SmoothRnd\x0 = Rnd(0, 1)
	SmoothRnd\x1 = Rnd(0, 1)
	SmoothRnd\dx = SmoothRnd\x1 - SmoothRnd\x0
	SmoothRnd\Tmin = Tmin
	SmoothRnd\Tmax = Tmax
	Return SmoothRnd
End Function

Function UpdateSmoothRnd#(SmoothRnd.TSmoothRnd, min#, max#, t#)
	If t > SmoothRnd\t1
		SmoothRnd\t0 = t;SmoothRnd\t1
		SmoothRnd\t1 = t + Rnd(SmoothRnd\Tmin, SmoothRnd\Tmax)
		SmoothRnd\dt = SmoothRnd\t1 - SmoothRnd\t0
		SmoothRnd\x0 = SmoothRnd\x1
		SmoothRnd\x1 = Rnd(0, 1)
		SmoothRnd\dx = SmoothRnd\x1 - SmoothRnd\x0
	EndIf
	
	dt# = (t-SmoothRnd\t0)/SmoothRnd\dt
	x# = SmoothRnd\x0 + CircleInterpolation(dt, 2)*SmoothRnd\dx
	
	Return min + (max-min)*x
End Function

Function CircleInterpolation#(t#, k# = 1)
	Local res#
	If (t < 0.5)
		t = 2*t
		res = (1-(1-t*t)^(k*0.5))*0.5
	Else
		t = 2*(t-1)
		res = (1+(1-t*t)^(k*0.5))*0.5
	EndIf
	Return res
End Function

Function CreateQuad(x0# = -1, y0# = -1, x1# = 1, y1# = 1)
	mesh = CreateMesh()
	srf = CreateSurface(mesh)
	AddVertex srf, x0, y0, 0, 0, 1
	AddVertex srf, x0, y1, 0, 0, 0
	AddVertex srf, x1, y0, 0, 1, 1
	AddVertex srf, x1, y1, 0, 1, 0
	AddTriangle srf, 0, 1, 2
	AddTriangle srf, 1, 3, 2
	UpdateNormals mesh
	Return mesh
End Function