#!BPY
"""
Name: 'MD2'
Blender: 233
Group: 'Export'
Tip: 'Export to Quake 2 file format. (.md2)'
"""
######################################################
# MD2 Exporter
# By:  Bob Holcomb
# Date: 23 JUL 04
# Ver: 0.14
######################################################
# This script imports a Quake 2 file (MD2), textures, 
# and animations into blender for editing.  Hopefully 
# this will make it into a future version as an import 
# feature of the menu.  Loader is based on MD2 loader
# from www.gametutorials.com-Thanks DigiBen! and the
# md3 blender loader by PhaethonH <phaethon@linux.ucla.edu>
#
#
#Also Help from Shadwolf, Skandal, and Roja
#Thanks Guys!
######################################################
import Blender
from Blender import NMesh, Object
from Blender.BGL import *
from Blender.Draw import *
from Blender.Window import *
from Blender.Image import *

import sys, struct, string
from types import *

import os
from os import path


######################################################
# Main Body
######################################################

def asciiz (s):
  n = 0
  while (ord(s[n]) != 0):
    n = n + 1
  return s[0:n]

  
######################################################
# MD2 Model Constants
######################################################
MD2_MAX_TRIANGLES=4096
MD2_MAX_VERTICES=2048
MD2_MAX_TEXCOORDS=2048
MD2_MAX_FRAMES=512
MD2_MAX_SKINS=32
MD2_MAX_FRAMESIZE=(MD2_MAX_VERTICES * 4 + 128)

#MD2_FRAME_NAME_LIST=(("stand",1,40),
#					("run",41,46),
#					("attack",47,54),
#					("pain1",55,58),
#					("pain2",59,62),
#					("pain3",63,66),
#					("jump",67,72),
#					("flip",73,84),
#					("salute", 85,95),
#					("taunt",96,112),
#					("wave",113,123),
#					("point",124,135),
#					("crstnd",136,154),
#					("crwalk",155,160),
#					("crattack",161,169),
#					("crpain",170,173),
#					("crdeath",174,178),
#					("death1",179,184),
#					("death2",185,190),
#					("death3",191,198))
MD2_FRAME_NAME_LIST=(("rasca",1,27),
					("quieto",28,120),
					("corre",121,128),
					("eleva",129,131),
					("pataleo",132,133))


######################################################
# MD2 data structures
######################################################
class md2_alias_triangle:
	vertices=[]
	lightnormalindex=0
	binary_format="<3BB"
	def __init__(self):
		self.vertices=[0]*3
		self.lightnormalindex=0
	def save(self, file):
		temp_data=[0]*4
		temp_data[0]=self.vertices[0]
		temp_data[1]=self.vertices[1]
		temp_data[2]=self.vertices[2]
		temp_data[3]=self.lightnormalindex
		data=struct.pack(self.binary_format, temp_data[0], temp_data[1], temp_data[2], temp_data[3])
		file.write(data)
	def dump(self):
		print "MD2 Alias_Triangle Structure"
		print "vertex: ", self.vertices[0]
		print "vertex: ", self.vertices[1]
		print "vertex: ", self.vertices[2]
		print "lightnormalindex: ",self.lightnormalindex
		print ""
		
class md2_face:
	vertex_index=[]
	texture_index=[]
	binary_format="<3h3h"
	def __init__(self):
		self.vertex_index = [ 0, 0, 0 ]
		self.texture_index = [ 0, 0, 0]
	def save(self, file):
		temp_data=[0]*6
		#swap vertices around so they draw right
		temp_data[0]=self.vertex_index[0]
		temp_data[1]=self.vertex_index[2]
		temp_data[2]=self.vertex_index[1]
		#swap texture vertices around so they draw right
		temp_data[3]=self.texture_index[0]
		temp_data[4]=self.texture_index[2]
		temp_data[5]=self.texture_index[1]
		data=struct.pack(self.binary_format,temp_data[0],temp_data[1],temp_data[2],temp_data[3],temp_data[4],temp_data[5])
		file.write(data)
	def dump (self):
		print "MD2 Face Structure"
		print "vertex index: ", self.vertex_index[0]
		print "vertex index: ", self.vertex_index[1]
		print "vertex index: ", self.vertex_index[2]
		print "texture index: ", self.texture_index[0]
		print "texture index: ", self.texture_index[1]
		print "texture index: ", self.texture_index[2]
		print ""
		
class md2_tex_coord:
	u=0
	v=0
	binary_format="<2h"
	def __init__(self):
		self.u=0
		self.v=0
	def save(self, file):
		temp_data=[0]*2
		temp_data[0]=self.u
		temp_data[1]=self.v
		data=struct.pack(self.binary_format, temp_data[0], temp_data[1])
		file.write(data)
	def dump (self):
		print "MD2 Texture Coordinate Structure"
		print "texture coordinate u: ",self.u
		print "texture coordinate v: ",self.v
		print ""

class md2_skin:
	name=""
	binary_format="<64s"
	def __init__(self):
		self.name=""
	def save(self, file):
		temp_data=self.name
		data=struct.pack(self.binary_format, temp_data)
		file.write(data)
	def dump (self):
		print "MD2 Skin"
		print "skin name: ",self.name
		print ""
		
class md2_alias_frame:
	scale=[]
	translate=[]
	name=[]
	vertices=[]
	binary_format="<3f3f16s"
	#did not add the "3bb" to the end of the binary format
	#because the alias_vertices will be read in through
	#thier own loader
	def __init__(self):
		self.scale=[0.0]*3
		self.translate=[0.0]*3
		self.name=""
		self.vertices=[]
	def save(self, file):
		temp_data=[0]*7
		temp_data[0]=float(self.scale[0])
		temp_data[1]=float(self.scale[1])
		temp_data[2]=float(self.scale[2])
		temp_data[3]=float(self.translate[0])
		temp_data[4]=float(self.translate[1])
		temp_data[5]=float(self.translate[2])
		temp_data[6]=self.name
		data=struct.pack(self.binary_format, temp_data[0],temp_data[1],temp_data[2],temp_data[3],temp_data[4],temp_data[5],temp_data[6])
		file.write(data)
	def dump (self):
		print "MD2 Alias Frame"
		print "scale x: ",self.scale[0]
		print "scale y: ",self.scale[1]
		print "scale z: ",self.scale[2]
		print "translate x: ",self.translate[0]
		print "translate y: ",self.translate[1]
		print "translate z: ",self.translate[2]
		print "name: ",self.name
		print ""
		
class md2_obj:
	#Header Structure
	ident=0				#int 0	This is used to identify the file
	version=0			#int 1	The version number of the file (Must be 8)
	skin_width=0		#int 2	The skin width in pixels
	skin_height=0		#int 3	The skin height in pixels
	frame_size=0		#int 4	The size in bytes the frames are
	num_skins=0			#int 5	The number of skins associated with the model
	num_vertices=0		#int 6	The number of vertices (constant for each frame)
	num_tex_coords=0	#int 7	The number of texture coordinates
	num_faces=0			#int 8	The number of faces (polygons)
	num_GL_commands=0	#int 9	The number of gl commands
	num_frames=0		#int 10	The number of animation frames
	offset_skins=0		#int 11	The offset in the file for the skin data
	offset_tex_coords=0	#int 12	The offset in the file for the texture data
	offset_faces=0		#int 13	The offset in the file for the face data
	offset_frames=0		#int 14	The offset in the file for the frames data
	offset_GL_commands=0#int 15	The offset in the file for the gl commands data
	offset_end=0		#int 16	The end of the file offset
	binary_format="<17i"  #little-endian (<), 17 integers (17i)
	#md2 data objects
	tex_coords=[]
	faces=[]
	frames=[]
	skins=[]
	def __init__ (self):
		self.tex_coords=[]
		self.faces=[]
		self.frames=[]
		self.skins=[]
	def save(self, file):
		temp_data=[0]*17
		temp_data[0]=self.ident
		temp_data[1]=self.version
		temp_data[2]=self.skin_width
		temp_data[3]=self.skin_height
		temp_data[4]=self.frame_size
		temp_data[5]=self.num_skins
		temp_data[6]=self.num_vertices
		temp_data[7]=self.num_tex_coords
		temp_data[8]=self.num_faces
		temp_data[9]=self.num_GL_commands
		temp_data[10]=self.num_frames
		temp_data[11]=self.offset_skins
		temp_data[12]=self.offset_tex_coords
		temp_data[13]=self.offset_faces
		temp_data[14]=self.offset_frames
		temp_data[15]=self.offset_GL_commands
		temp_data[16]=self.offset_end
		data=struct.pack(self.binary_format, temp_data[0],temp_data[1],temp_data[2],temp_data[3],temp_data[4],temp_data[5],temp_data[6],temp_data[7],temp_data[8],temp_data[9],temp_data[10],temp_data[11],temp_data[12],temp_data[13],temp_data[14],temp_data[15],temp_data[16])
		file.write(data)
		#write the skin data
		for i in xrange(0,self.num_skins):
			self.skins[i].save(file)
		#save the texture coordinates
		for i in xrange(0, self.num_tex_coords):
			self.tex_coords[i].save(file)
		#save the face info
		for i in xrange(0, self.num_faces):
			self.faces[i].save(file)
		#save the frames
		for i in xrange(0, self.num_frames):
			self.frames[i].save(file)
			for j in xrange(0,self.num_vertices):
				self.frames[i].vertices[j].save(file)
	def dump (self):
		print "Header Information"
		print "ident: ", self.ident
		print "version: ", self.version
		print "skin width: ", self.skin_width
		print "skin height: ", self.skin_height
		print "frame size: ", self.frame_size
		print "number of skins: ", self.num_skins
		print "number of texture coordinates: ", self.num_tex_coords
		print "number of faces: ", self.num_faces
		print "number of frames: ", self.num_frames
		print "number of vertices: ", self.num_vertices
		print "offset skins: ", self.offset_skins
		print "offset texture coordinates: ", self.offset_tex_coords
		print "offset faces: ", self.offset_faces
		print "offset frames: ",self.offset_frames
		print ""
		
######################################################
# Export functions
######################################################
def save_md2(filename):
	#intermediate data structures
	vert_list={}
	vert_count=0
	tex_list={}
	tex_count=0
	face_list={}
	face_count=0
	user_frame_list="default"
	
	
	#get the currently selected data structure
	mesh_obj = Blender.Object.GetSelected()
	
	#check if an object is selected
	if len(mesh_obj)==0:
		print "Fatal Error: Must select a mesh to output as MD2"
		print "Found nothing"
		result=Blender.Draw.PupMenu("Select an object to export%t|OK")
		Exit()
		
	#check if it's a mesh object
	if mesh_obj[0].getType()!="Mesh":
		print "Fatal Error: Must select a mesh to output as MD2"
		print "Found: ", mesh_obj[0].getType()
		result=Blender.Draw.PupMenu("Selected Object must be a mesh to output as MD2?%t|OK")
		Exit()
	
	#get access to the mesh data
	mesh=mesh_obj[0].getData()
	
	for face in mesh.faces:
		if len(face.v)!=3:
			print "Model not made entirely of triangles"
			result=Blender.Draw.PupMenu("Model not made entirely out of Triangles?%t|OK")
			Exit()
	
	#load up some intermediate data structures
	for vertex in mesh.verts:
		v = (vertex.co)
		vert_key=(v[0], v[1], v[2])
		if not vert_list.has_key(vert_key):
			vert_list[vert_key] = vert_count
			vert_count = vert_count + 1
			#print "added vert #",vert_count,": ", v
			
	if mesh.hasVertexUV():
		#print "I have Vertex UV"
		for vertex in mesh.verts:
			#are there UV coordinates?
			if(len(vertex.uvco)>0):
				t = (vertex.uvco)
				tex_key=(t[0], t[1])
				if not tex_list.has_key(tex_key):
					tex_list[tex_key]=tex_count
					tex_count=tex_count+1
					#print "added tex_vert #",tex_count,": ", t
			else:
				tex_key=(0.0,0.0)
				tex_list=[tex_key]=tex_count
				tex_count=tex_count+1
					
	elif mesh.hasFaceUV():
		#print "I have Face UV"
		for face in mesh.faces:
			face_count=face_count+1
			for i in range(0,3):
				#are there UV coordinates?
				if (len(face.uv)>0):
					t=(face.uv[i])
					tex_key=(t[0], t[1])
					if not tex_list.has_key(tex_key):
						tex_list[tex_key]=tex_count
						tex_count=tex_count+1
						#print "added tex_vert #",tex_count,": ", t
				else:
					tex_key=(0.0,0.0)
					tex_list=[tex_key]=tex_count
					tex_count=tex_count+1
	
	else:
		#print "I don't have any UV"
		for vertex in mesh.verts:
			tex_key=(0.0,0.0)
			tex_list=[tex_key]=tex_count
			tex_count=tex_count+1
		
	#make a blank md2 class and start filling it in
	md2=md2_obj()
	
	#header information
	md2.ident=844121161
	md2.version=8
	
	#I don't care about GL commands yet
	md2.num_GL_commands=0
	
	#setup other important numbers
	md2.num_vertices=vert_count
	md2.num_tex_coords=tex_count
	md2.num_faces=face_count
	
	#don't support exporting skins just yet.
	#get the skin information
	#use the first faces' image for the texture information
	"""
	#if there is a texture map
	mesh_image=mesh.faces[0].image
	size=mesh_image.getSize()
	#is this really what the user wants
	if (size[0]!=256 or size[1]!=256):
		result=Blender.Draw.PupMenu("Texture map size is not 256x256, it's: "+size[0]+"x"size[1]+": Continue?%t|Yes|No")
		if(result==2):
			Exit()
	md2.skin_width=size[0]
	md2.skin_height=size[1]
	md2.num_skins=1
	#add a skin node to the md2 data structure
	md2.skins.append(md2_skin())
	md2.skins[0].name=os.path.basename(mesh_image.getFilename())
	"""
	
	md2.skin_width=256
	md2.skin_height=256
	md2.num_skins=0
	
	#get the texture coordinates
	#print "tex_count: ", tex_count
	for this_tex in range (0, md2.num_tex_coords):
		md2.tex_coords.append(md2_tex_coord())
		
	for coord, index in tex_list.iteritems():
		#fill it with information
		md2.tex_coords[index].u=int(coord[0]*md2.skin_width)
		md2.tex_coords[index].v=int((1-coord[1])*md2.skin_height)

	#get the face info
	for this_face in xrange(0, md2.num_faces):
		#add another blankface to the face list
		md2.faces.append(md2_face())
		for i in range (0,3):
			#build the keys
			vert=(mesh.faces[this_face].v[i].co)
			vert_key=(vert[0], vert[1], vert[2])
			vert_index=vert_list[vert_key]
			#print "vet_index: ", vert_index
			md2.faces[this_face].vertex_index[i]=vert_index
			if mesh.hasVertexUV():
				uv_coord=(mesh.faces[this_face].v[i].uvco)
				tex_key=(uv_coord[0],uv_coord[1])
				#print "Vertex uv: ", uv
				tex_index=tex_list[tex_key]
				#print "tex_index: ", tex_index
				md2.faces[this_face].texture_index[i]=tex_index
			elif mesh.hasFaceUV():
				uv_coord=(mesh.faces[this_face].uv[i])
				tex_key=(uv_coord[0],uv_coord[1])
				#print "Face uv: ", uv
				tex_index=tex_list[tex_key]
				#print "tex_index: ", tex_index
				md2.faces[this_face].texture_index[i]=tex_index


	#get the frame data
	#one frame=frame header and all the verticies for that frame
	#the number of verticies is constant between frames
	#frame format=3f 3f 16s=12+12+16=40 bytes
	#a vertex format=3BB=3+1=4 bytes
	
	#calculate 1 frame size  + (1 vert size*num_verts)
	md2.frame_size=40+(vert_count*4) #in bytes
	
	#fill in each frame with frame info and all the vertices for that frame
	user_frame_list=get_frame_list()
	
	if user_frame_list=="default":
		md2.num_frames=198
	else:
		temp=user_frame_list[len(user_frame_list)-1]
		md2.num_frames=temp[2]+1

	print "number of frames: ", md2.num_frames

	for frame_counter in range(0,md2.num_frames):
		#add a frame
		md2.frames.append(md2_alias_frame())
		#update the mesh objects vertex positions for the animation
		#set blender to the correct frame
		Blender.Set("curframe", frame_counter)
		#update the mesh in blender to get the vertices modified by the animation
		mesh=NMesh.GetRawFromObject(mesh_obj[0].name)
		#each frame has a scale and transform value that gets the vertex value between 0-255
		#since the scale and transform are for the all the verts in the frame, we only neeed
		#to figure this out once per frame
		#initialize with the first vertex for both min and max
		frame_min_x=mesh.verts[0].co[1]
		frame_max_x=mesh.verts[0].co[1]
		frame_min_y=mesh.verts[0].co[0]
		frame_max_y=mesh.verts[0].co[0]
		frame_min_z=mesh.verts[0].co[2]
		frame_max_z=mesh.verts[0].co[2]
		#find the min and max of each
		#probably could do this neater/faster with the bounding box of the mesh from blender
		#bounding_box=mesh_obj[0].getBoundBox()
		#for vertex in bounding_box:
		for vertex in mesh.verts:
			if frame_min_x>vertex.co[1]: frame_min_x=vertex.co[1]
			if frame_max_x<vertex.co[1]: frame_max_x=vertex.co[1]
			if frame_min_y>vertex.co[0]: frame_min_y=vertex.co[0]
			if frame_max_y<vertex.co[0]: frame_max_y=vertex.co[0]
			if frame_min_z>vertex.co[2]: frame_min_z=vertex.co[2]
			if frame_max_z<vertex.co[2]: frame_max_z=vertex.co[2]
		
		#undo the import scale
		
		#the scale is the difference between the min and max (on that axis) / 255
		frame_scale_x=(frame_max_x-frame_min_x)/255
		frame_scale_y=(frame_max_y-frame_min_y)/255
		frame_scale_z=(frame_max_z-frame_min_z)/255
		
		#the adjust is 0-min to get the min vertex value on that axis to 0
		adjust_x=(0-frame_min_x)
		adjust_y=(0-frame_min_y)
		adjust_z=(0-frame_min_z)
		
		#translate value of the mesh to center it on the origin
		#frame_trans_x=-(frame_max_x-frame_min_x)/2
		#frame_trans_y=-(frame_max_y-frame_min_y)/2
		#frame_trans_z=-(frame_max_z-frame_min_z)/2
		frame_trans_x=frame_min_x
		frame_trans_y=frame_min_y
		frame_trans_z=frame_min_z
		
		#fill in the data
		md2.frames[frame_counter].scale=(-frame_scale_x, frame_scale_y, frame_scale_z)
		md2.frames[frame_counter].translate=(-frame_trans_x, frame_trans_y, frame_trans_z)
		
		#now for the vertices
		for vert_counter in range(0, md2.num_vertices):
			#add a vertex to the md2 structure
			md2.frames[frame_counter].vertices.append(md2_alias_triangle())
			#figure out the new coords based on scale and transform
			#then adjust the point so it's not less than 0
			#then scale it do it's between 0..255
			#print "frames", frame_scale_x, frame_scale_y, frame_scale_z
			new_x=(int((mesh.verts[vert_counter].co[1]+adjust_x)/frame_scale_x))
			new_y=int((mesh.verts[vert_counter].co[0]+adjust_y)/frame_scale_y)
			new_z=int((mesh.verts[vert_counter].co[2]+adjust_z)/frame_scale_z)
			#print "new x,y,z: ", new_x, new_y, new_z
			
			#put them in the structure
			md2.frames[frame_counter].vertices[vert_counter].vertices=(new_x, new_y, new_z)
			
			#don't care about this much either
			md2.frames[frame_counter].vertices[vert_counter].lightnormalindex=0
			

	if user_frame_list=="default":
		for frame_set in MD2_FRAME_NAME_LIST:
			for counter in range(frame_set[1]-1, frame_set[2]):
				md2.frames[counter].name=frame_set[0]+"_"+str(counter-frame_set[1]+2)
	else:
	    for frame_set in user_frame_list:
		for counter in range(frame_set[1], frame_set[2]+1):
		    md2.frames[counter].name=frame_set[0]+str(counter-frame_set[1])
			

	#compute these after everthing is loaded into a md2 structure
	header_size=17*4 #17 integers, and each integer is 4 bytes
	skin_size=64*md2.num_skins #64 char(or bytes) per skin * number of skins
	tex_coord_size=4*md2.num_tex_coords #2 short (2 bytes each) * number of texture coords
	face_size=12*md2.num_faces #3 shorts (2 bytes each=6 total) for vertex index, 3 shorts (2 bytes each=6 total) for tex index
	frames_size=(((12+12+16)+(4*md2.num_vertices))*md2.num_frames)
	#3 floats, 3 floats, 16 char for the frame header
	#4 bytes per vertex * number of vertex
	#all that times the number of frames
	
	#not worried about GL commands at this time
	GL_command_size=0
	
	#fill in the info about offsets
	md2.offset_skins=0+header_size
	
	
	md2.offset_tex_coords=md2.offset_skins+skin_size
	
	md2.offset_faces=md2.offset_tex_coords+tex_coord_size
	
	md2.offset_frames=md2.offset_faces+face_size
	
	md2.offset_GL_commands=md2.offset_frames+frames_size
	
	md2.offset_end=md2.offset_GL_commands+GL_command_size

	#actually write it to disk
	file=open(filename,"wb")
	md2.save(file)
	file.close()
	print "Closed the file"

def get_frame_list():
	global g_frame_filename
	frame_list=[]

	if g_frame_filename.val=="default":
		print "found default filename from GUI"
		return "default"

	else:
	#check for file
		if (os.path.isfile(g_frame_filename.val)):
			print "os.path says filename is valid"
			#open file and read it in
			file=open(g_frame_filename.val,"r")
			lines=file.readlines()
			file.close()
			print "opened and read ", len(lines), " lines"

			#check header (first line)
			if lines[0]<>"# MD2 Frame Name List\n":
				print "its not a valid file"
				return "default"
			else:
				#read in the data
				num_frames=0
				for counter in range(1, len(lines)):
					current_line=lines[counter]
					print "current line: ", current_line
					print "current_line[0]: ", current_line[0]

					if current_line[0]=="#":
						#found a comment
						pass

					else:
						data=current_line.split()
						print "data: ", data
						frame_list.append([data[0],num_frames+1, num_frames+int(data[1])])
						num_frames+=int(data[1])

				print "returning: "
				print	frame_list
				return frame_list
		
		else:
			return "default"
		

######################################################
# GUI Loader
######################################################

# Import globals
g_filename=Create("/home/marte/lalala.md2")
g_frame_filename=Create("default")

g_filename_search=Create("model")
g_frame_search=Create("default")

#Globals
g_scale=Create(1.0)

# Events
EVENT_NOEVENT=1
EVENT_SAVE_MD2=2
EVENT_CHOOSE_FILENAME=3
EVENT_CHOOSE_FRAME=4
EVENT_EXIT=100

######################################################
# Callbacks for Window functions
######################################################
def filename_callback(input_filename):
	global g_filename
	g_filename.val=input_filename

def frame_callback(input_frame):
	global g_frame_filename
	g_frame_filename.val=input_frame


def draw_gui():
	global g_scale
	global g_filename
	global g_frame_filename
	global EVENT_NOEVENT,EVENT_SAVE_MD2,EVENT_CHOOSE_FILENAME,EVENT_CHOOSE_FRAME,EVENT_EXIT

	########## Titles
	glClear(GL_COLOR_BUFFER_BIT)
	glRasterPos2d(8, 103)
	Text("MD2 Export")

	######### Parameters GUI Buttons
	g_filename = String("MD2 file to save: ", EVENT_NOEVENT, 10, 55, 210, 18,
                            g_filename.val, 255, "MD2 file to save")
	########## MD2 File Search Button
	Button("Search",EVENT_CHOOSE_FILENAME,220,55,80,18)

	g_frame_filename = String("Frame List file to load: ", EVENT_NOEVENT, 10, 35, 210, 18,
                                g_frame_filename.val, 255, "Frame List to load-overrides MD2 file")
	########## Texture Search Button
	Button("Search",EVENT_CHOOSE_FRAME,220,35,80,18)

	########## Scale slider-default is 1/8 which is a good scale for md2->blender
	g_scale= Slider("Scale Factor: ", EVENT_NOEVENT, 10, 75, 210, 18,
                    1.0, 0.001, 10.0, 1, "Scale factor for obj Model");

	######### Draw and Exit Buttons
	Button("Export",EVENT_SAVE_MD2 , 10, 10, 80, 18)
 	Button("Exit",EVENT_EXIT , 170, 10, 80, 18)

def event(evt, val):	
	if (evt == QKEY and not val):
		Exit()

def bevent(evt):
	global g_filename
	global g_frame_filename
	global EVENT_NOEVENT,EVENT_SAVE_MD2,EVENT_EXIT

	######### Manages GUI events
	if (evt==EVENT_EXIT):
		Blender.Draw.Exit()
	elif (evt==EVENT_CHOOSE_FILENAME):
		FileSelector(filename_callback, "MD2 File Selection")
	elif (evt==EVENT_CHOOSE_FRAME):
		FileSelector(frame_callback, "Frame Selection")
	elif (evt==EVENT_SAVE_MD2):
		if (g_filename.val == "model"):
			Blender.Draw.Exit()
		else:
			save_md2(g_filename.val)
			print "ended save_MD2"
			Blender.Draw.Exit()
			print "Blender.Draw.Exit() called"

Register(draw_gui, event, bevent)