import pandas as pd
import smtplib
import logging
from datetime import datetime, timedelta
from sqlalchemy import create_engine
import schedule
import time
from twilio.rest import Client  # Para SMS
import hashlib

class SistemaCribadoAutomatizado:
    def __init__(self):
        self.logger = self.configurar_logging()
        self.umbral_retraso = 30  # días máximos de retraso permitido
        
    def configurar_logging(self):
        logging.basicConfig(
            filename='logs/sistema_cribado.log',
            level=logging.INFO,
            format='%(asctime)s - %(levelname)s - %(message)s'
        )
        return logging.getLogger(__name__)
    
    def verificar_estado_citas(self):
        """Verifica el estado de todas las citas programadas"""
        try:
            # Conexión a base de datos
            engine = create_engine('postgresql://user:pass@localhost/db')
            
            # Query para detectar citas pendientes o retrasadas
            query = """
            SELECT 
                paciente_id,
                fecha_cita_programada,
                fecha_notificacion,
                estado,
                medio_notificacion
            FROM citas_cancer_mama
            WHERE estado IN ('pendiente', 'notificada')
            AND fecha_cita_programada <= CURRENT_DATE + INTERVAL '30 days'
            """
            
            citas = pd.read_sql(query, engine)
            
            # Detectar anomalías
            anomalias = self.detectar_anomalias(citas)
            
            if not anomalias.empty:
                self.alertar_sistema(anomalias)
                self.acciones_correctivas(anomalias)
            
            self.logger.info(f"Verificación completada: {len(citas)} citas revisadas")
            
        except Exception as e:
            self.logger.error(f"Error en verificación: {str(e)}")
            self.notificar_error_sistema(e)
    
    def detectar_anomalias(self, citas):
        """Detecta citas con problemas de notificación"""
        anomalias = []
        hoy = datetime.now().date()
        
        for _, cita in citas.iterrows():
            # Citas sin notificación 15 días antes
            if cita['fecha_notificacion'] is None:
                if (cita['fecha_cita_programada'] - hoy).days <= 15:
                    anomalias.append({
                        'tipo': 'sin_notificacion',
                        'paciente_id': cita['paciente_id'],
                        'dias_restantes': (cita['fecha_cita_programada'] - hoy).days
                    })
            
            # Citas con retraso acumulado
            if cita['estado'] == 'pendiente' and cita['fecha_cita_programada'] < hoy:
                dias_retraso = (hoy - cita['fecha_cita_programada']).days
                if dias_retraso > self.umbral_retraso:
                    anomalias.append({
                        'tipo': 'retraso_grave',
                        'paciente_id': cita['paciente_id'],
                        'dias_retraso': dias_retraso
                    })
        
        return pd.DataFrame(anomalias)