#ifndef __VECTORBASE_H__
#define __VECTORBASE_H__

#ifndef _USE_SSE

BEGIN_VLADO

/// A class representing a vector in 3d space.

class Vector {
public:
	/// The components of the vector
	real x, y, z;

	/// Default constructor - does nothing
	FORCEINLINE Vector(void) {};

	/// Initializes the vector with the given vector
	FORCEINLINE Vector(const Vector &a) { x=a.x; y=a.y; z=a.z; }

	/// Assignment operator
	FORCEINLINE Vector& operator=(const Vector &a) { x=a.x; y=a.y; z=a.z; return *this; }

	/// Initalizes the vector with the given components
	FORCEINLINE Vector(int ix, int iy, int iz) { x=real(ix); y=real(iy); z=real(iz); }

	/// Initalizes the vector with the given components
	FORCEINLINE Vector(float ix, float iy, float iz) { x=real(ix); y=real(iy); z=real(iz); }

	/// Initalizes the vector with the given components
	FORCEINLINE Vector(double ix, double iy, double iz) { x=real(ix); y=real(iy); z=real(iz); }

	/// Zeroes all components of the vector
	FORCEINLINE void makeZero(void) { x=y=z=0.0f; }

	/// Sets the compnents of the vector
	FORCEINLINE void set(real ix, real iy, real iz) { x=ix; y=iy; z=iz; }

	/// Adds the components of the given vector
	FORCEINLINE void operator +=(const Vector &a) { x+=a.x; y+=a.y; z+=a.z; }

	/// Subtracts the components of the given vector
	FORCEINLINE void operator -=(const Vector &a) { x-=a.x; y-=a.y; z-=a.z; }

	/// Multiplies all components by the given number
	FORCEINLINE void operator *=(real f) { x*=f; y*=f; z*=f; }

	/// Divides all components by the given number
	FORCEINLINE void operator /=(real f) { x/=f; y/=f; z/=f; }

	/// Reverses the sign of all components
	FORCEINLINE Vector operator-(void) const { return(Vector(-x,-y,-z)); } 

	/// Returns the i-th component (0 for x, 1 for y, 2 for z)
	FORCEINLINE real& operator [](const int index) { return (&x)[index]; }

	/// Returns the i-th component (0 for x, 1 for y, 2 for z) as a const
	FORCEINLINE const real& operator [](const int index) const { return (&x)[index]; }

	/// Returns the length of the vector
	FORCEINLINE real length(void) const { return real(sqrt(x*x+y*y+z*z)); }

	/// Returns the squared length of the vector
	FORCEINLINE real lengthSqr(void) const { return x*x+y*y+z*z; }

	/// Returns a unit vector along the direction of this one
	FORCEINLINE Vector getUnitVector(void) const {
		real len=length();
		return Vector(x/len, y/len, z/len);
	}

	/// Returns a unit vector along the direction of this one. If the length is <1e-12f returns the zero vector
	FORCEINLINE Vector getUnitVector0(void) const {
		real len=length();
		if (len<1e-12f) return Vector(0.0f, 0.0f, 0.0f);
		return Vector(x/len, y/len, z/len);
	}

	/// Makes this vector a unit vector
	FORCEINLINE void makeNormalized(void) {
		real len=length();
		x/=len;
		y/=len;
		z/=len;
	}

	/// Makes this vector a unit vector. If the length is <1e-12f, does nothing
	FORCEINLINE void makeNormalized0(void) {
		real len=length();
		if (len>1e-12f) {
			x/=len;
			y/=len;
			z/=len;
		}
	}

	/// Rotates the vector along the given axis. The length of the axis is the tangent of the angle of rotation
	FORCEINLINE void rotate(const Vector &axis);
};

/// \relates Vector
/// Multiplies the given vector by the given number.
FORCEINLINE Vector operator *(const Vector &a, real f) {
	Vector res(a);
	res*=f;
	return res;
}

/// \relates Vector
/// Multiplies all components of the given vector by the given number.
FORCEINLINE Vector operator *(real f, const Vector &a) {
	Vector res(a);
	res*=f;
	return res;
}

/// \relates Vector
/// Divides all components of the given vector by the given number.
FORCEINLINE Vector operator /(const Vector &a, real f) {
	Vector res(a);
	res/=f;
	return res;
}

/// \relates Vector
/// Dot product of two vectors
FORCEINLINE real operator *(const Vector &a, const Vector &b) {
	return a.x*b.x+a.y*b.y+a.z*b.z;
}

/// \relates Vector
/// Adds the components of the given vectors
FORCEINLINE Vector operator +(const Vector &a, const Vector &b) {
	Vector res(a);
	res+=b;
	return res;
}

/// \relates Vector
/// Subtracts the components of the given vectors
FORCEINLINE Vector operator -(const Vector &a, const Vector &b) {
	Vector res(a);
	res-=b;
	return res;
}

/// \relates Vector
/// Divides the components of the given vectors
FORCEINLINE Vector operator /(const Vector &a, const Vector &b) {
	return Vector(a.x/b.x, a.y/b.y, a.z/b.z);
}

/// \relates Vector
/// Mutiplies the components of the given vectors
FORCEINLINE Vector mul(const Vector &a, const Vector &b) {
	return Vector(a.x*b.x, a.y*b.y, a.z*b.z);
}

END_VLADO

#else

#include <fvec.h>

BEGIN_VLADO

#pragma pack(push,16)

__declspec(align(16)) class Vector {
	public:
		union {
			struct { real x, y, z; };
			float f[4];
			__m128 m;
		};

		operator __m128(void) const { return m; }
		Vector (__m128 im) { m=im; }
		
		FORCEINLINE Vector() {};

		FORCEINLINE Vector(const Vector &a) { x=a.x; y=a.y; z=a.z; }
		FORCEINLINE Vector& operator=(const Vector &a) { x=a.x; y=a.y; z=a.z; return *this; }


		FORCEINLINE Vector(int ix, int iy, int iz) { x=real(ix); y=real(iy); z=real(iz); }
		FORCEINLINE Vector(float ix, float iy, float iz) { x=real(ix); y=real(iy); z=real(iz); }
		FORCEINLINE Vector(double ix, double iy, double iz) { x=real(ix); y=real(iy); z=real(iz); }

		explicit Vector(float f) { m=_mm_set_ps1(f); }	

		FORCEINLINE void makeZero(void) { x=y=z=0.0f; }
		FORCEINLINE void set(real ix, real iy, real iz) { x=ix; y=iy; z=iz; }

		FORCEINLINE Vector& operator +=(const Vector &a) { return *this = _mm_add_ps(m,a); }
		FORCEINLINE Vector& operator -=(const Vector &a) { return *this = _mm_sub_ps(m,a); }   
		// FORCEINLINE Vector& operator *=(float k) { return *this = _mm_mul_ps(m,Vector(k)); } 
		// FORCEINLINE Vector& operator /=(float k) { return *this = _mm_div_ps(m,Vector(k)); }
		FORCEINLINE Vector& operator *=(float k) { x*=k; y*=k; z*=k; return *this; }
		FORCEINLINE Vector& operator /=(float k) { x/=k; y/=k; z/=k; return *this; }
/*
		FORCEINLINE Vector& operator *=(float k) {
#pragma vector aligned
			for (int i=0; i<4; i++) f[i]*=k;
			return *this;
		}
		FORCEINLINE Vector& operator /=(float k) {
#pragma vector aligned
			for (int i=0; i<4; i++) f[i]/=k;
			return *this;
		}

		FORCEINLINE Vector operator-(void) const {
			Vector res;
#pragma vector aligned
			for (int i=0; i<4; i++) res.f[i]=-f[i];
			return res;
		}*/
		FORCEINLINE Vector operator-(void) const { return Vector(-x,-y,-z); }

		FORCEINLINE real& operator [](const int index) { return (&x)[index]; }
		FORCEINLINE const real& operator [](const int index) const { return (&x)[index]; }

		FORCEINLINE real lengthSqr(void) const { return x*x+y*y+z*z; }
		FORCEINLINE real length(void) const { return sqrtf(lengthSqr()); }
		FORCEINLINE Vector normalize(void) const { Vector r(*this); r*=1.0f/length(); return r; }
		FORCEINLINE void makeNormalized(void) { *this*=1.0f/length(); }
		FORCEINLINE void makeNormalized0(void) {
			real len=length();
			if (len>1e-12f) *this*=1.0f/len;
		}

		FORCEINLINE void rotate(const Vector &axis);

};

FORCEINLINE Vector operator +(const Vector &a, const Vector &b) { return _mm_add_ps(a.m, b.m); }
FORCEINLINE Vector operator -(const Vector &a, const Vector &b) { return _mm_sub_ps(a,b); } 
/*
FORCEINLINE Vector operator *(const Vector &a, float k) { return _mm_mul_ps(a,Vector(k)); } 
FORCEINLINE Vector operator *(float k, const Vector &a) { return _mm_mul_ps(a,Vector(k)); } 
FORCEINLINE Vector operator /(const Vector &a, float k) { return _mm_div_ps(a,Vector(k)); }
*/
FORCEINLINE Vector operator /(const Vector &a, const Vector &b) { return _mm_div_ps(a,b); }
FORCEINLINE Vector mul(const Vector &a, const Vector &b) { return _mm_mul_ps(a,b); }

FORCEINLINE Vector operator *(const Vector &a, float k) { Vector r(a); r*=k; return r; }
FORCEINLINE Vector operator *(float k, const Vector &a) { Vector r(a); r*=k; return r; }
FORCEINLINE Vector operator /(const Vector &a, float k) { Vector r(a); r/=k; return r; }

FORCEINLINE real operator *(const Vector &a, const Vector &b) { return a.x*b.x+a.y*b.y+a.z*b.z; }

/*
FORCEINLINE Vector operator *(const Vector &a, float k) {
	Vector res;
#pragma vector aligned
	for (int i=0; i<4; i++) res.f[i]=a.f[i]*k;
	return res;
}

FORCEINLINE Vector operator *(float k, const Vector &a) {
	Vector res;
#pragma vector aligned
	for (int i=0; i<4; i++) res.f[i]=a.f[i]*k;
	return res;
}

FORCEINLINE Vector operator /(const Vector &a, float k) {
	Vector res;
#pragma vector aligned
	for (int i=0; i<4; i++) res.f[i]=a.f[i]/k;
	return res;
}

FORCEINLINE real operator *(const Vector &a, const Vector &b) {
	real res=0.0f;
#pragma vector aligned
	for (int i=0; i<3; i++) res+=a.f[i]*b.f[i];
	return res;
}
*/

END_VLADO

#pragma pack(pop)

#endif

#endif
