/*****************************************************
	FILE: vector.h
	DESCRIPTION: Vector in 3D
	Copyright (C) 2001-2003 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __VECTOR_H__
#define __VECTOR_H__

#include "vectorbase.h"

BEGIN_VLADO

FORCEINLINE real maxValue(const Vector &a) {
	if (a.x>a.y) {
		if (a.x>a.z) return a.x;
		else return a.z;
	} else {
		if (a.y>a.z) return a.y;
		else return a.z;
	}
}

FORCEINLINE real minValue(const Vector &a) {
	if (a.x<a.y) {
		if (a.x<a.z) return a.x;
		else return a.z;
	} else {
		if (a.y<a.z) return a.y;
		else return a.z;
	}
}

/// \relates Vector
/// Checks two vectors for exact equality
FORCEINLINE bool operator!=(const Vector &a, const Vector &b) {
	return a.x!=b.x || a.y!=b.y || a.z!=b.z;
}

/// \relates Vector
/// Returns the index (0, 1 or 2) of the component with the maximum absolute value
FORCEINLINE int maxComponent(const Vector &a) {
	int maxDim=0;
	if (fabs(a[1])>fabs(a[0])) maxDim=1;
	if (fabs(a[2])>fabs(a[maxDim])) maxDim=2;
	return maxDim;
}

/// \relates Vector
/// Returns the index (0, 1 or 2) of the component with the maximum value
FORCEINLINE int maxRealComponent(const Vector &a) {
	int maxDim=0;
	if (a[1]>a[0]) maxDim=1;
	if (a[2]>a[maxDim]) maxDim=2;
	return maxDim;
}

/// \relates Vector
/// Returns the index (0, 1 or 2) of the component with the minimum absolute value
FORCEINLINE int minComponent(const Vector &a) {
	int minDim=0;
	if (fabs(a[1])<fabs(a[0])) minDim=1;
	if (fabs(a[2])<fabs(a[minDim])) minDim=2;
	return minDim;
}

/// \relates Vector
/// Returns the cross product of two vectors
FORCEINLINE Vector operator ^(const Vector &a, const Vector &b) {
	return Vector(
		a.y*b.z-b.y*a.z,
		a.z*b.x-b.z*a.x,
		a.x*b.y-b.x*a.y);
}

/// \relates Vector
/// Returns the squared length of the given vector
FORCEINLINE real lengthSqr(const Vector &a) { return a.lengthSqr(); }

/// \relates Vector
/// Returns the length of the given vector
FORCEINLINE real length(const Vector &a) { return a.length(); }

/// \relates Vector
/// Returns a unit vector along the direction of the given one
FORCEINLINE Vector normalize(const Vector &a) { return a/a.length(); }

/// \relates Vector
/// Returns a vector with components which the the absolute values of the components of the given vector
FORCEINLINE Vector vabs(const Vector &a) {
	return Vector(fabsf(a.x), fabsf(a.y), fabsf(a.z));
}

/// \relates Vector
/// Returns a vector with components which are the minimum respective components of the two vectors
template<>
inline Vector Min<Vector>(const Vector &a, const Vector &b) {
	return Vector(Min(a.x, b.x), Min(a.y, b.y), Min(a.z, b.z));
}

/// \relates Vector
/// Returns a vector with components which are the maximum respective components of the two vectors
template<>
inline Vector Max<Vector>(const Vector &a, const Vector &b) {
	return Vector(Max(a.x, b.x), Max(a.y, b.y), Max(a.z, b.z));
}

FORCEINLINE void Vector::rotate(const Vector &axis) {
	real oldLen=length();

	Vector d=(*this)^axis;
	d+=(*this);

	real newLen=d.length();
	if (newLen>almostZero) (*this)=d*oldLen/newLen;
}

FORCEINLINE Vector rotate(const Vector &v, const Vector &axis) {
	real oldLen=length(v);

	Vector d=axis^v;
	d+=v;

	real newLen=length(d);
	if (newLen<almostZero) return Vector(0,0,0);

	return d*oldLen/newLen;
}

FORCEINLINE void vector_minMax(Vector &a, Vector &b) {
	Vector s=a+b;
	Vector d=vabs(a-b);
	a=0.5f*(s-d);
	b=0.5f*(s+d);
}

// Some macros
#define _add3(a,b,r) { (r)[0]=(a)[0]+(b)[0]; (r)[1]=(a)[1]+(b)[1]; (r)[2]=(a)[2]+(b)[2]; }
#define _sub3(a,b,r) { (r)[0]=(a)[0]-(b)[0]; (r)[1]=(a)[1]-(b)[1]; (r)[2]=(a)[2]-(b)[2]; }
#define _dot3(a,b) ((a)[0]*(b)[0]+(a)[1]*(b)[1]+(a)[2]*(b)[2])
#define _cross3(a,b,r) { (r)[0]=(a)[1]*(b)[2]-(b)[1]*(a)[2]; (r)[1]=(a)[2]*(b)[0]-(b)[2]*(a)[0]; (r)[2]=(a)[0]*(b)[1]-(b)[0]*(a)[1]; }

/// A class representing a 2d vector
class Vector2 {
public:
	/// The components of the vector
	real x,y;

	/// Default constructor - does nothing
	Vector2() {}

	/// Initializes the vector with the given components
	Vector2(real ix, real iy) { x=ix; y=iy; }

	/// Zeroes all components of the vector
	void makeZero(void) { x=y=0.0f; }

	/// Returns the i-th (0 or 1) components of the vector
	FORCEINLINE real& operator [](const int index) { return (&x)[index]; }

	/// Returns the i-th (0 or 1) components of the vector as const
	FORCEINLINE const real& operator [](const int index) const { return (&x)[index]; }

	/// Adds the components of the given vector
	FORCEINLINE Vector2& operator+=(const Vector2 &a) {
		x+=a.x;
		y+=a.y;
		return *this;
	}

	/// Multiplies the components of the vector by the given number
	FORCEINLINE Vector2& operator*=(real f) {
		x*=f;
		y*=f;
		return *this;
	}

	/// Sets the components of the vector
	void set(real ix, real iy) { x=ix; y=iy; }

	// Returns the squared length of the vector
	real lengthSqr(void) { return x*x+y*y; }
};

/// \relates Vector2
/// Adds the components of the given vectors
FORCEINLINE Vector2 operator+(const Vector2 &a, const Vector2 &b) {
	return Vector2(a.x+b.x, a.y+b.y);
}

/// \relates Vector2
/// Subtracts the components of the given vectors
FORCEINLINE Vector2 operator-(const Vector2 &a, const Vector2 &b) {
	return Vector2(a.x-b.x, a.y-b.y);
}

/// \relates Vector2
/// Multiplies the components of a vector by a number
FORCEINLINE Vector2 operator*(const Vector2 &a, real f) {
	return Vector2(a.x*f, a.y*f);
}

/// \relates Vector2
/// Divides the components of a vector by a number
FORCEINLINE Vector2 operator/(const Vector2 &a, real f) {
	return Vector2(a.x/f, a.y/f);
}

/// A class for storing compressed unit vectors.
/// The vector is stored with two unsigned 16-bit integer values
/// representing the solid-angle coordinates of the vector.
/// The distribution of the values is uniform over the sphere.
class CompressedUnitVector {
public:
	/// The two unsigned 16-bit values representing the vector
	uint16 u, v;

	/// Default constructor - does nothing
	CompressedUnitVector(void) {}

	/// Initializes the compressed vector with the passed vector, which must have unit length
	CompressedUnitVector(const Vector &a) { init(a); }

	/// Sets the compressed vector to the passed vector, which must have unit length
	void init(const Vector &a) {
		float fu=a.z*0.5f+0.5f;
		float fv=atan2f(a.y, a.x)*0.5f/pi()+0.5f;
		fu=clamp(fu, 0.0f, 1.0f);
		fv=clamp(fv, 0.0f, 1.0f);
		u=fast_floor(fu*65535.0f);
		v=fast_floor(fv*65535.0f);
	}

	/// Converts the compressed vector to a normal one
	operator Vector(void) {
		float fu=((float(u)/65535.0f)-0.5f)*2.0f;
		float fv=(float(v)/65535.0f);

		real thetaCos=sqrtf(1.0f-fu*fu);
		real phi=(fv-0.5f)*2.0f*pi();
		return Vector(cosf(phi)*thetaCos, sinf(phi)*thetaCos, fu);
	}
};

END_VLADO

#endif
