#ifndef __VGL_TRIMESH_H__
#define __VGL_TRIMESH_H__

#include "utils.h"
#include "rasterizer3d.h"

BEGIN_VLADO

class TriMeshFace {
	public:
		int v[3];
		int e[3];
		Color color;

		inline void set(int v0, int v1, int v2) { v[0]=v0; v[1]=v1; v[2]=v2; }
		inline void set(int v0, int v1, int v2, const Color &c) { v[0]=v0; v[1]=v1; v[2]=v2; color=c; }
		inline void setEdges(int e0, int e1, int e2) { e[0]=e0; e[1]=e1; e[2]=e2; }
};

class TriMeshEdge {
	public:
		int v[2];
		bool visible;
		Color color;

		int numFaces; // the number of faces sharing the edge
		int f[2]; // the faces sharing the edge
		int fe[2]; // the index in the faces sharing the edge
		int tv[2]; // the third vertex of the faces sharing the edge
};

class TriMesh {
	void createTetra(Color col);
	void buildVNormals(bool buildTransformed=false);
	public:
		int numFaces;
		TriMeshFace *faces;

		int numVerts;
		Vector *verts;
		Vector *vnormals; // vector normals;

		Vector *tverts; // transformed vertices
		Vector *tvnormals; // transformed vertex normals

		Vector *fnormals; // face normals
		float *farea2; // area of faces multiplied by 2

		Vector *tfnormals; // transformed face normals
		float *tfarea2; // transformed face area multiplied by 2

		// edges are built from faces, may be NULL
		int numEdges;
		TriMeshEdge *edges;

		int stripLength;
		int *strips; // used for drawing

		TriMesh(void) { init(); }
		TriMesh(TriMesh &mesh) { init(); init(mesh); }

		void init(void) {
			numVerts=0;
			verts=NULL;
			tverts=NULL;

			numFaces=0;
			faces=NULL;

			numEdges=0;
			edges=NULL;

			vnormals=NULL;
			tvnormals=NULL;

			fnormals=tfnormals=NULL;
			farea2=tfarea2=NULL;

			stripLength=0;
			strips=NULL;
		}

		void init(TriMesh &mesh);

		~TriMesh(void) {
			freeData();
		}

		void freeData(void) {
			if (verts) delete[] verts;
			verts=NULL;

			if (tverts) delete[] tverts;
			tverts=NULL;

			if (faces) delete[] faces;
			faces=NULL;

			if (edges) delete[] edges;
			edges=NULL;

			if (vnormals) delete[] vnormals;
			vnormals=NULL;

			if (tvnormals) delete[] tvnormals;
			tvnormals=NULL;

			if (fnormals) delete[] fnormals;
			fnormals=NULL;

			if (tfnormals) delete[] tfnormals;
			tfnormals=NULL;

			if (farea2) delete[] farea2;
			farea2=NULL;

			if (tfarea2) delete[] tfarea2;
			tfarea2=NULL;

			if (strips) delete[] strips;
			strips=NULL;
			stripLength=0;
		}

		// returns the normal of the i-th face
		inline Vector getFaceNrm(int i) {
			TriMeshFace &face=faces[i];
			return (verts[face.v[1]]-verts[face.v[0]])^(verts[face.v[2]]-verts[face.v[1]]);
		}

		// returns true if the given face is visible to the point
		inline bool isFaceVisible(Vector p, int faceIdx) {
			TriMeshFace &face=faces[faceIdx];
			return ((p-verts[face.v[0]])*fnormals[faceIdx])>=almostZero;
		}

		// Returns the interpolated normal for the given face
		Vector getNormal(int faceIdx, Vector bary);

		// initializes the mesh with random vertices
		void randomVerts(int n, float size, Vector center);

		// computes the faces of the convex hull for the vertices
		void convexHull(void);

		// removes unused vertices (for example, after convexHull())
		void collapseVerts(void);

		// computes normals
		void buildNormals(void) { buildVNormals(false); }
		void buildTNormals(void) { buildVNormals(true); }

		// compute edges
		void buildEdges(bool deleteOld=true);

		// removes degenerate faces
		void removeZeroFaces(real tolerance=1e-12f);

		// transform the vertices
		void transform(const Transform &tm);

		// builds strips
		void buildStrips(void);

		// returns the bounding box
		void boundingBox(Box &box);

		// splits quad faces
		void splitQuads(void);

		// merges the given mesh
		void merge(TriMesh &mesh);

		void saveToFile(FILE *fp);
		void saveToFile(char *fname);
		void loadFromFile(FILE *fp, real scale=1.0f);
		void loadFromFilePly(FILE *fp, real scale=1.0f);
		void loadFromFile(char *fname, real scale=1.0f);

		// various primitives
		void makeCube(real size=1.0f) { makeCube(size, size, size); }
		void makeCube(real a, real b, real c);

		void makeRegularPlane(real a, real b, int subdivsx, int subdivsy, real delone=0.0f, bool alternate=true);
		void makeDelonePlane(real a, real b, int edgeSubdivs, int numPoints);

		void makeBox(Box &box);
		void makeSphere(float radius, int subdivs);
		void makeCylinder(float radius, float height, int segments);

		// Drawing functions
		void drawWire(Rasterizer3D &rs, const Color &wireColor, bool drawTransformed=true);
		void drawShadedWire(Rasterizer3D &rs, const Color &wireColor, bool drawTransformed=true);
		void drawShaded(Rasterizer3D &rs, const Color &faceColor, bool drawTransformed=true);
		void drawFaceted(Rasterizer3D &rs, const Color &faceColor, bool drawTransformed=true);

		uint64 getMemUsage(void);
};

extern void loopSubdivide(TriMesh &mesh, TriMesh &smesh, int numSteps);

END_VLADO

#endif
