#ifndef _THREADS_WIN32_H_
#define _THREADS_WIN32_H_

#ifndef _WIN32
#	error This file is for Win32 platform only
#else

#define MAX_THREADS 8

BEGIN_VLADO

//**********************************************
// A simple critical section
class CriticalSection {
public:
	CRITICAL_SECTION csect;

	CriticalSection(void) { InitializeCriticalSection(&csect); }
	~CriticalSection(void) { DeleteCriticalSection(&csect); }

	void enter(void) { EnterCriticalSection(&csect); }
	void leave(void) { LeaveCriticalSection(&csect); }
	// int tryEnter(void) { return TryEnterCriticalSection(&csect); }
};

//**********************************************
// A spin lock. Note that a spin lock is NOT recursive
class SpinLock {
	long counter;
public:
	SpinLock(void) { counter=0; }
	void enter(void) { while (1==InterlockedExchange(&counter, 1)); }
	void leave(void) { InterlockedExchange(&counter, 0); }
};

//**********************************************
// A fast critical section
class FastCriticalSection {
	CRITICAL_SECTION csect;
public:
#if _WIN32_WINNT >= 0x0410
	FastCriticalSection(void) { InitializeCriticalSectionAndSpinCount(&csect, 0x7FFFFFFF); }
#else
#pragma message("[FastCriticalSection] _WIN32_WINNT < 0x0410; cannot use InitializeCriticalSectionAndSpinCount()")
	FastCriticalSection(void) { InitializeCriticalSection(&csect); }
#endif
	~FastCriticalSection(void) { DeleteCriticalSection(&csect); }

	void enter(void) { EnterCriticalSection(&csect); }
	void leave(void) { LeaveCriticalSection(&csect); }
};

/*/
class FastCriticalSection {
	SpinLock spinLock;
	int numReentries;
	DWORD owningThread;
public:
	FastCriticalSection(void) { numReentries=0; owningThread=0; }

	void enter(void) {
		DWORD threadID=GetCurrentThreadId();
		spinLock.enter();
		if (owningThread==threadID) numReentries++;
		else {
			while (owningThread!=0) {
				spinLock.leave();
				Sleep(0);
				spinLock.enter();
			}
			owningThread=threadID;
			numReentries++;
		}
		spinLock.leave();
	}
	void leave(void) {
		DWORD threadID=GetCurrentThreadId();
		spinLock.enter();
		numReentries--;
		if (numReentries==0) owningThread=0;
		spinLock.leave();
	}
};
*/
//**********************************************
// A simple blocking event; note that when the event is set, only ONE waiting thread will be woken
class Event {
	HANDLE event;
public:
	Event(void) { event=CreateEvent(NULL, FALSE, FALSE, NULL); }
	~Event(void) { CloseHandle(event); }

	// This waits for the event to be signalled; called by each thread that wants
	// to wait for the object; if timeout is other than zero, this waits for the specified
	// amount of time in milliseconds; returns true if the event was signalled
	// returns false if there was a timeout
	int wait(int timeout) {
		DWORD res=WaitForSingleObject(event, (timeout==0)? INFINITE : timeout);
		return (res!=WAIT_TIMEOUT);
	}

	// This wakes a single thread waiting for the event
	void set(void) { SetEvent(event); }
};

//**********************************************
// A simple variable with locked increment and decrement
class InterlockedCounter {
	volatile long counter;
public:
	long increment(void) { return InterlockedIncrement(&counter); }
	long decrement(void) { return InterlockedDecrement(&counter); }

	long set(long i) { return InterlockedExchange(&counter, i); }
	long get(void) { return counter; }

	long clear(void) { return set(0); }

	long operator++(int) { return increment(); }
	long operator--(int) { return decrement(); }
};

/**********************************************
	A read-write lock; reading can be asynchronous
	Writing is synchronised (i.e. several threads can read the object
	but only a single thread can write (modify) it).
	For proper work, you need to keep a separate ThreadCounter
	for each thread; you must pass this counter when requesting a lock
	This allows recursive read/write locks as well as mixing read and
	write locks (a thread can acquire a write lock when it already has
	a read lock).
************************************************/

struct ThreadCounter {
	int readers;
	ThreadCounter(void) { readers=0; }
};

typedef int (*RW_WAIT_CALLBACK)(void *ptr);

// Note that if a recursive read/write lock is required, the ThreadCounter parameter
// MUST be supplied and it MUST be the same for a given thread always
class FastReadWriteLock {
	FastCriticalSection lock;
	volatile int readers, writers, pendingWriters;
	ThreadCounter * volatile writerID;
public:
	FastReadWriteLock(void) { readers=writers=pendingWriters=0; writerID=NULL; }

	void lockRead(ThreadCounter &counter) {
		lock.enter();
		readers++;
		counter.readers++;
		lock.leave();
	}

	void unlockRead(ThreadCounter &counter) {
		lock.enter();
		readers--;
		counter.readers--;
		lock.leave();
	}

	int lockWrite(ThreadCounter &counter, int exclusive=false, RW_WAIT_CALLBACK waitFunc=NULL, void *funcParam=NULL) {
		lock.enter();
		// If the current thread is a writer already, just increment the writer count
		if (writerID==&counter) {
			writers++;
			return true;
		}
		// Subtract the readers for the current thread
		readers-=counter.readers;
		if (!exclusive) pendingWriters++;
		// Wait for other reading and writing operations to complete
		while (readers || writerID || (exclusive && pendingWriters)) {
			lock.leave();
			Sleep(0);
			lock.enter();
			// Check to see if the callback signals that we don't have to wait any longer
			// This is only possible if there are no other writers
			if (!writerID && waitFunc && waitFunc(funcParam)) {
				readers+=counter.readers;
				if (!exclusive) pendingWriters--;
				lock.leave();
				return false;
			}
		}
		if (!exclusive) pendingWriters--;
		writers++;
		writerID=&counter;
		return true;
	}

	void unlockWrite(ThreadCounter &counter) {
		writers--;
		if (writers==0) {
			writerID=NULL;
			readers+=counter.readers;
		}
		lock.leave();
	}
};

/**********************************************
	A simple read-write lock; reading can be asynchronous;
	writing is synchronised (i.e. several threads can read the object
	but only a single thread can write (modify) it).
	This lock allows recursion with respect to reading and writing
	separately, but does not allow mixing them - a deadlock will occur
	(a thread that has a read lock cannot obtain a write lock).
************************************************/

class SimpleReadWriteLock {
	FastCriticalSection lock;
	volatile int readers, writers;
	volatile DWORD writerID;
public:
	SimpleReadWriteLock(void) { readers=writers=0; writerID=0; }

	void lockRead(void) {
		lock.enter();
		readers++;
		lock.leave();
	}

	void unlockRead(void) {
		lock.enter();
		readers--;
		lock.leave();
	}

	void lockWrite(void) {
		lock.enter();
		DWORD threadID=GetCurrentThreadId();
		if (writerID==threadID) {
			writers++;
			return;
		}
		while (readers || writerID) {
			lock.leave();
			Sleep(0);
			lock.enter();
		}
		writers++;
		writerID=threadID;
		// lock.leave();
	}

	void unlockWrite(void) {
		// lock.enter();
		writers--;
		if (writers==0) writerID=0;
		lock.leave();
	}
};

class DoneEvent {
	friend class ThreadManager;
	CriticalSection csect;

	int count;
	HANDLE handle;
public:

	DoneEvent(void) {
		handle=NULL;
		count=0;
	}

	~DoneEvent(void) {
		if (handle) CloseHandle(handle);
	}

	// Sets the number of threads to be waited for; returns cnt if
	// the count was set, zero otherwise
	int setCount(int cnt) {
		csect.enter();
		if (count) { csect.leave(); return 0; }
		if (!handle) handle=CreateEvent(NULL, TRUE, FALSE, NULL);
		count=cnt;
		csect.leave();

		ResetEvent(handle);

		return cnt;
	}

	// Decrements the counter by one when a thread is finished
	// If the counter becomes zero, releases the waiting thread
	void decCount(void) {
		csect.enter();
		count--;
		int newCount=count;
		csect.leave();

		if (newCount==0) SetEvent(handle);
	}

	int wait(int waitInterval) { return WaitForSingleObject(handle, waitInterval); }
	int waitEvents(int waitInterval) { return MsgWaitForMultipleObjects(1, &handle, FALSE, waitInterval, QS_ALLPOSTMESSAGE | QS_ALLINPUT); }
};

class ThreadLocalStorage {
public:
	DWORD index;

	ThreadLocalStorage(void) { index=TlsAlloc(); }
	~ThreadLocalStorage(void) { TlsFree(index); }

	void set(void *ptr) { TlsSetValue(index, ptr); }
	void* get(void) { return TlsGetValue(index); }
};

END_VLADO

#endif

#endif
