/**********************************************************************
	FILE: sequence.h
	DESCRIPTION: A helper class for storing a large (variable) number of things
	Copyright (c) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32

	HISTORY:
		13 March 2003
			- Sequence and SequenceST are both based on a common template class _Sequence
		13 June 2002
			- added the SEQUENCE_ALIGN option which will cause aligned allocation of memory
		4 Jan 2002
			- fixed newArray when the size of the array is exactly equal to the block size
		2 Jan 2002
			- added class SequenceST - a single-threaded version of Sequence
***********************************************************************/

#ifndef __THREADS_H__
#define __THREADS_H__

#include "progress.h"

#ifdef _WIN32
#	include "threads_win32.h"
#else
#	include "threads_linux.h"
#endif

BEGIN_VUTILS

// A volatile resource class, that can be either valid, or invalid
// In addition to being lock-able
class MovableResource {
	volatile int valid;
	volatile int deleteIt;
	InterlockedCounter locked;
	CriticalSection csect;
public:
	MovableResource(void) { clear(); }
	void clear(void) { valid=false; locked.set(0); deleteIt=false; }

	void* fastLock(void); // Returns a pointer to the resource and locks it, if the resource is valid; NULL otherwise
	void* lock(void); // Validates the resource, locks it, and a returns a pointer to it
	void unlock(void); // Unlocks the resource; invalidates it if the resource is marked for invalidating
	void invalidate(void); // Invalidates the resource, if it is unlocked; otherwise just marks it for invalidating

	virtual void makeValid(void)=0; // This is called to actually validate the resource; thread-safety guaranteed
	virtual void makeInvalid(void)=0; // This is called to actually invalidate the resource; thread-safety guaranteed
	virtual void *getData(void)=0; // Returns a pointer to the resource
};

class ThreadManager;

//**********************************************
/// A basic multithreaded class. An instance of this class is passed to ThreadManager::runThreads().
class MultiThreaded {
public:
	/// Executes the thread procedure with the given number of threads. This simply calls threadman->runThreads().
	/// @param threadman The thread manager that will be called to execute the threads.
	/// @param numThreads The number of threads to run in parallel. 0 means to run as many threads as there are processors.
	/// @param prog A callback to track the progress of the calculations.
	/// @param showProgress If true, the callback will be called to track the progress.
	void run(ThreadManager *threadman, int numThreads=0, Vlado::ProgressCallback *prog=NULL, int showProgress=true);

	/// This is the thread procedure that performs the actual work.
	/// @param abort You should periodically check this parameter; if true, the calculations must be aborted.
	/// @param progress This must be updated (incremented) as the calculations progress.
	/// @param threadIdx The zero-based index of this thread.
	/// @param numThreads The total number of threads, execting the thread procedure in parallel.
	virtual int threadProc(volatile int &abort, volatile int &progress, int threadIdx, int numThreads)=0;
};

/// An instance of this class is passed to ThreadManager::runThread().
class ThreadProcedure {
public:
	/// The procedure that will be executed in a separate thread.
	/// @param param This is the parameter passed to ThreadManager::runThread().
	virtual void proc(void *param)=0;
};

//**********************************************
// A helper multithreaded class for a simple "for" cycle
class MultiThreadedFor: private MultiThreaded {
	CriticalSection csect;
	unsigned char *processed;
	int count;
	int threadProc(volatile int &abort, volatile int &progress, int threadIdx, int numThreads);
public:
	// This is called to run the cycle
	void run(ThreadManager *threadman, int numIterations, int numThreads=0, Vlado::ProgressCallback *p=NULL, int showProgress=true);

	// This does the actual work
	virtual void body(int index, volatile int &abort, int threadIdx, int numThreads)=0;
};

//**********************************************
// Global functions

// Returns the zero-based index of the currently running thread
int getThreadIndex(void);

/// @relates ThreadManager
/// Returns the number of system processors.
int getNumProcessors(void);

/// @relates ThreadManager
/// Thread priorities supported by a thread manager
enum ThreadPriority { normalPriority, lowPriority };

/// This class provides high-level support for multithreading, independent of the OS
class ThreadManager {
public:
	/// Frees any memory associated with the thread manager; destroys all threads.
	virtual void freeMem(void)=0;

	/// Executes the given multithreaded class.
	/// @param numThreads The number of threads to run in parallel; 0 will create as many threads as there are processors in the system.
	/// @param multi The procedure to execute in parallel.
	/// @param prog An optional callback to track the progress of the calculations.
	/// @param showProgress If this is true, the passed progress callback will be called to update itself during the course of the calculations.
	virtual int runThreads(int numThreads, MultiThreaded *multi, VUtils::ProgressCallback *prog, int showProgress=true)=0;

	/// Sets the thread priority
	/// @param threadPriority The desired thread priority.
	virtual void setThreadPriority(ThreadPriority threadPriority)=0;

	/// Executes the given function in a separate thread; returns control immediately.
	/// @param threadProc The thread procedure to execute.
	/// @param param An optional parameter to pass to the thread procedure.
	virtual void runThread(ThreadProcedure &threadProc, void *param=NULL)=0;

	/// Sets the stack size for newly created threads; this does not affect already created threads.
	/// If you want this to affect all threads, you must first destroy any current threads with
	/// freeMem().
	/// @param stackSize The stack (in bytes) for the newly created threads; 0 means the same as the calling thread.
	virtual void setThreadStack(uint32 stackSize)=0;
};

/// @relates ThreadManager
/// Creates a new default thread manager
extern ThreadManager *newDefaultThreadManager(void);

/// @relates ThreadManager
/// Destroys a thread manager returned from newDefaultThreadManager()
extern void deleteDefaultThreadManager(ThreadManager *threadman);

END_VUTILS

#endif
