/**********************************************************************
	FILE: sequence.h
	DESCRIPTION: A helper class for storing a large (variable) number of things
	Copyright (c) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32

	HISTORY:
		13 March 2003
			- Sequence and SequenceST are both based on a common template class _Sequence
		13 June 2002
			- added the SEQUENCE_ALIGN option which will cause aligned allocation of memory
		4 Jan 2002
			- fixed newArray when the size of the array is exactly equal to the block size
		2 Jan 2002
			- added class SequenceST - a single-threaded version of Sequence
***********************************************************************/
#ifndef __SEQUENCE_H__
#define __SEQUENCE_H__

#include "threads.h"

BEGIN_VLADO

template<class T, bool justAlloc>
class _SequenceBlock {
	public:
		_SequenceBlock *next, *prev;
		int numElements, maxElements;
		T *elements;

		_SequenceBlock(void) { next=prev=NULL; elements=NULL; numElements=0; maxElements=0; }
		_SequenceBlock(int numEls) {
			next=prev=NULL;

#ifndef SEQUENCE_ALIGN
			if (justAlloc) {
				elements=(T*) malloc(sizeof(T)*numEls);
			} else {
				elements=new T[numEls];
			}
#else
			elements=(T*) _aligned_malloc(sizeof(T)*numEls, __alignof(T));
#endif
			if (!elements) throw "[_SequenceBlock] Could not allocate memory";

			numElements=0;
			maxElements=numEls;
		}
		~_SequenceBlock(void) { freeData(); }

		void freeData(void) {
#ifndef SEQUENCE_ALIGN
			if (elements) {
				if (justAlloc) free(elements);
				else delete[] elements;
			}
#else
			if (elements) _aligned_free(elements);
#endif
			elements=NULL;
		}
};

template<class T, int justAlloc>
class _SequenceCounter {
	public:
		_SequenceBlock<T, justAlloc> *cblock;
		int cindex;
};

template<class T, int maxEls, bool justAlloc, bool threaded>
class _Sequence {
	public:
		typedef _SequenceBlock<T, justAlloc> Block;
		typedef _SequenceCounter<T, justAlloc> Counter;
	private:
		Block *list, *lastBlock, *cblock;
		int cindex;
		int cnt;
		int maxBlockElements;
		CriticalSection csect;
		void _removeLastArray(T *arr, int numItems) {
			if ((lastBlock->numElements)>=numItems) {
				lastBlock->numElements-=numItems;
				memcpy(arr, lastBlock->elements+lastBlock->numElements, sizeof(T)*numItems);
				cnt-=numItems;
			} else {
				int numCopy=(lastBlock->numElements);
				memcpy(arr+numItems-numCopy, lastBlock->elements, sizeof(T)*numCopy);
				cnt-=numCopy;
				lastBlock->numElements=0;
				if (lastBlock->prev) lastBlock=lastBlock->prev;
				_removeLastArray(arr, numItems-numCopy);
			}
		}
	public:
		_Sequence(void) { init(); }

		_Sequence(const _Sequence &s) { init(s); }

		~_Sequence(void) {
			unInit();
		}

		void init(int maxBlockElements=maxEls) {
			list=NULL;
			lastBlock=NULL;
			cblock=NULL;
			cindex=0;
			cnt=0;
			this->maxBlockElements=maxBlockElements;
		}

		void init(const _Sequence &s) {
			list=NULL;
			lastBlock=NULL;
			cblock=NULL;
			cindex=0;
			cnt=0;
			maxBlockElements=s.maxBlockElements;
		}

		void unInit(void) {
			freeData();
		}

		// Does not release memory
		void clear(void) {
			Block *ptr=list;
			while (ptr) {
				ptr->numElements=0;
				ptr=ptr->next;
			}
			lastBlock=list;
			cnt=0;
			cblock=list;
			cindex=0;
		}

		// Frees the memory
		void freeData(void) {
			Block *ptr=list;
			while (ptr) {
				Block *next=ptr->next;
				ptr->freeData();
				delete ptr;
				ptr=next;
			}
			list=NULL;
			cblock=NULL;
			cindex=0;
			lastBlock=NULL;
			cnt=0;
		}

		void setMaxBlockElements(int maxElements) { maxBlockElements=maxElements; }

		T *newElement(void) {
			if (threaded) csect.enter();
			Block *prev=lastBlock? lastBlock->prev: NULL;
			while (lastBlock) {
				if ((lastBlock->numElements) < (lastBlock->maxElements)) {
					cnt++;
					T *res=(lastBlock->elements)+(lastBlock->numElements++);
					if (threaded) csect.leave();
					return res;
				}
				prev=lastBlock;
				lastBlock=lastBlock->next;
			}

			// No free space, a new block must be allocated
			lastBlock=new Block(maxBlockElements);
			lastBlock->prev=prev;
			if (prev) prev->next=lastBlock;
			else { list=lastBlock; }

			cnt++;
			T *res=(lastBlock->elements)+(lastBlock->numElements++);

			if (threaded) csect.leave();
			return res;
		}

		void storeElement(T &element) { *newElement()=element; }

		// Returns a pointer to an array of elements
		T *newArray(int numEls) {
			if (threaded) csect.enter();
			Block *prev=lastBlock? lastBlock->prev: NULL;
			while (lastBlock) {
				if ((lastBlock->numElements)+numEls <= (lastBlock->maxElements)) {
					cnt+=numEls;
					T *res=(lastBlock->elements)+(lastBlock->numElements);
					(lastBlock->numElements)+=numEls;
					if (threaded) csect.leave();
					return res;
				}
				prev=lastBlock;
				lastBlock=lastBlock->next;
			}
			// No free space, a new block must be allocated
			lastBlock=new Block(Max(maxBlockElements, numEls));
			lastBlock->prev=prev;
			if (prev) prev->next=lastBlock;
			else { list=lastBlock; }

			cnt+=numEls;
			T *res=(lastBlock->elements);
			(lastBlock->numElements)+=numEls;

			if (threaded) csect.leave();
			return res;
		}

		// Returns a pointer to the first element in the sequence
		T *firstElement(void) {
			if (!list) return NULL;
			return list->elements;
		}

		// Returns a pointer to the last element in the sequence
		T *lastElement(void) {
			if (!lastBlock) return NULL;
			if (lastBlock->numElements==0) {
				Block *b=lastBlock->prev;
				if (!b) return NULL;
				return (b->elements)+(b->numElements-1);
			}
			return (lastBlock->elements)+(lastBlock->numElements-1);
		}

		void removeElement(T *element) {
			if (threaded) csect.enter();

			if (!lastBlock) return;
			if (lastBlock->numElements==0) {
				lastBlock=lastBlock->prev;
				if (!lastBlock) return;
			}
			*element=lastBlock->elements[--lastBlock->numElements];
			cnt--;

			if (threaded) csect.leave();
		}

		// Removes the last element from the sequence
		void removeLast(void) {
			if (threaded) csect.enter();
			if (lastBlock) {
				if (lastBlock->numElements==0) lastBlock=lastBlock->prev;
				if (lastBlock) {
					(lastBlock->numElements)--;
					cnt--;
				}
			}
			if (threaded) csect.leave();
		}

		// Removes the last element and copies it into the given pointer
		// Does nothing if there aren't any elements into the sequence
		void removeLast(T *ptr) {
			if (threaded) csect.enter();

			if (!lastBlock) return;
			if (lastBlock->numElements==0) { lastBlock=lastBlock->prev; if (!lastBlock) return; }

			*ptr=lastBlock->elements[--(lastBlock->numElements)];
			cnt--;

			if (threaded) csect.leave();
		}

		// Removes and stores the given number of elements from the end
		void removeLastArray(T *arr, int numItems) {
			if (threaded) csect.enter();
			_removeLastArray(arr, numItems);
			if (threaded) csect.leave();
		}

		// Returns the number of elements in the sequence
		int count(void) {
			return cnt;
		}

		// Resets the counter back to the beginning
		void resetCounter(void) {
			cblock=list;
			cindex=0;
		}

		// Resets the given counter to the start of the sequence
		void resetCounter(Counter &counter) {
			counter.cblock=list;
			counter.cindex=0;
		}

		// Returns a pointer to the current element and moves the counter forward
		T *nextElement(void) {
			if (!cblock) return NULL;
			if (cindex<cblock->numElements) return &(cblock->elements[cindex++]);
			if (!cblock->next) return NULL;
			cblock=cblock->next;
			cindex=0;
			return nextElement();
		}

		// Returns a pointer to the current element and moves the counter forward
		T *nextElement(Counter &counter) {
			if (!counter.cblock) return NULL;
			if (counter.cindex<counter.cblock->numElements) return &(counter.cblock->elements[counter.cindex++]);
			if (!counter.cblock->next) return NULL;
			counter.cblock=counter.cblock->next;
			counter.cindex=0;
			return nextElement(counter);
		}

		// Returns the counter corresponding to the element with the given index
		void getElementCounter(int index, Counter &counter) {
			Block *plist=list;
			int cidx=0;
			while (plist) {
				if (cidx+plist->numElements > index) {
					counter.cblock=plist;
					counter.cindex=index-cidx;
					return;
				}
				cidx+=plist->numElements;
				plist=plist->next;
			}
			counter.cblock=NULL;
			counter.cindex=0;
		}

		// Returns the amount of memory allocated for the dynamic structures
		unsigned int getMemUsage(void) {
			Block *block=list;
			unsigned int nbytes=0;
			while (block) {
				nbytes+=block->maxElements*sizeof(T)+sizeof(Block);
				block=block->next;
			}
			return nbytes;
		}

		_Sequence &operator=(const _Sequence &s) {
			maxBlockElements=s.maxBlockElements;
			return *this;
		}
};

template<class T, int maxEls=20000, bool justAlloc=false>
class Sequence: public _Sequence<T, maxEls, justAlloc, true> {
public:
	Sequence(int maxElements=maxEls) { init(maxElements); }
	Sequence(Sequence &s) { init(s); }
	~Sequence(void) { unInit(); }
};

template<class T, int maxEls, bool justAlloc=false>
class SequenceST: public _Sequence<T, maxEls, justAlloc, false> {
public:
	SequenceST(int maxElements=maxEls) { init(maxElements); }
	SequenceST(SequenceST &s) { init(s); }
	~SequenceST(void) { unInit(); }
};

END_VLADO

#endif
