#ifndef __VGL_CAMERAS_H__
#define __VGL_CAMERAS_H__

#include "misc_ray.h"

BEGIN_VLADO

struct RayDeriv {
	Vector dPdx;
	Vector dPdy;
	Vector dDdx;
	Vector dDdy;
	void Init()	{ dPdx = dPdy = dDdx = dDdy = Vector(0,0,0); }
};

class Camera {
	public:
		// Returns false when there are no rays for the associated pixel
		virtual bool GetScreenRay(double u, double v, float dof_uc, float dof_vc, Ray& ray, RayDeriv* p=NULL)=0;
		virtual void MapToScreen(const Vector &p, float &xs, float &ys)=0;
		virtual void update(float time) {}

		virtual void LookAt(const Vector &from, const Vector &to, const Vector &upDir) {
			mPos	 = from;
			mForward = normalize(to-from);
			mRight	 = normalize(mForward^upDir);
			mUp		 = normalize(mRight^mForward);
		}

		virtual void SetPos(const Vector &pos, const Vector &forward, const Vector &up, const Vector &right) {
			mPos=pos;
			mRight=right;
			mUp=up;
			mForward=forward;
		}

		virtual void SetClippingPlanes(bool usePlanes, float nearPlane, float farPlane) {
			useClippingPlanes=usePlanes;
			clipNear=nearPlane;
			clipFar=farPlane;
		}

		void ClipRay(const Ray &ray, float &mint, float &maxt) {
			if (!useClippingPlanes) { mint=0.0f, maxt=1e18f; return; }

			float zslope=ray.dir*mForward;
			if (fabs(zslope)<1e-12f) return;

			float zval=GetZValue(ray.p);

			float t0=(clipNear-zval)/zslope;
			float t1=(clipFar-zval)/zslope;

			if (t0<0.0f) t0=0.0f;

			mint=t0;
			maxt=t1;
		}

		virtual float GetZValue(const Vector &p) {
			return (p-mPos)*mForward;
		}

		virtual Vector mapToCamSpace(const Vector &p) { return p; }

		// Camera position & orientation
		Vector	mPos;
		Vector	mRight;
		Vector	mUp;
		Vector	mForward;
		float width, height;

		bool useClippingPlanes;
		float clipNear, clipFar;

	protected:
		Vector GetNormDeriv(const Vector& deriv, const Vector& dir, float dirLen)
		{ return ((deriv*dirLen)-(dir*((dir/dirLen)*deriv)))/(dirLen*dirLen); }
};

class PinholeCamera: public Camera {
	public:
		float fov;
		float xscale, yscale;
		float width, height;
		float dofShutter, dofDist;
		float biasPower;
		int sidesOn, numSides;
		float rotation;

		PinholeCamera(void) { Init(); }
		PinholeCamera(float w, float h, float fov, float aspect=1.0f) { Init(w, h, fov, aspect); }

		void Init(void) {
			width=320.0f;
			height=400.0f;
			mPos.makeZero();
			mForward.set(0.0f, 1.0f, 0.0f);
			mUp.set(0.0f, 0.0f, 1.0f);
			mRight.set(1.0f, 0.0f, 0.0f);
			fov=45*pi()/180.0f;
			biasPower=1.0f;
		}

		void Init(float w, float h, float fieldOfView, float aspect=1.0f, float shutter=0.0f, float focalDist=1.0f, float bias=0.0f, int sidesOn=false, int numSides=6, float rotation=0.0f) {
			width=w;
			height=h;
			dofShutter=shutter;
			dofDist=focalDist;
			fov=fieldOfView; // fieldOfView*pi/180.0f;
			xscale=tanf(fov*0.5f)*2.0f/width;
			yscale=-xscale/aspect;
			biasPower=(bias>0.0f)? 1.0f/(1.0f+bias) : 1.0f-bias;
			this->sidesOn=sidesOn;
			this->numSides=numSides;
			this->rotation=rotation;
		}

		// From Camera
		bool GetScreenRay(double xs, double ys, float dof_uc, float dof_vc, Ray& ray, RayDeriv* pDeriv) {
			float fx, fy;
			if (!sidesOn) Vlado::getDiscPoint(powf(dof_uc, biasPower), dof_vc, fx, fy);
			else Vlado::getNGonPoint(numSides, rotation, powf(dof_uc, biasPower), dof_vc, fx, fy);
			fx*=dofShutter;
			fy*=dofShutter;

			ray.p=mPos+mRight*fx+mUp*fy;

			double u=(xs-width*0.5f)*xscale;
			double v=(ys-height*0.5f)*yscale;

			// The line below replaces these two lines (keeps precision better)
			// --> Vector fpt=mPos+(mRight*u - mUp*v + mForward)*dofDist;
			// --> ray.dir=fpt-ray.p;

			ray.dir=mRight*float(u*dofDist-fx) + mUp*float(v*dofDist-fy) + mForward*dofDist;
			float dirLen = length(ray.dir);

			// if Ray derivatives are needed, then compute them
			if (pDeriv)
			{
				pDeriv->dPdx = Vector(0,0,0);
				pDeriv->dPdy = Vector(0,0,0);

				pDeriv->dDdx = mRight*dofDist*xscale;
				pDeriv->dDdy = mUp*dofDist*yscale;
				pDeriv->dDdx = GetNormDeriv(pDeriv->dDdx, ray.dir, dirLen);
				pDeriv->dDdy = GetNormDeriv(pDeriv->dDdy, ray.dir, dirLen);
			}

			ray.dir/=dirLen;

			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			Vector offs=p-mPos;

			Vector sp(offs*mRight, offs*mUp, offs*mForward);
			if (sp.z<0.0f) { xs=ys=0.0f; return; }
			sp/=sp.z;

			xs=sp.x/xscale+width*0.5f;
			ys=sp.y/yscale+height*0.5f;
		}

		void update(float time){}

		Vector mapToCamSpace(const Vector &p) {
			Vector offs=p-mPos;

			Vector sp(offs*mRight, offs*mUp, offs*mForward);

			sp.x=sp.x/(sp.z*xscale)+width*0.5f;
			sp.y=sp.y/(sp.z*yscale)+height*0.5f;
			sp.z=1.0f/sp.z;

			return sp;
		}
};

class OrthoCamera: public Camera
{
	public:
		OrthoCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 100.0f, 100.0f);
		}

		void Init(float w, float h, float CamWidth, float CamHeight)
		{
			width		= w;
			height		= h;
			mCamWidth	= CamWidth*2.0f;
			mCamHeight  = CamHeight*2.0f*(h/w);
			mXScale     = 1.0f/w;
			mYScale		= 1.0f/h;
		}

		virtual bool GetScreenRay(double xs, double ys, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{

			CamRay.p	= mPos + float(xs*mXScale-0.5f)*mCamWidth*mRight + float(0.5f-ys*mYScale)*mCamHeight*mUp;

//			CamRay.p	= mPos + (u-0.5f)*mWidth*mRight + (0.5f-v)*mHeight*mUp;
			CamRay.dir	= mForward;
			if(pDeriv)
			{
				pDeriv->dPdx = (mCamWidth*mXScale)*mRight*1.0f;
				pDeriv->dPdy = (-mCamHeight*mYScale)*mUp*1.0f;
				pDeriv->dDdx = Vector(0,0,0);
				pDeriv->dDdy = Vector(0,0,0);
			}
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float mCamWidth;
		float mCamHeight;
		float mXScale;
		float mYScale;
};

class BoxCamera: public Camera
{
	public:
		BoxCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 1.0f, 1.0f, 1.0f, 0.0f);
		}


		void Init(float w, float h, float x, float y, float z, float borderSize) 
		{ Init(w, h, Vector(x, y, z), borderSize); }

		void Init(float w, float h, const Vector& box, float borderSize)
		{
			width		= w;
			height		= h;
			mBox		= box;
			mHalfBox	= box*0.5f;
			mScaleU		= (2.0f*mBox.z + mBox.x)/width;
			mScaleV		= 2.0f*(mBox.z + mBox.y)/height;
			mX1 = mBox.z;
			mX2 = mX1 + mBox.x;
			mY1 = mBox.y;
			mY2 = mY1 + mBox.z;
			mY3 = mY1 + mY2;
			mBorderSize=borderSize;
		}

		virtual bool GetScreenRay(double sx, double sy, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			Vector dir;	
			if(!BoxIsec(float(sx*mScaleU), float(sy*mScaleV), dir, pDeriv))
				return false;
			float dirLen = length(dir);

			if(pDeriv)
			{
				pDeriv->dPdx = Vector(0,0,0);
				pDeriv->dPdy = Vector(0,0,0);
				pDeriv->dDdx = GetNormDeriv(pDeriv->dDdx, dir, dirLen);
				pDeriv->dDdy = GetNormDeriv(pDeriv->dDdy, dir, dirLen);
				pDeriv->dDdx = mRight*pDeriv->dDdx.x + mForward*pDeriv->dDdx.y + mUp*pDeriv->dDdx.z;
				pDeriv->dDdy = mRight*pDeriv->dDdy.x + mForward*pDeriv->dDdy.y + mUp*pDeriv->dDdy.z;
			}
			CamRay.dir	= normalize( mRight*dir.x + mForward*dir.y + mUp*dir.z);
			CamRay.p	= mPos;
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		Vector	mHalfBox;
		Vector	mBox;
		float	mScaleU;
		float	mScaleV;
		float   mX1;
		float   mX2;
		float   mY1;
		float   mY2;
		float   mY3;
		float		mBorderSize;

		/*	 0  1  0
		 *   2  3  4
		 *	 0  5  0
		 *	 0  6  0
		 */

		bool BoxIsec(float u, float v, Vector& vec,RayDeriv* pDeriv)
		{
			if (u <= mX1 && v <= mY2+mBorderSize*mScaleV && v >= mY1-mBorderSize*mScaleV) {
				// region 2
				vec.x = 0;
				vec.y = mBox.y + u - mX1;
				vec.z = mBox.z - v + mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->dDdx = Vector(0.0f, mScaleU,     0.0f);
					pDeriv->dDdy = Vector(0.0f,    0.0f, -mScaleV);
				}
				return true;
			}

			if (u >= mX2 && v <= mY2+mBorderSize*mScaleV && v >= mY1-mBorderSize*mScaleV) {
				// region 4
				vec.x = mBox.x;
				vec.y = mBox.y + mX2 - u;
				vec.z = mBox.z - v + mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->dDdx = Vector(0.0f, -mScaleU,     0.0f);
					pDeriv->dDdy = Vector(0.0f,     0.0f, -mScaleV);
				}
				return true;
			}

			if (u >= mX1-mBorderSize*mScaleU && u <= mX2+mBorderSize*mScaleU && v <= mY1) {
				// region 1
				vec.y = v;
				vec.z = mBox.z;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->dDdy = Vector(0.0f, mScaleV, 0.0f);
					pDeriv->dDdx = Vector(mScaleU, 0.0f, 0.0f);
				}
				return true;
			}

			if (u >= mX1-mBorderSize*mScaleU && u <= mX2+mBorderSize*mScaleU && v >= mY3) {
				// region 6
				vec.y = 0.0f;
				vec.z = v - mY3;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->dDdy = Vector(0.0f, 0.0f, +mScaleV);
					pDeriv->dDdx = Vector(mScaleU, 0.0f, 0.0f);
				}
				return true;
			}

			if (u >= mX1-mBorderSize*mScaleU && u <= mX2+mBorderSize*mScaleU && v <= mY3 && v>mY2) {
				// region 5
				vec.y = mBox.y + mY2 - v;
				vec.z = 0.0f;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if (pDeriv) {
					pDeriv->dDdy = Vector(0.0f, -mScaleV, 0.0f);
					pDeriv->dDdx = Vector(mScaleU, 0.0f, 0.0f);
				}
				return true;
			}

			if (u >= mX1 && u <= mX2 && v <= mY2 && v >= mY1) {
				// region 3
				vec.y = mBox.y;
				vec.z = mBox.z - v + mY1;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->dDdx = Vector(mScaleU, 0.0f, 0.0f);
					pDeriv->dDdy = Vector(0.0f, 0.0f, -mScaleV);
				}
				return true;
			}

			return false;
		}
};


/*
 *  class SphereCamera
 */
class SphereCamera: public Camera
{
	public:
		SphereCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f);
		}

  		void Init(float w, float h, float FovX, float FovY)
		{
			width=w;
			height=h;
			float aspect=(height/width)*pi();

			xscale=FovX*0.5f/(width*pi());
			yscale=Vlado::Min(FovY*aspect*0.5f, pi())/(height*pi());
		}

		virtual bool GetScreenRay(double xs, double ys, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			double u=(xs-0.5f*width)*xscale+0.25f;
			double v=(0.5f*height-ys)*yscale*2.0f;
			if (v<-1.0f) v+=2.0f;
			if (v>1.0f) v-=2.0f;

			double CosPhi = cos(u*2.0f*pi());
			double SinPhi = sin(u*2.0f*pi());
			double CosTheta = cos(pi()*0.5f*v) ;// sqrtf(1.0f - v*v);
			double SinTheta = sin(pi()*0.5f*v); // -v;

			// now calculate the direction:
			Vector dir		= float(-CosPhi*CosTheta)*mRight + float(SinPhi*CosTheta)*mForward + float(SinTheta)*mUp;
			float dirLen	= length( dir );
			if(pDeriv)
			{
				pDeriv->dPdx = Vector(0,0,0);
				pDeriv->dPdy = Vector(0,0,0);
				// we just need the direction of the derivatives, so looking at the following exact
				// formulas we can omit the unnecessary operations
				
				// pDeriv->dDdx = xscale*2.0f*pi*CosTheta*(SinPhi*mRight + CosPhi*mForward); // derivative dx, exact formula
				// if (fabs(CosTheta)<1e-12f) pDeriv->dDdy.makeZero();
				// else pDeriv->dDdy = 2.0f*v*yscale/CosTheta*(CosPhi*mRight-SinPhi*mForward)-2.0f*yscale*mUp;
				double du_dx=xscale;
				double dv_dy=-2.0f*yscale;

				double dphi_dx=pi()*2.0f*du_dx;
				double dtheta_dy=pi()*0.5f*dv_dy;

				pDeriv->dDdx=float(SinPhi*dphi_dx*CosTheta)*mRight+float(CosPhi*dphi_dx*CosTheta)*mForward;
				pDeriv->dDdy=float(CosPhi*SinTheta*dtheta_dy)*mRight-float(SinPhi*SinTheta*dtheta_dy)*mForward+float(CosTheta*dtheta_dy)*mUp;

				// now normalize the derivatives:
				pDeriv->dDdx = GetNormDeriv(pDeriv->dDdx, dir, dirLen);
				pDeriv->dDdy = GetNormDeriv(pDeriv->dDdy, dir, dirLen);
			}
			CamRay.p	= mPos;
			CamRay.dir 	= dir/dirLen;
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float xscale, yscale;
		float mFovX;
		float mFovY;
};

/*
 *  class WarpedSphereCamera
 */
class WarpedSphereCamera: public Camera
{
	public:
		WarpedSphereCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f);
		}

  		void Init(float w, float h, float FovX, float FovY)
		{
			width=w;
			height=h;
			float aspect=(height/width)*pi();

			xscale=FovX*0.5f/(width*pi());
			yscale=Vlado::Min(FovY*aspect*0.5f, pi())/(height*pi());
		}

		virtual bool GetScreenRay(double xs, double ys, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			double u=(xs-0.5f*width)*xscale+0.25f;
			double v=(0.5f*height-ys)*yscale*2.0f;
			if (v<-1.0f) v+=2.0f;
			if (v>1.0f) v-=2.0f;

			double CosPhi = cos(u*2.0f*pi());
			double SinPhi = sin(u*2.0f*pi());
			double CosTheta = sqrt(1.0f - v*v);
			double SinTheta = v;

			// now calculate the direction:
			Vector dir		= float(-CosPhi*CosTheta)*mRight + float(SinPhi*CosTheta)*mForward + float(SinTheta)*mUp;
			float dirLen	= length( dir );
			if (pDeriv)
			{
				pDeriv->dPdx = Vector(0,0,0);
				pDeriv->dPdy = Vector(0,0,0);
				// we just need the direction of the derivatives, so looking at the following exact
				// formulas we can omit the unnecessary operations
				
				pDeriv->dDdx = float(xscale*2.0f*pi()*CosTheta)*(float(SinPhi)*mRight + float(CosPhi)*mForward); // derivative dx, exact formula
				if (fabs(CosTheta)<1e-12f) pDeriv->dDdy.makeZero();
				else pDeriv->dDdy = float(2.0f*v*yscale/CosTheta)*(float(-CosPhi)*mRight+float(SinPhi)*mForward)-2.0f*yscale*mUp;
				/*
				float du_dx=xscale;
				float dv_dy=-2.0f*yscale;

				float dphi_dx=pi*2.0f*du_dx;
				float dtheta_dy=pi*0.5f*dv_dy;

				pDeriv->dDdx=(SinPhi*dphi_dx*CosTheta)*mRight+(CosPhi*dphi_dx*CosTheta)*mForward;
				pDeriv->dDdy=(CosPhi*SinTheta*dtheta_dy)*mRight-(SinPhi*SinTheta*dtheta_dy)*mForward+(CosTheta*dtheta_dy)*mUp;
				*/

				// now normalize the derivatives:
				pDeriv->dDdx = GetNormDeriv(pDeriv->dDdx, dir, dirLen);
				pDeriv->dDdy = GetNormDeriv(pDeriv->dDdy, dir, dirLen);
			}
			CamRay.p	= mPos;
			CamRay.dir 	= dir/dirLen;
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float xscale, yscale;
		float mFovX;
		float mFovY;
};

/*
 *  class FishEyeCamera - implements the Fisheye efect. This is acheived by intersecting
 *  camera rays with a virtual sphere. Then the sphere normal at the intersection point
 *  is returned as the Screen Ray (the sphere reflects the scene into camera)
 *
 */
class FishEyeCamera: public Camera
{
	public:
		FishEyeCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f, false, 2.0f, 1.0f);
		}

		void Init(float w, float h, float FovX, float FovY, int AutoFit, float Delta=2.0f, float Curv=1.0f, float shutter=0.0f, float focalDist=1.0f)
		{
			Delta		= Max(Delta, 1.0f);
			dofShutter=shutter;
			dofDist=focalDist;

			width		= w;
			height		= h;
			float aspect=(height/width);

			mOffsetX	= width*0.5f;
			mOffsetY	= height*0.5f;
			mFovX		= FovX;
			mFovY		= FovY;

			if(AutoFit)
			{
	//			float SphSin	= 1.0f / mDelta;
	//			float MaxAngle  = asin(SphSin);
				Delta  = 1.0f /	(float)sin( Min( mFovX, mFovY ));
				if(Delta < 1.0f) Delta = 1.0f;
			}

			mFovX = (float)tan(mFovX)*2.0f/width;
			mFovY = (float)tan(mFovY)*2.0f*aspect/height;

			mSphCenter	= Vector(0.0f, Delta, 0.0f);
			mCurvFactor = Curv;
		}

		virtual bool GetScreenRay(double sx, double sy, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			float fx, fy;
			Vlado::getDiscPoint(dof_uc, dof_vc, fx, fy);
			fx*=dofShutter;
			fy*=dofShutter;

			CamRay.p=Vector(fx, 0.0f, fy); // mPos+mRight*fx+mUp*fy;
			// CamRay.p = Vector(0,0,0);

			double u = (sx - mOffsetX)*mFovX;
			double v = (sy - mOffsetY)*mFovY;

			CamRay.dir=Vector(-float(u), 1.0f, -float(v));

			float time;
			if(intersectSphere(CamRay, mSphCenter, 1.0f, time))
			{
				Vector point	= CamRay.p + CamRay.dir*time;	// intersection point
				Vector normal	= point - mSphCenter;			// sphere normal at intersection point
				// mCurvFactor==1.0f is equivalent to dir = getReflectDir( CamRay.dir, normal );
				Vector dir = CamRay.dir*mCurvFactor - 2.0f*normal*(normal*CamRay.dir)*(2.0f - mCurvFactor);
				if(pDeriv)
				{
					// derivatives of starting ray:
					Vector DirDX  = Vector(-mFovX, 0.0f,   0.0f); 
					Vector DirDY  = Vector(  0.0f, 0.0f, -mFovY); 
	//				DirDX = GetNormDeriv(DirDX, CamRay.dir, length(CamRay.dir));
	//				DirDY = GetNormDeriv(DirDY, CamRay.dir, length(CamRay.dir));

					// the transfer equation (the ray hits the spehre):
					pDeriv->dPdx = time*DirDX;
					pDeriv->dPdy = time*DirDY;

					float  dTdX =  -(pDeriv->dPdx*normal)/(CamRay.dir*normal);
					float  dTdY =  -(pDeriv->dPdy*normal)/(CamRay.dir*normal);

					pDeriv->dPdx += dTdX*CamRay.dir;
					pDeriv->dPdy += dTdY*CamRay.dir;

					// the reflection equation (the ray is reflected by the sphere):
					Vector NormDX = pDeriv->dPdx;	// derivative of the normal
					Vector NormDY = pDeriv->dPdy;	// derivative of the normal

					// compute derivatives according to the reflection formula:
					pDeriv->dDdx   = DirDX*mCurvFactor - 2.0f*(2.0f - mCurvFactor)*
									( NormDX*(normal*CamRay.dir) + (NormDX*CamRay.dir + DirDX*normal)*normal );

					pDeriv->dDdy   = DirDY*mCurvFactor - 2.0f*(2.0f - mCurvFactor)*
									( NormDY*(normal*CamRay.dir) + (NormDY*CamRay.dir + DirDY*normal)*normal );

					// normalize the derivatives:
					float dirLen = length(dir);
					pDeriv->dDdx   = GetNormDeriv(pDeriv->dDdx, dir, dirLen);
					pDeriv->dDdy   = GetNormDeriv(pDeriv->dDdy, dir, dirLen);

					// now orientate derivatives in the correct position:
					pDeriv->dDdx	  = (pDeriv->dDdx.x*mRight + pDeriv->dDdx.y*mForward + pDeriv->dDdx.z*mUp);
					pDeriv->dDdy	  = (pDeriv->dDdy.x*mRight + pDeriv->dDdy.y*mForward + pDeriv->dDdy.z*mUp);

	//				pDeriv->dPdx = Vector(0,0,0);
	//				pDeriv->dPdy = Vector(0,0,0);
				}

				CamRay.dir	= normalize( mRight*dir.x + mForward*dir.y + mUp*dir.z );
				point		-= mSphCenter;
				CamRay.p	= point.x*mRight+point.y*mForward+point.z*mUp + mPos;
	//			CamRay.p	= mPos;
				return true;
			}
			else
			{
				//	CamRay.p   = mPos;
				//	CamRay.dir = normalize( mRight*CamRay.dir.x + mForward*CamRay.dir.y + mUp*CamRay.dir.z );
				return false;
			}
		}

		// camera is actually looking in the opposite direction (mForward = normalize(from-to))
		// (at a virtual sphere that reflects the scene)
		virtual void LookAt(const Vector &from, const Vector &to, const Vector &upDir) 
		{
			mPos	 = from;
			mForward = normalize(from-to);
			mRight	 = normalize(mForward^upDir);
			mUp		 = normalize(mRight^mForward);
		}
		virtual void SetPos(const Vector &pos, const Vector &forward, const Vector &up, const Vector &right) {
			mPos=pos;
			mRight=-right;
			mUp=up;
			mForward=-forward;
		}


		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float	mFovX;
		float	mFovY;
		float	mOffsetX;
		float	mOffsetY;
		float   mCurvFactor;
		Vector	mSphCenter;
		float dofShutter, dofDist;
};


/*
 *  class CylindricCamera
 */
class CylindricCamera: public Camera
{
	public:
		CylindricCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f, 1, 100.0f);
		}

		virtual bool GetScreenRay(double sx, double sy, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv=NULL)
		{
			Vector		dir;

			double u		= (0.5f-sx*mXScale)*mFovX;
			double v		= sy*mYScale;
			double CosA = (float)cos(u);
			double SinA = (float)sin(u);


			// if the rays are beamd from the cylinder's center:
			if (mbFromCenter)
			{
				double z  = (0.5f - v)*mFovY;
//				float y = 1.0f;

				// rotate dir around z-axis. dir is normalized, so is the result
				dir	= float(-SinA)*mRight + float(CosA)*mForward + float(z)*mUp;
				float dirLen = length(dir);

				if(pDeriv)
				{
					pDeriv->dPdx = Vector(0,0,0);
					pDeriv->dPdy = Vector(0,0,0);

					pDeriv->dDdx = (float(CosA*mFovX)*mRight + float(SinA*mFovX)*mForward)*mXScale;
					pDeriv->dDdy = -mFovY*mUp*mYScale;

					pDeriv->dDdx = GetNormDeriv(pDeriv->dDdx, dir, dirLen);
					pDeriv->dDdy = GetNormDeriv(pDeriv->dDdy, dir, dirLen);
				}
				
				CamRay.p	= mPos;
				CamRay.dir	= dir / dirLen;
			}
			else
			{
				// rays are parallel with xy-plane
				dir = float(-SinA)*mRight + float(CosA)*mForward;
				float dirLen = length(dir);

				if(pDeriv)
				{
					pDeriv->dPdx = Vector(0,0,0);
					pDeriv->dPdy = -mHeight*mUp*mYScale;
					pDeriv->dDdx = (float(CosA)*mRight + float(SinA)*mForward)*mXScale*mFovX;
					pDeriv->dDdy = Vector(0,0,0);
					pDeriv->dDdx = GetNormDeriv(pDeriv->dDdx, dir, dirLen);
				}

				CamRay.p	= mPos + float(0.5f-v)*mHeight*mUp;
				CamRay.dir	= dir / dirLen;
			}
			return true;
		}

		void Init(float w, float h, float FovX, float FovY, int FromCenter=1, float Height=100.0f) {
//			if(mFovX < 0.0f	 ) mFovX =   0.0f;
//			if(mFovX > 360.0f) mFovX = 360.0f;
//			if(mFovY < 0.0f	 ) mFovY =   0.0f;
//			if(mFovY > 180.0f) mFovY = 180.0f;

//			mFovX = FovX*pi/180.0f;									// FOVX [0..2pi]
//			mFovY = (float)tan(FovY*pi/360.0f)*2.0f;

			width			= w;
			height			= h;

			float aspect=(height/width);

			mXScale			= 1.0f/width;
			mYScale			= 1.0f/height;
			mFovX			= FovX;
			mFovY			= tanf(FovY*0.5f)*2.0f*aspect;
			mHeight			= Height;
			mbFromCenter	= FromCenter;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float	mFovX;			// X viewing angle
		float	mFovY;			// FovY, only used when mbFromCenter is false (0)
		float   mHeight;		// cylinder height, only used when mbFromCenter is true (1)
		int		mbFromCenter;	// wheter to beam the rays from cylinder's center
		float	mXScale;
		float	mYScale;
};

END_VLADO

#endif
