/*****************************************************
	FILE: ray.h
	DESCRIPTION: A ray in 3D space; also ray-box intersection
	Copyright (C) 2001-2003 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __RAY_H__
#define __RAY_H__

BEGIN_VLADO

/// A class representing a ray in 3d space
struct Ray {
	Vector p; ///< The origin of the ray
	Vector dir; ///< The direction of the ray (may be non-unit)

	Ray(void) {} ///< Default constructor; does nothing
	Ray(const Vector &pt, const Vector &d):p(pt), dir(d) {} ///< Initializes the ray with the given origin and direction
	Ray(const Ray &ray):p(ray.p), dir(ray.dir) {} ///< Initializes the ray from the given ray

	void set(const Vector &pt, const Vector &d) { p=pt; dir=d; } ///< Sets the origin and direction of the ray
};

/// @relates Ray
/// Multiplies a ray by the given matrix
FORCEINLINE Ray operator*(const Matrix &m, const Ray &ray) {
	return Ray(m*ray.p, m*ray.dir);
}

/// @relates Ray
/// Transforms a ray with the given transformation
FORCEINLINE Ray operator*(const Transform &tm, const Ray &ray) {
	return Ray(tm*ray.p, tm.transformVec(ray.dir));
}

typedef double Ireal;

/// A ray specifically optimized for intersection with many boxes
struct IRay: Ray {
	Ireal rdir[3];
	int dirFlags[3];
	Ireal cmint, cmaxt;

	IRay(void) {}
	IRay(const Vector &p, const Vector &d, Ireal mint=0.0f, Ireal maxt=1e18f):Ray(p,d) { init(mint, maxt); }
	IRay(const Ray &ray, Ireal mint=0.0f, Ireal maxt=1e18f):Ray(ray) { init(mint, maxt); }

	void init(Ireal mint, Ireal maxt) {
		cmint=mint;
		cmaxt=maxt;

		for (int i=2; i>=0; i--) {
			Ireal a=dir[i];
			if (fabs(a)<1e-18f) {
				if (a>0.0f) { dirFlags[i]=0; rdir[i]=1e18f; }
				else { dirFlags[i]=1; rdir[i]=-1e18f; }
			} else {
				rdir[i]=1.0f/a;
				if (a>0.0f) dirFlags[i]=0;
				else dirFlags[i]=1;
			}
		}
	}

	void init(Vector &pt, Vector &d, Ireal mint, Ireal maxt) {
		set(pt, d);
		init(mint, maxt);
	}

	FORCEINLINE void intersectBoxDim(int dim, Box &b) {
		cmint=Max((b.c(dirFlags[dim])[dim]-p[dim])*rdir[dim], cmint);
		cmaxt=Min((b.c(1-dirFlags[dim])[dim]-p[dim])*rdir[dim], cmaxt);
	}

	FORCEINLINE void intersectBox1(Box &b, int bboxFlags) {
		if (bboxFlags & 0x110000) intersectBoxDim(0, b);
		if (bboxFlags & 0x220000) intersectBoxDim(1, b);
		if (bboxFlags & 0x440000) intersectBoxDim(2, b);
	}

	FORCEINLINE int intersectBox(Box &b, int bboxFlags) {
		if (bboxFlags & 0x110000) intersectBoxDim(0, b);
		if (bboxFlags & 0x220000) intersectBoxDim(1, b);
		if (bboxFlags & 0x440000) intersectBoxDim(2, b);
		return cmint<=cmaxt*1.000001f;
	}

	FORCEINLINE int intersectBox(Box &b) {
		intersectBoxDim(0, b);
		intersectBoxDim(1, b);
		intersectBoxDim(2, b);
		return cmint<=cmaxt*1.000001f;
	}

	FORCEINLINE void intersectBoxDim(int dim, Box &b, Ireal &cmint, Ireal &cmaxt) {
		cmint=Max((b.c(dirFlags[dim])[dim]-p[dim])*rdir[dim], cmint);
		cmaxt=Min((b.c(1-dirFlags[dim])[dim]-p[dim])*rdir[dim], cmaxt);
	}

	FORCEINLINE int intersectBox(Box &b, Ireal &mint, Ireal &maxt) {
		intersectBoxDim(0, b, mint, maxt);
		intersectBoxDim(1, b, mint, maxt);
		intersectBoxDim(2, b, mint, maxt);
		return mint<=maxt*1.000001f;
	}

	FORCEINLINE int testIntersectBox(Box &b) {
		Ireal mint=cmint, maxt=cmaxt;
		return intersectBox(b, mint, maxt);
	}
};

END_VLADO

#endif
