#ifndef __QUATERNION_H__
#define __QUATERNION_H__

BEGIN_VLADO

class Quaternion {
public:
	real s;
	Vector v;

	Quaternion(void) {}
	Quaternion(real s, const Vector &v) {
		this->s=s;
		this->v=v;
	}
	Quaternion(Vector x) {
		v=x;
		s=0.0f;
	}

	void makeZero(void) {
		s=0.0f;
		v.set(0.0f, 0.0f, 0.0f);
	}

	void makeIdentity(void) {
		s=1.0f;
		v.set(0.0f, 0.0f, 0.0f);
	}

	void makeRotation(Vector axis, real angle) {
		s=cosf(angle*0.5f);
		v=sinf(angle*0.5f)*normalize(axis);
	}

	void makeNormalize(void) {
		real len=(real) sqrt(s*s+lengthSqr(v));
		s/=len;
		v/=len;
	}

	void makeConjugate(void) {
		v=-v;
	}

	void makeInverse(void) {
		real lenSq=s*s+lengthSqr(v);
		s/=lenSq;
		v=-v/lenSq;
	}

	Vector rotate(const Vector &a) {
		Vector b=(v^a)+s*a;
		return (v*a)*v+s*b-(b^v);
	}

	Vector rotateInverse(const Vector &a) {
		Vector b=s*a-(v^a);
		return (v*a)*v+s*b+(b^v);
	}
};

inline Quaternion operator*(const Quaternion &a, const Quaternion &b) {
	Quaternion r;
	r.s=(a.s)*(b.s)-(a.v)*(b.v);
	r.v=(a.s)*(b.v)+(b.s)*(a.v)+((a.v)^(b.v));
	return r;
}

inline Quaternion operator+(const Quaternion &a, const Quaternion &b) {
	Quaternion r;
	r.s=a.s+b.s;
	r.v=a.v+b.v;
	return r;
}

inline Quaternion operator-(const Quaternion &a, const Quaternion &b) {
	Quaternion r;
	r.s=a.s-b.s;
	r.v=a.v-b.v;
	return r;
}

inline Quaternion operator*(real f, const Quaternion &q) {
	Quaternion r;
	r.s=f*q.s;
	r.v=f*q.v;
	return r;
}

inline Matrix toMatrix(const Quaternion &q) {
	Matrix r;
	const real &s=q.s;
	const Vector &v=q.v;

	real x2=v.x+v.x, y2=v.y+v.y, z2=v.z+v.z;
	real xx, yy, zz, xy, yz, xz, wx, wy, wz;

	xx=v.x*x2; xy=v.x*y2; xz = v.x*z2;
	yy=v.y*y2; yz=v.y*z2; zz = v.z*z2;
	wx=s*x2; wy=s*y2; wz=s*z2;

	r.f[0].set(1.0f-(yy+zz), xy-wz, xz+wy);
	r.f[1].set(xy+wz, 1.0f-(xx+zz), yz-wx);
	r.f[2].set(xz-wy, yz+wx, 1.0f-(xx+yy));

	return r;
}

inline real lengthSqr(const Quaternion &q) {
	return sqr(q.s)+lengthSqr(q.v);
}

inline real length(const Quaternion &q) {
	return (real) sqrt(lengthSqr(q));
}

inline Quaternion normalize(const Quaternion &q) {
	Quaternion r;
	real len=length(q);

	r.s=q.s/len;
	r.v=q.v/len;

	return r;
}

inline Quaternion conjugate(const Quaternion &q) {
	Quaternion r;
	r.s=q.s;
	r.v=-q.v;
	return r;
}

inline Vector operator*(const Quaternion &q, const Vector &r) {
	return ((q*Quaternion(r))*conjugate(q)).v;
}

inline Quaternion inverse(const Quaternion &q) {
	real lenSq=lengthSqr(q);

	Quaternion r;
	r.s=q.s/lenSq;
	r.v=-q.v/lenSq;
	return r;
}

// This rotates the given quaternion by the given vector
inline Quaternion rotate(const Quaternion &q, const Vector &r) {
	real angle=length(r);
	if (fabsf(angle)<1e-6f) return q;

	Quaternion rot(cosf(angle*0.5f), sinf(angle*0.5f)*r/angle);

	return rot*q;
}

// This returns a vector that will rotate quaternion a so that it matches quaternion b
inline Vector rotationDiff(const Quaternion &a, const Quaternion &b) {
	Quaternion d=b*inverse(a);
	Vector n=normalize(d.v);
	real angle=acosf(d.s)*2.0f;
	if (fabsf(angle)<1e-6f) {
		return Vector(0,0,0);
	}
	return angle*n;
}

END_VLADO

#endif