#ifndef __PROGRESS_H__
#define __PROGRESS_H__

#include "defines.h"

#define PERR_START 0
#define PERR_END (PERR_START+9999)
#define PWRN_START (PERR_END+1)
#define PWRN_END (PWRN_START+9999)
#define PINF_START (PWRN_END+1)
#define PINF_END (PINF_START+9999)
#define PDBG_START (PINF_END+1)
#define PDBG_END (PDBG_START+9999)

BEGIN_VLADO

/// A class for displaying calculation progress, various diagnostic messages etc. Typically, an instance of this
/// class is passed to ThreadManager::runThreads() to track the calculation progress.
class ProgressCallback {
public:
	int markerLine;
	const tchar *markerFile;
	const tchar *markerDescription;

	virtual int Init()=0; ///< Must be called once to initialize the console.
	virtual int Done()=0; ///< Must be called once to free any data associated with the console.
	//
	virtual void SetTotal(int)=0; ///< Sets the total number of items that will be processed for the current task.
	virtual int GetTotal()=0; ///< Returns the total number of items for the current task.
	//
	virtual void SetProgress(int progress)=0; ///< Sets the number of currently processed items for the current task.
	virtual int GetProgress()=0; ///< Returns the number of currently processed items for the current task.
	//
	virtual void SetMainTitle(const tchar *title)=0; ///< Sets a name for the console.
	virtual void SetProgTitle(const tchar *task)=0; ///< Sets the name for the current task.
	//abort / pause flags
	//
	//TRUE -> abort requested, FALSE - > just CONTINUE
	virtual int CheckForAbort()=0; ///< Checks for user abort and returns the abort status. Must be called from time to time during the calculations.
	virtual int GetAbort()=0;	 ///< Returns the abort status. Does not check for user abort.
	virtual void SetAbort(int abort)=0; ///< Sets the abort status.

	/// This must return true if the progress callback does custom message processing. In that case, ProcessMessages() will be called
	/// to process the messages.
	virtual int DoesMessageProcessing()=0; 
	virtual void ProcessMessages()=0; ///< If DoesMessageProcessing() returns true, this function will be called to process the messages
	//
	//always returns FALSE... but we will leave it for future use by independent progress callbacks
	virtual int GetPause()=0;	///< Returns the pause status.
	virtual void SetPause(int)=0; ///< Sets the pause status.
	//
	virtual ProgressCallback& operator<<(const tchar *); ///< Outputs a message to the console. By default calls DumpMessage() with PINF_END.

	/// Dumps a message to the progress output.
	/// @param module An application-defined identifier of the module outputting the message.
	/// @param level Level of importance for the message:
	///   PERR_START..PERR_END: critical errors (very important)
	///   PWRN_START..PWRN_END: warnings (non-critical errors)
	///   PINF_START..PINF_END: informative messages
	///   PDBG_START..PDBG_END: debug messages
	/// @param format The message format string (similar to printf()).
	virtual void DumpMsg(int module, int level, const tchar *format, ...)=0;

	virtual void DisableDump(void)=0; ///< Disables display of messages.
	virtual void EnableDump(void)=0; ///< Enables display of messages.
	//
	virtual void IncMsgIndent(void)=0; ///< Increments the message indentation; can be used to group messages.
	virtual void DecMsgIndent(void)=0; ///< Decrements the message indentation.

	// This method may be called *very* often; that's why it's not virtual
	/// Sets a program marker; this is used by SetException() to display information.
	/// @param line The source line for the marker.
	/// @param fname The source file name for the marker.
	/// @param description A short description for the marker, typically the function name.
	void SetMarker(int line, const tchar *fname, const tchar *description) {
		markerLine=line; markerFile=fname; markerDescription=description;
	}

	virtual void ClearException(void)=0; ///< Clears the exception flag.

	/// Sets the exception flag. Typically, a message box with the exception description and the last marker will be displayed.
	/// The user may be given a choice to ignore the exception, or to terminate the calculations.
	virtual void SetException(int module, const tchar *description)=0;

	virtual void StartProgress(void) {} ///< Starts a progress tracker. Typically this displays a progress bar.
	virtual void EndProgress(void) {} /// Ends a progress tracker. Typically, hides the progress bar displayed by StartProgress().

	virtual void debug(const tchar *format, ...); ///< Outputs a debug message. By default calls DumpMessage() with level PDBG_END.
	virtual void warning(const tchar *format, ...); ///< Outputs a warning message. By default calls DumpMessage() with PWRN_END.
	virtual void error(const tchar *format, ...); ///< Outputs an error message. By default calls DumpMessage() with PERR_END.
	virtual void info(const tchar *format, ...); /// Outputs an informational message. By default calls DumpMessage() with PINF_END.
};

class ConsoleProgressCallback: public ProgressCallback {
public:
	virtual int CreateConsole()=0;
	virtual int DestroyConsole()=0;

	virtual void ShowConsole()=0;
	virtual void HideConsole()=0;
	virtual void ClearConsole()=0;
};

END_VLADO

#endif
