/*****************************************************
	FILE: matrix.h
	DESCRIPTION: Matrices and transformations in 3D
	Copyright (C) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __MATRIX_H__
#define __MATRIX_H__

BEGIN_VLADO

// A pure 3x3 matrix
class Matrix {
	public:
		Vector f[3]; // The COLUMNS of the matrix

		FORCEINLINE Vector& operator [](const int index) { return f[index]; }
		FORCEINLINE const Vector& operator [](const int index) const { return f[index]; }

		Matrix(void) {}
		Matrix(int i) {
			if (i==1) makeIdentity();
			if (i==0) makeZero();
		}
		Matrix(const Vector &a, const Vector &b, const Vector &c) {
			f[0]=a;
			f[1]=b;
			f[2]=c;
		}

		void set(const Vector &a, const Vector &b, const Vector &c) {
			f[0]=a; f[1]=b; f[2]=c;
		}

		void setCol(int i, const Vector &a) { f[i]=a; }
		void setRow(int i, const Vector &a) { f[0][i]=a.x; f[1][i]=a.y; f[2][i]=a.z; }

		void makeZero(void) {
			memset(f, 0, sizeof(Vector)*3);
		}

		void makeIdentity(void) {
			f[0].set(1.0f, 0.0f, 0.0f);
			f[1].set(0.0f, 1.0f, 0.0f);
			f[2].set(0.0f, 0.0f, 1.0f);
		}

		void makeDiagonal(const Vector &a) {
			makeZero();
			f[0][0]=a.x;
			f[1][1]=a.y;
			f[2][2]=a.z;
		}

		void makeDotProd(const Vector &a, const Vector &b) {
			f[0]=a.x*b;
			f[1]=a.y*b;
			f[2]=a.z*b;
		}

		void makeTranspose(void) {
			real t;
			t=f[0][1]; f[0][1]=f[1][0]; f[1][0]=t;
			t=f[0][2]; f[0][2]=f[2][0]; f[2][0]=t;
			t=f[1][2]; f[1][2]=f[2][1]; f[2][1]=t;
		}

		void makeInverse(void) {
			Matrix r;
			real D=f[0]*(f[1]^f[2]);
			r.setRow(0, (f[1]^f[2])/D);
			r.setRow(1, (f[2]^f[0])/D);
			r.setRow(2, (f[0]^f[1])/D);
			*this=r;
		}

		void addDiagonal(const Vector &a) {
			f[0][0]+=a[0];
			f[1][1]+=a[1];
			f[2][2]+=a[2];
		}

		void addTranspose(const Matrix &b) {
			f[0][0]+=b.f[0][0];
			f[0][1]+=b.f[1][0];
			f[0][2]+=b.f[2][0];
			
			f[1][0]+=b.f[0][1];
			f[1][1]+=b.f[1][1];
			f[1][2]+=b.f[2][1];

			f[2][0]+=b.f[0][2];
			f[2][1]+=b.f[1][2];
			f[2][2]+=b.f[2][2];
		}

		void operator *=(real x) { for (int i=0; i<3; i++) f[i]*=x; }
		void operator +=(const Matrix &m) {
			f[0]+=m.f[0];
			f[1]+=m.f[1];
			f[2]+=m.f[2];
		}
		void operator -=(const Matrix &m) {
			f[0]-=m.f[0];
			f[1]-=m.f[1];
			f[2]-=m.f[2];
		}

		void rotate(const Vector &axis) {
			f[0].rotate(axis);
			f[1].rotate(axis);
			f[2].rotate(axis);
		}

		void makeOrthogonal(void) {
			Matrix t;
			t.f[0]=normalize(f[0]);
			t.f[1]=normalize(f[2]^f[0]);
			t.f[2]=t.f[0]^t.f[1];
			f[0]=length(f[0])*t.f[0];
			f[1]=length(f[1])*t.f[1];
			f[2]=length(f[2])*t.f[2];
		}

		void makeOrthonormal(void) {
			Matrix t;
			t.f[0]=normalize(f[0]);
			t.f[1]=normalize(f[2]^f[0]);
			t.f[2]=t.f[0]^t.f[1];
		}
};

static Matrix ID_mat(1);
static Matrix ZERO_mat(0);

FORCEINLINE Matrix inverse(const Matrix &m) {
	Matrix r;
	real D=m.f[0]*(m.f[1]^m.f[2]);
	r.setRow(0, (m.f[1]^m.f[2])/D);
	r.setRow(1, (m.f[2]^m.f[0])/D);
	r.setRow(2, (m.f[0]^m.f[1])/D);
	return r;
}

// FORCEINLINE Vector operator *(const Vector &a, const Matrix &m) { return a.x*m.f[0]+a.y*m.f[1]+a.z*m.f[2]; }
// FORCEINLINE Vector operator *(const Matrix &m, const Vector &a) { return Vector(a*m.f[0], a*m.f[1], a*m.f[2]); }
FORCEINLINE Vector operator *(const Vector &a, const Matrix &m) { return Vector(a*m.f[0], a*m.f[1], a*m.f[2]); }
FORCEINLINE Vector operator *(const Matrix &m, const Vector &a) { return a.x*m.f[0]+a.y*m.f[1]+a.z*m.f[2]; }

FORCEINLINE Matrix operator ^(const Matrix &m, const Vector &a) {
	return Matrix(m.f[0]^a, m.f[1]^a, m.f[2]^a);
}

FORCEINLINE Matrix operator ^(const Vector &a, const Matrix &m) {
	return Matrix(a^m.f[0], a^m.f[1], a^m.f[2]);
}

FORCEINLINE Matrix operator *(const Matrix &m, real x) {
	return Matrix(m.f[0]*x, m.f[1]*x, m.f[2]*x);
}

FORCEINLINE Matrix operator *(real x, const Matrix &m) {
	return Matrix(x*m.f[0], x*m.f[1], x*m.f[2]);
}

FORCEINLINE Matrix operator /(const Matrix &m, real x) {
	return Matrix(m.f[0]/x, m.f[1]/x, m.f[2]/x);
}

FORCEINLINE real operator /(real x, const Matrix &m) {
	return x/(m.f[0]*(m.f[1]^m.f[2]));
}

FORCEINLINE Matrix operator +(const Matrix &a, const Matrix &b) {
	return Matrix(a.f[0]+b.f[0], a.f[1]+b.f[1], a.f[2]+b.f[2]);
}

FORCEINLINE Matrix operator -(const Matrix &a, const Matrix &b) {
	return Matrix(a.f[0]-b.f[0], a.f[1]-b.f[1], a.f[2]-b.f[2]);
}

FORCEINLINE Matrix operator *(const Matrix &a, const Matrix &b) {
	Matrix r;
	r.f[0].set(
		a.f[0][0]*b.f[0][0]+a.f[1][0]*b.f[0][1]+a.f[2][0]*b.f[0][2],
		a.f[0][1]*b.f[0][0]+a.f[1][1]*b.f[0][1]+a.f[2][1]*b.f[0][2],
		a.f[0][2]*b.f[0][0]+a.f[1][2]*b.f[0][1]+a.f[2][2]*b.f[0][2]);
	r.f[1].set(
		a.f[0][0]*b.f[1][0]+a.f[1][0]*b.f[1][1]+a.f[2][0]*b.f[1][2],
		a.f[0][1]*b.f[1][0]+a.f[1][1]*b.f[1][1]+a.f[2][1]*b.f[1][2],
		a.f[0][2]*b.f[1][0]+a.f[1][2]*b.f[1][1]+a.f[2][2]*b.f[1][2]);
	r.f[2].set(
		a.f[0][0]*b.f[2][0]+a.f[1][0]*b.f[2][1]+a.f[2][0]*b.f[2][2],
		a.f[0][1]*b.f[2][0]+a.f[1][1]*b.f[2][1]+a.f[2][1]*b.f[2][2],
		a.f[0][2]*b.f[2][0]+a.f[1][2]*b.f[2][1]+a.f[2][2]*b.f[2][2]);
	return r;
}

FORCEINLINE Matrix operator-(const Matrix &a) {
	return Matrix(-a.f[0], -a.f[1], -a.f[2]);
}

FORCEINLINE Matrix dotMatrix(const Vector &a, const Vector &b) {
	return Matrix(a.x*b, a.y*b, a.z*b);
}

FORCEINLINE Matrix transpose(const Matrix &m) {
	Matrix r;
	r.f[0][0]=m.f[0][0]; r.f[1][0]=m.f[0][1]; r.f[2][0]=m.f[0][2];
	r.f[0][1]=m.f[1][0]; r.f[1][1]=m.f[1][1]; r.f[2][1]=m.f[1][2];
	r.f[0][2]=m.f[2][0]; r.f[1][2]=m.f[2][1]; r.f[2][2]=m.f[2][2];
	return r;
}

FORCEINLINE Matrix rotate(const Matrix &m, const Vector &axis) {
	return Matrix(rotate(m.f[0], axis), rotate(m.f[1], axis), rotate(m.f[2], axis));
}

FORCEINLINE Matrix normalize(const Matrix &m) {
	Matrix r;
	r.f[0]=normalize(m.f[0]);
	r.f[1]=normalize(m.f[2]^m.f[0]);
	r.f[2]=r.f[0]^r.f[1];
	return r;
}

FORCEINLINE Matrix makeRotMatrixX(real xrot) {
	Matrix t;
	t.f[0]=Vector(1.0f,0.0f,0.0f);
	t.f[1]=Vector(0.0f,cosf(xrot), -sinf(xrot));
	t.f[2]=Vector(0.0f,sinf(xrot), cosf(xrot));
	return t;
}

FORCEINLINE Matrix makeRotMatrixY(real yrot){
	Matrix t;
	t.f[0]=Vector(cosf(yrot), 0.0f, sinf(yrot));
	t.f[1]=Vector(0.0f, 1.0f, 0.0f);
	t.f[2]=Vector(-sinf(yrot), 0.0f, cosf(yrot));
	return t;
}

FORCEINLINE Matrix makeRotMatrixZ(real zrot) {
	Matrix t;
	t.f[0]=Vector(cosf(zrot), -sinf(zrot), 0.0f);
	t.f[1]=Vector(sinf(zrot), cosf(zrot), 0.0f);
	t.f[2]=Vector(0.0f, 0.0f, 1.0f);
	return t;
}

// This returns a matrix for transforming normals for objects transformed with the given matrix
FORCEINLINE Matrix normalTransformMatrix(const Matrix &m) {
	return Matrix(m[1]^m[2], m[2]^m[0], m[0]^m[1]);
}

// A transformation in 3D (3x3 matrix + offset)
class Transform {
	public:
		Matrix m;
		Vector offs;

		Transform(void) {}
		Transform(int i) {
			if (i==1) makeIdentity();
			if (i==0) makeZero();
		}
		Transform(const Matrix &im, const Vector &ioffs) {
			m=im; offs=ioffs;
		}

		void makeZero(void) {
			m.makeZero();
			offs.set(0,0,0);
		}

		void makeIdentity(void) {
			m.makeIdentity();
			offs.set(0,0,0);
		}

		void operator *=(real x) { m*=x; offs*=x; }

		void operator +=(const Transform &tm) {
			m+=tm.m;
			offs+=tm.offs;
		}

		void operator -=(const Transform &tm) {
			m-=tm.m;
			offs-=tm.offs;
		}

		Vector transformVec(const Vector &a) const {
			return m*a;
		}

		void makeInverse(void) {
			m.makeInverse();
			offs=-m*offs;
		}
};

FORCEINLINE Vector operator *(const Transform &tm, const Vector &a) { return tm.m*a+tm.offs; }
FORCEINLINE Vector operator *(const Vector &a, const Transform &tm) { return (a+tm.offs)*tm.m; }

FORCEINLINE Transform operator-(const Transform &tm) {
	return Transform(-tm.m, -tm.offs);
}

FORCEINLINE Transform operator *(const Transform &tm, real x) {
	return Transform(tm.m*x, tm.offs*x);
}

FORCEINLINE Transform operator *(real x, const Transform &tm) {
	return Transform(x*tm.m, x*tm.offs);
}

FORCEINLINE Transform operator /(const Transform &tm, real x) {
	return Transform(tm.m/x, tm.offs/x);
}

FORCEINLINE Transform operator +(const Transform &a, const Transform &b) {
	return Transform(a.m+b.m, a.offs+b.offs);
}

FORCEINLINE Transform operator -(const Transform &a, const Transform &b) {
	return Transform(a.m-b.m, a.offs-b.offs);
}

FORCEINLINE Transform operator *(const Transform &a, const Transform &b) {
	return Transform(a.m*b.m, a.m*b.offs+a.offs);
}

FORCEINLINE Transform inverse(const Transform &tm) {
	Transform r;
	r.m=inverse(tm.m);
	r.offs=-r.m*tm.offs;
	return r;
}

END_VLADO

#endif
