#ifndef __LOOPSUBDIVIDE__
#define __LOOPSUBDIVIDE__

/*
	Template class for Loop subdivision scheme
	The argument class must implement the following functions:

	int numVerts()
*/

#include "EdgeLister.h"

BEGIN_VLADO

template<class Mesh>
class MeshEdges {
	public:
		const Mesh &mesh;

		int numEdges;
		EdgeData *edges;
		int *faceEdges;

		MeshEdges(const Mesh &m):mesh(m) {
			numEdges=0;
			edges=NULL;
			faceEdges=new int[mesh.getNumFaces()*3];
		}

		~MeshEdges(void) { freeData(); }

		int getNumVerts(void) { return mesh.getNumVerts(); }
		int getNumFaces(void) { return mesh.getNumFaces(); }
		int getFaceVertex(int face, int vertex) { return mesh.getFaceVertex(face, vertex); }

		void setNumEdges(int numEdges) {
			this->numEdges=numEdges;
			edges=new EdgeData[numEdges];
		}
		void setEdge(int i, EdgeData &edge) {
			edges[i]=edge;
		}
		void setFaceEdge(int face, int edgeFace, int edgeIdx) {
			faceEdges[face*3+edgeFace]=edgeIdx;
		}

		void freeData(void) {
			numEdges=0;
			if (edges) delete[] edges;
			edges=NULL;
			if (faceEdges) delete[] faceEdges;
			faceEdges=NULL;
		}

		void buildEdges(void) {
			EdgeLister<MeshEdges<Mesh> > edgeLister;
			edgeLister.enumEdges(*this);
		}
};

template<class Mesh>
void _loopSubdivideStep(Mesh &inMesh, Mesh &outMesh) {
	// Build the edge data for the input mesh
	MeshEdges<Mesh> edgeList(inMesh);
	edgeList.buildEdges();

	// Set the size of the output mesh
	outMesh.setNumVerts(inMesh.getNumVerts()+edgeList.numEdges);
	outMesh.setNumFaces(inMesh.getNumFaces()*4);

	// Mark the boundary vertices
	int *boundary=new int[inMesh.getNumVerts()];
	memset(boundary, 0, sizeof(int)*inMesh.getNumVerts());

	for (int i=0; i<edgeList.numEdges; i++) {
		EdgeData &edge=edgeList.edges[i];
		if (edge.numFaces<2) {
			boundary[edge.v[0]]=1;
			boundary[edge.v[1]]=1;
		}
	}

	// Find and add the neighboring vertices
	int *numNgbs=new int[inMesh.getNumVerts()];
	memset(numNgbs, 0, sizeof(int)*inMesh.getNumVerts());

	Vector *psum=new Vector[inMesh.getNumVerts()];
	memset(psum, 0, sizeof(Vector)*inMesh.getNumVerts());

	for (i=0; i<edgeList.numEdges; i++) {
		EdgeData &edge=edgeList.edges[i];

		if (!boundary[edge.v[0]] || boundary[edge.v[1]]) {
			psum[edge.v[0]]+=inMesh.getVert(edge.v[1]);
			numNgbs[edge.v[0]]++;
		}
		if (!boundary[edge.v[1]] || boundary[edge.v[0]]) {
			psum[edge.v[1]]+=inMesh.getVert(edge.v[0]);
			numNgbs[edge.v[1]]++;
		}
	}

	// Set the vertices of the new mesh
	for (i=0; i<inMesh.getNumVerts(); i++) {
		Vector p=psum[i];

		float n=float(numNgbs[i]);
		if (!boundary[i]) {
			float a=5.0f/8.0f-sqr(3.0f+2.0f*cosf(2.0f*pi()/n))/64.0f;
			float alpha=n*(1.0f-a)/a;

			p+=alpha*inMesh.getVert(i);
			p/=alpha+n;
		} else {
			p+=6.0f*inMesh.getVert(i);
			p/=(6.0f+n);
		}

		outMesh.setAndCopyVert(i, p, inMesh, i);
	}

	for (i=0; i<edgeList.numEdges; i++) {
		EdgeData &edge=edgeList.edges[i];

		Vector p=3.0f*inMesh.getVert(edge.v[0])+3.0f*inMesh.getVert(edge.v[1]);

		if (edge.numFaces==1) p+=p/3.0f;
		else p+=inMesh.getVert(edge.tv[0])+inMesh.getVert(edge.tv[1]);

		p/=8.0f;

		outMesh.setAndSplitVert(inMesh.getNumVerts()+i, p, inMesh, edge.v[0], edge.v[1]);
	}

	// Set the faces of the new mesh
	for (i=0; i<inMesh.getNumFaces(); i++) {
		int nv0=inMesh.getNumVerts()+edgeList.faceEdges[i*3];
		int nv1=inMesh.getNumVerts()+edgeList.faceEdges[i*3+1];
		int nv2=inMesh.getNumVerts()+edgeList.faceEdges[i*3+2];

		outMesh.setFace(i*4, nv0, nv2, inMesh.getFaceVertex(i, 0));
		outMesh.setFace(i*4+1, nv1, nv0, inMesh.getFaceVertex(i, 1));
		outMesh.setFace(i*4+2, nv2, nv1, inMesh.getFaceVertex(i, 2));
		outMesh.setFace(i*4+3, nv0, nv1, nv2);
	}

	delete[] psum;
	delete[] numNgbs;
	delete[] boundary;
}

END_VLADO

#endif