/*****************************************************
	FILE: Color.h
	DESCRIPTION: RGB color
	Copyright (C) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __COLOR_H__
#define __COLOR_H__

BEGIN_VLADO

typedef uint32 RGB32;

/// A class for storing a color in RGB space.
/// A pure black color has all components set to 0.0f.
/// A pure white color has all components set to 1.0f.
class Color {
public:
	// The red, greed and blue components
	real r, g, b;

	Color() {} ///< Default constructor, does nothing
	Color(real ir, real ig, real ib):r(ir), g(ig), b(ib) {} ///< Initializes the color with the given components

	/// Initializes the color with the given integer RGB components.
	/// Note that white is (255, 255, 255)
	Color(int ir, int ig, int ib) {
		r=real(ir)/255.0f;
		g=real(ig)/255.0f;
		b=real(ib)/255.0f;
	}

	explicit Color(int col) {
		r=real((col>>16)&0xFF)/255.0f;
		g=real((col>>8)&0xFF)/255.0f;
		b=real(col&0xFF)/255.0f;
	}

	/// Sets the components of the color
	void set(real ir, real ig, real ib) { r=ir; g=ig; b=ib; }

	/// Makes all components zero
	void makeZero(void) { r=g=b=0.0f; }

	/// Makes all components 1.0f
	void makeWhite(void) { r=g=b=1.0f; }

	/// Returns true if color is black (all components < 1e-12f)
	int isBlack(void) const { return (r<1e-12f && g<1e-12f && b<1e-12f); }

	/*FORCEINLINE operator RGB32() {
		return 
			(fast_floor(realMax(realMin(b*255.0f, 255.0f), 0.0f)))+
			(fast_floor(realMax(realMin(g*255.0f, 255.0f), 0.0f))<<8)+
			(fast_floor(realMax(realMin(r*255.0f, 255.0f), 0.0f))<<16);
	}*/
	RGB32 toRGB32(void) const {
		return 
			(fast_floor(Max(Min(b*255.0f, 255.0f), 0.0f)))+
			(fast_floor(Max(Min(g*255.0f, 255.0f), 0.0f))<<8)+
			(fast_floor(Max(Min(r*255.0f, 255.0f), 0.0f))<<16);
	}
	RGB32 toBGR32(void) const {
		return 
			(fast_floor(Max(Min(r*255.0f, 255.0f), 0.0f)))+
			(fast_floor(Max(Min(g*255.0f, 255.0f), 0.0f))<<8)+
			(fast_floor(Max(Min(b*255.0f, 255.0f), 0.0f))<<16);
	}
	void fromRGB32( int col) {
		r=real((col>>16)&0xFF)/255.0f;
		g=real((col>>8)&0xFF)/255.0f;
		b=real(col&0xFF)/255.0f;
	}
	void fromBGR32( int col) {
		b=real((col>>16)&0xFF)/255.0f;
		g=real((col>>8)&0xFF)/255.0f;
		r=real(col&0xFF)/255.0f;
	}

	void operator +=(const Color &a) { r+=a.r; g+=a.g; b+=a.b; } ///< Adds the components of another color to this one
	void operator -=(const Color &a) { r-=a.r; g-=a.g; b-=a.b; } ///< Subtracts the components of another color from this one
	void operator *=(const Color &a) { r*=a.r; g*=a.g; b*=a.b; } ///< Multiplies the components by those of another color
	void operator /=(const Color &a) { r/=a.r; g/=a.g; b/=a.b; } ///< Divides the components by those of another color
	void operator /=(real f) { r/=f; g/=f; b/=f; } ///< Divides all components by the given number
	void operator *=(real f) { r*=f; g*=f; b*=f; } ///< Multiplies all components by the given number

	Color operator-(void) const { return(Color(-r,-g,-b)); } ///< Reverses the sign of all components

	real& operator [](int index) { return (&r)[index]; } ///< Returns the i-th component (i=0, 1, or 2)
	const real& operator [](int index) const { return (&r)[index]; } ///< Returns the i-th component (i=0, 1, or 2) as const

	bool operator==(const Color &c) { return (r==c.r) && (g==c.g) && (b==c.b); } ///< Returns true if the components of this color exactly match those of another one

	real intensity(void) const { return (r+g+b)/3.0f; } ///< Returns the intensity of the color (r+g+b)/3
	real lengthSqr(void) const { return r*r+g*g+b*b; } ///< Returns the squared length of the color
	real length(void) const { return sqrtf(r*r+g*g+b*b); } ///< Returns the length of the color
	real maxComponentValue(void) const { return Max(r, g, b); } ///< Returns the value of the largest component
	real minComponentValue(void) const { return Min(r, g, b); } ///< Returns the value of the smallest component

	/// Clamps all components to the range [0,1]
	void clampMinMax(void) {
		if (r<0.0f) r=0.0f;
		if (g<0.0f) g=0.0f;
		if (b<0.0f) b=0.0f;

		if (r>1.0f) r=1.0f;
		if (g>1.0f) g=1.0f;
		if (b>1.0f) b=1.0f;
	}

	/// Clamps all components to be non-negative
	void clampMin(void) {
		if (r<0.0f) r=0.0f;
		if (g<0.0f) g=0.0f;
		if (b<0.0f) b=0.0f;
	}

	/// Clamps all components to be less than or equal to 1.0f
	void clampMax(void) {
		if (r>1.0f) r=1.0f;
		if (g>1.0f) g=1.0f;
		if (b>1.0f) b=1.0f;
	}

	/// (De)saturates the given color.
	/// @param[in] k k=0.0 means color is greyed;
	/// k=1.0 means the color is unchanged;
	/// k>1.0 adds more color
	void saturate(real k) {
		real grey=(r+g+b)/3.0f;
		r=interpolate(k, grey, r);
		g=interpolate(k, grey, g);
		b=interpolate(k, grey, b);
	}

	/// Applies a contrast correction to the color.
	/// @param[in] k the contrast scale - 1.0f leaves the color unchanged; 0.0f makes the color equal to the middle
	/// @param[in] middle the contrast base - the level that remains unchanged.
	void contrast(real k, real middle=0.5f) {
		r=(r-middle)*k+middle;
		g=(g-middle)*k+middle;
		b=(b-middle)*k+middle;
	}

	/// Returns the complement to white for this color, i.e. Color(1.0, 1.0, 1.0)-(*this)
	Color whiteComplement(void) {
		return Color(1.0f-r, 1.0f-g, 1.0f-b);
	}
};

/// @relates Color
/// Divides the components of a color by a number
FORCEINLINE Color operator /(const Color &a, real f) { return Color(a.r/f, a.g/f, a.b/f); }

/// @relates Color
/// Multiplies the components of a color by a number
FORCEINLINE Color operator *(const Color &a, real f) { return Color(a.r*f, a.g*f, a.b*f); }

/// @relates Color
/// Multiplies the components of a color by a number
FORCEINLINE Color operator *(real f, const Color &a) { return Color(a.r*f, a.g*f, a.b*f); }

/// @relates Color
/// Adds two colors
FORCEINLINE Color operator +(const Color &a, const Color &b) { return Color(a.r+b.r, a.g+b.g, a.b+b.b); }

/// @relates Color
/// Subtracts two colors
FORCEINLINE Color operator -(const Color &a, const Color &b) { return Color(a.r-b.r, a.g-b.g, a.b-b.b); }

/// @relates Color
/// Multiplies two colors component-wise (different from the operator* on vectors)
FORCEINLINE Color operator *(const Color &a, const Color &b) { return Color(a.r*b.r, a.g*b.g, a.b*b.b); }

/// @relates Color
/// Divides two colors component-wise
FORCEINLINE Color operator /(const Color &a, const Color &b) { return Color(a.r/b.r, a.g/b.g, a.b/b.b); }

/// @relates Color
/// Raises all components of a color to a given power
FORCEINLINE Color pow(const Color &color, real f) { return Color(powf(color.r, f), powf(color.g, f), powf(color.b, f)); }

/// @relates Color
/// Returns the squared length of a color
FORCEINLINE real lengthSqr(const Color &a) { return a.lengthSqr(); }

/// @relates Color
/// Returns the length of a color
FORCEINLINE real length(const Color &a) { return a.length(); }

/// @relates Color
/// Returns the intensity (r+g+b)/3 of a color
FORCEINLINE real intensity(const Color &a) { return a.intensity(); }

/// @relates Color
/// Converts a rgb color r,g,b in [0,1] into a HSV color (h in [0, 360], s and v in [0,1])
FORCEINLINE void RGBtoHSV(real r, real g, real b, real &h, real &s, real &v) {
	real mn=Min(r,g,b);
	real mx=Max(r,g,b);

	v=mx; // Value

	real delta=mx-mn;
	if (mx!=0.0f) s=delta/mx; // Saturation
	else { s=0.0f; h=-1.0f; return; }

	if (r==mx) h=(g-b)/delta; // Between yellow & magenta
	else if (g==mx) h=2.0f+(b-r)/delta;	// Between cyan & yellow
	else h=4.0f+(r-g)/delta; // Between magenta & cyan

	h*=60.0f; // Degrees
	if (h<0) h+=360.0f;
}

/// @relates Color
/// Converts a hsv color h,s,v (h in [0, 360], s and v in [0,1]) into a rgb color (r, g, b in [0,1])
FORCEINLINE void HSVtoRGB(real h, real s, real v, real &r, real &g, real &b) {
	if (s==0.0f) { r=g=b=v; return; } // Achromatic (grey)

	h/=60.0f; // Sector 0 to 5
	int i=fast_floor(h); // Integer part of h
	real f=h-real(i); // Factorial part of h

	real p=v*(1.0f-s);
	real q=v*(1.0f-s*f);
	real t=v*(1.0f-s*(1.0f-f));

	switch (i) {
		case 0: { r=v; g=t; b=p; break; }
		case 1: { r=q; g=v; b=p; break; }
		case 2: { r=p; g=v; b=t; break; }
		case 3: { r=p; g=q; b=v; break; }
		case 4: { r=t; g=p; b=v; break; }
		default: { r=v; g=p; b=q; break; } // case 5
	}
}

/// @relates Color
/// Converts a rgb color r,g,b in [0,1] into a HSV color (h in [0, 360], s and v in [0,1])
FORCEINLINE void RGBtoHSV(Color &color, real &h, real &s, real &v) { RGBtoHSV(color.r, color.g, color.b, h, s, v); }

/// @relates Color
/// Converts a hsv color h,s,v (h in [0, 360], s and v in [0,1]) into a rgb color (r, g, b in [0,1])
FORCEINLINE Color HSVtoRGB(real h, real s, real v) {
	Color color;
	HSVtoRGB(h, s, v, color.r, color.g, color.b);
	return color;
}

/// @relates Color
/// Returns a color with components which are the minimum respective components of the given colors
template<>
FORCEINLINE Color Min<Color>(const Color &a, const Color &b) {
	return Color(Min(a[0], b[0]), Min(a[1], b[1]), Min(a[2], b[2]));
}

/// @relates Color
/// Returns a color with components which are the maximum respective components of the given colors
template<>
FORCEINLINE Color Max<Color>(const Color &a, const Color &b) {
	return Color(Max(a[0], b[0]), Max(a[1], b[1]), Max(a[2], b[2]));
}
/*
class AColor {
public:
	real r, g, b, a;

	AColor(void) {}
	AColor(float ir, float ig, float ib, float ia) { r=ir; g=ig; b=ib; a=ia; }

	void operator +=(const AColor &c) { r+=c.r; g+=c.g; b+=c.b; a+=c.a; }
	void operator -=(const AColor &c) { r-=c.r; g-=c.g; b-=c.b; a-=c.a; }
	void operator *=(const AColor &c) { r*=c.r; g*=c.g; b*=c.b; a*=c.a; }
	void operator /=(const AColor &c) { r/=c.r; g/=c.g; b/=c.b; a/=c.a; }
	void operator /=(real f) { r/=f; g/=f; b/=f; a/=f; }
	void operator *=(real f) { r*=f; g*=f; b*=f; a*=f; }

	real lengthSqr(void) { return r*r+g*g+b*b+a*a; }
};

FORCEINLINE AColor operator /(const AColor &a, real f) { return AColor(a.r/f, a.g/f, a.b/f, a.a/f); }
FORCEINLINE AColor operator *(const AColor &a, real f) { return AColor(a.r*f, a.g*f, a.b*f, a.a*f); }
FORCEINLINE AColor operator *(real f, const AColor &a) { return AColor(a.r*f, a.g*f, a.b*f, a.a*f); }
FORCEINLINE AColor operator +(const AColor &a, const AColor &b) { return AColor(a.r+b.r, a.g+b.g, a.b+b.b, a.a+b.a); }
FORCEINLINE AColor operator -(const AColor &a, const AColor &b) { return AColor(a.r-b.r, a.g-b.g, a.b-b.b, a.a-b.a); }
FORCEINLINE AColor operator *(const AColor &a, const AColor &b) { return AColor(a.r*b.r, a.g*b.g, a.b*b.b, a.a*b.a); }
FORCEINLINE AColor operator /(const AColor &a, const AColor &b) { return AColor(a.r/b.r, a.g/b.g, a.b/b.b, a.a/b.a); }
*/

/// A 32-bit compressed color, based on Ward's real pixels
class CompressedColor {
public:
	int8 r, g, b;
	uint8 e;

	CompressedColor(void) {} ///< Default constructor; does nothing
	CompressedColor(const Color &color) { init(color); } ///< Initializes the compressed color with the given regular Color
	CompressedColor(const Vector &a) { init(Color(a.x, a.y, a.z)); } ///< Initializes the compressed color with the given Vector (treated as Color)

	void makeZero(void) { r=g=b=e=0; } ///< Makes all components zero

	/// Sets the compressed color with the closest match to the given regular Color
	void init(const Color &color) {
		float v=Max(fabsf(color.r), fabsf(color.g), fabsf(color.b));
		if (v<1e-32) { r=g=b=e=0; }
		else {
			int exp;
			v=frexpf(v,&exp)*128.0f/v;
			r=int8(color.r*v);
			g=int8(color.g*v);
			b=int8(color.b*v);
			e=int8(exp+128);
		}
	}

	/// Converts the compressed color to a regular Color
	operator Color(void) {
		float f=ldexpf(1.0, int(e)-135);
		return Color(r*f, g*f, b*f);
	}

	/// Converts the compressed color to a Vector
	operator Vector(void) {
		Color c=Color(*this);
		return Vector(c.r, c.g, c.b);
	}
};

class AColor {
public:
	Color color;
	real alpha;

	AColor(void) {}
	AColor(const Color &c): color(c), alpha(1.0f) {}
	AColor(const Color &c, float a): color(c), alpha(a) {}
	AColor(real r, real g, real b, real a): color(r,g,b), alpha(a) {}

	void operator +=(const AColor &a) { color+=a.color; alpha+=a.alpha; }
	void operator -=(const AColor &a) { color-=a.color; alpha-=a.alpha; }
	void operator *=(const AColor &a) { color*=a.color; alpha*=a.alpha; }
	void operator /=(const AColor &a) { color/=a.color; alpha/=a.alpha; }
	void operator /=(real f) { color/=f; alpha/=f; }
	void operator *=(real f) { color*=f; alpha*=f; }

	AColor operator-(void) { return AColor(-color, -alpha); }

	void set(const Color &c, real a) { color=c; alpha=a; }
	void set(real r, real g, real b, real a) { color.set(r,g,b); alpha=a; }

	void makeZero(void) { color.makeZero(); alpha=0.0f; }

	real lengthSqr(void) { return color.lengthSqr()+sqr(alpha); }

	real& operator[](int i) { return ((float*) (&color))[i]; }
	const real& operator[](int i) const { return ((float*) (&color))[i]; }
};

FORCEINLINE AColor operator /(const AColor &a, real f) { return AColor(a.color/f, a.alpha/f); }
FORCEINLINE AColor operator *(const AColor &a, real f) { return AColor(a.color*f, a.alpha*f); }
FORCEINLINE AColor operator *(real f, const AColor &a) { return AColor(a.color*f, a.alpha*f); }
FORCEINLINE AColor operator +(const AColor &a, const AColor &b) { return AColor(a.color+b.color, a.alpha+b.alpha); }
FORCEINLINE AColor operator -(const AColor &a, const AColor &b) { return AColor(a.color-b.color, a.alpha-b.alpha); }
FORCEINLINE AColor operator *(const AColor &a, const AColor &b) { return AColor(a.color*b.color, a.alpha*b.alpha); }
FORCEINLINE AColor operator /(const AColor &a, const AColor &b) { return AColor(a.color/b.color, a.alpha*b.alpha); }

template<>
FORCEINLINE AColor Min<AColor>(const AColor &a, const AColor &b) {
	return AColor(Min(a.color, b.color), Min(a.alpha, b.alpha));
}

template<>
FORCEINLINE AColor Max<AColor>(const AColor &a, const AColor &b) {
	return AColor(Max(a.color, b.color), Max(a.alpha, b.alpha));
}

END_VLADO

#endif
