/***********************************************************
	Implementation of an interval tree
***********************************************************/

#ifndef __VGL_BVTREE_H__
#define __VGL_BVTREE_H__

#include "box.h"
#include "fdop.h"

BEGIN_VLADO

#define BV_undefined 2

typedef Box BVol;
// typedef FDop BVol;

struct BVolume {
	BVol bv; // The bounding volume for the element
	int flags; // Some flags
};

class BBCallback {
	public:
		virtual bool intersect(BVolume *b0, BVolume *b1)=0;
};

struct BVIntersection;
struct BVRay {
	IRay iray;
	Ireal mint, maxt;
	BVIntersection *is;
	int flags;
};

struct BVIntersection {
	Ireal t;
};

// template<class B>
struct BBNode {
	BVol bv; // the bounding volume of the nodes below this node
	int maxd; // the splitting axis
	float middle; // the split value
	int flags;

	BVolume *bvolume; // if this is a leaf, this is a pointer to the stored bbox
	BBNode *left, *right; // If the node is not a leaf, these are pointers to the left and right subtrees

	void init(void) {
		left=right=NULL;
		bvolume=NULL;
		bv.init();
	}

	void freeData(void) {
		if (left) {
			left->freeData();
			left=NULL;
		}
		if (right) {
			right->freeData();
			right=NULL;
		}
	}

	FORCEINLINE BBNode* subs(int i) { return (&left)[i]; }
};

inline void intersectNodes(BBNode *node0, BBNode *node1, BBCallback &cb, real tolerance, int doFlags=false) {
	BVol &bv0=node0->bv;
	BVol &bv1=node1->bv;

	if (doFlags) {
		if (node0->flags==node1->flags && ((node0->flags-1) & node0->flags)==0) return;
	}

	if (!bv0.intersectTolerance(bv1, tolerance)) return;

	if (node0->bvolume && node1->bvolume && node0!=node1)
		cb.intersect(node0->bvolume, node1->bvolume);

	// choose the biggest bounding box
	real v0=bv0.volume();
	real v1=bv1.volume();

	// split the biggest box
	bool splitLeft=v0>v1;
	if (node1->bvolume) splitLeft=true;
	else if (node0->bvolume) splitLeft=false;

	if (splitLeft) {
		if (node0->left) intersectNodes(node0->left, node1, cb, tolerance);
		if (node0->right) intersectNodes(node0->right, node1, cb, tolerance);
	} else {
		if (node1->left) intersectNodes(node0, node1->left, cb, tolerance);
		if (node1->right) intersectNodes(node0, node1->right, cb, tolerance);
	}
}

struct BVStackElement {
	BBNode *node;
	Ireal mint, maxt;
};

#define BVPOP if (sptr==stack) break; sptr--; node=sptr->node; ray.iray.cmint=sptr->mint; ray.iray.cmaxt=sptr->maxt;

template<class T>
class BVTree {
	public:
		SequenceST<T, 100> bvolumes;
		BBNode *root; // the root of the tree
		SequenceST<BBNode, 100> nodes; // the nodes of the tree

		int depth;

		BVTree(void) { init(); }

		void init(void) { root=NULL; depth=0; }

		void freeData(void) {
			bvolumes.freeData();
			nodes.freeData();
			root=NULL;
			depth=0;
		}

		// Stores a new element in the tree
		void storeBBox(T &b) { *(bvolumes.newElement())=b; }

		// Returns a pointer to a new element
		T* newElement(void) { return bvolumes.newElement(); }

		// Builds the tree hierarchy
		void build(void) {
			nodes.freeData();
			depth=0;

			int count=bvolumes.count();
			if (count<=0) return;

			BVolume **pbvolumes=new BVolume*[count];

			bvolumes.resetCounter();
			for (int i=0; i<count; i++) pbvolumes[i]=bvolumes.nextElement();

			BVol bv;
			bv.init();
			root=buildNode(pbvolumes, count, 1, bv);

			delete[] pbvolumes;
		}

		// Refits the hierarchy to the current bvolumes
		void refit(void) { refitNode(root); }

		// Intersects the tree with the given arbitrary bounding volume; calls the given callback
		// for every elementary volume that intersects the given one
		void intersect(BVolume &b, BBCallback &cb, real tolerance) {
			if (root) intersectNode(root, b, cb, tolerance);
		}

		// Intersects the elementary volumes in the tree with one another
		void selfIntersect(BBCallback &cb, real tolerance, int doFlags) {
			if (root) selfIntersectNode(root, cb, tolerance, doFlags);
		}

		// Intersects the elementary volumes in the tree with the given ray;
		// stops on first intersection; uses the current value of BVRay::is->t
		int intersectRay(BVRay &ray) {
			BVStackElement *stack=(BVStackElement*) alloca((depth*2+5)*sizeof(BVStackElement));
			BVStackElement *sptr=stack;

			BBNode *node=root;

			int res=0;
			do {
				if (!node || !ray.iray.intersectBox(node->bv, node->maxd)) {
					BVPOP
					continue;
				}

				if (node->bvolume) {
					Ireal pmaxt=ray.iray.cmaxt;
					ray.iray.cmint=Max(ray.mint, ray.iray.cmint*0.9999f);
					ray.iray.cmaxt=Min(ray.maxt, ray.iray.cmaxt*1.0001f);

					res|=((T*) node->bvolume)->intersectRay(ray);

					if (res==1 && ((ray.flags & TRACE_FINDANY) || ray.is->t<=pmaxt)) return res;

					BVPOP
					continue;
				}

				int maxd=node->maxd & 7;

				int dir=ray.iray.dirFlags[maxd];
				sptr->node=node->subs(1-dir);
				sptr->maxt=ray.iray.cmaxt;
				sptr->mint=ray.iray.cmint;
				sptr++;

				node=node->subs(dir);
			} while (1);

			return res;
		}

	private:
		BBNode *buildNode(BVolume **bvolumes, int numBVolumes, int level, BVol &parentBVol) {
			if (numBVolumes<=0) return NULL;

			if (level>depth) depth=level;

			BBNode *node=nodes.newElement();
			node->init();

			// if there is only one point - store it in the node
			if (numBVolumes==1) {
				node->bvolume=bvolumes[0];
				node->bv+=bvolumes[0]->bv;
				node->flags=node->bvolume->flags;
				node->maxd=0x770000;
				return node;
			}

			// find the bounding volume of the points
			BVol bv;
			bv.init();
			for (int i=0; i<numBVolumes; i++) bv+=bvolumes[i]->bv;
			node->bv=bv;

			// find the middle of the longest side
			int maxd=bv.maxDimension();
			float middle=(bv.pmin[maxd]+bv.pmax[maxd])*0.5f;
			node->maxd=maxd | bv.compare(parentBVol);
			node->middle=middle;

			// divide the points in two groups
			int nLeft=0, nRight=numBVolumes-1;
			while (nLeft<=nRight) {
				if (bvolumes[nLeft]->bv.pmin[maxd]<=middle) nLeft++;
				else {
					BVolume *t=bvolumes[nLeft];
					bvolumes[nLeft]=bvolumes[nRight];
					bvolumes[nRight]=t;
					nRight--;
				}
			}

			if (nLeft==0) nLeft++;
			if (nRight==numBVolumes-1) nLeft--;

			node->left=buildNode(bvolumes, nLeft, level+1, bv);
			node->right=buildNode(bvolumes+nLeft, numBVolumes-nLeft, level+1, bv);

			node->flags=0;
			if (node->left) node->flags|=node->left->flags;
			if (node->right) node->flags|=node->right->flags;

			return node;
		}

		void refitNode(BBNode *node) {
			if (!node) return;

			node->bv.init();
			bool fitted=false;

			if (node->bvolume) {
				node->bv=node->bvolume->bv;
				node->flags=node->bvolume->flags;
				fitted=true;
				return;
			}

			node->flags=0;
			if (node->left) {
				refitNode(node->left);
				node->bv+=node->left->bv;
				node->flags|=node->left->flags;
				fitted=true;
			}
			if (node->right) {
				refitNode(node->right);
				node->bv+=node->right->bv;
				node->flags|=node->right->flags;
				fitted=true;
			}
			if (!fitted/*node->bbox.pmin.z>50.0f*/) {
				OutputDebugString("Error...\n");
			}
		}

		void intersectNode(BBNode *node, BVolume &b, BBCallback &cb, real tolerance) {
			if (!node) return;

			if (node->bv) {
				if (node->bvolume->bv.intersectTolerance(b.bv, tolerance)) cb.intersect(node->bvolume, &b);
				return;
			}

			if (!node->bv.intersect(b.bv)) return;

			intersectNode(node->left, b, cb);
			intersectNode(node->right, b, cb);
		}

		void selfIntersectNode(BBNode *node, BBCallback &cb, real tolerance, int doFlags=false) {
			if (!node) return;

			if (doFlags) {
				if ((node->flags & (node->flags-1))==0) return;
			}

			if (node->bvolume) cb.intersect(node->bvolume, node->bvolume);
			selfIntersectNode(node->left, cb, tolerance, doFlags);
			selfIntersectNode(node->right, cb, tolerance, doFlags);

			if (node->left && node->right) {
				intersectNodes(node->left, node->right, cb, tolerance, doFlags);
			}
		}
};

END_VLADO

#endif
