/*****************************************************
	FILE: box.h
	DESCRIPTION: Axis-aligned box in 3D
	Copyright (C) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/

#ifndef __BOX_H__
#define __BOX_H__

BEGIN_VLADO

#define BOX_FLAGS 0xFF0000

/// A class representing a box in 3d space.

class Box {
public:
	Vector pmin; ///< The lower bounds for the box along the three axes
	Vector pmax; ///< The upper bounds for the box along the three axes

	/// Default constructor - does nothing
	Box(void) {}

	/// Initializes the box to contain the given point
	Box(const Vector &p) { pmin=pmax=p; }

	/// Initializes the box with the given extents
	Box(const Vector &imin, const Vector &imax) { pmin=imin; pmax=imax; }

	/// Initializes the box to either an empty one, or to an infinite one
	Box(int i) {
		if (i==0) init();
		else if (i==1) makeInfinite();
	}

	/// Initializes the box to an empty one
	void init(void) {
		pmin.set(1e20f, 1e20f, 1e20f);
		pmax.set(-1e20f, -1e20f, -1e20f);
	}

	/// Initializes the box to an infinite one
	void makeInfinite(void) {
		pmin.set(-1e20f, -1e20f, -1e20f);
		pmax.set(1e20f, 1e20f, 1e20f);
	}

	/// Initializes the box to contain the given point
	void init(const Vector &p) { pmin=pmax=p; }

	real dx() const { return pmax.x-pmin.x; } ///< Returns the size of the box along the x axis
	
	real dy() const { return pmax.y-pmin.y; } ///< Returns the size of the box along the y axis
	
	real dz() const { return pmax.z-pmin.z; } ///< Returns the size of the box along the z axis

	/// Expands the box (if necessary) to contain the given point
	void operator +=(const Vector &p) {
		pmin=Min(pmin, p);
		pmax=Max(pmax, p);
		/*
		if (p[0]<pmin[0]) pmin[0]=p[0];
		if (p[0]>pmax[0]) pmax[0]=p[0];

		if (p[1]<pmin[1]) pmin[1]=p[1];
		if (p[1]>pmax[1]) pmax[1]=p[1];

		if (p[2]<pmin[2]) pmin[2]=p[2];
		if (p[2]>pmax[2]) pmax[2]=p[2];
		*/
	}

	// Add a box to the box
	Box& operator +=(const Box &b) {
		pmin=Min(pmin, b.pmin);
		pmax=Max(pmax, b.pmax);
		/*
		if (b.pmin[0]<pmin[0]) pmin[0]=b.pmin[0];
		if (b.pmax[0]>pmax[0]) pmax[0]=b.pmax[0];

		if (b.pmin[1]<pmin[1]) pmin[1]=b.pmin[1];
		if (b.pmax[1]>pmax[1]) pmax[1]=b.pmax[1];

		if (b.pmin[2]<pmin[2]) pmin[2]=b.pmin[2];
		if (b.pmax[2]>pmax[2]) pmax[2]=b.pmax[2];
		*/

		return *this;
	}

	// Intersect the box with a box
	Box& operator &=(const Box &b) {
		pmin=Max(pmin, b.pmin);
		pmax=Min(pmax, b.pmax);
		/*
		if (b.pmin[0]>pmin[0]) pmin[0]=b.pmin[0];
		if (b.pmax[0]<pmax[0]) pmax[0]=b.pmax[0];

		if (b.pmin[1]>pmin[1]) pmin[1]=b.pmin[1];
		if (b.pmax[1]<pmax[1]) pmax[1]=b.pmax[1];

		if (b.pmin[2]>pmin[2]) pmin[2]=b.pmin[2];
		if (b.pmax[2]<pmax[2]) pmax[2]=b.pmax[2];
		*/

		return *this;
	}

	// Returns true if the given point is in the box
	int isInside(const Vector &p, real tolerance=0.0f) const {
		if (pmin.x-tolerance<=p.x && p.x<=pmax.x+tolerance && 
			pmin.y-tolerance<=p.y && p.y<=pmax.y+tolerance &&
			pmin.z-tolerance<=p.z && p.z<=pmax.z+tolerance) return true;
		return false;
	}

	// Returns true if the given box is inside this one
	int contains(const Box &b) const {
		return
			(pmin[0]<=b.pmin[0] && b.pmax[0]<=pmax[0]) &&
			(pmin[1]<=b.pmin[1] && b.pmax[1]<=pmax[1]) &&
			(pmin[2]<=b.pmin[2] && b.pmax[2]<=pmax[2]);
	}

	// Returns true if the given box intersects this one
	int intersect(const Box &b) const {
		if (pmax.x<b.pmin.x || pmin.x>b.pmax.x) return false;
		if (pmax.y<b.pmin.y || pmin.y>b.pmax.y) return false;
		if (pmax.z<b.pmin.z || pmin.z>b.pmax.z) return false;
		return true;
	}

	int intersectTolerance(const Box &b, real tolerance) const {
		if (pmax.x+tolerance<b.pmin.x || pmin.x-tolerance>b.pmax.x) return false;
		if (pmax.y+tolerance<b.pmin.y || pmin.y-tolerance>b.pmax.y) return false;
		if (pmax.z+tolerance<b.pmin.z || pmin.z-tolerance>b.pmax.z) return false;
		return true;
	}

	Vector width(void) const { return pmax-pmin; }
	Vector center(void) const { return (pmin+pmax)*0.5f; }

	void expand(real dist) {
		pmin.x-=dist;
		pmin.y-=dist;
		pmin.z-=dist;

		pmax.x+=dist;
		pmax.y+=dist;
		pmax.z+=dist;
	}

	void scale(real k) {
		for (int i=0; i<3; i++) {
			real d=k*(pmax[i]-pmin[i]);
			pmin[i]-=d;
			pmax[i]+=d;
		}
	}

	void scale(const Vector &k) {
		for (int i=0; i<3; i++) {
			real d=k[i]*(pmax[i]-pmin[i]);
			pmin[i]-=d;
			pmax[i]+=d;
		}
	}

	void move(Vector &a) {
		pmin+=a;
		pmax+=a;
	}

	void _scale(real f) {
		for (int i=0; i<3; i++) {
			pmin[i]-=(1.0f+fabsf(pmin[i]))*f;
			pmax[i]+=(1.0f+fabsf(pmax[i]))*f;
		}
	}

	real volume(void) const {
		return (pmax[0]-pmin[0])*(pmax[1]-pmin[1])*(pmax[2]-pmin[2]);
	}

	real surface(void) const {
		Vector w=width();
		return 2.0f*(w[0]*w[1]+w[1]*w[2]+w[2]*w[0]);
	}

	int maxDimension(void) const {
		return maxComponent(width());
	}

	int empty(real tolerance=0.0f) const {
		if (pmin[0]>pmax[0]+tolerance) return true;
		if (pmin[1]>pmax[1]+tolerance) return true;
		if (pmin[2]>pmax[2]+tolerance) return true;
		return false;
	}

	int empty(int dim, real tolerance=0.0f) const {
		return pmin[dim]>pmax[dim]+tolerance;
	}

	Vector operator[](int corner) const {
		switch (corner) {
			case 0: return pmin;
			case 1: return Vector(pmax.x, pmin.y, pmin.z);
			case 2: return Vector(pmin.x, pmax.y, pmin.z);
			case 3: return Vector(pmax.x, pmax.y, pmin.z);
			case 4: return Vector(pmin.x, pmin.y, pmax.z);
			case 5: return Vector(pmax.x, pmin.y, pmax.z);
			case 6: return Vector(pmin.x, pmax.y, pmax.z);
			case 7: return Vector(pmax.x, pmax.y, pmax.z);
		}
		return pmin;
	}

	Vector clamp(const Vector &p) const {
		return Vector(
			Vlado::clamp(p.x, pmin.x, pmax.x),
			Vlado::clamp(p.y, pmin.y, pmax.y),
			Vlado::clamp(p.z, pmin.z, pmax.z));
	}

	int compare(const Box &b) const {
		return
			(b.pmin[0]!=pmin[0])<<16 |
			(b.pmax[0]!=pmax[0])<<(16+4) |
			(b.pmin[1]!=pmin[1])<<17 |
			(b.pmax[1]!=pmax[1])<<(17+4) |
			(b.pmin[2]!=pmin[2])<<18 |
			(b.pmax[2]!=pmax[2])<<(18+4);
	}

	void split(int maxd, real middle, Box &bLeft, Box &bRight) {
		bLeft=bRight=*this;
		bLeft.pmax[maxd]=middle;
		bRight.pmin[maxd]=middle;
	}

	FORCEINLINE Vector &c(int i) { return (&pmin)[i]; }

	// Returns the closest distance from a point to the box
	// Returns 0.0f if the point is inside the box
	float minDistSqr(const Vector &pt);
};

FORCEINLINE Box transform(const Box &b, const Vlado::Transform &tm) {
	Box r;
	r.init();
	if (!b.empty()) {
		for (int i=0; i<8; i++) r+=tm*b[i];
	}
	return r;
}

FORCEINLINE int compareBBoxes(const Box &b0, const Box &b1) {
	return b0.compare(b1);
}

FORCEINLINE Box operator&(const Box &b0, const Box &b1) {
	return Box(b0)&=b1;
}

FORCEINLINE Box operator+(const Box &b0, const Box &b1) {
	return Box(b0)+=b1;
}

END_VLADO

#endif
