#include "utils.h"
#include "table.h"
#include "sequence.h"
#include "bvtree.h"

#include "collisiondetector.h"

#define VF_INVALID 16

USE_VLADO

struct MeshInfo;

struct BVertex: public BVolume {
	MeshInfo *mi;
	int idx;
	int flags;

	Vector p;
};

struct BEdge: public BVolume {
	MeshInfo *mi;
	int edgeIdx;
	int idx[2];
	int flags;

	Vector p;
	Vector dir;
};

struct BFace: public BVolume {
	MeshInfo *mi;
	int faceIdx;
	int idx[3];
	int flags;

	Transform tm; // transform from world space to face space
	Vector normal;
};

struct BMesh: public BVolume {
	MeshInfo *mi;
};

typedef BVTree<BVertex> BVertexTree;
typedef BVTree<BEdge> BEdgeTree;
typedef BVTree<BFace> BFaceTree;
typedef BVTree<BMesh> BMeshTree;

struct MeshInfo {
	CollisionMesh *mesh;

	int numVerts;
	int deflector, selfCollide;
	real friction;

	Vector *verts;
	Vector *pverts;
	Vector *origVerts;
	int *flags;

	BVertexTree vertexTree;
	BEdgeTree edgeTree;
	BFaceTree faceTree;

	void freeMem(void) {
		vertexTree.freeData();
		edgeTree.freeData();
		faceTree.freeData();
		if (verts) { delete[] verts; verts=NULL; }
		if (pverts) { delete[] pverts; pverts=NULL; }
		if (origVerts) { delete[] origVerts; origVerts=NULL; }
		if (flags) { delete[] flags; flags=NULL; }
	}
};

class DefaultCollisionDetector: public CollisionDetector {
	CollisionList clist;
	Table<MeshInfo*> meshes;

	BMeshTree meshTree;
public:
	DefaultCollisionDetector(void) { init(); }
	~DefaultCollisionDetector(void) { freeMem(); }

	void init(void);
	void freeMem(void);

	void addCollisionMesh(CollisionMesh *cmesh);
	void update(int rebuildAll=true);
	real collide(real tolerance, int meshTypes, int selfTypes);
	int checkIntersection(int checkSelfIntersection);
	void accept(void);
	void applyImpulses(void);
	CollisionList& getCollisionList(void) { return clist; }
};

CollisionDetector* Vlado::newDefaultCollisionDetector(void) { return new DefaultCollisionDetector; }

void DefaultCollisionDetector::init(void) {
}

void DefaultCollisionDetector::freeMem(void) {
	meshTree.freeData();
	clist.freeData();
	for (int i=0; i<meshes.count(); i++) { meshes[i]->freeMem(); delete meshes[i]; meshes[i]=NULL; }
	meshes.freeData();
}

//*******************************************************************
//
void DefaultCollisionDetector::addCollisionMesh(CollisionMesh *mesh) {
	if (!mesh) return;
	MeshInfo *m=new MeshInfo;
	meshes+=m;

	m->mesh=mesh;
	m->numVerts=mesh->getNumCollisionVerts();

	m->verts=new Vector[m->numVerts];
	m->pverts=new Vector[m->numVerts];
	m->origVerts=new Vector[m->numVerts];
	m->flags=new int[m->numVerts];

	BVertex v;
	for (int i=0; i<mesh->getNumCollisionVerts(); i++) {
		v.mi=m;
		v.idx=i;
		m->vertexTree.storeBBox(v);
		m->pverts[i]=m->origVerts[i]=mesh->getCollisionVertexPos(i);
	}

	BEdge e;
	for (i=0; i<mesh->getNumCollisionEdges(); i++) {
		e.mi=m;
		mesh->getCollisionEdgeVertices(i, e.idx);
		e.edgeIdx=i;
		m->edgeTree.storeBBox(e);
	}

	BFace f;
	for (i=0; i<mesh->getNumCollisionFaces(); i++) {
		f.mi=m;
		mesh->getCollisionFaceVertices(i, f.idx);
		f.faceIdx=i;
		m->faceTree.storeBBox(f);
	}

	BMesh bm;
	bm.mi=m;
	meshTree.storeBBox(bm);
}

//*******************************************************************
//
void DefaultCollisionDetector::update(int rebuildAll) {
	for (int i=0; i<meshes.count(); i++) {
		MeshInfo *m=meshes[i];
		CollisionMesh *mesh=m->mesh;

		mesh->updateCollisionInfo();

		m->deflector=true;
		m->selfCollide=false;
		m->friction=mesh->getFriction();

		Vector *verts=m->verts;
		Vector *pverts=m->pverts;
		int *flags=m->flags;

		for (int j=0; j<m->numVerts; j++) {
			verts[j]=mesh->getCollisionVertexPos(j);
			// pverts[j]=mesh->getVertexPosPrev(j);
			flags[j]=mesh->getCollisionVertexFlags(j);
			if (flags[j] & VF_SELFCOLLIDE) m->selfCollide=true;
			if ((flags[j] & VF_FIXED)==0) m->deflector=false;
		}

		m->vertexTree.bvolumes.resetCounter();
		for (int j=0; j<m->vertexTree.bvolumes.count(); j++) {
			BVertex *v=m->vertexTree.bvolumes.nextElement();

			Vector &p=verts[v->idx];
			v->p=p;
			v->bv.init(p);
			// v->bv+=pverts[v->idx];
			v->flags=flags[j];
		}

		m->edgeTree.bvolumes.resetCounter();
		for (j=0; j<m->edgeTree.bvolumes.count(); j++) {
			BEdge *be=m->edgeTree.bvolumes.nextElement();

			Vector &p0=verts[be->idx[0]];
			Vector &p1=verts[be->idx[1]];

			be->bv.init(p0);
			be->bv+=p1;
			// be->bv+=pverts[be->idx[0]];
			// be->bv+=pverts[be->idx[1]];

			be->p=p0;
			be->dir=p1-p0;

			be->flags=flags[be->idx[0]] & flags[be->idx[1]];
		}

		m->faceTree.bvolumes.resetCounter();
		for (j=0; j<m->faceTree.bvolumes.count(); j++) {
			BFace *bf=m->faceTree.bvolumes.nextElement();

			Vector p0=verts[bf->idx[0]];
			Vector p1=verts[bf->idx[1]];
			Vector p2=verts[bf->idx[2]];

			bf->bv.init(p0);
			bf->bv+=p1;
			bf->bv+=p2;
			// bf->bv+=pverts[bf->idx[0]];
			// bf->bv+=pverts[bf->idx[1]];
			// bf->bv+=pverts[bf->idx[2]];

			Vector n=(p1-p0)^(p2-p0);
			float len=length(n);
			if (len<1e-8f) {
				bf->flags=VF_INVALID;
				continue;
			}

			bf->normal=n/len;
			Transform tm;
			tm.m.f[0]=p1-p0;
			tm.m.f[1]=p2-p0;
			tm.m.f[2]=bf->normal;
			tm.offs=p0;

			bf->tm=inverse(tm);

			bf->flags=flags[bf->idx[0]] & flags[bf->idx[1]] & flags[bf->idx[2]];
		}

		if (rebuildAll) {
			m->vertexTree.build();
			m->edgeTree.build();
			m->faceTree.build();
		} else {
			m->vertexTree.refit();
			m->edgeTree.refit();
			m->faceTree.refit();
		}
	}

	meshTree.bvolumes.resetCounter();
	for (int j=0; j<meshTree.bvolumes.count(); j++) {
		BMesh *m=meshTree.bvolumes.nextElement();
		MeshInfo *mi=m->mi;

		m->bv.init();
		for (int i=0; i<mi->numVerts; i++) {
			m->bv+=mi->verts[i];
			// m->bv+=mi->pverts[i];
		}
	}

	if (rebuildAll) meshTree.build();
	else meshTree.refit();
}

void DefaultCollisionDetector::accept(void) {
	/*
	for (int i=0; i<meshes.count(); i++) {
		MeshInfo *m=meshes[i];
		CollisionMesh *mesh=m->mesh;

		Vector *verts=m->verts;
		Vector *pverts=m->pverts;

		for (int j=0; j<m->numVerts; j++) pverts[j]=verts[j];
	}
	*/
}

//*******************************************************************
//
int checkPlane(Vector p[4], Vector q[4], double &t) {
	return false;
	Vector dp[4];
	for (int i=0; i<4; i++) dp[i]=q[i]-p[i];

	Vector e0=p[2]-p[1], e1=p[3]-p[1];
	double pn[3];
	_cross3(e0, e1, pn);

	Vector e2=q[2]-q[1], e3=q[3]-q[1];
	double qn[3];
	_cross3(e2, e3, qn);

	double dn[3];
	_sub3(qn, pn, dn);

	Vector pf=p[0]-p[1], qf=q[0]-q[1];
	Vector df=qf-pf;

	double a=_dot3(dn, df);
	double b=_dot3(pf,dn)+_dot3(df,pn);
	double c=_dot3(pn,pf);

	double f0=c, f1=a+b+c;
	if (f0*f1<-1e-3f) return true;
	return false;
}

class VertexVertexCollide: public BBCallback {
public:
	real tolerance, toleranceSq;
	CollisionList *clist;
	real *minDist;

	void init(real dist, CollisionList &collisions, real *mdist) {
		tolerance=dist;
		toleranceSq=sqr(tolerance);
		clist=&collisions;
		minDist=mdist;
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		BVertex *bv0=(BVertex*) b0;
		BVertex *bv1=(BVertex*) b1;

		int selfCollision=(bv0->mi==bv1->mi);
		int flags=(bv0->flags) & (bv1->flags);
		if (flags & VF_FIXED) return true;
		if ((flags & VF_COLLIDE)==0) return true;

		if (selfCollision) {
			if ((flags & VF_SELFCOLLIDE)==0) return true;
			if (bv0->idx==bv1->idx) return true;
		}

		Vector dp=(bv0->p)-(bv1->p);
		real d=lengthSqr(dp);

		if (d>=toleranceSq) return true;
		if (selfCollision && d<1e-5f) return true;

		real tdistSq=toleranceSq, tdist=tolerance;
		// compute the original distance
		if (selfCollision) {
			Vector *origp=bv0->mi->origVerts;
			Vector q0=origp[bv0->idx];
			Vector q1=origp[bv1->idx];

			real origDistSq=lengthSqr(q0-q1);
			if (origDistSq<=tdistSq) {
				float k=origDistSq/tdistSq;
				tdistSq=origDistSq*k;
				if (d>=tdistSq) return true;
				tdist=sqrtf(tdistSq);
			}
		}

		*minDist=Min(*minDist, d/tdistSq);

		// A collision was detected
		Collision &c=*(clist->newElement());

		c.m0=bv0->mi->mesh;
		c.vv.vert0=bv0->idx;

		c.m1=bv1->mi->mesh;
		c.vv.vert1=bv1->idx;

		c.friction=0.5f*(bv0->mi->friction+bv1->mi->friction);
		c.dist=tdist;
		c.type=c_vertex_vertex;
		c.normal=normalize(dp);
		c.deflectorCollision=(bv0->mi->deflector | bv1->mi->deflector);
		c.strength=1.0f;

		if (!c.init()) clist->removeLast();

		return true;
	}
};

class VertexEdgeCollide: public BBCallback {
public:
	real tolerance, toleranceSq;
	CollisionList *clist;
	real *minDist;

	void init(real dist, CollisionList &collisions, real *mdist) {
		tolerance=dist;
		toleranceSq=sqr(tolerance);
		clist=&collisions;
		minDist=mdist;
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		BVertex *bv=(BVertex*) b0;
		BEdge *be=(BEdge*) b1;

		int selfCollision=(bv->mi==be->mi);
		int flags=(bv->flags) & (be->flags);
		if (flags & VF_FIXED) return true;
		if ((flags & VF_COLLIDE)==0) return true;

		if (selfCollision) {
			if ((flags & VF_SELFCOLLIDE)==0) return true;
			if (bv->idx==be->idx[0] || bv->idx==be->idx[1]) return true;
		}

		real t=((bv->p)-(be->p))*(be->dir)/lengthSqr(be->dir);
		if (t<0.0f || t>1.0f) return true;

		Vector q=(be->p)+(be->dir)*t;
		Vector dp=(bv->p)-q;
		real d=lengthSqr(dp);

		if (d>=toleranceSq) return true;
		if (selfCollision && d<1e-5f) return true;

		real tdistSq=toleranceSq, tdist=tolerance;
		// compute the original distance
		if (selfCollision) {
			Vector *origp=bv->mi->origVerts;
			Vector q0=origp[bv->idx];
			Vector q1=origp[be->idx[0]]*(1.0f-t)+origp[be->idx[1]]*t;;

			real origDistSq=lengthSqr(q0-q1);
			if (origDistSq<=tdistSq) {
				float k=origDistSq/tdistSq;
				tdistSq=origDistSq*k;
				if (d>=tdistSq) return true;
				tdist=sqrtf(tdistSq);
			}
		}

		*minDist=Min(*minDist, d/tdistSq);

		// A collision was detected
		Collision &c=*(clist->newElement());

		c.m0=bv->mi->mesh;
		c.ve.vert=bv->idx;

		c.m1=be->mi->mesh;
		c.ve.edge=be->edgeIdx;
		c.ve.t=t;

		c.friction=0.5f*(bv->mi->friction+be->mi->friction);
		c.dist=tdist;
		c.type=c_vertex_edge;
		c.normal=normalize(dp);
		c.deflectorCollision=(bv->mi->deflector | be->mi->deflector);
		c.strength=1.0f;

		if (!c.init()) clist->removeLast();

		return true;
	}
};

class VertexFaceCollide: public BBCallback {
public:
	real tolerance, toleranceSq;
	CollisionList *clist;
	int *iipath;
	real *minDist;

	void init(real dist, CollisionList &collisions, int *ipath, real *mdist) {
		tolerance=dist;
		toleranceSq=sqr(tolerance);
		clist=&collisions;
		iipath=ipath;
		minDist=mdist;
	}

	float det(Vector &a, Vector &b, Vector &c) { return (a^b)*c; }

	int checkVertexFacePath(BVertex *bv, BFace *bf, Vector ip[4]) {
		Vector p[4], q[4];

		p[0]=bv->mi->pverts[bv->idx];
		p[1]=bf->mi->pverts[bf->idx[0]];
		p[2]=bf->mi->pverts[bf->idx[1]];
		p[3]=bf->mi->pverts[bf->idx[2]];

		q[0]=bv->mi->verts[bv->idx];
		q[1]=bf->mi->verts[bf->idx[0]];
		q[2]=bf->mi->verts[bf->idx[1]];
		q[3]=bf->mi->verts[bf->idx[2]];

		// Find if and when the four points lie in a plane
		double t=1.0f;
		int res=checkPlane(p, q, t);

		// Compute the position of the points at that moment
		for (int i=0; i<4; i++) ip[i]=q[i]; // p[i]*float(1.0f-t)+q[i]*float(t);
		return res;
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		BVertex *bv=(BVertex*) b0;
		BFace *bf=(BFace*) b1;

		if (bf->flags & VF_INVALID) return true;

		int selfCollision=(bv->mi==bf->mi);
		int flags=(bv->flags) & (bf->flags);
		if (flags & VF_FIXED) return true;
		if ((flags & VF_COLLIDE)==0) return true;

		Vector offs=bf->mi->verts[bf->idx[0]]-(bv->p);

		if (selfCollision) {
			if ((flags & VF_SELFCOLLIDE)==0) return true;
			if (bv->idx==bf->idx[0] || bv->idx==bf->idx[1] || bv->idx==bf->idx[2]) return true;
		} else {
			// if (bf->mi->deflector && offs*bf->normal>0.0f) return true;
		}

		/*
		// Find the closest point (u,v) in the triangle
		Vector e0=(bf->mi->verts[bf->idx[1]]-bf->mi->verts[bf->idx[0]]);
		Vector e1=(bf->mi->verts[bf->idx[2]]-bf->mi->verts[bf->idx[0]]);
		real ac=e0*e0, bc=e0*e1, cc=e1*e1, dc=e0*offs, ec=e1*offs;

		real det=ac*cc-bc*bc, u=bc*ec-cc*dc, v=bc*dc-ac*ec;
		if (u+v<det) {
			if (u<0.0f) {
				if (v<0.0f) {
					// region 4
					u=v=0.0f;
				} else {
					// region 3
					u=0.0f;
					v=clamp(-ec/cc, 0.0f, 1.0f);
				}
			} else if (v<0.0f) {
				// region 5
				v=0.0f;
				u=clamp(-dc/ac, 0.0f, 1.0f);
			} else {
				// region 0
				real invDet=1.0f/det;
				u*=invDet;
				v*=invDet;
			}
		} else {
			if (u<0.0f) {
				// region 2
				u=0.0f;
				v=1.0f;
			} else if (v<0.0f) {
				// region 6
				u=1.0f;
				v=0.0f;
			} else {
				// region 1
				u=clamp((cc+ec-bc-dc)/(ac-2.0f*bc+cc), 0.0f, 1.0f);
				v=1.0f-u;
			}
		}

		Vector dp=-(offs+e0*u+e1*v);
		real d=lengthSqr(dp);
		*/

		const Vector &p=bv->p;

		const Transform &tm=bf->tm;
		real sd=(p-(bf->mi->verts[bf->idx[0]]))*bf->normal;

		real uc=p.x*(tm.m.f[0].x)+p.y*(tm.m.f[1].x)+p.z*(tm.m.f[2].x)+(tm.offs.x);
		if (uc<1e-6f) return true;

		real vc=p.x*(tm.m.f[0].y)+p.y*(tm.m.f[1].y)+p.z*(tm.m.f[2].y)+(tm.offs.y);
		if (vc<1e-6f || uc+vc>1.0f-1e-6f) return true;

		Vector dp=sd*bf->normal;
		real d=sd*sd;

		if (d>=toleranceSq) return true;
		if (selfCollision && d<1e-5f) return true;

		real tdistSq=toleranceSq, tdist=tolerance;
		// compute the original distance
		if (selfCollision) {
			Vector *origp=bv->mi->origVerts;
			Vector q0=origp[bv->idx];
			Vector q1=(1.0f-uc-vc)*origp[bf->idx[0]]+uc*origp[bf->idx[1]]+vc*origp[bf->idx[2]];

			real origDistSq=lengthSqr(q0-q1);
			if (origDistSq<=tdistSq) {
				float k=origDistSq/tdistSq;
				tdistSq=origDistSq*k;
				if (d>=tdistSq) return true;
				tdist=sqrtf(tdistSq);
			}
		}

		*minDist=Min(*minDist, d/tdistSq);

		// A collision was detected
		Collision &c=*(clist->newElement());

		c.m0=bv->mi->mesh;
		c.vf.vert=bv->idx;

		c.m1=bf->mi->mesh;
		c.vf.face=bf->faceIdx;

		c.vf.uc=uc;
		c.vf.vc=vc;

		c.friction=0.5f*(bv->mi->friction+bf->mi->friction);
		c.dist=tdist;
		c.type=c_vertex_face;
		c.normal=normalize(dp);
		c.deflectorCollision=(bv->mi->deflector | bf->mi->deflector);
		c.strength=1.0f;

		if (!c.init()) clist->removeLast();

		return true;
	}
};

class EdgeEdgeCollide: public BBCallback {
public:
	real tolerance, toleranceSq;
	CollisionList *clist;
	int *iipath;
	real *minDist;

	void init(real dist, CollisionList &collisions, int *ipath, real *mdist) {
		tolerance=dist;
		toleranceSq=sqr(tolerance);
		clist=&collisions;
		iipath=ipath;
		minDist=mdist;
	}

	int checkEdgeEdgePath(BEdge *be0, BEdge *be1, Vector ip[4]) {
		Vector p[4], q[4];

		p[0]=be0->mi->pverts[be0->idx[0]];
		p[1]=be0->mi->pverts[be0->idx[1]];
		p[2]=be1->mi->pverts[be1->idx[0]];
		p[3]=be1->mi->pverts[be1->idx[1]];

		q[0]=be0->mi->verts[be0->idx[0]];
		q[1]=be0->mi->verts[be0->idx[1]];
		q[2]=be1->mi->verts[be1->idx[0]];
		q[3]=be1->mi->verts[be1->idx[1]];

		// Find if and when the four points lie in a plane
		double t=1.0f;
		int res=checkPlane(p, q, t);

		// Compute the position of the points at that moment
		for (int i=0; i<4; i++) ip[i]=q[i]; // p[i]*float(1.0f-t)+q[i]*float(t);
		return res;
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		BEdge *be0=(BEdge*) b0;
		BEdge *be1=(BEdge*) b1;

		int selfCollision=(be0->mi==be1->mi);
		int flags=(be0->flags) & (be1->flags);
		if (flags & VF_FIXED) return true;
		if ((flags & VF_COLLIDE)==0) return true;

		if (selfCollision) {
			if ((flags & VF_SELFCOLLIDE)==0) return true;
			if (be0->idx[0]==be1->idx[0] || be0->idx[1]==be1->idx[0] || be0->idx[0]==be1->idx[1] || be0->idx[1]==be1->idx[1]) return true;
		}

		Vector &p0=be0->p;
		Vector &q0=be1->p;
		Vector dir0=be0->dir, dir1=be1->dir;

		Vector qp=dir0^dir1;
		float qpLen=length(qp);
		if (qpLen<1e-3f) return true;
		qp/=qpLen;

		float len0=dir0*dir0;
		if (len0<toleranceSq) return true;

		float len1=dir1*dir1;
		if (len1<toleranceSq) return true;

		Vector offs=p0-q0;
		float mx=dir0*dir1;

		float D=len0*len1-mx*mx;
		if (fabs(D)<1e-3f) return true;

		float offs_e0=offs*dir0;
		float offs_e1=offs*dir1;

		float t0=(offs_e1*mx-offs_e0*len1)/D;
		float t1=(offs_e1*len0-offs_e0*mx)/D;

		// t0=clamp(t0, 0.0f, 1.0f);
		// t1=clamp(t1, 0.0f, 1.0f);
		if (t0<1e-6f || t0>1.0f-1e-6f || t1<1e-6f || t1>1.0f-1e-6f) return true;

		// compute current distance
		Vector p=p0+dir0*t0;
		Vector q=q0+dir1*t1;
		qp=p-q;
		real d=lengthSqr(qp);
		// real sd=(q-p)*qp;
		// real d=sd*sd;

		if (d>=toleranceSq) return true;
		if (selfCollision && d<1e-5f) return true;

		real tdistSq=toleranceSq, tdist=tolerance;
		// compute the original distance
		if (selfCollision) {
			Vector *origp=be0->mi->origVerts;
			Vector q0=(1.0f-t0)*origp[be0->idx[0]]+t0*origp[be0->idx[1]];
			Vector q1=(1.0f-t1)*origp[be1->idx[0]]+t1*origp[be1->idx[1]];

			real origDistSq=lengthSqr(q0-q1);
			if (origDistSq<=tdistSq) {
				float k=origDistSq/tdistSq;
				tdistSq=origDistSq*k;
				if (d>=tdistSq) return true;
				tdist=sqrtf(tdistSq);
			}
		}

		*minDist=Min(*minDist, d/tdistSq);

		// a collision was detected
		Collision &c=*(clist->newElement());

		c.m0=be0->mi->mesh;
		c.ee.edge0=be0->edgeIdx;
		c.ee.t0=t0;

		c.m1=be1->mi->mesh;
		c.ee.edge1=be1->edgeIdx;
		c.ee.t1=t1;

		c.friction=0.5f*(be0->mi->friction+be1->mi->friction);
		c.dist=tdist;
		c.type=c_edge_edge;
		// if (!ipath && sd>0.0f) qp=-qp;
		// c.normal=qp;
		c.normal=normalize(qp);
		c.deflectorCollision=(be0->mi->deflector | be1->mi->deflector);
		c.strength=1.0f;

		if (!c.init()) clist->removeLast();

		return true;
	}
};

class MeshMeshCollide: public BBCallback {
public:
	real tolerance, toleranceSq;

	VertexFaceCollide vfCollide;
	EdgeEdgeCollide eeCollide;
	VertexVertexCollide vvCollide;
	VertexEdgeCollide veCollide;
	int meshMeshCollisions, selfCollisions;

	MeshMeshCollide(real dist, CollisionList &collisions, int meshMeshCollisions, int selfCollisions, int *ipath, real *minDist) {
		tolerance=dist;
		toleranceSq=sqr(tolerance);
		this->meshMeshCollisions=meshMeshCollisions;
		this->selfCollisions=selfCollisions;

		vfCollide.init(dist, collisions, ipath, minDist);
		eeCollide.init(dist, collisions, ipath, minDist);
		vvCollide.init(dist, collisions, minDist);
		veCollide.init(dist, collisions, minDist);
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		BMesh *bm0=(BMesh*) b0;
		BMesh *bm1=(BMesh*) b1;

		// If these are two delfectors - do not collide
		if (bm0->mi->deflector && bm1->mi->deflector) return true;

		if (bm0->mi==bm1->mi) {
			// Self-collisions
			if (!bm0->mi->selfCollide) return true;
			if (selfCollisions & CT_VERTEX_FACE) intersectNodes(bm0->mi->vertexTree.root, bm0->mi->faceTree.root, vfCollide, tolerance, false);
			if (selfCollisions & CT_EDGE_EDGE) bm0->mi->edgeTree.selfIntersect(eeCollide, tolerance, false);
			if (selfCollisions & CT_VERTEX_VERTEX) bm0->mi->vertexTree.selfIntersect(vvCollide, tolerance, false);
			if (selfCollisions & CT_VERTEX_EDGE) intersectNodes(bm0->mi->vertexTree.root, bm0->mi->edgeTree.root, veCollide, tolerance, false);
			return true;
		}

		if (meshMeshCollisions & CT_VERTEX_FACE) {
			intersectNodes(bm0->mi->vertexTree.root, bm1->mi->faceTree.root, vfCollide, tolerance);
			intersectNodes(bm1->mi->vertexTree.root, bm0->mi->faceTree.root, vfCollide, tolerance);
		}
		if (meshMeshCollisions & CT_EDGE_EDGE) intersectNodes(bm0->mi->edgeTree.root, bm1->mi->edgeTree.root, eeCollide, tolerance);
		if (meshMeshCollisions & CT_VERTEX_VERTEX) intersectNodes(bm0->mi->vertexTree.root, bm1->mi->vertexTree.root, vvCollide, tolerance);
		if (meshMeshCollisions & CT_VERTEX_EDGE) {
			intersectNodes(bm0->mi->vertexTree.root, bm1->mi->edgeTree.root, veCollide, tolerance);
			intersectNodes(bm1->mi->vertexTree.root, bm0->mi->edgeTree.root, veCollide, tolerance);
		}

		return true;
	}
};

real DefaultCollisionDetector::collide(real tolerance, int meshTypes, int selfTypes) {
	clist.clear();
	int ipath=false;
	real minDist=tolerance*tolerance;
	MeshMeshCollide mmCollide(tolerance, clist, meshTypes, selfTypes, &ipath, &minDist);
	meshTree.selfIntersect(mmCollide, tolerance, false);
	return minDist;

	/*
	real maxRelVel=0.0f;

	clist.resetCounter();
	Vlado::Collision *c;
	while (c=clist.nextElement()) {
		real relVel=-(c->relVel()*c->normal)/c->dist;
		if (relVel>maxRelVel) maxRelVel=relVel;
	}

	return maxRelVel;
	*/
}

//*******************************************************************
//
class EdgeFaceCollide: public BBCallback {
public:
	int *isect;

	void init(int *intersection) {
		isect=intersection;
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		if (*isect) return false;
		BEdge *be=(BEdge*) b0;
		BFace *bf=(BFace*) b1;

		if (bf->flags & VF_INVALID) return true;
		int flags=be->flags & bf->flags;

		int selfCollide=(be->mi==bf->mi);
		if (selfCollide) {
			if ((flags & VF_SELFCOLLIDE)==0) return true;
		}

		if (flags & VF_FIXED) return true;
		if ((flags & VF_INTERSECT)==0) return true;
		if ((flags & VF_COLLIDE)==0) return true;

		Vector &p=be->p;
		Vector &dir=be->dir;
		Vector q=p+dir;

		Transform &tm=bf->tm;

		real s0=(p-(bf->mi->verts[bf->idx[0]]))*bf->normal;
		real s1=(q-(bf->mi->verts[bf->idx[0]]))*bf->normal;

		if (s0<s1) {
			if (s1-s0<1e-6f) return true;
			if (s1<1e-6f) return true;
			if (s0>-1e-6f) return true;
		} else {
			if (s0-s1<1e-6f) return true;
			if (s0<1e-6f) return true;
			if (s1>-1e-6f) return true;
		}

		real s=s0/(s0-s1);
		Vector m=p+dir*s;

		real u=m.x*(tm.m.f[0].x)+m.y*(tm.m.f[1].x)+m.z*(tm.m.f[2].x)+(tm.offs.x);
		if (u<0.0f) return true;

		real v=m.x*(tm.m.f[0].y)+m.y*(tm.m.f[1].y)+m.z*(tm.m.f[2].y)+(tm.offs.y);
		if (v<0.0f) return true;
		if (u+v>1.0f) return true;

		/*
		if (selfCollide) {
			Vector *origp=be->mi->origVerts;
			Vector p=(1.0f-s)*origp[be->idx[0]]+s*origp[be->idx[1]];
			Vector q=origp[bf->idx[0]]*(1.0f-u-v)+origp[bf->idx[1]]*u+origp[bf->idx[2]]*v;
			real origDist=lengthSqr(q-p);
			if (origDist<toleranceSq) return true;
		}
		*/
		Vector i0=p+dir*s, i1=(bf->mi->verts[bf->idx[0]])*(1.0f-u-v)+(bf->mi->verts[bf->idx[1]])*u+(bf->mi->verts[bf->idx[2]])*v;

		*isect=true;
		return false;
	}
};

class MeshMeshIntersect: public BBCallback {
public:
	int *isect;
	int selfSect;
	EdgeFaceCollide efIntersect;

	MeshMeshIntersect(int *intersection, int checkSelfIntersections) {
		isect=intersection;
		selfSect=checkSelfIntersections;
		efIntersect.init(isect);
	}

	bool intersect(BVolume *b0, BVolume *b1) {
		if (*isect) return false;
		BMesh *bm0=(BMesh*) b0;
		BMesh *bm1=(BMesh*) b1;

		if (bm0->mi==bm1->mi) {
			if (!selfSect || !bm0->mi->selfCollide) return true;

			intersectNodes(bm0->mi->edgeTree.root, bm0->mi->faceTree.root, efIntersect, 0.0f);
			if (*isect) return false;
			else return true;
		}

		if (bm0->mi->deflector && bm1->mi->deflector) return true;

		intersectNodes(bm0->mi->edgeTree.root, bm1->mi->faceTree.root, efIntersect, false);
		intersectNodes(bm1->mi->edgeTree.root, bm0->mi->faceTree.root, efIntersect, false);
		if (*isect) return false;
		else return true;
	}
};

int DefaultCollisionDetector::checkIntersection(int checkSelfIntersections) {
	int intersection=false;
	MeshMeshIntersect mmIntersect(&intersection, checkSelfIntersections);
	meshTree.selfIntersect(mmIntersect, 0.0f, false);

	return intersection;
}

void DefaultCollisionDetector::applyImpulses(void) {
	clist.resetCounter();
	Vlado::Collision *c;
	while (c=clist.nextElement()) c->applyImpulses();
}

//*************************************************************
//
int Collision::init(void) {
	dist*=1.0f;

	Vector rvel=relVel();
	float nv=rvel*normal;

	if (nv>1e-4f) valid=false;
	else valid=true;

	float len=length(rvel);
	if (len>1e-12f) nv/=len; else nv=0.0f;

	friction*=fabsf(nv);
	if (friction<0.0f) friction=0.0f;

	float np=fabsf(relPos()*normal);
	if (np<1e-4f) valid=false;

	strength=1.0f; // Min(dist/np, 1000.0f);
	// if (deflectorCollision) strength*=2.0f;

	/*
	if (type==c_vertex_face) {
		s0=m0->getVertImpulseStrength(vert);
		s1=m1->getFaceImpulseStrength(face, uc, vc);
	} else {
		s0=m0->getEdgeImpulseStrength(edge0, t0);
		s1=m1->getEdgeImpulseStrength(edge1, t1);
	}
	*/

	return valid;
}

Vector Collision::relPos(void) {
	switch (type) {
		case c_vertex_face: return (m0->getCollisionVertPos(vf.vert))-(m1->getCollisionFacePos(vf.face, vf.uc, vf.vc));
		case c_edge_edge: return (m0->getCollisionEdgePos(ee.edge0, ee.t0))-(m1->getCollisionEdgePos(ee.edge1, ee.t1));
		case c_vertex_vertex: return (m0->getCollisionVertPos(vv.vert0))-(m1->getCollisionVertPos(vv.vert1));
		case c_vertex_edge: return (m0->getCollisionVertPos(ve.vert))-(m1->getCollisionEdgePos(ve.edge, ve.t));
	}
	return Vector(0,0,0);
}

Vector Collision::relVel(void) {
	switch (type) {
		case c_vertex_face: return (m0->getCollisionVertVel(vf.vert))-(m1->getCollisionFaceVel(vf.face, vf.uc, vf.vc));
		case c_edge_edge: return (m0->getCollisionEdgeVel(ee.edge0, ee.t0))-(m1->getCollisionEdgeVel(ee.edge1, ee.t1));
		case c_vertex_vertex: return (m0->getCollisionVertVel(vv.vert0))-(m1->getCollisionVertVel(vv.vert1));
		case c_vertex_edge: return (m0->getCollisionVertVel(ve.vert))-(m1->getCollisionEdgeVel(ve.edge, ve.t));
	}
	return Vector(0,0,0);
}

Vector Collision::relForce(void) {
	switch (type) {
		case c_vertex_face: return (m0->getCollisionVertForce(vf.vert))-(m1->getCollisionFaceForce(vf.face, vf.uc, vf.vc));
		case c_edge_edge: return (m0->getCollisionEdgeForce(ee.edge0, ee.t0))-(m1->getCollisionEdgeForce(ee.edge1, ee.t1));
		case c_vertex_vertex: return (m0->getCollisionVertForce(vv.vert0))-(m1->getCollisionVertForce(vv.vert1));
		case c_vertex_edge: return (m0->getCollisionVertForce(ve.vert))-(m1->getCollisionEdgeForce(ve.edge, ve.t));
	}
	return Vector(0,0,0);
}

void Collision::applyImpulses(void) {
	if (!valid) return;
	switch (type) {
		case c_vertex_face: {
			Vector rpos=(m0->getCollisionVertPos(vf.vert))-(m1->getCollisionFacePos(vf.face, vf.uc, vf.vc));
			Vector rvel=(m0->getCollisionVertVel(vf.vert))-(m1->getCollisionFaceVel(vf.face, vf.uc, vf.vc));
			// Vector rforce=(m0->getCollisionVertForce(vf.vert))-(m1->getCollisionFaceForce(vf.face, vf.uc, vf.vc));

			real np=rpos*normal, nv=rvel*normal;
			Vector pimp=normal*(dist-np), vimp=normal*(-nv);
			Vector fimp=(rvel-nv*normal)*(-friction);
			pimp*=strength; vimp*=strength;

			m0->applyVertImpulse(vf.vert, pimp, vimp, fimp, normal);
			m1->applyFaceImpulse(vf.face, vf.uc, vf.vc, -pimp, -vimp, -fimp, normal);
			break;
		}
		case c_edge_edge: {
			Vector rpos=(m0->getCollisionEdgePos(ee.edge0, ee.t0))-(m1->getCollisionEdgePos(ee.edge1, ee.t1));
			Vector rvel=(m0->getCollisionEdgeVel(ee.edge0, ee.t0))-(m1->getCollisionEdgeVel(ee.edge1, ee.t1));
			// Vector rforce=(m0->getCollisionEdgeForce(ee.edge0, ee.t0))-(m1->getCollisionEdgeForce(ee.edge1, ee.t1));

			real np=rpos*normal, nv=rvel*normal;
			Vector pimp=normal*(dist-np), vimp=normal*(-nv);
			Vector fimp=(rvel-nv*normal)*(-friction);
			pimp*=strength; vimp*=strength;

			m0->applyEdgeImpulse(ee.edge0, ee.t0, pimp, vimp, fimp, normal);
			m1->applyEdgeImpulse(ee.edge1, ee.t1, -pimp, -vimp, -fimp, normal);
			break;
		}
		case c_vertex_vertex: {
			Vector rpos=(m0->getCollisionVertPos(vv.vert0))-(m1->getCollisionVertPos(vv.vert1));
			Vector rvel=(m0->getCollisionVertVel(vv.vert0))-(m1->getCollisionVertVel(vv.vert1));
			// Vector rforce=(m0->getCollisionVertForce(vv.vert0))-(m1->getCollisionVertForce(vv.vert1));

			real np=rpos*normal, nv=rvel*normal;
			Vector pimp=normal*(dist-np), vimp=normal*(-nv);
			Vector fimp=(rvel-nv*normal)*(-friction);
			pimp*=strength; vimp*=strength;

			m0->applyVertImpulse(vv.vert0, pimp, vimp, fimp, normal);
			m1->applyVertImpulse(vv.vert1, -pimp, -vimp, -fimp, normal);
			break;
		}
		case c_vertex_edge: {
			Vector rpos=(m0->getCollisionVertPos(ve.vert))-(m1->getCollisionEdgePos(ve.edge, ve.t));
			Vector rvel=(m0->getCollisionVertVel(ve.vert))-(m1->getCollisionEdgeVel(ve.edge, ve.t));
			Vector rforce=(m0->getCollisionVertForce(ve.vert))-(m1->getCollisionEdgeForce(ve.edge, ve.t));

			real np=rpos*normal, nv=rvel*normal;
			Vector pimp=normal*(dist-np), vimp=normal*(-nv);
			Vector fimp=(rvel-nv*normal)*(-friction);
			pimp*=strength; vimp*=strength;

			m0->applyVertImpulse(ve.vert, pimp, vimp, fimp, normal);
			m1->applyEdgeImpulse(ve.edge, ve.t, -pimp, -vimp, -fimp, normal);
			break;
		}
	}
}
