#ifndef __COLLISIONDETECTOR_H__
#define __COLLISIONDETECTOR_H__

#include "utils.h"
#include "sequence.h"
#include "bvtree.h"

BEGIN_VLADO

#define VF_COLLIDE 1
#define VF_SELFCOLLIDE 2
#define VF_INTERSECT 4
#define VF_FIXED 8

enum CollisionType {
	c_vertex_vertex,
	c_vertex_edge,
	c_vertex_face,
	c_edge_edge,
};

class CollisionMesh {
public:
	// Collision detection stuff
	virtual real getFriction(void)=0;

	virtual int getNumCollisionVerts(void)=0;

	virtual int getNumCollisionEdges(void)=0;
	virtual void getCollisionEdgeVertices(int edge, int v[2])=0;

	virtual int getNumCollisionFaces(void)=0;
	virtual void getCollisionFaceVertices(int face, int v[3])=0;

	virtual void updateCollisionInfo(void)=0;
	virtual Vector getCollisionVertexPos(int i)=0;
	virtual int getCollisionVertexFlags(int i)=0;

	// Collision response stuff
	virtual Vector getCollisionVertPos(int vertIdx)=0;
	virtual Vector getCollisionVertVel(int vertIdx)=0;
	virtual Vector getCollisionVertForce(int vertIdx)=0;
	virtual Vector getCollisionEdgePos(int edgeIdx, real t)=0;
	virtual Vector getCollisionEdgeVel(int edgeIdx, real t)=0;
	virtual Vector getCollisionEdgeForce(int edgeIdx, real t)=0;
	virtual Vector getCollisionFacePos(int faceIdx, real uc, real vc)=0;
	virtual Vector getCollisionFaceVel(int faceIdx, real uc, real vc)=0;
	virtual Vector getCollisionFaceForce(int faceIdx, real uc, real vc)=0;

	virtual void applyVertImpulse(int vertIdx, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vector &nrm)=0;
	virtual void applyEdgeImpulse(int edgeIdx, real t, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vector &nrm)=0;
	virtual void applyFaceImpulse(int faceIdx, real uc, real vc, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vector &nrm)=0;

	virtual float getVertImpulseStrength(int vertIdx)=0;
	virtual float getEdgeImpulseStrength(int edgeIdx, real t)=0;
	virtual float getFaceImpulseStrength(int faceIdx, real uc, real vc)=0;
};

class Collision {
	real wsum;
public:
	CollisionType type;
	real friction;
	real strength;
	int deflectorCollision;

	CollisionMesh *m0, *m1;
	real s0, s1;
	union {
		struct {
			int vert, face;
			real uc, vc;
		} vf;
		struct {
			int edge0, edge1;
			real t0, t1;
		} ee;
		struct {
			int vert0, vert1;
		} vv;
		struct {
			int vert, edge;
			real t;
		} ve;
	};

	Vector normal;
	real dist;
	int ipath;
	int valid;

	int init(void);
	Vector relPos(void);
	Vector relVel(void);
	Vector relForce(void);
	void applyImpulses(void);
};

typedef Sequence<Collision> CollisionList;

#define CT_VERTEX_FACE 1
#define CT_EDGE_EDGE 2
#define CT_VERTEX_EDGE 4
#define CT_VERTEX_VERTEX 8

#define CT_ALL (CT_VERTEX_FACE | CT_EDGE_EDGE | CT_VERTEX_EDGE | CT_VERTEX_VERTEX)

class CollisionDetector {
public:
	virtual ~CollisionDetector(void) {}

	virtual void init(void)=0;
	virtual void freeMem(void)=0;

	virtual void addCollisionMesh(CollisionMesh *cmesh)=0;
	virtual void update(int rebuildAll=true)=0;
	virtual real collide(real tolerance, int meshTypes=CT_ALL, int selfTypes=CT_ALL)=0;
	virtual int checkIntersection(int checkSelfIntersection)=0;
	virtual void accept(void)=0;
	virtual void applyImpulses(void)=0;
	virtual CollisionList& getCollisionList(void)=0;
};

extern CollisionDetector* newDefaultCollisionDetector(void);

END_VLADO

#endif