/******************************************************
	FILE: SimClothToMax.h
	DESCRIPTION: This is header file for the connection between the SimCloth engine and the Max modifier
	CREATED: 1st December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) 2000-2002 by Vladimir Koylazov

	HISTORY:
		December 1st: file created

******************************************************/

#ifndef __SIMCLOTHTOMAX_H__
#define __SIMCLOTHTOMAX_H__

#include "ClothForces.h"
#include "EdgeLister.h"
#include "resource.h"
#include "quaternion.h"

class SimClothObject: public DynamicObject, public Vlado::CollisionMesh {
public:
	// Some Max-specific data
	INode *node;
	SimCloth3 *mod;
	SimClothLocalData *localData;

	Matrix3 tm;
	TimeValue worldTime;

	// The geometry of the object
	int numVerts;
	/*
	Vlado::Vector *objVerts; // Vertices of the mesh in object space
	Vlado::Vector *vertexNormals;
	*/

	int numEdges;
	Vlado::EdgeData *edges;

	int numFaces;
	Vlado::FaceData *faces;

	float friction;
	float worldScale;

	SimClothObject(void);
	~SimClothObject(void);

	void init(INode *node, SimCloth3 *mod, SimClothLocalData *localData);
	void freeMem(void);

	// From DynamicObject
	void beginSimulation(TimeValue t);
	void updateSimulation(TimeValue t);
	void endSimulation(void);
	void saveCache(TimeValue t);

	// From CollisionMesh
	Vlado::real getFriction(void) { return friction; }
	int getNumCollisionVerts(void) { return numVerts; }
	int getNumCollisionEdges(void) { return numEdges; }
	int getNumCollisionFaces(void) { return numFaces; }
	void getCollisionEdgeVertices(int edge, int v[2]) { v[0]=edges[edge].v[0]; v[1]=edges[edge].v[1]; }
	void getCollisionFaceVertices(int face, int v[3]) { v[0]=faces[face].v[0]; v[1]=faces[face].v[1]; v[2]=faces[face].v[2]; }

	// Other methods
	virtual void prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm)=0;
	virtual void updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm)=0;
	virtual void saveCache(Point3 *points, Matrix3 &itm)=0;
};

struct ClothParticle {
	Vlado::real mass, mass_inv;
	Vlado::real origMass_inv;

	Vlado::Vector p, v; // position and velocity
	Vlado::Vector np, nv; // new position and velocity

	Vlado::Vector pimp, vimp; // collision impulses
	int numImpulses;

	Vlado::Vector p0, p1, p2; // the position at the start and end of the frame, if the particle is fixed
	Vlado::Vector v0, v1; // the velocity at the start and end of the frame, if the particle is fixed

	Vlado::Vector force; // the sum of the forces acting on the particle

	Vlado::Matrix mat;

	// Support for cloth forces
	Vlado::Vector getPos(void) { return np; }
	Vlado::Vector getVel(void) { return nv; }
	void applyForce(Vlado::Vector &f) { force+=f; }
};

class ClothObject: public SimClothObject {
	// The cloth particles
	int numParts;
	ClothParticle *parts;

	float particleMass;
	float airDrag;
	int selfCollideOn;

	Vlado::Table<ForceField*> forceFields;

	//**************************************************
	// The internal forces of the cloth

	// Stretch forces
	int stretchType;

	int numSprings;
	Vlado::SpringForce<ClothParticle> *springs;
	float spring_stiffness, spring_damping;

	int numStretchShears;
	Vlado::StretchShearForce<ClothParticle> *stretchShears;
	float stretch_stiffness, shear_stiffness, stretchShear_damping;

	int numTrisprings;
	Vlado::Trispring<ClothParticle> *trisprings;
	float trispring_stiffness, trispring_damping;

	// Bend forces
	int bendType;

	int numBendSprings;
	Vlado::SpringForce<ClothParticle> *bendSprings;
	float bendSpring_stiffness, bendSpring_damping;

	int numBendAngles;
	Vlado::BendAngle<ClothParticle, Vlado::FaceData> *bendAngles;
	float bendAngle_stiffness, bendAngle_damping;

	int numBendProjections;
	Vlado::BendProjection<ClothParticle> *bendProjections;
	float bendProjection_stiffness, bendProjection_damping;

	int skipSelectedEdges;

public:
	ClothObject(INode *node, SimCloth3 *mod, SimClothLocalData *localData) { init(node, mod, localData); }
	~ClothObject(void) { freeMem(); }

	void init(INode *node, SimCloth3 *mod, SimClothLocalData *localData);
	void freeMem(void);

	// Integration methods
	int getNumVars(void);
	void reset(Vlado::real subTime);
	void computeChanges(Vlado::real *x, float dt);
	void addChanges(Vlado::real *x, Vlado::real scale);
	void acceptChanges(void);

	void clearForces(void);
	void applyForces(void);
	void applyGravity(const Vlado::Vector &accel);

	// Collision detection methods
	Vlado::CollisionMesh *getCollisionMesh(void);

	// From SimClothObject
	void prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
	void updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
	void saveCache(Point3 *points, Matrix3 &itm);

	// From CollisionMesh
	void updateCollisionInfo(void) {}
	Vlado::Vector getCollisionVertexPos(int i);
	int getCollisionVertexFlags(int i);

	Vlado::Vector getCollisionVertPos(int vertIdx);
	Vlado::Vector getCollisionVertVel(int vertIdx);
	Vlado::Vector getCollisionVertForce(int vertIdx);
	Vlado::Vector getCollisionEdgePos(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionEdgeVel(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionEdgeForce(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionFacePos(int faceIdx, Vlado::real uc, Vlado::real vc);
	Vlado::Vector getCollisionFaceVel(int faceIdx, Vlado::real uc, Vlado::real vc);
	Vlado::Vector getCollisionFaceForce(int faceIdx, Vlado::real uc, Vlado::real vc);

	void applyVertImpulse(int vertIdx, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &nrm);
	void applyEdgeImpulse(int edgeIdx, Vlado::real t, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &nrm);
	void applyFaceImpulse(int faceIdx, Vlado::real uc, Vlado::real vc, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &nrm);

	float getVertImpulseStrength(int vertIdx);
	float getEdgeImpulseStrength(int edgeIdx, Vlado::real t);
	float getFaceImpulseStrength(int faceIdx, Vlado::real uc, Vlado::real vc);
};

struct DeflectorParticle {
	Vlado::Vector p, v; // position and velocity

	Vlado::Vector p0, p1, p2; // the position at the start and end of the frame
	Vlado::Vector v0, v1; // the velocity at the start and end of the frame
};

class DeflectorObject: public SimClothObject {
	int numParts;
	DeflectorParticle *parts;
public:
	DeflectorObject(INode *node, SimCloth3 *mod, SimClothLocalData *localData) { init(node, mod, localData); }
	~DeflectorObject(void) { freeMem(); }

	void init(INode *node, SimCloth3 *mod, SimClothLocalData *localData);
	void freeMem(void);

	// Integration methods
	int getNumVars(void) { return 0; }
	void reset(Vlado::real subTime);
	void computeChanges(Vlado::real *x, float dt) {}
	void addChanges(Vlado::real *x, Vlado::real scale) {}
	void acceptChanges(void) {}

	void clearForces(void) {}
	void applyForces(void) {}
	void applyGravity(const Vlado::Vector &accel) {}

	// Collision detection methods
	Vlado::CollisionMesh *getCollisionMesh(void);

	// From SimClothObject
	void prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
	void updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
	void saveCache(Point3 *points, Matrix3 &itm);

	// From CollisionMesh
	void updateCollisionInfo(void) {}
	Vlado::Vector getCollisionVertexPos(int i);
	int getCollisionVertexFlags(int i);

	Vlado::Vector getCollisionVertPos(int vertIdx);
	Vlado::Vector getCollisionVertVel(int vertIdx);
	Vlado::Vector getCollisionVertForce(int vertIdx) { return Vlado::Vector(0,0,0); }
	Vlado::Vector getCollisionEdgePos(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionEdgeVel(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionEdgeForce(int edgeIdx, Vlado::real t) { return Vlado::Vector(0,0,0); }
	Vlado::Vector getCollisionFacePos(int faceIdx, Vlado::real uc, Vlado::real vc);
	Vlado::Vector getCollisionFaceVel(int faceIdx, Vlado::real uc, Vlado::real vc);
	Vlado::Vector getCollisionFaceForce(int faceIdx, Vlado::real uc, Vlado::real vc) { return Vlado::Vector(0,0,0); }

	void applyVertImpulse(int vertIdx, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &nrm) {}
	void applyEdgeImpulse(int edgeIdx, Vlado::real t, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &nrm) {}
	void applyFaceImpulse(int faceIdx, Vlado::real uc, Vlado::real vc, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &nrm) {}

	float getVertImpulseStrength(int vertIdx) { return 0.0f; }
	float getEdgeImpulseStrength(int edgeIdx, Vlado::real t) { return 0.0f; }
	float getFaceImpulseStrength(int faceIdx, Vlado::real uc, Vlado::real vc) { return 0.0f; }
};

class RigidObject: public SimClothObject {
	// Mass
	Vlado::real particleMass;
	Vlado::real mass, mass_inv;
	float *vmass;

	// Forces
	Vlado::Vector force, torque;

	// Collision impulses
	Vlado::Vector pos_imp, linear_imp, angular_imp, rot_imp;

	// The rigid body state
	Vlado::Vector pos; // Position of center of mass in world space
	Vlado::Matrix tm; // Orientation
	// Vlado::Quaternion rot; // Rotation
	Vlado::Vector linearVel; // Linear velocity
	Vlado::Vector angularVel; // Angular velocity
	// Vlado::Vector linearMoment, angularMoment;

	Vlado::Vector i_pos;
	Vlado::Matrix i_tm;
	Vlado::Vector i_rot;
	Vlado::Vector i_linearVel, i_angularVel;

	// New state
	Vlado::Vector n_pos;
	Vlado::Matrix n_tm;
	Vlado::Vector n_rot;
	Vlado::Vector n_linearVel, n_angularVel;
	// Vlado::Quaternion n_rot;
	// Vlado::Vector n_linearMoment, n_angularMoment;

	// Inertia matrices in object space
	Vlado::Matrix iobj, iobj_inv;

	Vlado::Vector cmass; // Center of mass (in object space)
	Vlado::Vector *verts; // The vertices of the mesh (in object space, relative to the center of mass)

	// The transformation from object space to world space
	Vlado::Vector pt;
public:
	RigidObject(INode *node, SimCloth3 *mod, SimClothLocalData *localData) { init(node, mod, localData); }
	~RigidObject(void) { freeMem(); }

	void init(INode *node, SimCloth3 *mod, SimClothLocalData *localData);
	void freeMem(void);

	// Integration methods
	int getNumVars(void);
	void reset(Vlado::real subTime);
	void computeTargets(float dt);
	void computeChanges(Vlado::real *x, float dt);
	void addChanges(Vlado::real *x, Vlado::real scale);
	void acceptChanges(void);

	void clearForces(void);
	void applyForces(void);
	void applyGravity(const Vlado::Vector &accel);

	// Collision detection methods
	Vlado::CollisionMesh *getCollisionMesh(void);

	// From CollisionMesh
	void prepareCollisionDetection(void);

	// From SimClothObject
	void prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
	void updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
	void saveCache(Point3 *points, Matrix3 &itm);

	// From CollisionMesh
	void updateCollisionInfo(void) {}
	Vlado::Vector getCollisionVertexPos(int i);
	int getCollisionVertexFlags(int i);

	Vlado::Vector getCollisionVertPos(int vertIdx);
	Vlado::Vector getCollisionVertVel(int vertIdx);
	Vlado::Vector getCollisionVertForce(int vertIdx);
	Vlado::Vector getCollisionEdgePos(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionEdgeVel(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionEdgeForce(int edgeIdx, Vlado::real t);
	Vlado::Vector getCollisionFacePos(int faceIdx, Vlado::real uc, Vlado::real vc);
	Vlado::Vector getCollisionFaceVel(int faceIdx, Vlado::real uc, Vlado::real vc);
	Vlado::Vector getCollisionFaceForce(int faceIdx, Vlado::real uc, Vlado::real vc);

	void applyVertImpulse(int vertIdx, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &normal);
	void applyEdgeImpulse(int edgeIdx, Vlado::real t, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &normal);
	void applyFaceImpulse(int faceIdx, Vlado::real uc, Vlado::real vc, const Vlado::Vector &pimp, const Vlado::Vector &vimp, const Vlado::Vector &fimp, const Vlado::Vector &normal);

	float getVertImpulseStrength(int vertIdx) { return 1.0f; }
	float getEdgeImpulseStrength(int edgeIdx, Vlado::real t) { return 1.0f; }
	float getFaceImpulseStrength(int faceIdx, Vlado::real uc, Vlado::real vc) { return 1.0f; }
};

class SimClothToMax: public SimClothConnection {
	Vlado::Table<SimClothObject*> simClothObjects;
	HWND hWnd;
	int substeps;
	int cgIterations;
public:
	HFONT hFont;
	int abort;
	int calculating;

	SimClothToMax(void);
	~SimClothToMax(void);

	// SimClothEngine connection
	int numDynamicObjects(void) { return simClothObjects.count(); }
	DynamicObject* getDynamicObject(int i) { return simClothObjects[i]; }
	void getSimClothEngineParams(SimClothEngineParams &params);

	void displayMessage(TCHAR *str) { SetWindowText(GetDlgItem(hWnd, st_stats), str); }
	void setFrameProgressRange(int minRange, int maxRange) {
		SendMessage(GetDlgItem(hWnd, pb_frame), PBM_SETRANGE32, minRange, maxRange);
		substeps=maxRange;
	}
	void setFrameProgressPos(int progressPos) {
		SendMessage(GetDlgItem(hWnd, pb_frame), PBM_SETPOS, progressPos, 0);
		TCHAR str[512];
		_stprintf(str, "Current frame (substep %i of %i)", progressPos, substeps);
		SetWindowText(GetDlgItem(hWnd, st_frame), str);
	}
	void setNumCollisions(int collisions) {
		TCHAR str[512];
		_stprintf(str, "%i", collisions);
		SetWindowText(GetDlgItem(hWnd, st_collisions), str);
	}
	void setNumCGIterations(int iterations) {
		TCHAR str[512];
		_stprintf(str, "%i", iterations);
		SetWindowText(GetDlgItem(hWnd, st_cgIterations), str);
		cgIterations+=iterations;
	}
	int getAbort(void) {
		if (GetAsyncKeyState(VK_ESCAPE)) return true;
		if (abort) return true;
		if (!GetCOREInterface()->CheckMAXMessages()) return true;
		return false;
	}


	// Other methods
	int simulate(SimClothEngine *scEngine);
	void init(void);
	void freeData(void);

	void listNodes(INode *root);
};

#endif