#include "Headers.h"

#include "Utils.h"
#include "SimClothEngine.h"
#include "resource.h"
#include "SimClothModifier.h"
#include "SimClothToMax.h"
#include "ToMax.h"
#include "VertexGroup.h"
#include "tomax.h"

USE_VLADO

#define FORCES_USE_SSE

#ifdef FORCES_USE_SSE
#	include "fvec.h"
#endif

extern ParamBlockDesc2 sc3_global_pblock;

const float forceScale=1000.0f;
const float vscale=1024.0f;

INT_PTR CALLBACK statsDlgProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch (uMsg) {
		case WM_INITDIALOG: {
			SetWindowLongPtr(hWnd, GWLP_USERDATA, lParam);
			SimClothToMax *connection=(SimClothToMax*) lParam;

			HWND progressBar=GetDlgItem(hWnd, pb_frame);
			SendMessage(progressBar, PBM_SETPOS, 0, 0);
			SendMessage(progressBar, PBM_SETRANGE32, 0, 100);

			LOGFONT logFont;
			memset(&logFont, 0, sizeof(logFont));

			HDC hDC=GetDC(hWnd);
			logFont.lfHeight=-MulDiv(8, GetDeviceCaps(hDC, LOGPIXELSY), 72);
			logFont.lfWeight=FW_BOLD;
			_tcscpy(logFont.lfFaceName, _T("Microsoft Sans Serif"));
			ReleaseDC(hWnd, hDC);

			connection->hFont=CreateFontIndirect(&logFont);
			SendDlgItemMessage(hWnd, st_collisions, WM_SETFONT, (LPARAM) connection->hFont, FALSE);
			SendDlgItemMessage(hWnd, st_lastFrameTime, WM_SETFONT, (LPARAM) connection->hFont, FALSE);
			SendDlgItemMessage(hWnd, st_cgIterations, WM_SETFONT, (LPARAM) connection->hFont, FALSE);
			SendDlgItemMessage(hWnd, st_simulationTime, WM_SETFONT, (LPARAM) connection->hFont, FALSE);
			SendDlgItemMessage(hWnd, st_estimatedTimeLeft, WM_SETFONT, (LPARAM) connection->hFont, FALSE);

			return TRUE;
		}
		case WM_DESTROY: {
			SimClothToMax *connection=(SimClothToMax*) GetWindowLongPtr(hWnd, GWLP_USERDATA);
			DeleteObject(connection->hFont);
			connection->hFont=NULL;
			return TRUE;
		}
		case WM_COMMAND: {
			SimClothToMax *connection=(SimClothToMax*) GetWindowLongPtr(hWnd, GWLP_USERDATA);
			if (LOWORD(wParam)==bn_cancel) connection->abort=true;
			return TRUE;
		}
	}
	return FALSE;
}

void printTime(TCHAR *str, int milliseconds) {
	if (milliseconds<60*1000) _stprintf(str, "%.1fs", float(milliseconds)/1000.0f);
	else if (milliseconds<60*60*1000) {
		int minutes=milliseconds/(60*1000);
		milliseconds-=minutes*(60*1000);
		_stprintf(str, "%im %.1fs", minutes, float(milliseconds)/1000.0f);
	} else {
		int hours=milliseconds/(60*60*1000);
		milliseconds-=hours*(60*60*1000);
		int minutes=milliseconds/(60*1000);
		milliseconds-=minutes*(60*1000);
		_stprintf(str, "%ih %im %.1fs", hours, minutes, float(milliseconds)/1000.0f);
	}
}

SimClothToMax::SimClothToMax(void) {
	calculating=false;
}

SimClothToMax::~SimClothToMax(void) {
	freeData();
}

int SimClothToMax::simulate(SimClothEngine *scEngine) {
	scEngine->setConnection(this);

	Interface *ip=GetCOREInterface();
  int ticksPerFrame=GetTicksPerFrame();

	// Determine the simulation interval
	int startFrame, endFrame;
	if (sc3_global_pblock.GetInt(sc3_global_useActiveTimeSegment, 0)) {
		Interval animRange=ip->GetAnimRange();
		startFrame=animRange.Start()/ticksPerFrame;
		endFrame=animRange.End()/ticksPerFrame;
	} else {
		startFrame=sc3_global_pblock.GetInt(sc3_global_startFrame, 0);
		endFrame=sc3_global_pblock.GetInt(sc3_global_endFrame, 0);
	}

	// Prepare the objects in the simulation
	try { for (int i=0; i<simClothObjects.count(); i++) simClothObjects[i]->beginSimulation(startFrame*ticksPerFrame); }
	catch (...) { return SC3_ERR_EXCEPTION; }

	// Prepare the SimCloth engine
	try { scEngine->beginSimulation(); }
	catch (...) { return SC3_ERR_EXCEPTION; }

	// Record the start of the simulation (system ticks)
	int ticksStart=GetTickCount();

	int res=0;

	hWnd=CreateDialogParam(hInstance, MAKEINTRESOURCE(IDD_STATS), GetCOREInterface()->GetMAXHWnd(), &statsDlgProc, (LPARAM) this);

	HWND simProgress=GetDlgItem(hWnd, pb_simulation);
	SendMessage(simProgress, PBM_SETPOS, startFrame, 0);
	SendMessage(simProgress, PBM_SETRANGE32, startFrame, endFrame);
	abort=false;

#ifdef _INTERNAL
	FILE *fdebug_i=fopen("w:/temp/simcloth_i.out", "wt");
	FILE *fdebug_f=fopen("w:/temp/simcloth_f.out", "wt");
	FILE *fdebug_t=fopen("w:/temp/simcloth_t.out", "wt");
#endif

	// Loop through all the frames in simulation and compute the state of the objects
	float dt=1.0f/float(GetFrameRate());
	TCHAR str[512];
	for (int i=startFrame; i<=endFrame; i++) {
		int frameTicks=GetTickCount();

		TimeValue time=i*ticksPerFrame;

		cgIterations=0;

		// Update the objects for the current frame
		try { for (int j=0; j<simClothObjects.count(); j++) simClothObjects[j]->updateSimulation(time+ticksPerFrame); }
		catch (...) { res=SC3_ERR_EXCEPTION; break; }

		// Call the SimCloth engine to compute the object states for the frame
		try { res=scEngine->step(dt); }
		catch (...) { res=SC3_ERR_EXCEPTION; break; }
		if (res!=0) break;

		// Save the state of the objects
		try { for (int j=0; j<simClothObjects.count(); j++) simClothObjects[j]->saveCache(time); }
		catch (...) { res=SC3_ERR_EXCEPTION; break; }

		// Update the viewports
		ip->SetTime(time);

		// If the Esc key is pressed, abort the simulation
		if (getAbort()) { res=SC3_ERR_INTERRUPTED; break; }

		printTime(str, GetTickCount()-frameTicks);
		SetWindowText(GetDlgItem(hWnd, st_lastFrameTime), str);

		int simTime=GetTickCount()-ticksStart;
		printTime(str, simTime);
		SetWindowText(GetDlgItem(hWnd, st_simulationTime), str);

		int estimatedTimeLeft=(endFrame-i)*simTime/(i-startFrame+1);
		printTime(str, estimatedTimeLeft);
		SetWindowText(GetDlgItem(hWnd, st_estimatedTimeLeft), str);

		SendMessage(simProgress, PBM_SETPOS, i, 0);
		_stprintf(str, "Total simulation (frame %i of %i)", i-startFrame+1, endFrame-startFrame+1);
		SetWindowText(GetDlgItem(hWnd, st_simulation), str);
#ifdef _INTERNAL
		fprintf(fdebug_i, "%i %i\n", time, cgIterations);
		fprintf(fdebug_f, "%i %i\n", time, scEngine->forceEvals);
		fprintf(fdebug_t, "%i %i\n", time, GetTickCount()-frameTicks);
#endif
	}

	// Record the end of the simulation and compute the total simulation time (in system ticks)
	int ticksEnd=GetTickCount();
	DebugPrint("Simulation took %i ticks\n", ticksEnd-ticksStart);

	// Free any data associated with the SimCloth engine
	try { scEngine->endSimulation(); }
	catch (...) { res=SC3_ERR_EXCEPTION; }

	// Free any data associated with the objects in the simulation
	try { for (i=0; i<simClothObjects.count(); i++) simClothObjects[i]->endSimulation(); }
	catch (...) { res=SC3_ERR_EXCEPTION; }

	DestroyWindow(hWnd);
	hWnd=NULL;

#ifdef _INTERNAL
	fclose(fdebug_i);
	fclose(fdebug_f);
	fclose(fdebug_t);
#endif

	// Put the calculation time in the status bar
	printTime(str, ticksEnd-ticksStart);
	TCHAR msg[512];
	_stprintf(msg, "Simulation took %s", str);
	ip->ReplacePrompt(msg);

	return res;
}

void SimClothToMax::listNodes(INode *root) {
	for (int i=0; i<root->NumberOfChildren(); i++) {
		INode *node=root->GetChildNode(i);
		listNodes(node);
		if (node->IsNodeHidden()) continue;

		Object *obj=node->GetObjectRef();
		if (!obj) continue;
		while (obj->SuperClassID()==GEN_DERIVOB_CLASS_ID) {
			IDerivedObject *derivObj=(IDerivedObject*) obj;

			for (int i=0; i<derivObj->NumModifiers(); i++) {
				Modifier *mod=derivObj->GetModifier(i);
				if (!mod->IsEnabled()) continue;

				if (mod->ClassID()==SIMCLOTH3_CLASS_ID) {
					ModContext *mc=derivObj->GetModContext(i);
					if (mc->localData==NULL) mc->localData=new SimClothLocalData;

					SimCloth3 *sc3=(SimCloth3*) mod;
					SimClothObject *scObj=NULL;
					switch(sc3->GetParamBlockByID(sc3_general)->GetInt(sc3_general_type)) {
						case 0: scObj=new DeflectorObject(node, sc3, (SimClothLocalData*) mc->localData); break;
						case 1: scObj=new ClothObject(node, sc3, (SimClothLocalData*) mc->localData); break;
						case 2: scObj=new RigidObject(node, sc3, (SimClothLocalData*) mc->localData); break;
					}
					if (scObj) *simClothObjects.newElement()=scObj;
				}
			}
			obj=derivObj->GetObjRef();
		}
	}
}

void SimClothToMax::init(void) {
	// Scan the scene for simcloth objects
	calculating=true;
	Interface *ip=GetCOREInterface();
	INode *root=ip->GetRootNode();
	listNodes(root);
}

void SimClothToMax::freeData(void) {
	for (int i=0; i<simClothObjects.count(); i++) {
		simClothObjects[i]->freeMem();
		delete simClothObjects[i];
	}
	simClothObjects.freeData();
	calculating=false;
}

void SimClothToMax::getSimClothEngineParams(SimClothEngineParams &params) {
	params.substeps=sc3_global_pblock.GetInt(sc3_global_substeps, 0);
	params.gravity.set(0.0f, 0.0f, -sc3_global_pblock.GetFloat(sc3_global_gravity, 0));
	params.collisionTolerance=sc3_global_pblock.GetFloat(sc3_global_collisionTolerance, 0);
	params.solverPrecision=sc3_global_pblock.GetFloat(sc3_global_solverPrecision, 0);
	params.checkIntersections=sc3_global_pblock.GetInt(sc3_global_checkIntersections, 0);
	params.adaptiveSubdivs=sc3_global_pblock.GetInt(sc3_global_adaptiveSubdivs, 0);
	params.useSSE=sc3_global_pblock.GetInt(sc3_global_useSSE, 0);
	params.worldScale=sc3_global_pblock.GetFloat(sc3_global_worldScale, 0);
}

//******************************************************
// SimClothObject

inline Vector combineImpulse(const Vector &a, const Vector &imp, float k) {
	// return a/(1.0f+imp.length())+imp;
	return a+imp;

	float lenSq=imp.lengthSqr();
	if (lenSq<1e-6f) return a+imp;
	Vector h=imp*((a*imp)/lenSq);
	Vector p=a-h;

	return p+imp;
}

SimClothObject::SimClothObject(void) {
	node=NULL;
	mod=NULL;
	localData=NULL;
}

SimClothObject::~SimClothObject(void) {
	freeMem();
}

void SimClothObject::init(INode *node, SimCloth3 *mod, SimClothLocalData *localData) {
	this->node=node;
	this->mod=mod;
	this->localData=localData;

	numEdges=0;
	edges=NULL;

	numFaces=0;
	faces=NULL;

	friction=mod->pblockCollisions->GetFloat(sc3_collisions_friction, 0);
	worldScale=sc3_global_pblock.GetFloat(sc3_global_worldScale, 0);
}

void SimClothObject::freeMem(void) {
	node=NULL;
	mod=NULL;
	localData=NULL;

	if (edges) { delete[] edges; edges=NULL; }
	numEdges=0;

	if (faces) { delete[] faces; faces=NULL; }
	numFaces=0;
}

TriObject *GetTriObjectFromNode(TimeValue t, INode *node, int &deleteIt) {
	deleteIt=FALSE;
	Object *obj=node->EvalWorldState(t).obj;
	if (obj->CanConvertToType(triObjectClassID)) { 
		TriObject *tri = (TriObject *) obj->ConvertToType(t, triObjectClassID);
		if (obj!=tri) deleteIt=TRUE;
		return tri;
	} else return NULL;
}

class MeshEdgeLister {
	Mesh &mesh;
	SimClothObject *scObj;
public:
	MeshEdgeLister(Mesh &m, SimClothObject *iscObj):mesh(m), scObj(iscObj) {}

	int getNumVerts() { return mesh.numVerts; }
	int getNumFaces() { return mesh.numFaces; }

	int getFaceVertex(int faceIdx, int vertIdx) { return mesh.faces[faceIdx].v[vertIdx]; }

	void setNumEdges(int numEdges) {
		scObj->numEdges=numEdges;
		scObj->edges=new EdgeData[numEdges];
	}
	void setEdge(int idx, EdgeData &edge) {
		scObj->edges[idx]=edge;
	}
	void setFaceEdge(int faceIdx, int edgeFaceIdx, int edgeIdx) {}
};

void SimClothObject::beginSimulation(TimeValue t) {
	localData->cache.freeData();

	node->InvalidateWS();

	mod->dontModify=true;
	mod->NotifyDependents(FOREVER, PART_ALL, REFMSG_CHANGE);
	int deleteIt;
	Matrix3 tm=node->GetObjTMAfterWSM(t);
	TriObject *tri=GetTriObjectFromNode(t, node, deleteIt);

	Mesh &mesh=tri->GetMesh();

	// Record the mesh topology
	MeshEdgeLister meshEdgeLister(mesh, this);
	EdgeLister<MeshEdgeLister> edgeLister;
	edgeLister.enumEdges(meshEdgeLister);

	numFaces=mesh.numFaces;
	faces=new FaceData[numFaces];
	for (int i=0; i<numFaces; i++) {
		for (int j=0; j<3; j++) faces[i].v[j]=mesh.faces[i].v[j];
	}

	numVerts=mesh.numVerts;

	// Call prepareMesh() to build the actual dynamic object
	Matrix3 stm=tm;
	stm*=1.0f/worldScale;
	prepareMesh(tri, mesh, stm);

	if (deleteIt) tri->DeleteMe();
	mod->dontModify=false;

	worldTime=t;
	clearForces();

	mod->beginSimulation(t, this);
}

void SimClothObject::updateSimulation(TimeValue t) {
	worldTime=t;
	mod->dontModify=true;
	mod->updateSimulation(t, this);

	int deleteIt;
	TriObject *tri=GetTriObjectFromNode(t, node, deleteIt);

	Mesh &mesh=tri->GetMesh();
	tm=node->GetObjTMAfterWSM(t);

	Matrix3 stm=tm;
	stm*=1.0f/worldScale;
	updateMesh(tri, mesh, stm);

	if (deleteIt) tri->DeleteMe();
	mod->dontModify=false;
}

void SimClothObject::endSimulation(void) {
	mod->endSimulation();
	freeMem();
}

void SimClothObject::saveCache(TimeValue t) {
	SimClothKey *key=localData->cache.keys.newElement();
	key->time=t;
	key->numPoints=numVerts;
	key->points=new Point3[numVerts];

	Matrix3 stm=tm;
	stm*=1.0f/worldScale;

	Matrix3 itm=Inverse(stm);
	saveCache(key->points, itm);

	mod->NotifyDependents(FOREVER, PART_ALL, REFMSG_CHANGE);
}

//*******************************************************
// ClothObject

void ClothObject::init(INode *node, SimCloth3 *mod, SimClothLocalData *localData) {
	SimClothObject::init(node, mod, localData);

	numParts=0;
	parts=NULL;

	numSprings=0;
	springs=NULL;

	numStretchShears=0;
	stretchShears=NULL;

	numTrisprings=0;
	trisprings=NULL;

	numBendSprings=0;
	bendSprings=NULL;

	numBendAngles=0;
	bendAngles=NULL;

	numBendProjections=0;
	bendProjections=NULL;

	//*******************************************
	stretchType=mod->pblockIntegrity->GetInt(sc3_integrity_stretchType);

	spring_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_springsStiffness, 0);
	spring_damping=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_springsDamping, 0);

	stretch_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_stretch_stiffness, 0);
	shear_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_shear_stiffness, 0);
	stretchShear_damping=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_stretchShear_damping, 0);

	trispring_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_trispringStiffness, 0);
	trispring_damping=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_trispringDamping, 0);

	//*******************************************
	bendType=mod->pblockIntegrity->GetInt(sc3_integrity_bendType);

	bendSpring_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_bendSpring_stiffness, 0);
	bendSpring_damping=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_bendSpring_damping, 0);

	bendAngle_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_bendAngle_stiffness, 0);
	bendAngle_damping=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_bendAngle_damping, 0);

	bendProjection_stiffness=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_bendProjection_stiffness, 0);
	bendProjection_damping=forceScale*mod->pblockIntegrity->GetFloat(sc3_integrity_bendProjection_damping, 0);

	skipSelectedEdges=mod->pblockIntegrity->GetInt(sc3_integrity_excludeSelectedEdges, 0);

	//*******************************************
	particleMass=mod->pblockGeneral->GetFloat(sc3_general_particleMass, 0);
	airDrag=mod->pblockIntegrity->GetFloat(sc3_integrity_airDrag, 0)*0.1f;
	selfCollideOn=mod->pblockCollisions->GetInt(sc3_collisions_selfCollide, 0)==TRUE;

	forceFields.clear();
	for (int n=0; n<mod->pblockIntegrity->Count(sc3_integrity_spacewarps); n++) {
		INode *node=mod->pblockIntegrity->GetINode(sc3_integrity_spacewarps, 0, n);
		Object *obj=node->GetObjectRef()->FindBaseObject();
		if (obj->SuperClassID()==WSM_OBJECT_CLASS_ID) {
			WSMObject *wsmObject=(WSMObject*) obj;
			ForceField *ff=wsmObject->GetForceField(node);
			if (ff) *forceFields.newElement()=ff;
		}
	}
}

void ClothObject::freeMem(void) {
	SimClothObject::freeMem();

	if (parts) { delete[] parts; parts=NULL; }
	numParts=0;

	if (springs) { delete[] springs; springs=NULL; }
	numSprings=0;

	if (stretchShears) { delete[] stretchShears; stretchShears=NULL; }
	numStretchShears=0;

	if (numTrisprings) { delete[] trisprings; trisprings=NULL; }
	numTrisprings=0;

	if (bendSprings) { delete[] bendSprings; bendSprings=NULL; }
	numBendSprings=0;

	if (bendAngles) { delete[] bendAngles; bendAngles=NULL; }
	numBendAngles=0;

	if (bendProjections) { delete[] bendProjections; bendProjections=NULL; }
	numBendProjections=0;

	for (int i=0; i<forceFields.count(); i++) forceFields[i]->DeleteThis();
	forceFields.freeData();
}

int ClothObject::getNumVars(void) { return numParts*6; }

void ClothObject::reset(real subTime) {
	for (int i=0; i<numParts; i++) {
		ClothParticle &p=parts[i];
		if (p.mass_inv>1e-6f) {
			// A free particle
			p.np=p.p;
			p.nv=p.v;
		} else {
			// A constrained particle
			p.p=p.np=p.p0+(p.p1-p.p0)*subTime;
			p.v=p.nv=p.v0+(p.v1-p.v0)*subTime;
		}
	}
}

inline Vector cut(Vector &a, Vector &b) {
	float lenSq=b.lengthSqr();
	if (lenSq<1e-6f) return a;
	return a-b*((a*b)/lenSq);
}

void ClothObject::computeChanges(real *x, float dt) {
	for (int i=0; i<numParts; i++) {
		ClothParticle &p=parts[i];

		Vector nv=p.v+p.force*(p.mass_inv*dt);
		Vector np=p.p+p.nv*dt;

		Vector dp=combineImpulse(np-p.np, p.pimp, 1.0f);
		Vector dv=combineImpulse(nv-p.nv, p.vimp, 1.0f)/vscale;

		*(x++)=dp[0];
		*(x++)=dp[1];
		*(x++)=dp[2];

		*(x++)=dv[0];
		*(x++)=dv[1];
		*(x++)=dv[2];
	}
}

void ClothObject::addChanges(real *x, real scale) {
	for (int i=0; i<numParts; i++) {
		ClothParticle &p=parts[i];

		p.np[0]+=*(x++)*scale;
		p.np[1]+=*(x++)*scale;
		p.np[2]+=*(x++)*scale;

		p.nv[0]+=*(x++)*scale*vscale;
		p.nv[1]+=*(x++)*scale*vscale;
		p.nv[2]+=*(x++)*scale*vscale;
	}
}

void ClothObject::acceptChanges(void) {
	for (int i=0; i<numParts; i++) {
		ClothParticle &p=parts[i];

		p.p=p.np;
		p.v=p.nv;
	}
}

void ClothObject::clearForces(void) {
	for (int i=0; i<numParts; i++) {
		ClothParticle &p=parts[i];

		p.force.makeZero();
		// p.pimp.init();
		// p.vimp.init();
		p.pimp.makeZero();
		p.vimp.makeZero();
	}
}

void ClothObject::applyForces(void) {
	// Compute stretch forces
	for (int i=0; i<numSprings; i++) springs[i].applyForces();
	for (int i=0; i<numStretchShears; i++) stretchShears[i].applyForces();
	for (int i=0; i<numTrisprings; i++) trisprings[i].applyForces();

	// Compute bend forces
	for (int i=0; i<numFaces; i++) {
		FaceData &f=faces[i];

		Vector e0=parts[f.v[1]].np-parts[f.v[0]].np;
		Vector e1=parts[f.v[2]].np-parts[f.v[0]].np;

		Vector n=e0^e1;
		float len=n.length();

		f.normalLength=len;
		f.normalLength_inv=1.0f/len;
		f.normal=n/len;
	}

	for (int i=0; i<numBendSprings; i++) bendSprings[i].applyForces();
	for (int i=0; i<numBendAngles; i++) bendAngles[i].applyForces();
	for (int i=0; i<numBendProjections; i++) bendProjections[i].applyForces();

	// Compute force fields
	for (i=0; i<forceFields.count(); i++) {
		ForceField *ff=forceFields[i];
		for (int k=0; k<numFaces; k++) {
			Vector pos=(parts[faces[k].v[0]].np+parts[faces[k].v[1]].np+parts[faces[k].v[2]].np)/3.0f;
			Vector vel=(parts[faces[k].v[0]].nv+parts[faces[k].v[1]].nv+parts[faces[k].v[2]].nv)/3.0f;

			Vector f=toVector(ff->Force(worldTime, toPoint3(pos), toPoint3(vel), 0))*1e6f;
			f=faces[k].normal*(faces[k].normal*f);

			parts[faces[k].v[0]].force+=f;
			parts[faces[k].v[1]].force+=f;
			parts[faces[k].v[2]].force+=f;
		}
	}

	// Compute air drag
	if (airDrag>1e-6f) {
		for (int k=0; k<numFaces; k++) {
			Vector pos=(parts[faces[k].v[0]].np+parts[faces[k].v[1]].np+parts[faces[k].v[2]].np)/3.0f;
			Vector vel=(parts[faces[k].v[0]].nv+parts[faces[k].v[1]].nv+parts[faces[k].v[2]].nv)/3.0f;

			Vector f=faces[k].normal*((faces[k].normal*vel)*(-airDrag));

			parts[faces[k].v[0]].force+=f;
			parts[faces[k].v[1]].force+=f;
			parts[faces[k].v[2]].force+=f;
		}
	}
}

void ClothObject::applyGravity(const Vector &accel) {
	for (int i=0; i<numParts; i++) parts[i].force+=parts[i].mass*accel;
}

void ClothObject::prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm) {
	// Create the particles
	numParts=mesh.numVerts;
	parts=new ClothParticle[numParts];

	memset(parts, 0, sizeof(parts[0])*numParts);

	Point3 *verts=mesh.verts;
	for (int i=0; i<mesh.numVerts; i++) {
		ClothParticle &p=parts[i];
		p.p=p.np=p.p0=p.p1=p.p2=toVector(tm*verts[i]);
		p.v=p.v0=p.v1=Vector(0,0,0);
		p.mass=1.0f;
	}

	/*
	for (int i=0; i<numFaces; i++) {
		FaceData &f=faces[i];
		float area=0.5f*length((parts[f.v[1]].p-parts[f.v[0]].p)^(parts[f.v[2]].p-parts[f.v[0]].p));
		area/=3.0f;
		parts[f.v[0]].mass+=area;
		parts[f.v[1]].mass+=area;
		parts[f.v[2]].mass+=area;
	}
	*/

	float *vweights=mesh.getVSelectionWeights();
	for (int i=0; i<mesh.numVerts; i++) {
		ClothParticle &p=parts[i];

		float softSel=1.0f;
		if (vweights) softSel=vweights[i];
		else if (!tri->IsPointSelected(i)) softSel=0.0f;

		if (p.mass>0.0f) p.mass_inv=p.origMass_inv=(1.0f-softSel)/(p.mass*particleMass);
		else p.mass_inv=p.origMass_inv=0.0f;
	}

	// Create the stretch forces
	switch (stretchType) {
		case 0: {
			springs=new SpringForce<ClothParticle>[numEdges];
			numSprings=0;
			for (int i=0; i<numEdges; i++) {
				EdgeData &e=edges[i];
				springs[numSprings++].init(parts[e.v[0]], parts[e.v[1]], spring_stiffness, spring_damping);
			}
			break;
		}
		case 1: {
			stretchShears=new StretchShearForce<ClothParticle>[numFaces];
			numStretchShears=0;
			for (int i=0; i<numFaces; i++) {
				FaceData &f=faces[i];

				Vector e0=parts[f.v[1]].p-parts[f.v[0]].p;
				Vector e1=parts[f.v[2]].p-parts[f.v[0]].p;

				Vector n=e0^e1;
				Vector u=Vector(0.0f, 1.0f, 0.0f)^n;
				if (u.lengthSqr()<1e-12f) u=n^Vector(1.0f, 0.0f, 0.0f);
				u.makeNormalized();
				Vector v=normalize(n^u);

				float uc[3]={ 0.0f, e0*u, e1*u };
				float vc[3]={ 0.0f, e0*v, e1*v };

				stretchShears[numStretchShears++].init(parts[f.v[0]], parts[f.v[1]], parts[f.v[2]], stretch_stiffness, shear_stiffness, stretchShear_damping, uc, vc);
			}
			break;
		}
		case 2: {
			trisprings=new Trispring<ClothParticle>[numFaces];
			numTrisprings=0;
			for (int i=0; i<numFaces; i++) {
				FaceData &f=faces[i];
				trisprings[numTrisprings++].init(parts[f.v[0]], parts[f.v[1]], parts[f.v[2]], trispring_stiffness, trispring_damping);
			}
			break;
		}
	}

	// Create the bend forces
	switch (bendType) {
		case 0:
			break;
		case 1: {
			bendSprings=new SpringForce<ClothParticle>[numEdges];
			numBendSprings=0;
			for (int i=0; i<numEdges; i++) {
				EdgeData &e=edges[i];
				if (e.numFaces==2) {
					bendSprings[numBendSprings++].init(parts[e.v[0]], parts[e.v[1]], bendSpring_stiffness, bendSpring_damping);
				}
			}
			break;
		}
		case 2: {
			bendAngles=new BendAngle<ClothParticle, FaceData>[numEdges];
			numBendAngles=0;
			for (int i=0; i<numEdges; i++) {
				EdgeData &e=edges[i];
				if (e.numFaces==2) {
					bendAngles[numBendAngles++].init(
						parts[e.v[0]], parts[e.v[1]],
						parts[e.tv[0]], parts[e.tv[1]],
						faces[e.f[0]], faces[e.f[1]],
						bendAngle_stiffness, bendAngle_damping
					);
				}
			}
			break;
		}
		case 3: {
			bendProjections=new BendProjection<ClothParticle>[numEdges];
			numBendProjections=0;
			for (int i=0; i<numEdges; i++) {
				EdgeData &e=edges[i];
				if (e.numFaces==2) {
					bendProjections[numBendProjections++].init(
						parts[e.v[0]], parts[e.v[1]],
						parts[e.tv[0]], parts[e.tv[1]],
						bendProjection_stiffness, bendProjection_damping
					);
				}
			}
			break;
		}
	}
}

void ClothObject::updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm) {
	for (int i=0; i<numParts; i++) {
		ClothParticle &p=parts[i];

		p.p0=p.p1;
		p.p1=p.p2;
		p.v0=p.v1;

		Vector pos=toVector(tm*mesh.verts[i]);
		if (p.origMass_inv==0.0f) {
			p.p2=pos;
			p.v1=(p.p2-p.p1)*float(GetFrameRate());
			p.mass_inv=0.0f;
		} else if (mod->getAttachPos(i, pos)) {
			p.p2=pos;
			p.v1=(p.p2-p.p1)*float(GetFrameRate());
			p.mass_inv=0.0f;
		} else {
			p.mass_inv=parts[i].origMass_inv;
			p.p2=parts[i].np;
		}
	}
}

void ClothObject::saveCache(Point3 *points, Matrix3 &itm) {
	for (int i=0; i<numParts; i++) points[i]=itm*toPoint3(parts[i].p);
}

CollisionMesh *ClothObject::getCollisionMesh(void) {
	return (CollisionMesh*) this;
}

Vector ClothObject::getCollisionVertexPos(int i) {
	return parts[i].np;
}

int ClothObject::getCollisionVertexFlags(int i) {
	int res=VF_COLLIDE | VF_INTERSECT;
	if (parts[i].mass_inv==0.0f) res|=VF_FIXED;
	if (selfCollideOn) res|=VF_SELFCOLLIDE;
	return res;
}

Vector ClothObject::getCollisionVertPos(int vertIdx) { return parts[vertIdx].np; }
Vector ClothObject::getCollisionVertVel(int vertIdx) { return parts[vertIdx].nv; }
Vector ClothObject::getCollisionVertForce(int vertIdx) { return parts[vertIdx].force; }

Vector ClothObject::getCollisionEdgePos(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	return
		(parts[v[0]].np)*(1.0f-t)+
		(parts[v[1]].np)*t;
}

Vector ClothObject::getCollisionEdgeVel(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	return
		(parts[v[0]].nv)*(1.0f-t)+
		(parts[v[1]].nv)*t;
}

Vector ClothObject::getCollisionEdgeForce(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	return
		(parts[v[0]].force)*(1.0f-t)+
		(parts[v[1]].force)*t;
}

Vector ClothObject::getCollisionFacePos(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	return
		(parts[v[0]].np)*(1.0f-uc-vc)+
		(parts[v[1]].np)*uc+
		(parts[v[2]].np)*vc;
}

Vector ClothObject::getCollisionFaceVel(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	return
		(parts[v[0]].nv)*(1.0f-uc-vc)+
		(parts[v[1]].nv)*uc+
		(parts[v[2]].nv)*vc;
}

Vector ClothObject::getCollisionFaceForce(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	return
		(parts[v[0]].force)*(1.0f-uc-vc)+
		(parts[v[1]].force)*uc+
		(parts[v[2]].force)*vc;
}

inline Vector diff(const Vector &b, const Vector &x) {
	return x;
	/*
	real lenSq=b.lengthSqr();
	if (lenSq<1e-6f) return x;
	Vector h=b*((b*x)/lenSq);
	return x-h;
	*/
}

void ClothObject::applyVertImpulse(int vertIdx, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vector &nrm) {
	ClothParticle &p=parts[vertIdx];

	//p.force-=nrm*(nrm*p.force);

	p.pimp+=diff(p.pimp, pimp*p.mass_inv);
	p.vimp+=diff(p.vimp, vimp*p.mass_inv);
	p.force+=fimp*p.mass_inv;
}

void ClothObject::applyEdgeImpulse(int edgeIdx, real t, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vlado::Vector &nrm) {
	int *v=edges[edgeIdx].v;
	ClothParticle &p0=parts[v[0]], &p1=parts[v[1]];

	//p0.force-=nrm*(nrm*p0.force);
	real w0=(1.0f-t)*p0.mass_inv;
	p0.pimp+=diff(p0.pimp, pimp*w0);
	p0.vimp+=diff(p0.vimp, vimp*w0);
	p0.force+=fimp*w0;

	//p1.force-=nrm*(nrm*p1.force);
	real w1=t*p1.mass_inv;
	p1.pimp+=diff(p1.pimp, pimp*w1);
	p1.vimp+=diff(p1.vimp, vimp*w1);
	p1.force+=fimp*w1;
}

void ClothObject::applyFaceImpulse(int faceIdx, real uc, real vc, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vlado::Vector &nrm) {
	int *v=faces[faceIdx].v;
	ClothParticle &p0=parts[v[0]], &p1=parts[v[1]], &p2=parts[v[2]];

	//p0.force-=nrm*(nrm*p0.force);
	real w0=(1.0f-uc-vc)*p0.mass_inv;
	p0.pimp+=diff(p0.pimp, pimp*w0);
	p0.vimp+=diff(p0.vimp, vimp*w0);
	p0.force+=fimp*w0;

	//p1.force-=nrm*(nrm*p1.force);
	real w1=uc*p1.mass_inv;
	p1.pimp+=diff(p1.pimp, pimp*w1);
	p1.vimp+=diff(p1.vimp, vimp*w1);
	p1.force+=fimp*w1;

	//p2.force-=nrm*(nrm*p2.force);
	real w2=vc*p2.mass_inv;
	p2.pimp+=diff(p2.pimp, pimp*w2);
	p2.vimp+=diff(p2.vimp, vimp*w2);
	p2.force+=fimp*w2;
}

float ClothObject::getVertImpulseStrength(int vertIdx) {
	return 1.0f;
	return parts[vertIdx].mass_inv; 
}

float ClothObject::getEdgeImpulseStrength(int edgeIdx, real t) {
	return 1.0f;
	int *v=edges[edgeIdx].v;
	return
		sqr(parts[v[0]].mass_inv*(1.0f-t))+
		sqr(parts[v[1]].mass_inv*t);
}
float ClothObject::getFaceImpulseStrength(int faceIdx, real uc, real vc) {
	return 1.0f;
	int *v=faces[faceIdx].v;
	return
		sqr(parts[v[0]].mass_inv*(1.0f-uc-vc))+
		sqr(parts[v[1]].mass_inv*uc)+
		sqr(parts[v[2]].mass_inv*vc);
}

//*******************************************************
// DeflectorObject

void DeflectorObject::init(INode *node, SimCloth3 *mod, SimClothLocalData *localData) {
	SimClothObject::init(node, mod, localData);
}

void DeflectorObject::freeMem(void) {
	SimClothObject::freeMem();
}

void DeflectorObject::reset(real subTime) {
	for (int i=0; i<numParts; i++) {
		DeflectorParticle &p=parts[i];

		p.p=p.p0+(p.p1-p.p0)*subTime;
		p.v=p.v0+(p.v1-p.v0)*subTime;
	}
}

void DeflectorObject::prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm) {
	// Create the particles
	numParts=mesh.numVerts;
	parts=new DeflectorParticle[numParts];
	for (int i=0; i<numParts; i++) {
		DeflectorParticle &p=parts[i];

		p.p=p.p0=p.p1=p.p2=toVector(tm*mesh.verts[i]);
		p.v0=p.v1=Vector(0,0,0);
	}
}

void DeflectorObject::updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm) {
	for (int i=0; i<numParts; i++) {
		DeflectorParticle &p=parts[i];

		p.p0=p.p1;
		p.p1=p.p2;
		p.v0=p.v1;

		p.p2=toVector(tm*mesh.verts[i]);
		p.v1=(p.p1-p.p0)*float(GetFrameRate());
	}
}

void DeflectorObject::saveCache(Point3 *points, Matrix3 &itm) {
	for (int i=0; i<numParts; i++) points[i]=itm*toPoint3(parts[i].p);
}

CollisionMesh* DeflectorObject::getCollisionMesh(void) {
	return (CollisionMesh*) this;
}

Vector DeflectorObject::getCollisionVertexPos(int i) {
	return parts[i].p;
}

int DeflectorObject::getCollisionVertexFlags(int i) {
	return VF_COLLIDE | VF_INTERSECT | VF_FIXED;
}

Vector DeflectorObject::getCollisionVertPos(int vertIdx) {
	return parts[vertIdx].p;
}

Vector DeflectorObject::getCollisionVertVel(int vertIdx) {
	return parts[vertIdx].v;
}

Vector DeflectorObject::getCollisionEdgePos(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	return parts[v[0]].p*(1.0f-t)+parts[v[1]].p*t;
}

Vector DeflectorObject::getCollisionEdgeVel(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	return parts[v[0]].v*(1.0f-t)+parts[v[1]].v*t;
}

Vector DeflectorObject::getCollisionFacePos(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	return parts[v[0]].p*(1.0f-uc-vc)+parts[v[1]].p*uc+parts[v[2]].p*vc;
}

Vector DeflectorObject::getCollisionFaceVel(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	return parts[v[0]].v*(1.0f-uc-vc)+parts[v[1]].v*uc+parts[v[2]].v*vc;
}

//*******************************************************
// RigidObject

void RigidObject::init(INode *node, SimCloth3 *mod, SimClothLocalData *localData) {
	SimClothObject::init(node, mod, localData);
	verts=NULL;
	vmass=NULL;
	particleMass=mod->pblockGeneral->GetFloat(sc3_general_particleMass, 0);
}

void RigidObject::freeMem(void) {
	SimClothObject::freeMem();
	if (verts) delete[] verts;
	verts=NULL;
	if (vmass) delete[] vmass;
	vmass=NULL;
}

int RigidObject::getNumVars(void) { return 3+3+3+3; }

void RigidObject::reset(real subTime) {
	n_pos=pos;
	n_rot.makeZero(); // =rot;
	n_linearVel=linearVel;
	n_angularVel=angularVel;
	n_tm=tm;
}

void RigidObject::computeTargets(float dt) {
}

void RigidObject::computeChanges(real *x, float dt) {
	Vector t_linearVel=linearVel+force*(mass_inv*dt);
	Vector t_pos=pos+n_linearVel*dt;

	Vector t_angularVel=angularVel+torque*(mass_inv*dt);
	Vector t_rot=n_angularVel*dt;

	Vector d_pos=combineImpulse(t_pos-n_pos, pos_imp, 1.0f);
	*(x++)=d_pos[0];
	*(x++)=d_pos[1];
	*(x++)=d_pos[2];

	Vector d_linearVel=combineImpulse(t_linearVel-n_linearVel, linear_imp, 1.0f)/vscale;
	*(x++)=d_linearVel[0];
	*(x++)=d_linearVel[1];
	*(x++)=d_linearVel[2];

	Vector d_rot=combineImpulse(t_rot-n_rot, rot_imp, 1.0f);
	*(x++)=d_rot[0];
	*(x++)=d_rot[1];
	*(x++)=d_rot[2];

	Vector d_angularVel=combineImpulse(t_angularVel-n_angularVel, angular_imp, 1.0f)/vscale;
	*(x++)=d_angularVel[0];
	*(x++)=d_angularVel[1];
	*(x++)=d_angularVel[2];
}

void RigidObject::addChanges(real *x, real scale) {
	n_pos[0]+=*(x++)*scale;
	n_pos[1]+=*(x++)*scale;
	n_pos[2]+=*(x++)*scale;

	n_linearVel[0]+=*(x++)*scale*vscale;
	n_linearVel[1]+=*(x++)*scale*vscale;
	n_linearVel[2]+=*(x++)*scale*vscale;

	n_rot[0]+=*(x++)*scale;
	n_rot[1]+=*(x++)*scale;
	n_rot[2]+=*(x++)*scale;

	n_angularVel[0]+=*(x++)*scale*vscale;
	n_angularVel[1]+=*(x++)*scale*vscale;
	n_angularVel[2]+=*(x++)*scale*vscale;

	n_tm=tm+(n_rot^tm);
	n_tm.makeOrthonormal();
}

void RigidObject::acceptChanges(void) {
	pos=n_pos;
	n_tm.makeOrthonormal();
	tm=n_tm;
	linearVel=n_linearVel;
	angularVel=n_angularVel;
}

void RigidObject::clearForces(void) {
	force.makeZero();
	torque.makeZero();

	pos_imp.makeZero();
	linear_imp.makeZero();
	rot_imp.makeZero();
	angular_imp.makeZero();
}

void RigidObject::applyForces(void) {
	/*
	Vector p=n_tm*verts[0]+n_pos;

	Vector d=pt-p;
	d*=100.0f;
	force+=d;
	torque+=(p-n_pos)^d;
	*/
}

void RigidObject::applyGravity(const Vector &accel) {
	force+=mass*accel;
}

CollisionMesh *RigidObject::getCollisionMesh(void) {
	return (CollisionMesh*) this;
}

void RigidObject::prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm) {
	// Compute center of mass (in object space)
	cmass.makeZero();
	for (int i=0; i<numVerts; i++) cmass+=toVector(mesh.verts[i]);
	cmass/=float(numVerts);

	iobj.makeZero();

	verts=new Vector[numVerts];
	for (int i=0; i<numVerts; i++) {
		Vector p=toVector(mesh.verts[i])-cmass;

		verts[i]=p;

		iobj[0][0]+=p.y*p.y+p.z*p.z;
		iobj[1][1]+=p.z*p.z+p.x*p.x;
		iobj[2][2]+=p.x*p.x+p.y*p.y;

		iobj[1][0]-=p.x*p.y;
		iobj[0][1]-=p.x*p.y;

		iobj[2][0]-=p.x*p.z;
		iobj[0][2]-=p.x*p.z;

		iobj[1][2]-=p.y*p.z;
		iobj[2][1]-=p.y*p.z;
	}

	iobj_inv=inverse(iobj);

	vmass=new float[numVerts];
	memset(vmass, 0, sizeof(vmass[0])*numVerts);
	for (int i=0; i<numFaces; i++) {
		FaceData &f=faces[i];
		float area=0.5f*length((verts[f.v[1]]-verts[f.v[0]])^(verts[f.v[2]]-verts[f.v[0]]));
		area/=3.0f;
		vmass[f.v[0]]+=area;
		vmass[f.v[1]]+=area;
		vmass[f.v[2]]+=area;
	}
	for (int i=0; i<numVerts; i++) vmass[i]=1.0f; // vmass[i]/=(vscale*particleMass);

	Transform ttm=toTransform(tm);

	pos=ttm*cmass;
	this->tm=ttm.m;
	linearVel.makeZero();
	angularVel.makeZero();

	mass=mass_inv=1.0f;

	// angularVel.set(0.0f, 1.0f, 1.0f);

	pt=this->tm*verts[0]+pos;
}

void RigidObject::updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm) {
}

void RigidObject::saveCache(Point3 *points, Matrix3 &itm) {
	for (int i=0; i<numVerts; i++) points[i]=itm*toPoint3(n_pos+n_tm*verts[i]);
}

Vector RigidObject::getCollisionVertexPos(int i) {
	return n_tm*verts[i]+n_pos;
}

int RigidObject::getCollisionVertexFlags(int i) {
	return VF_COLLIDE | VF_INTERSECT;
}

Vector RigidObject::getCollisionVertPos(int vertIdx) {
	return n_tm*verts[vertIdx]+n_pos;
}

Vector RigidObject::getCollisionVertVel(int vertIdx) {
	Vector p=verts[vertIdx];
	return n_linearVel+((n_angularVel)^(n_tm*p));
}

Vector RigidObject::getCollisionVertForce(int vertIdx) {
	Vector p=verts[vertIdx];
	return force+((torque)^(n_tm*p));
}

Vector RigidObject::getCollisionEdgePos(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	Vector p=verts[v[0]]*(1.0f-t)+verts[v[1]]*t;
	return n_tm*p+n_pos;
}

Vector RigidObject::getCollisionEdgeVel(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	Vector p=verts[v[0]]*(1.0f-t)+verts[v[1]]*t;
	return n_linearVel+((n_angularVel)^(n_tm*p));
}

Vector RigidObject::getCollisionEdgeForce(int edgeIdx, real t) {
	int *v=edges[edgeIdx].v;
	Vector p=verts[v[0]]*(1.0f-t)+verts[v[1]]*t;
	return force+((torque)^(n_tm*p));
}

Vector RigidObject::getCollisionFacePos(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	Vector p=verts[v[0]]*(1.0f-uc-vc)+verts[v[1]]*uc+verts[v[2]]*vc;
	return n_tm*p+n_pos;
}

Vector RigidObject::getCollisionFaceVel(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	Vector p=verts[v[0]]*(1.0f-uc-vc)+verts[v[1]]*uc+verts[v[2]]*vc;
	return n_linearVel+((n_angularVel)^(n_tm*p));
}

Vector RigidObject::getCollisionFaceForce(int faceIdx, real uc, real vc) {
	int *v=faces[faceIdx].v;
	Vector p=verts[v[0]]*(1.0f-uc-vc)+verts[v[1]]*uc+verts[v[2]]*vc;
	return force+((torque)^(n_tm*p));
}

void RigidObject::applyVertImpulse(int vertIdx, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vlado::Vector &nrm) {
	pos_imp+=pimp*vmass[vertIdx];
	linear_imp+=vimp*vmass[vertIdx];

	Vector r=n_tm*verts[vertIdx];
	r/=lengthSqr(r);

	force+=fimp*vmass[vertIdx];
	torque+=(r^fimp)*vmass[vertIdx];

	rot_imp+=(r^pimp)*vmass[vertIdx];
	angular_imp+=(r^vimp)*vmass[vertIdx];
}

void RigidObject::applyEdgeImpulse(int edgeIdx, real t, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vlado::Vector &nrm) {
	int *v=edges[edgeIdx].v;

	real w0=(1.0f-t);
	applyVertImpulse(v[0], pimp*w0, vimp*w0, fimp*w0, nrm);

	real w1=t;
	applyVertImpulse(v[1], pimp*w1, vimp*w1, fimp*w1, nrm);
}

void RigidObject::applyFaceImpulse(int faceIdx, real uc, real vc, const Vector &pimp, const Vector &vimp, const Vector &fimp, const Vlado::Vector &nrm) {
	int *v=faces[faceIdx].v;

	real w0=(1.0f-uc-vc);
	applyVertImpulse(v[0], pimp*w0, vimp*w0, fimp*w0, nrm);

	real w1=uc;
	applyVertImpulse(v[1], pimp*w1, vimp*w1, fimp*w1, nrm);

	real w2=vc;
	applyVertImpulse(v[2], pimp*w2, vimp*w2, fimp*w2, nrm);
}

//*************************************************************************
// Old stuff

/*
// Find the eigenvalues of a symmetric matrix
// Algorithm from Graphics Gems IV
void eigenValues(float ixx, float iyy, float izz, float ixy, float iyz, float izx, double &l0, double &l1, double &l2) {
	double c=ixx*iyy, d=iyz*iyz, e=ixy*ixy, f=izx*izx;
	double p=-ixx-iyy-izz;
	double q=c+(ixx+iyy)*izz-d-e-f;
	double r=(e-c)*izz+d*ixx-2.0*(ixy*iyz*izx)+f*iyy;

	double a=q-p*p/3.0;
	double b=2.0*p*p*p/27.0-p*q/3.0+r;

	double m=2.0*sqrt(-a/3.0);
	double theta=acos(3.0*b/(a*m))/3.0;

	double cs=cos(theta), sn=sin(theta);

	double sqrt3=sqrt(3.0);

	l0=m*cs-p/3.0;
	l1=m*(-cs+sqrt3*sn)*0.5f-p/3.0;
	l2=m*(-cs-sqrt3*sn)*0.5f-p/3.0;

	if (l0<0.0) l0=0.0;
	if (l1<0.0) l1=0.0;
	if (l2<0.0) l2=0.0;
}

Vector findSolution(float ixx, float iyy, float izz, float ixy, float iyz, float izx) {
	Matrix m;
	m.f[0].set(ixx, ixy, izx);
	m.f[1].set(ixy, iyy, iyz);
	m.f[2].set(izx, iyz, izz);
	Matrix im=inverse(m);

	// Find the largest element in the inverse matrix
	float maxf=0.0f;
	int row=0, col=0;
	for (int i=0; i<3; i++) {
		for (int j=0; j<3; j++) {
			if (fabs(im.f[i][j])>maxf) { maxf=fabs(im.f[i][j]); row=i; col=j; }
		}
	}

	Vector res;
	res[col]=1.0f;

	float a=m.f[(row+1)%3][(col+1)%3];
	float b=m.f[(row+1)%3][(col+2)%3];

	float c=m.f[(row+2)%3][(col+1)%3];
	float d=m.f[(row+2)%3][(col+2)%3];

	float e=-m.f[(row+1)%3][(col)%3];
	float f=-m.f[(row+2)%3][(col)%3];

	float D=a*d-b*c;
	res[(col+1)%3]=(e*d-b*f)/D;
	res[(col+2)%3]=(a*f-e*c)/D;

	return normalize(res);
}

Vector eigenVector(float ixx, float iyy, float izz, float ixy, float iyz, float izx, double l) {
	return findSolution(ixx-float(l), iyy-float(l), izz-float(l), ixy, iyz, izx);
}

Transform computeInertiaTransform(Mesh &mesh) {
	Vector gcenter(0.0f, 0.0f, 0.0f);
	float wsum=0.0f;
	for (int i=0; i<mesh.numVerts; i++) {
		Vector p=toVector(mesh.verts[i]);
		float mass=1.0f;

		gcenter+=p*mass;
		wsum+=mass;
	}
	gcenter/=wsum;

	float ixx=0.0f, ixy=0.0f, izx=0.0f, iyy=0.0f, iyz=0.0f, izz=0.0f;

	Box b;
	b.init();

	for (i=0; i<mesh.numVerts; i++) {
		Vector p=toVector(mesh.verts[i]);
		float mass=1.0f;

		b+=p;

		Vector rp=(p-gcenter)*mass;
		ixx+=p.y*p.y+p.z*p.z;
		iyy+=p.x*p.x+p.z*p.z;
		izz+=p.x*p.x+p.y*p.y;

		ixy-=p.x*p.y;
		iyz-=p.y*p.z;
		izx-=p.z*p.x;
	}

	// Find the inertia matrix
	Matrix I;
	I.f[0].set(ixx, ixy, izx);
	I.f[1].set(ixy, iyy, iyz);
	I.f[2].set(izx, iyz, izz);

	// Find a rotation that will convert the inertia matrix
	// to a diagonal matrix

	// Find the eigenvalues of the inertia matrix
	double l0, l1, l2;
	eigenValues(ixx, iyy, izz, ixy, iyz, izx, l0, l1, l2);

	Vector u=eigenVector(ixx, iyy, izz, ixy, iyz, izx, l0);
	Vector v=eigenVector(ixx, iyy, izz, ixy, iyz, izx, l1);
	Vector w=eigenVector(ixx, iyy, izz, ixy, iyz, izx, l2);

	Transform res;

	if (l0<1e-6) l0=1.0f;
	if (l1<1e-6) l1=1.0f;
	if (l2<1e-6) l2=1.0f;

	res.m.set(u/l0, v/l1, w/l2);
	// res.m=res.m*inverse(I*res.m);
	res.offs=gcenter;
	// res.makeInverse();

	Vector wd=b.width();
	int maxd=maxComponent(wd);
	float m=wd[maxd];

	res.m.f[0].set(1.0f, 0.0f, 0.0f);
	res.m.f[1].set(0.5f, sqrtf(3.0f)*0.5f, 0.0f);
	res.m.f[2].set(0.5f, sqrtf(3.0f)/6.0f, sqrtf(6.0f)/3.0f);

	res.m.f[0]*=wd.x;
	res.m.f[1]*=wd.y;
	res.m.f[2]*=wd.z;

	res.offs=b.center()-(res.m.f[0]+res.m.f[1]+res.m.f[2])*0.25f; // b.center()-Vector(0.25f, 0.25f, 0.25f);

	Vector test=res*Vector(0,0,0)+res*Vector(1,0,0)+res*Vector(0,1,0)+res*Vector(0,0,1);

	return res;
}

inline int SimClothObject::getParticleCollide(int index) {
	return mod->getVertexCollide(index);
}

inline int SimClothObject::getParticleSelfCollide(int index) {
	return mod->getVertexSelfCollide(index);
}

inline int SimClothObject::getParticleCheckIntersections(int index) {
	return mod->getVertexCheckIntersections(index);
}

void genUVcoords(Vector &p0, Vector &p1, Vector &p2, float *uc, float *vc) {
	Vector e0=p1-p0;
	Vector e1=p2-p0;

	Vector n=e0^e1;
	Vector u=Vector(0.0f, 1.0f, 0.0f)^n;
	if (u.lengthSqr()<1e-12f) u=n^Vector(1.0f, 0.0f, 0.0f);
	u.makeNormalized();
	Vector v=normalize(n^u);

	uc[0]=0.0f; vc[0]=0.0f;
	uc[1]=e0*u; vc[1]=e0*v;
	uc[2]=e1*u; vc[2]=e1*v;
}

void SimClothObject::updateForceScale(void) {
	for (int j=0; j<springsArray.count(); j++) {
		springsArray.scale[j]=1.0f;

		int numGroups=0;
		float sum=0.0f;

		int v0=springsArray.p0[j]-parts;
		int v1=springsArray.p1[j]-parts;

		for (int i=0; i<mod->groups.Count(); i++) {
			VertexGroup &group=*(mod->groups[i]);
			if (group.numPts==numParts && group.selected[v0] && group.selected[v1]) { sum+=group.scale; numGroups++; }
		}

		if (numGroups>0) springsArray.scale[j]=sum/float(numGroups);
	}

	for (j=0; j<numStretchShears; j++) {
		stretchShears[j].scale=1.0f;

		int numGroups=0;
		float sum=0.0f;

		int v0=stretchShears[j].parts[0]-parts;
		int v1=stretchShears[j].parts[1]-parts;
		int v2=stretchShears[j].parts[2]-parts;

		for (int i=0; i<mod->groups.Count(); i++) {
			VertexGroup &group=*(mod->groups[i]);
			if (group.numPts==numParts && group.selected[v0] && group.selected[v1] && group.selected[v2]) { sum+=group.scale; numGroups++; }
		}

		if (numGroups>0) stretchShears[j].scale=sum/float(numGroups);
	}
}

void SimClothObject::buildNormals(void) {
	// DebugPrint("Building normals...");
	for (int i=0; i<numVerts; i++) vertexNormals[i].makeZero();

	for (i=0; i<numFaces; i++) {
		Vector n;
		if (!isRigidBody()) {
			Vector &p0=parts[faces[i].v[0]].np;
			Vector &p1=parts[faces[i].v[1]].np;
			Vector &p2=parts[faces[i].v[2]].np;

			n=(p1-p0)^(p2-p0);

			triplets[i].normalLength=n.length();
			triplets[i].normal=n/triplets[i].normalLength;
		} else {
			Vector &p0=vertexPos(faces[i].v[0]);
			Vector &p1=vertexPos(faces[i].v[1]);
			Vector &p2=vertexPos(faces[i].v[2]);

			n=(p1-p0)^(p2-p0);

			triplets[i].normalLength=n.length();
			triplets[i].normal=n/triplets[i].normalLength;
		}
		for (int j=0; j<3; j++) vertexNormals[faces[i].v[j]]+=n;
	}

	for (i=0; i<numVerts; i++) vertexNormals[i].makeNormalized();

	for (i=0; i<numEdges; i++) {
		EdgeData &edge=edges[i];
		if (edge.numFaces<2) edgeNormals[i]=triplets[edge.f[0]].normal;
		else edgeNormals[i]=normalize(triplets[edge.f[0]].normal+triplets[edge.f[1]].normal);
	}
	// DebugPrint("done.\n");
}

void SimClothObject::updateMesh(TriObject *obj, Mesh &mesh, Matrix3 &tm) {
	// Update the transformation matrix
	this->tm=tm;

	// Update the position of attached particles
	for (int i=0; i<numParts; i++) {
		parts[i].q0=parts[i].q1;
		parts[i].q1=parts[i].q2;
		parts[i].qv0=parts[i].qv1;

		if (objType==obj_cloth) {
			Vector p;
			if (mod->getAttachPos(i, p)) {
				parts[i].q2=p;
				parts[i].qv1=(parts[i].q2-parts[i].q1)*float(GetFrameRate());
				parts[i].invMass=0.0f;
			} else {
				parts[i].invMass=parts[i].origInvMass;
				parts[i].q2=parts[i].np;
			}
		}
		if (parts[i].origInvMass<1e-6f) {
			parts[i].q2=toVector(tm*mesh.verts[i]);
			parts[i].qv1=(parts[i].q2-parts[i].q1)*float(GetFrameRate());
		} 
	}

	// Update the collision and self-collision flags
	if (objType==obj_cloth) {
		for (int i=0; i<numParts; i++) {
			parts[i].collide=getParticleCollide(i);
			parts[i].selfCollide=getParticleSelfCollide(i);
			parts[i].checkIntersections=getParticleCheckIntersections(i);
		}
	}

	buildNormals();
}
*/