/********************************************************
	FILE: SimClothEngine.h
	DESCRIPTION: Header file for the simcloth engine interface
	CREATED: 1 December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) by Vladimir Koylazov (Vlado)

	HISTORY:

*********************************************************/

#ifndef __SIMCLOTHENGINE_H__
#define __SIMCLOTHENGINE_H__

#include "utils.h"
#include "table.h"
// #include "collisionmesh.h"
// #include "collision.h"
#include "collisiondetector.h"

// SimCloth error return codes (0 means no error)
#define SC3_ERR_INTEGRITY_LOSS (-1)
#define SC3_ERR_INTERRUPTED (-2)
#define SC3_ERR_INTERSECTION (-3)
#define SC3_ERR_EXCEPTION (-4)

// A structure with the parameters for the SimCloth engine
struct SimClothEngineParams {
	int substeps;
	int adaptiveSubdivs;
	Vlado::Vector gravity;
	float collisionTolerance;
	float solverPrecision;
	int checkIntersections;
	int useSSE;
	float worldScale;
};

// A class that prepares the data for the SimCloth engine
class DynamicObject;
class SimClothConnection {
public:
	virtual int numDynamicObjects(void)=0;
	virtual DynamicObject* getDynamicObject(int i)=0;
	virtual void getSimClothEngineParams(SimClothEngineParams &params)=0;

	virtual void displayMessage(TCHAR *str)=0;
	virtual void setFrameProgressRange(int minRange, int maxRange)=0;
	virtual void setFrameProgressPos(int progressPos)=0;
	virtual void setNumCollisions(int collisions)=0;
	virtual void setNumCGIterations(int iterations)=0;
	virtual int getAbort(void)=0;
};

enum DynamicObjectType { obj_deflector, obj_cloth, obj_rigid };

// A dynamic object (a collection of particles)
class DynamicObject {
public:
	virtual ~DynamicObject(void) {}

	//**************************************************
	// Force methods
	//**************************************************

	// Clears the forces applied on the object
	virtual void clearForces(void)=0;

	// Computes the internal forces of the object
	virtual void applyForces(void)=0;

	// Applies a gravity force to the object
	virtual void applyGravity(const Vlado::Vector &gforce)=0;

	//**************************************************
	// Integration methods
	//**************************************************

	// The number of variables (in floats)
	virtual int getNumVars(void)=0;

	// Initialize the current state variables at the given sub-frame time (0.0 to 1.0)
	virtual void reset(Vlado::real subTime)=0;

	virtual void computeTargets(float dt) {}

	// Stores in x the difference between the current state and the desired target state
	virtual void computeChanges(Vlado::real *x, float dt)=0;

	// Adds the given values to the current state
	virtual void addChanges(Vlado::real *x, Vlado::real scale)=0;

	// Accept the current state variables
	virtual void acceptChanges(void)=0;

	//**************************************************
	// Collision detection/response methods
	//**************************************************

	virtual Vlado::CollisionMesh *getCollisionMesh(void)=0;
};

// The SimCloth engine itself
class SimClothEngine {
private:
	SimClothConnection *connection; // The SimCloth connection
	Vlado::Table<DynamicObject*> objects; // The objects in the simulation
	Vlado::CollisionDetector *cdetector; // The collision detector
	SimClothEngineParams params; // The SimCloth engine parameters
	Vlado::real *r, *p, *r1; // These are passed to cgSolveVectorTarget() as working variables

	int numVars; // The total number of variables in the simulation

	float dt, dt_inv; // Time step and the inverse of the time step
	int integrationSteps, integrationLevel;
	int fastSelfCollide;
	int collisionsOnly;
	int intersection, lockStep;

	int substeps;
	int incStep;
	int sum, numSum;
	int substepWait;
	int subdivLevel;
	int savePos;
	int okSteps;
	int integrationLevels;
	int solveWhat;

	// Make a single substep of the simulation
	int makeSubstep(float dt, int first, float substepOffset, float substepSize);

	// Compute the forces and collision impulses acting on the particles
	void applyForces(float dt);

	// Integration - computes the position and velocity of the particles at the next moment of time
	int implicitEuler(float dt, float substepOffset, float substepSize);

	void reset(float subTime);
public:
	int forceEvals;

	// Set the simulation environment
	void setConnection(SimClothConnection *connection);

	// Prepares the simulation
	void beginSimulation(void);

	// Make one simulation step
	int step(float timeStep);

	// Cleanes up after simulation
	void endSimulation(void);

	// Support for cgSolveVectorTarget
	Vlado::real *get_r(void) { return r; } // Returns some temporary storage
	Vlado::real *get_r1(void) { return r1; } // Returns some temporary storage
	Vlado::real *get_p(void) { return p; } // Returns some temporary storage
	int getNumVars(void); // Returns the number of variables
	void addChanges(Vlado::real *x, Vlado::real scale); // Adds the given vectors to the simulated variables
	void computeChanges(Vlado::real *x); // Computes the difference between the current variables and the target state
};

#endif