#include <Windows.h>
#include "fvec.h"

#include "utils.h"
#include "cgsolver.h"

#include "simclothengine.h"
#include "collisiondetector.h"

const float vscale=1024.0f;

// Set the simulation environment
void SimClothEngine::setConnection(SimClothConnection *connection) {
	this->connection=connection;
}

// Prepares the simulation
void SimClothEngine::beginSimulation(void) {
	cdetector=Vlado::newDefaultCollisionDetector();

	numVars=0;
	for (int i=connection->numDynamicObjects()-1; i>=0; i--) {
		DynamicObject *obj=connection->getDynamicObject(i);

		numVars+=obj->getNumVars();
		cdetector->addCollisionMesh(obj->getCollisionMesh());
		*(objects.newElement())=obj;
	}
	connection->getSimClothEngineParams(params);

	// Init the implicit target
	r=new Vlado::real[numVars];
	p=new Vlado::real[numVars];
	r1=new Vlado::real[numVars];

	integrationSteps=params.substeps;
	integrationLevel=0;
	fastSelfCollide=false;
	intersection=0;
	lockStep=false;

	substeps=params.substeps;
	incStep=3;
	sum=0;
	numSum=0;
	substepWait=0;
	subdivLevel=0;
	okSteps=4;
	integrationLevels=0;

	collisionsOnly=false;
	savePos=true;
}

// Make one simulation step
int SimClothEngine::step(float timeStep) {
	int err=implicitEuler(timeStep, 0.0f, 1.0f);
	return err;
}

// Cleanes up after simulation
void SimClothEngine::endSimulation(void) {
	cdetector->freeMem();
	delete cdetector;
	cdetector=NULL;

	objects.freeData();

	if (r) delete[] r;
	if (r1) delete[] r1;
	if (p) delete[] p;
	r=r1=p=NULL;
}

int SimClothEngine::makeSubstep(float dt, int first, float substepOffset, float substepSize) {
	int err=0;
	err=implicitEuler(dt, substepOffset, substepSize);
	return err;
/*
	// Save the particle positions
	for (int i=0; i<objects.count(); i++) {
		DynamicObject &obj=*(objects[i]);
		for (int k=0; k<obj.numParts; k++) {
			Vlado::Particle<DynamicObject> &p=obj.parts[k];
			p.sv=p.v;
			p.sp=p.p;
			p.spp=p.pp;
		}
	}

	float s=0.0f;
	float ds=1.0f;
	fastSelfCollide=true;

	int frst=true;
	while (1) {
		if (GetAsyncKeyState(VK_ESCAPE)) break;
		// Make one step
		// cdetector.update(first==0, dt);
		// cdetector.collide(dt, params.collisionTolerance, CT_VERTEX_FACE | CT_EDGE_EDGE, CT_VERTEX_FACE | CT_EDGE_EDGE, fastSelfCollide);
		int err=implicitEuler(dt*ds, substepOffset+substepSize*s, substepSize*ds);
		if (err) return err;

		s+=ds;

		// Check for intersections
		cdetector.update(frst, dt);
		frst=false;
		cdetector.collide(dt, params.collisionTolerance, CT_VERTEX_FACE | CT_EDGE_EDGE, CT_VERTEX_FACE | CT_EDGE_EDGE, fastSelfCollide);

		// Check if there are intersections
		int correctPos=true;
		cdetector.clist->resetCounter();
		Vlado::Collision<DynamicObject> *c;
		while (c=cdetector.clist->nextElement()) {
			if (c->relPos()*c->normal<0.0f) { correctPos=false; break; }
		}

		// Save the particle positions if we are at a correct position
		if (correctPos) {
			for (i=0; i<objects.count(); i++) {
				DynamicObject &obj=*(objects[i]);
				for (int k=0; k<obj.numParts; k++) {
					Vlado::Particle<DynamicObject> &p=obj.parts[k];
					p.pp=p.p;
				}
			}
		}

		if (!params.checkIntersections || !cdetector.checkIntersection()) {
			if (fabsf(1.0f-s)<1e-6f) {
				// All is ok
				fastSelfCollide=true;
				return 0;
			} else {
				// Try to make the remaining step
				ds=1.0f-s;

				for (int i=0; i<objects.count(); i++) {
					DynamicObject &obj=*(objects[i]);
					for (int k=0; k<obj.numParts; k++) {
						Vlado::Particle<DynamicObject> &p=obj.parts[k];
						p.sv=p.v;
						p.sp=p.p;
						p.spp=p.pp;
					}
				}

				continue;
			}
		}

		// Cannot subdivide further, just go on simulating
		if (ds<1.0f/float(1<<params.adaptiveSubdivs)) continue;

		// An intersection has occurred, must increase the substeps
		// First roll back
		s-=ds;
		for (i=0; i<objects.count(); i++) {
			DynamicObject &obj=*(objects[i]);
			for (int k=0; k<obj.numParts; k++) {
				Vlado::Particle<DynamicObject> &p=obj.parts[k];
				p.v=p.sv;
				p.p=p.sp;
				p.pp=p.spp;
			}
		}
		fastSelfCollide=false;
		cdetector.collide(dt, params.collisionTolerance, CT_VERTEX_FACE | CT_EDGE_EDGE, CT_VERTEX_FACE | CT_EDGE_EDGE, fastSelfCollide);

		ds*=0.5f;
	}

	return 0;*/
}

/**************************************
	Integration

	SimCloth uses implicit Euler as the integration method:

	v(t+dt)=v(t)+M^(-1)*f(x+dt)*dt
	x(t+dt)=x(t)+v(x+dt)*dt

	Both the position and the velocity are considered unknowns, in difference from
	other methods like Baraff and Volino, where only the velocity is considered an
	unknown and the position is deduced from that; or Verlet integration where the
	position is considered unknown and the velocity is derived from that. Reliable
	collision response requires us to have control over both the position and velocity
	of the particles - in one time step. Otherwise we stumble across various
	difficulties and artifacts.

***************************************/

// The number of variables is equal to totalParticles*2 - these the particles'
// position and velocity
int SimClothEngine::getNumVars(void) { return numVars; }

void SimClothEngine::reset(Vlado::real subTime) {
	for (int i=0; i<objects.count(); i++) objects[i]->reset(subTime);
	/*
	for (int i=0; i<objects.count(); i++) {
		DynamicObject &obj=*(objects[i]);
		for (int k=0; k<obj.numParts; k++) {
			Vlado::Particle<DynamicObject> &p=obj.parts[k];
			if (p.invMass>1e-6f) {
				p.np=p.p;
				p.nv=p.v;
			} else {
				p.p=p.np=p.q0+(p.q1-p.q0)*frameTime;
				p.v=p.nv=p.qv0+(p.qv1-p.qv0)*frameTime;
			}
		}
	}
	*/
}

// Adds the given vectors to the simulated variables
void SimClothEngine::addChanges(Vlado::real *x, Vlado::real scale) {
	for (int i=0; i<objects.count(); i++) {
		DynamicObject *obj=objects[i];

		obj->addChanges(x, scale);
		x+=obj->getNumVars();
	}
/*
			for (int k=0; k<obj.numParts; k++) {
				Vlado::Particle<DynamicObject> &p=obj.parts[k];

				switch (solveFor) {
					case solve_positionAndVelocity:
						p.np+=x[cnt++]*scale;
						p.nv+=x[cnt++]*(scale*vscale);
						break;
					case solve_position:
						p.np+=x[cnt++]*scale;
						p.nv=(p.np-p.p)*dt_inv;
						break;
					case solve_velocity:
						p.nv+=x[cnt++]*scale*vscale;
						p.np=p.p+p.nv*dt;
						break;
				}
			}
		}
	}
	*/
}


/**************************************
	Computes the difference between the current variables and the target state
	This is computed from the above formulas as follows:

	v(t+dt)=v(t)+M^(-1)*f(x+dt)*dt translates into
	0=v(t)+M^(-1)*f(x+dt)*dt-v(t+dt)
	0=p.v+(p.invMass*dt)*p.f-p.nv for each particle p

	x(t+dt)=x(t)+v(x+dt)*dt translates into
	0=x(t)+v(x+dt)*dt-x(t+dt)
	0=p.p+nv*dt-p.np for each particle p

	The condition for the velocity is additionally scaled by dt to bring the positions and
	velocities to roughly the same scale, since the velocities are usually much larger numbers
	that the positions, which slows down the implicit solver, since higher precision is required
***************************************/

void SimClothEngine::computeChanges(Vlado::real *x) {
	applyForces(dt);
	for (int i=0; i<objects.count(); i++) {
		DynamicObject *obj=objects[i];
		obj->computeChanges(x, dt);
		x+=obj->getNumVars();
	}
}

// Integration - computes the position and velocity of the particles at the next moment of time
int SimClothEngine::implicitEuler(float timeStep, float substepOffset, float substepSize) {
	dt=timeStep/float(integrationSteps);
	dt_inv=1.0f/this->dt;

	const int adaptive=true;

	// The number of iterations at which the current result will be discarded and timestep decreased
	const int decreaseThreshOne=32;

	// The number of iterations at which the current result is accepted, but timestep is decreased
	const int decreaseThreshTwo=16;

	// The number of iterations below which the timestep is increased
	const int increaseThresh=8;

	// The maximum number of iterations allowed
	const int maxSteps=201;

	int err=0;
	int numIter=0;
	int decreaseCount=0;

	int lastUpdated=false;
	for (int k=0; k<integrationSteps;) {
		if (connection->getAbort()) { err=SC3_ERR_INTERRUPTED; break; }
		// Initialize the "next" state variables and compute the positions
		// and velocties of the fixed particles
		reset(substepOffset+substepSize*float(k+1)/float(integrationSteps));

		// Detect collisions
		if (!lastUpdated) cdetector->update(k==0);
		cdetector->collide(params.collisionTolerance, CT_ALL, CT_ALL);
		connection->setNumCollisions(cdetector->getCollisionList().count());

		// Solve for the next state
		int numSteps=Vlado::cgSolveTarget<SimClothEngine>(*this, params.solverPrecision, maxSteps, 0.9f, 1.0f);
		if (numSteps<0 && integrationSteps>8192) { err=SC3_ERR_INTEGRITY_LOSS; break; }

		Vlado::real minDist=1e20f;
		Vlado::CollisionList &clist=cdetector->getCollisionList();
		Vlado::Collision *c;
		clist.resetCounter();
		while (c=clist.nextElement()) if (c->valid) minDist=Vlado::Min(minDist, (c->relPos()*c->normal)/(c->dist));

		// if (numSteps>=0) numIter+=numSteps+1;

		int stepChange=0;
		if (!adaptive) {
			cdetector->update(k==0);
			cdetector->collide(params.collisionTolerance, CT_ALL, CT_ALL);
			cdetector->accept();
			k++;
		} else {
			// Adaptively determine the timestep
			if ((numSteps<0 || numSteps>=decreaseThreshOne) && integrationLevels<=params.adaptiveSubdivs) stepChange=1;
			else if (numSteps>decreaseThreshTwo && integrationLevels<=params.adaptiveSubdivs) stepChange=2;
			else if (!lockStep && numSteps<increaseThresh && integrationSteps>params.substeps && (k+1)/2<integrationSteps/2 && ((k+1)&1)==0) stepChange=3;

			if (stepChange!=1) {

				numIter++;

				if (params.checkIntersections) {
					cdetector->update(false);
					lastUpdated=true;
					int isect=cdetector->checkIntersection(true);
					if (minDist<0.1f || isect) {
						// Intersection is detected
						// If possible, reduce the time step
						// Otherwise, if this is the first time we detected an intersection, try a few more
						// steps to see if it is resolved
						// If not, break out with an error
						lockStep=true;
						if (integrationSteps<=8192) stepChange=1;
						else {
							intersection++;
							if (isect && intersection>=50) return SC3_ERR_INTERSECTION;
						}
					} else /*if (minDist<0.5f) {
						if (integrationSteps<=512) stepChange=2;
					} else*/ {
						lockStep=false;
						intersection=0;
						cdetector->accept();
					}
				} else {
					lockStep=false;
					intersection=0;
					cdetector->accept();
				}
			}

			if (stepChange==0) {
				k++;
			} else if (stepChange==1) {
				// Too many iterations, decrease time step and repeat
				integrationSteps*=2;
				// if (numSteps==maxSteps) params.substeps=integrationSteps;
				dt*=0.5f;
				dt_inv*=2.0f;
				k*=2;
				integrationLevels++;
				// numIter-=numSteps+1;
				decreaseCount=32;
				lastUpdated=false;
			} else if (stepChange==2) {
				k++;
				// Too many iterations, decrease time step
				integrationSteps*=2;
				dt*=0.5f;
				dt_inv*=2.0f;
				decreaseCount=16;
				integrationLevels++;
				k*=2;
			} else if (stepChange==3) {
				k++;
				if (decreaseCount==0) {
					// Too few iterations, increase the time step
					decreaseCount=4;
					integrationSteps/=2;
					dt*=2.0f;
					dt_inv*=0.5f;
					k/=2;
					integrationLevels--;
				} else decreaseCount--;
			}
		}

		connection->setFrameProgressRange(0, integrationSteps);
		connection->setFrameProgressPos(k);

		// Set the new variables
		if (stepChange!=1) {
			connection->setNumCGIterations(numSteps);
			for (int i=0; i<objects.count(); i++) objects[i]->acceptChanges();
		}
	}

	return err;
}

// Compute the forces acting on the particles
void SimClothEngine::applyForces(float dt) {
	forceEvals++;
	for (int i=0; i<objects.count(); i++) objects[i]->clearForces();

	Vlado::Vector g=params.gravity;
	for (i=0; i<objects.count(); i++) {
		DynamicObject *obj=objects[i];

		obj->applyForces();
		obj->applyGravity(g);
	}

	for (i=0; i<objects.count(); i++) {
		DynamicObject *obj=objects[i];
		obj->computeTargets(dt);
	}

	cdetector->applyImpulses();
}
