#include "Headers.h"
#include "fvec.h"

#include "Utils.h"
#include "SimClothEngine.h"
#include "Forces.h"

USE_VLADO

void SpringsArray::freeMem(void) {
	if (ks) delete[] ks;
	if (kd) delete[] kd;
	if (restLen) delete[] restLen;
	if (scale) delete[] scale;

	if (p0) delete[] p0;
	if (p1) delete[] p1;

	ks=kd=restLen=scale=NULL;
	p0=p1=NULL;
	idx=numSprings=0;
}

void SpringsArray::setCount(int count, int useSSE) {
	freeMem();
	numSprings=count;

	this->useSSE=useSSE;

	ks=new float[numSprings];
	kd=new float[numSprings];
	restLen=new float[numSprings];
	scale=new float[numSprings];

	p0=new SCParticle*[numSprings];
	p1=new SCParticle*[numSprings];
	idx=0;
}

void SpringsArray::addSpring(SCParticle &p0, SCParticle &p1, float ks, float kd) {
	float len=length(p0.getPos()-p1.getPos());
	if (len<1e-6f) return;

	restLen[idx]=len;
	this->p0[idx]=&p0;
	this->p1[idx]=&p1;
	this->ks[idx]=fabsf(ks);
	this->kd[idx]=kd;
	scale[idx]=1.0f;
	idx++;
}

void SpringsArray::applyForces(void) {
	for (int i=0; i<idx; i++) doSprings1(i);
}

//***********************************************
inline void SpringsArray::doSprings1(int index) {
	Vector d=p1[index]->getPos()-p0[index]->getPos();

	float dlen=d.length();
	float dp=(scale[index]-dlen/restLen[index])*ks[index];
	Vector f=d*dp/dlen;

	p0[index]->f-=f;
	p1[index]->f+=f;
}

//***********************************************
// class BendsArray

void BendsArray::freeMem(void) {
	if (ks) delete[] ks;
	if (kd) delete[] kd;
	if (restLen) delete[] restLen;

	if (parts) delete[] parts;

	ks=kd=NULL;
	parts=NULL;
	restLen=NULL;
	idx=numBends=0;
}

void BendsArray::setCount(int count, int useSSE, int method) {
	freeMem();
	numBends=count;

	this->useSSE=useSSE;
	this->method=method;

	ks=new float[numBends];
	kd=new float[numBends];
	restLen=new float[numBends];

	parts=new SCParticle*[numBends*4];
	tri0=new ForceTriplet*[numBends];
	tri1=new ForceTriplet*[numBends];

	idx=0;
}

void BendsArray::addBend(SCParticle &p0, SCParticle &p1, SCParticle &p2, SCParticle &p3, ForceTriplet &t0, ForceTriplet &t1, float ks, float kd) {
	int idx4=idx*4;

	this->parts[idx4]=&p0;
	this->parts[idx4+1]=&p1;
	this->parts[idx4+2]=&p2;
	this->parts[idx4+3]=&p3;

	this->ks[idx]=ks;
	this->kd[idx]=kd;

	float len=length(p2.getPos()-p3.getPos());
	if (len<1e-6f) return;
	this->restLen[idx]=len;

	this->tri0[idx]=&t0;
	this->tri1[idx]=&t1;

	// Compute the inverse of the inertia matrix
	SCParticle **pt=parts+idx4;

	Vector p[4];
	for (int i=0; i<4; i++) p[i]=pt[i]->getPos();

	Vector u=p[1]-p[0], v=p[3]-p[2];
	Vector n=u^v;
	Vector n1=n/(n*n);

	Vector P=(p[0]+p[1]+p[2]+p[3])/4.0f;
	for (int i=0; i<4; i++) p[i]-=P;

	idx++;
}

void BendsArray::applyForces(void) {
	for (int i=0; i<idx; i++) doBends1(i);
}

inline void BendsArray::doBends1(int index) {
	if (method==1) {
		SCParticle **pt=parts+index*4;

		Vector p[4], v[4];
		for (int i=0; i<4; i++) { p[i]=pt[i]->getPos(); v[i]=pt[i]->getVel(); }

		Vector uc=p[1]-p[0], vc=p[3]-p[2];
		Vector n=uc^vc;
		Vector n1=n/(n*n);

		Vector P=(p[0]+p[1]+p[2]+p[3])/4.0f;
		Vector V=(v[0]+v[1]+v[2]+v[3])/4.0f;
		float kp[4], kv[4];
		Vector f[4];
		for (int i=0; i<4; i++) { p[i]-=P; v[i]-=V; kv[i]=v[i]*n1; v[i]=n*kv[i]; kp[i]=-p[i]*n1; f[i]=n*kp[i]; p[i]+=f[i]; }

		Vector q0=p[0];

		float u0=(q0^vc)*n1, v0=(uc^q0)*n1;
		float u1=u0+1.0f, v1=v0;
		float u2=-u0-0.5f, v2=-v0-0.5f;
		float u3=u2, v3=v2+1.0f;

		float uu=uc*uc, uv=uc*vc, vv=vc*vc;

		float pc=4.0f*u0*(u0+1.0f)+1.5f; // u0*u0+u1*u1+u2*u2+u3*u3;
		float qc=4.0f*u0*v0+2.0f*v0; // u0*v0+u1*v1+u2*v2+u3*v3;
		float rc=4.0f*v0*v0+0.5f; // v0*v0+v1*v1+v2*v2+v3*v3;

		float a=uv*qc+vv*rc, b=-uv*pc-vv*qc;
		float c=-uu*qc-uv*rc, d=uu*pc+uv*qc;

		float D=1.0f/(a*d-b*c);

		float su=u0*kv[0]+u1*kv[1]+u2*kv[2]+u3*kv[3];
		float sv=v0*kv[0]+v1*kv[1]+v2*kv[2]+v3*kv[3];

		float Lu=uv*su+vv*sv;
		float Lv=-uu*su-uv*sv;

		float wu=(Lu*d-Lv*b)*D;
		float wv=(a*Lv-c*Lu)*D;

		Vector av=uc*wu+vc*wv;

		float psu=u0*kp[0]+u1*kp[1]+u2*kp[2]+u3*kp[3];
		float psv=v0*kp[0]+v1*kp[1]+v2*kp[2]+v3*kp[3];

		float pLu=uv*psu+vv*psv;
		float pLv=-uu*psu-uv*psv;

		float pwu=(pLu*d-pLv*b)*D;
		float pwv=(a*pLv-c*pLu)*D;

		Vector pav=uc*pwu+vc*pwv;

		for (int i=0; i<4; i++) {
			Vector q=p[i];
			Vector fv=(av^q)-v[i];
			Vector fp=-(pav^q)+f[i];
			pt[i]->f+=fp*ks[index];
			pt[i]->f+=fv*kd[index];
		}
	} else {
		// Bend force
		SCParticle **p=parts+index*4;
		// Damping
		Vector ed=(p[1]->getPos())-(p[0]->getPos());
		float c=length(ed);

		Vector &n0=tri0[index]->normal;
		real a=tri0[index]->invLength;

		Vector &n1=tri1[index]->normal;
		real b=tri1[index]->invLength;

		real sinalpha=(n0^n1)*ed/c;
		real cosalpha=n0*n1;

		// the bend condition
		real bend=atan2f(sinalpha, cosalpha);

		// the derivatives
		Vector cd[4];
		cd[2]=n0*(-a*c);
		cd[3]=n1*(-b*c);

		float kt0=((p[2]->getPos()-p[0]->getPos())*ed)/(c*c);
		float kt1=((p[3]->getPos()-p[0]->getPos())*ed)/(c*c);

		cd[1]=cd[2]*(-kt0)+cd[3]*(-kt1);
		cd[0]=cd[2]*(kt0-1.0f)+cd[3]*(kt1-1.0f);

		// compute bend/dt
		real bend_dt=0.0f;
		for (int i=0; i<4; i++) bend_dt+=cd[i]*p[i]->getVel();

		// compute the forces
		real k=-(ks[index]*bend+kd[index]*bend_dt);

		for (i=0; i<4; i++) p[i]->applyForce(k*cd[i]);
	}
}
