/**********************************************************************
	FILE: sequence.h
	DESCRIPTION: A helper class for storing a large (variable) number of things
	Copyright (c) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32

	HISTORY:
		13 March 2003
			- Sequence and SequenceST are both based on a common template class _Sequence
		13 June 2002
			- added the SEQUENCE_ALIGN option which will cause aligned allocation of memory
		4 Jan 2002
			- fixed newArray when the size of the array is exactly equal to the block size
		2 Jan 2002
			- added class SequenceST - a single-threaded version of Sequence
***********************************************************************/

#ifndef __THREADS_H__
#define __THREADS_H__

#include "progress.h"

#ifdef _WIN32
#	include "threads_win32.h"
#else
#	include "threads_linux.h"
#endif

BEGIN_VLADO

// A volatile resource class, that can be either valid, or invalid
// In addition to being lock-able
class MovableResource {
	volatile int valid;
	volatile int deleteIt;
	InterlockedCounter locked;
	CriticalSection csect;
public:
	MovableResource(void) { clear(); }
	void clear(void) { valid=false; locked.set(0); deleteIt=false; }

	void* fastLock(void); // Returns a pointer to the resource and locks it, if the resource is valid; NULL otherwise
	void* lock(void); // Validates the resource, locks it, and a returns a pointer to it
	void unlock(void); // Unlocks the resource; invalidates it if the resource is marked for invalidating
	void invalidate(void); // Invalidates the resource, if it is unlocked; otherwise just marks it for invalidating

	virtual void makeValid(void)=0; // This is called to actually validate the resource; thread-safety guaranteed
	virtual void makeInvalid(void)=0; // This is called to actually invalidate the resource; thread-safety guaranteed
	virtual void *getData(void)=0; // Returns a pointer to the resource
};

//**********************************************
// A basic multithreaded class
class MultiThreaded {
public:
	// Call the run() method to execute the thread procedure in parallel in the given
	// number of threads (0 means a thread per processor)
	void run(int numThreads, Vlado::ProgressCallback *p=NULL, int showProgress=true);

	// The thread function performing the actual work
	// You should periodically check abort, and return if it is true
	virtual int threadProc(volatile int &abort, volatile int &progress, int threadIdx, int numThreads)=0;
};

//**********************************************
// A helper multithreaded class for a simple "for" cycle
class MultiThreadedFor: private MultiThreaded {
	CriticalSection csect;
	unsigned char *processed;
	int count;
	int threadProc(volatile int &abort, volatile int &progress, int threadIdx, int numThreads);
public:
	// This is called to run the cycle
	void run(int numIterations, int numThreads=0, Vlado::ProgressCallback *p=NULL, int showProgress=true);

	// This does the actual work
	virtual void body(int index, volatile int &abort)=0;
};

//**********************************************
// Global functions

// Executes the given multithreaded class (actually MultiThreaded::run() uses this method)
int multiThreads(int numThreads, MultiThreaded *multi, Vlado::ProgressCallback *p, int showProgress=true);

// Returns the zero-based index of the currently running thread
int getThreadIndex(void);

// Returns the number of processors in the system
int getNumProcessors(void);

// Sets the thread priority
enum ThreadPriority { normalPriority, lowPriority };
void setThreadPriority(ThreadPriority threadPriority);

END_VLADO

#endif
