#ifndef __VGL_FDOP_H__
#define __VGL_FDOP_H__

BEGIN_VLADO

#define NUM_DIRS 7

#define FDOP_MUL 1.7320508075688772935274463415059f
#define FDOP_DIV (1.0f/FDOP_MUL)

static Vector fdop_dirs[NUM_DIRS]={
	Vector(1,0,0),
	Vector(0,1,0),
	Vector(0,0,1),

	normalize(Vector(1,1,1)),
	normalize(Vector(1,-1,1)),
	normalize(Vector(1,-1,-1)),
	normalize(Vector(1,1,-1))
};

inline void refit(real &low, real &high, real x) {
	if (x<low) low=x;
	if (x>high) high=x;
}

inline void refitLow(real &low, real x) {
	if (x<low) low=x;
}

inline void refitHigh(real &high, real x) {
	if (x>high) high=x;
}

inline bool inInterval(real low, real high, real x) {
	if (x<low || x>high) return false;
	return true;
}

class FDop {
	public:
		real pmin[7], pmax[7];

		FDop(void) {}
		FDop(Vector &p) {
			init(p);
		}

		void init(void) {
			pmin[0]=1e20f;
			pmin[1]=1e20f;
			pmin[2]=1e20f;
			pmin[3]=1e20f;
			pmin[4]=1e20f;
			pmin[5]=1e20f;
			pmin[6]=1e20f;

			pmax[0]=-1e20f;
			pmax[1]=-1e20f;
			pmax[2]=-1e20f;
			pmax[3]=-1e20f;
			pmax[4]=-1e20f;
			pmax[5]=-1e20f;
			pmax[6]=-1e20f;
		}

		void init(Vector &p) {
			pmin[0]=pmax[0]=p.x;
			pmin[1]=pmax[1]=p.y;
			pmin[2]=pmax[2]=p.z;

			pmin[3]=pmax[3]=p.x+p.y+p.z;
			pmin[4]=pmax[4]=p.x-p.y+p.z;
			pmin[5]=pmax[5]=p.x-p.y-p.z;
			pmin[6]=pmax[6]=p.x+p.y-p.z;
		}

		// Add a point to the fdop
		void operator +=(Vector &p) {
			refit(pmin[0], pmax[0], p.x);
			refit(pmin[1], pmax[1], p.y);
			refit(pmin[2], pmax[2], p.z);

			refit(pmin[3], pmax[3], p.x+p.y+p.z);
			refit(pmin[4], pmax[4], p.x-p.y+p.z);
			refit(pmin[5], pmax[5], p.x-p.y-p.z);
			refit(pmin[6], pmax[6], p.x+p.y-p.z);
		}

		// Add an fdop
		void operator +=(FDop &f) {
			refitLow(pmin[0], f.pmin[0]);
			refitLow(pmin[1], f.pmin[1]);
			refitLow(pmin[2], f.pmin[2]);
			refitLow(pmin[3], f.pmin[3]);
			refitLow(pmin[4], f.pmin[4]);
			refitLow(pmin[5], f.pmin[5]);
			refitLow(pmin[6], f.pmin[6]);

			refitHigh(pmax[0], f.pmax[0]);
			refitHigh(pmax[1], f.pmax[1]);
			refitHigh(pmax[2], f.pmax[2]);
			refitHigh(pmax[3], f.pmax[3]);
			refitHigh(pmax[4], f.pmax[4]);
			refitHigh(pmax[5], f.pmax[5]);
			refitHigh(pmax[6], f.pmax[6]);
		}

		// Intersect with an fdop
		void operator &=(FDop &f) {
			refitHigh(pmin[0], f.pmin[0]);
			refitHigh(pmin[1], f.pmin[1]);
			refitHigh(pmin[2], f.pmin[2]);
			refitHigh(pmin[3], f.pmin[3]);
			refitHigh(pmin[4], f.pmin[4]);
			refitHigh(pmin[5], f.pmin[5]);
			refitHigh(pmin[6], f.pmin[6]);

			refitLow(pmax[0], f.pmax[0]);
			refitLow(pmax[1], f.pmax[1]);
			refitLow(pmax[2], f.pmax[2]);
			refitLow(pmax[3], f.pmax[3]);
			refitLow(pmax[4], f.pmax[4]);
			refitLow(pmax[5], f.pmax[5]);
			refitLow(pmax[6], f.pmax[6]);
		}

		// Returns true if the given point is in the fdop
		bool isInside(Vector &p) {
			if (!inInterval(pmin[0], pmax[0], p.x)) return false;
			if (!inInterval(pmin[1], pmax[1], p.y)) return false;
			if (!inInterval(pmin[2], pmax[2], p.z)) return false;
			if (!inInterval(pmin[3], pmax[3], p.x+p.y+p.z)) return false;
			if (!inInterval(pmin[4], pmax[4], p.x-p.y+p.z)) return false;
			if (!inInterval(pmin[5], pmax[5], p.x-p.y-p.z)) return false;
			if (!inInterval(pmin[6], pmax[6], p.x+p.y-p.z)) return false;
			return true;
		}

		// Returns true if the given fdop intersects this one
		bool intersect(FDop &f) {
			if (pmax[0]<f.pmin[0] || pmin[0]>f.pmax[0]) return false;
			if (pmax[1]<f.pmin[1] || pmin[1]>f.pmax[1]) return false;
			if (pmax[2]<f.pmin[2] || pmin[2]>f.pmax[2]) return false;
			if (pmax[3]<f.pmin[3] || pmin[3]>f.pmax[3]) return false;
			if (pmax[4]<f.pmin[4] || pmin[4]>f.pmax[4]) return false;
			if (pmax[5]<f.pmin[5] || pmin[5]>f.pmax[5]) return false;
			if (pmax[6]<f.pmin[6] || pmin[6]>f.pmax[6]) return false;
			return true;
		}

		bool intersectTolerance(FDop &f, real tolerance) {
			if (pmax[0]+tolerance<f.pmin[0] || pmin[0]-tolerance>f.pmax[0]) return false;
			if (pmax[1]+tolerance<f.pmin[1] || pmin[1]-tolerance>f.pmax[1]) return false;
			if (pmax[2]+tolerance<f.pmin[2] || pmin[2]-tolerance>f.pmax[2]) return false;

			tolerance*=FDOP_MUL;

			if (pmax[3]+tolerance<f.pmin[3] || pmin[3]-tolerance>f.pmax[3]) return false;
			if (pmax[4]+tolerance<f.pmin[4] || pmin[4]-tolerance>f.pmax[4]) return false;
			if (pmax[5]+tolerance<f.pmin[5] || pmin[5]-tolerance>f.pmax[5]) return false;
			if (pmax[6]+tolerance<f.pmin[6] || pmin[6]-tolerance>f.pmax[6]) return false;
			return true;
		}

		// Expands the fdop by dist
		void expand(real dist) {
			pmin[0]-=dist;
			pmin[1]-=dist;
			pmin[2]-=dist;

			pmax[0]+=dist;
			pmax[1]+=dist;
			pmax[2]+=dist;

			dist*=FDOP_MUL;

			pmin[3]-=dist;
			pmin[4]-=dist;
			pmin[5]-=dist;
			pmin[6]-=dist;

			pmax[3]+=dist;
			pmax[4]+=dist;
			pmax[5]+=dist;
			pmax[6]+=dist;
		}

		// Gives an approximation to the volume
		real volume(void) {
			return (pmax[0]-pmin[0])*(pmax[1]-pmin[1])*(pmax[2]-pmin[2]);
		}

		int maxDimension(void) {
			int maxd=0;
			real d=pmax[0]-pmin[0];

			if (pmax[1]-pmin[1]>d) { d=pmax[1]-pmin[1]; maxd=1; }
			if (pmax[2]-pmin[2]>d) { d=pmax[2]-pmin[2]; maxd=2; }
			if ((pmax[3]-pmin[3])*FDOP_DIV>d) { d=pmax[3]-pmin[3]; maxd=3; }
			if ((pmax[4]-pmin[4])*FDOP_DIV>d) { d=pmax[4]-pmin[4]; maxd=4; }
			if ((pmax[5]-pmin[5])*FDOP_DIV>d) { d=pmax[5]-pmin[5]; maxd=5; }
			if ((pmax[6]-pmin[6])*FDOP_DIV>d) { d=pmax[6]-pmin[6]; maxd=6; }
			return maxd;
		}
};

END_VLADO

#endif
