/*****************************************************
	FILE: Color.h
	DESCRIPTION: RGB color
	Copyright (C) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __COLOR_H__
#define __COLOR_H__

BEGIN_VLADO

typedef uint32 RGB32;

class Color {
	public:
		real r, g, b;

		Color() {};
		Color(real ir, real ig, real ib) { r=ir; g=ig; b=ib; }
		Color(int ir, int ig, int ib) {
			r=real(ir)/255.0f;
			g=real(ig)/255.0f;
			b=real(ib)/255.0f;
		}
		/*inline Color(int col) {
			r=real((col>>16)&0xFF)/255.0f;
			g=real((col>>8)&0xFF)/255.0f;
			b=real(col&0xFF)/255.0f;
		}*/

		void set(real ir, real ig, real ib) {
			r=ir; g=ig; b=ib;
		}

		void makeZero(void) {
			r=g=b=0.0f;
		}

		bool isBlack(void) {
			if (r<1e-12 && g<1e-12 && b<1e-12) return true;
			else return false;
		}

		/*inline operator RGB32() {
			return 
				(FLOOR(realMax(realMin(b*255.0f, 255.0f), 0.0f)))+
				(FLOOR(realMax(realMin(g*255.0f, 255.0f), 0.0f))<<8)+
				(FLOOR(realMax(realMin(r*255.0f, 255.0f), 0.0f))<<16);
		}*/
		RGB32 toRGB32(void) const {
			return 
				(FLOOR(Max(Min(b*255.0f, 255.0f), 0.0f)))+
				(FLOOR(Max(Min(g*255.0f, 255.0f), 0.0f))<<8)+
				(FLOOR(Max(Min(r*255.0f, 255.0f), 0.0f))<<16);
		}

		void operator +=(const Color &a) { r+=a.r; g+=a.g; b+=a.b; }
		void operator -=(const Color &a) { r-=a.r; g-=a.g; b-=a.b; }
		void operator *=(const Color &a) { r*=a.r; g*=a.g; b*=a.b; }
		void operator /=(real f) { r/=f; g/=f; b/=f; }
		void operator *=(real f) { r*=f; g*=f; b*=f; }

		Color operator-(void) const { return(Color(-r,-g,-b)); } 

		real& operator [](int index) { return (&r)[index]; }
		const real& operator [](int index) const { return (&r)[index]; }

		real intensity(void) { return (r+g+b)/3.0f; }
		real lengthSqr(void) { return r*r+g*g+b*b; }
		void clampMinMax(void) {
			if (r<0.0f) r=0.0f;
			if (g<0.0f) g=0.0f;
			if (b<0.0f) b=0.0f;

			if (r>1.0f) r=1.0f;
			if (g>1.0f) g=1.0f;
			if (b>1.0f) b=1.0f;
		}

		void clampMin(void) {
			if (r<0.0f) r=0.0f;
			if (g<0.0f) g=0.0f;
			if (b<0.0f) b=0.0f;
		}

		void clampMax(void) {
			if (r>1.0f) r=1.0f;
			if (g>1.0f) g=1.0f;
			if (b>1.0f) b=1.0f;
		}
};

INLINE Color operator /(const Color &a, real f) { return Color(a.r/f, a.g/f, a.b/f); }
INLINE Color operator *(const Color &a, real f) { return Color(a.r*f, a.g*f, a.b*f); }
INLINE Color operator *(real f, const Color &a) { return Color(a.r*f, a.g*f, a.b*f); }
INLINE Color operator +(const Color &a, const Color &b) { return Color(a.r+b.r, a.g+b.g, a.b+b.b); }
INLINE Color operator -(const Color &a, const Color &b) { return Color(a.r-b.r, a.g-b.g, a.b-b.b); }
INLINE Color operator *(const Color &a, const Color &b) { return Color(a.r*b.r, a.g*b.g, a.b*b.b); }
INLINE Color operator /(const Color &a, const Color &b) { return Color(a.r/b.r, a.g/b.g, a.b/b.b); }

INLINE Color pow(const Color &color, real f) { return Color(powf(color.r, f), powf(color.g, f), powf(color.b, f)); }

INLINE real lengthSqr(const Color &a) { return a.r*a.r+a.g*a.g+a.b*a.b; }
INLINE real length(const Color &a) { return sqrtf(a.r*a.r+a.g*a.g+a.b*a.b); }
INLINE real intensity(const Color &a) { return (a.r+a.g+a.b)/3.0f; }

END_VLADO

#endif
