#ifndef __VGL_COLLISIONDETECT_H__
#define __VGL_COLLISIONDETECT_H__

#include "CollisionMesh.h"
#include "Collision.h"

#define CT_VERTEX_VERTEX 1
#define CT_VERTEX_FACE 2
#define CT_VERTEX_EDGE 4
#define CT_EDGE_EDGE 8

#define CT_ALL (CT_VERTEX_VERTEX | CT_VERTEX_FACE | CT_VERTEX_EDGE | CT_EDGE_EDGE)

#define CT_CLOTH_CLOTH 16
#define CT_CLOTH_DEFLECTOR 32

BEGIN_VLADO

template<class Mesh>
class CollisionDetector {
	public:
		typedef BVTree<BVertex<Mesh> > BVertexTree;
		typedef BVTree<BEdge<Mesh> > BEdgeTree;
		typedef BVTree<BFace<Mesh> > BFaceTree;
		typedef BVTree<BMesh<Mesh> > BMeshTree;

		typedef SequenceST<Collision<Mesh>, 20000 > CollisionList;
		typedef CollisionList::Counter CollisionListCounter;

		CollisionList *clist, *clistOld;
		CollisionList clist0, clist1;

		BMeshTree cmeshTree;
		real tolerance;

		CollisionDetector(void) {
			init(0.2f);
		}

		void init(float tolerance) {
			clist=&clist0;
			clistOld=&clist1;
			this->tolerance=tolerance;
		}

		void freeData(void) {
			cmeshTree.bvolumes.resetCounter();
			cmeshTree.freeData();
			clist0.freeData();
			clist1.freeData();
			clist=&clist0;
			clistOld=&clist1;
		}

		// Add a new collision mesh
		void addCollisionMesh(Mesh *cmesh);

		// Meshes must be transformed!
		void update(bool rebuildAll, float timestep);
		void updateTrees(Mesh *mesh, int rebuildAll=true);

		void collide(real timestep, real tolerance, int meshMeshCollisions, int selfCollisions, int fastSelfCollide);
		int checkIntersection(void);
		float getMinCollision(void);
		void saveCollisions(void) {
			clistOld->clear();
			clist->resetCounter();
			Collision<Mesh> *c;
			while (c=clist->nextElement()) *clistOld->newElement()=*c;
		}
		void restoreCollisions(void) {
			clist->clear();
			clistOld->resetCounter();
			Collision<Mesh> *c;
			while (c=clistOld->nextElement()) *clist->newElement()=*c;
		}
};

//*********************************************
// Implementation
static real collisionSpring=1.0f;
static real cspring=1.0f;

#define EPS 1e-3f


//*********************************************
//
template<class Mesh>
void CollisionDetector<Mesh>::addCollisionMesh(Mesh *cmesh) {
	cmesh->allocOrigp(cmesh->getNumVerts());

	BVertexTree &vertexTree=cmesh->getVertexTree();
	BEdgeTree &edgeTree=cmesh->getEdgeTree();
	BFaceTree &faceTree=cmesh->getFaceTree();

	BVertex<Mesh> v;
	for (int i=0; i<cmesh->getNumVerts(); i++) {
		v.cmesh=cmesh;
		v.idx=i;
		v.fixed=cmesh->isVertexFixed(i);
		vertexTree.storeBBox(v);
		cmesh->getOrigp(i)=cmesh->vertexPos(i);
	}

	BEdge<Mesh> e;
	for (i=0; i<cmesh->getNumEdges(); i++) {
		e.cmesh=cmesh;
		e.idx[0]=cmesh->getEdgeVertex(i, 0);
		e.idx[1]=cmesh->getEdgeVertex(i, 1);
		e.fixed=cmesh->isVertexFixed(e.idx[0]) || cmesh->isVertexFixed(e.idx[1]);
		e.edgeIdx=i;
		edgeTree.storeBBox(e);
	}

	BFace<Mesh> f;
	for (i=0; i<cmesh->getNumFaces(); i++) {
		f.cmesh=cmesh;
		f.idx[0]=cmesh->getFaceVertex(i, 0);
		f.idx[1]=cmesh->getFaceVertex(i, 1);
		f.idx[2]=cmesh->getFaceVertex(i, 2);
		f.fixed=cmesh->isVertexFixed(f.idx[0]) || cmesh->isVertexFixed(f.idx[1]) || cmesh->isVertexFixed(f.idx[2]);
		faceTree.storeBBox(f);
	}

	BMesh<Mesh> bm;
	bm.cmesh=cmesh;
	cmeshTree.storeBBox(bm);
}

template<class Mesh>
void CollisionDetector<Mesh>::update(bool rebuildAll, float timestep) {
	// OutputDebugString("Updating collision detector...\n");
	cmeshTree.bvolumes.resetCounter();
	for (int i=0; i<cmeshTree.bvolumes.count(); i++) {
		BMesh<Mesh> *bm=cmeshTree.bvolumes.nextElement();
		bm->cmesh->prepareCollisionDetection();
		updateTrees(bm->cmesh, rebuildAll);
		bm->bv=(bm->cmesh->getVertexTree().root->bv);
	}

	if (rebuildAll) cmeshTree.build();
	else cmeshTree.refit();
	// OutputDebugString("Finished updating collision detector.\n");
}

// Checks if the four given points come to lie the same plane somewhere in the interval [0,1]
// return 0 if there is no intersection
static int checkPlane(Vector p[4], Vector q[4], double &t) {
	Vector dp[4];
	for (int i=0; i<4; i++) dp[i]=q[i]-p[i];

	Vector e0=p[2]-p[1], e1=p[3]-p[1];
	double pn[3];
	_cross3(e0, e1, pn);

	Vector e2=q[2]-q[1], e3=q[3]-q[1];
	double qn[3];
	_cross3(e2, e3, qn);

	double dn[3];
	_sub3(qn, pn, dn);

	Vector pf=p[0]-p[1], qf=q[0]-q[1];
	Vector df=qf-pf;

	double a=_dot3(dn, df);
	double b=_dot3(pf,dn)+_dot3(df,pn);
	double c=_dot3(pn,pf);

	if (fabs(a)<1e-6f) {
		double t0;
		if (fabs(b)>1e-6f) t0=-c/b;
		else return false;

		if (t0<1e-6f || t0>1.0f-1e-6f) return false;
		t=t0;
		return true;
	} else {
		double D=b*b-4.0f*a*c;
		if (D<1e-6f) return false;

		D=sqrt(D);
		double t0=(-b-D)/(2.0f*a);
		double t1=(-b+D)/(2.0f*a);
		if (t0>t1) swap(t0, t1);

		if (t0>1.0f-1e-6f || t1<1e-6f) return false;
		if (t0>1e-6f) {
			t=t0;
			return true;
		}
		if (t1<1.0f-1e-6f) {
			t=t1;
			return true;
		}
		return false;
	}
}

template<class Mesh>
class VertexFaceCollide: public BBCallback {
	public:
		real timestep, tolerance, toleranceSq;
		CollisionDetector<Mesh>::CollisionList *collisions;

		void init(real dt, real dist, CollisionDetector<Mesh>::CollisionList *clist) {
			timestep=dt;
			tolerance=dist;
			toleranceSq=sqr(tolerance);
			collisions=clist;
		}

		float det(Vector &a, Vector &b, Vector &c) {
			return (a^b)*c;
		}

		int checkVertexFacePath(BVertex<Mesh> *bv, BFace<Mesh> *bf, Vector ip[4]) {
			Vector p[4], q[4];

			p[0]=(bv->cmesh)->vertexPosPrev(bv->idx);
			p[1]=(bf->cmesh)->vertexPosPrev(bf->idx[0]);
			p[2]=(bf->cmesh)->vertexPosPrev(bf->idx[1]);
			p[3]=(bf->cmesh)->vertexPosPrev(bf->idx[2]);

			q[0]=(bv->cmesh)->vertexPos(bv->idx);
			q[1]=(bf->cmesh)->vertexPos(bf->idx[0]);
			q[2]=(bf->cmesh)->vertexPos(bf->idx[1]);
			q[3]=(bf->cmesh)->vertexPos(bf->idx[2]);

			// Find if and when the four points lie in a plane
			double t=1.0f;
			int res=checkPlane(p, q, t);

			// Compute the position of the points at that moment
			for (int i=0; i<4; i++) ip[i]=p[i]*float(1.0f-t)+q[i]*float(t);
			return res;
		}

		bool intersect(BVolume *b0, BVolume *b1) {
			// if (*isect) return false;
			BVertex<Mesh> *bv=(BVertex<Mesh>*) b0;
			BFace<Mesh> *bf=(BFace<Mesh>*) b1;

			if (!bf->valid) return true;

			int selfCollision=(bv->cmesh==bf->cmesh);
			if (selfCollision && !(bf->cmesh)->selfCollide()) return true;
			if (bv->fixed && bf->fixed) return true;

			if (!selfCollision) {
				if (!bv->collide || !bf->collide) return true;
			} else {
				if (!bv->selfCollide || !bf->selfCollide) return true;
			}

			// First check if vertex path intersects the face
			Vector ip[4]/*={
				(bv->cmesh)->vertexPos(bv->idx),
				(bf->cmesh)->vertexPos(bf->idx[0]),
				(bf->cmesh)->vertexPos(bf->idx[1]),
				(bf->cmesh)->vertexPos(bf->idx[2])
			}*/;
			int ipath=/* false; // */checkVertexFacePath(bv, bf, ip);

			Vector u=ip[2]-ip[1];
			Vector v=ip[3]-ip[1];
			Vector n=u^v;
			float nlen=length(n);
			if (nlen<1e-3f) return true;
			n/=nlen;

			Vector offs=ip[0]-ip[1];
			float D=(u^v)*n;
			if (fabsf(D)<1e-3f) return true;

			float uc=((offs^v)*n)/D;
			if (uc<1e-6f) return true;

			float vc=((u^offs)*n)/D;
			if (vc<1e-6f || uc+vc>=1.0f-1e-6f) return true;

			float sd=n*offs;
			real d=sd*sd;

			Vector test=ip[1]+u*uc+v*vc;

			if (d>=toleranceSq) return true;
			if (!ipath && selfCollision && d<1e-6f) return true;

			real tdistSq=toleranceSq, tdist=tolerance;
			// compute the original distance
			if (selfCollision) {
				Vector *origp=&(bv->cmesh->getOrigp(0));
				Vector q0=origp[bv->idx];
				Vector q1=(1.0f-uc-vc)*origp[bf->idx[0]]+uc*origp[bf->idx[1]]+vc*origp[bf->idx[2]];

				real origDist=lengthSqr(q0-q1);
				if (origDist<=tdistSq) {
					tdistSq=origDist;
					if (d>=tdistSq*0.999f) return true;
					tdist=sqrtf(tdistSq);
				}
			}

			// A collision was detected
			Collision<Mesh> &c=*(collisions->newElement());
			c.b0=b0;
			c.b1=b1;
			c.dist=tdist*collisionSpring;
			c.ipath=ipath;
			c.type=c_vertex_face;
			if (!ipath && sd<0.0f) n=-n;
			c.normal=n;
			c.uc=uc;
			c.vc=vc;

			if (!c.initParticles()) collisions->removeLast();

			return true;
		}
};

template<class Mesh>
class EdgeEdgeCollide: public BBCallback {
	public:
		real timestep, tolerance, toleranceSq;
		CollisionDetector<Mesh>::CollisionList *collisions;

		void init(real dt, real dist, CollisionDetector<Mesh>::CollisionList *clist) {
			timestep=dt;
			tolerance=dist;
			toleranceSq=sqr(tolerance);
			collisions=clist;
		}

		int checkEdgeEdgePath(BEdge<Mesh> *be0, BEdge<Mesh> *be1, Vector ip[4]) {
			Vector p[4], q[4];

			p[0]=(be0->cmesh)->vertexPosPrev(be0->idx[0]);
			p[1]=(be0->cmesh)->vertexPosPrev(be0->idx[1]);
			p[2]=(be1->cmesh)->vertexPosPrev(be1->idx[0]);
			p[3]=(be1->cmesh)->vertexPosPrev(be1->idx[1]);

			q[0]=(be0->cmesh)->vertexPos(be0->idx[0]);
			q[1]=(be0->cmesh)->vertexPos(be0->idx[1]);
			q[2]=(be1->cmesh)->vertexPos(be1->idx[0]);
			q[3]=(be1->cmesh)->vertexPos(be1->idx[1]);

			// Find if and when the four points lie in a plane
			double t=1.0f;
			int res=checkPlane(p, q, t);

			// Compute the position of the points at that moment
			for (int i=0; i<4; i++) ip[i]=p[i]*float(1.0f-t)+q[i]*float(t);
			return res;
		}

		bool intersect(BVolume *b0, BVolume *b1) {
			BEdge<Mesh> *be0=(BEdge<Mesh>*) b0;
			BEdge<Mesh> *be1=(BEdge<Mesh>*) b1;

			int selfCollision=(be0->cmesh==be1->cmesh);
			if (selfCollision && !(be0->cmesh)->selfCollide()) return true;
			if (be0->fixed && be1->fixed) return true;
			if (be0->invisible || be1->invisible) return true;

			if (!selfCollision) {
				if (!be0->collide || !be1->collide) return true;
			} else {
				if (!be0->selfCollide || !be1->selfCollide) return true;
			}

			// First check if the edges intersected
			Vector ip[4]/*={
				(be0->cmesh)->vertexPos(be0->idx[0]),
				(be0->cmesh)->vertexPos(be0->idx[1]),
				(be1->cmesh)->vertexPos(be1->idx[0]),
				(be1->cmesh)->vertexPos(be1->idx[1])
			}*/;
			int ipath=/*false; // */checkEdgeEdgePath(be0, be1, ip);

			Vector &p0=ip[0];
			Vector &q0=ip[2];
			Vector dir0=ip[1]-ip[0], dir1=ip[3]-ip[2];

			Vector qp=dir0^dir1;
			float qpLen=length(qp);
			if (qpLen<1e-3f) return true;
			qp/=qpLen;

			float len0=dir0*dir0;
			if (len0<toleranceSq) return true;

			float len1=dir1*dir1;
			if (len1<toleranceSq) return true;

			Vector offs=p0-q0;
			float mx=dir0*dir1;

			float D=len0*len1-mx*mx;
			if (fabs(D)<1e-3f) return true;

			float offs_e0=offs*dir0;
			float offs_e1=offs*dir1;

			float t0=(offs_e1*mx-offs_e0*len1)/D;
			float t1=(offs_e1*len0-offs_e0*mx)/D;

			if (t0<1e-6f || t0>1.0f-1e-6f || t1<1e-6f || t1>1.0f-1e-6f) return true;

			// compute current distance
			Vector p=p0+dir0*t0;
			Vector q=q0+dir1*t1;
			// qp=p-q;
			// real sd=length(qp);
			// if (fabs(sd)>1e-6f) qp/=sd;
			real sd=(q-p)*qp;
			real d=sd*sd;

			if (d>=toleranceSq) return true;
			if (!ipath && selfCollision && d<1e-6f) return true;

			real tdistSq=toleranceSq, tdist=tolerance;
			// compute the original distance
			if (selfCollision) {
				Vector *origp=&(be0->cmesh->getOrigp(0));
				Vector q0=(1.0f-t0)*origp[be0->idx[0]]+t0*origp[be0->idx[1]];
				Vector q1=(1.0f-t1)*origp[be1->idx[0]]+t1*origp[be1->idx[1]];

				real origDist=lengthSqr(q0-q1);
				if (origDist<=tdistSq) {
					tdistSq=origDist;
					if (d>=tdistSq*0.999f) return true;
					tdist=sqrtf(tdistSq);
				}
			}

			// a collision was detected
			Collision<Mesh> &c=*(collisions->newElement());
			c.b0=b0;
			c.b1=b1;
			c.dist=tdist*collisionSpring;
			c.ipath=ipath;
			c.type=c_edge_edge;
			if (!ipath && sd>0.0f) qp=-qp;
			c.normal=qp;
			c.t0=t0;
			c.t1=t1;

			if (!c.initParticles()) collisions->removeLast();

			return true;
		}
};

template<class Mesh>
class MeshMeshCollide: public BBCallback {
	public:
		real timestep, tolerance, toleranceSq;
		int fastSelfCollide;
		CollisionDetector<Mesh>::CollisionList *collisions;

		VertexFaceCollide<Mesh> vfCollide;
		EdgeEdgeCollide<Mesh> eeCollide;
		int meshMeshCollisions, selfCollisions;

		MeshMeshCollide(real dt, real dist, CollisionDetector<Mesh>::CollisionList *clist, int meshMeshCollisions, int selfCollisions, int fastSelfCollide) {
			timestep=dt;
			tolerance=dist;
			toleranceSq=sqr(tolerance);
			collisions=clist;
			this->meshMeshCollisions=meshMeshCollisions;
			this->selfCollisions=selfCollisions;
			this->fastSelfCollide=fastSelfCollide;

			vfCollide.init(dt, dist, clist);
			eeCollide.init(dt, dist, clist);
		}

		bool intersect(BVolume *b0, BVolume *b1) {
			BMesh<Mesh> *bm0=(BMesh<Mesh>*) b0;
			BMesh<Mesh> *bm1=(BMesh<Mesh>*) b1;

			// If these are two delfectors - do not collide
			if (bm0->cmesh->objType==obj_deflector && bm1->cmesh->objType==obj_deflector) return true;

			if (bm0->cmesh==bm1->cmesh) {
				// Self-collisions
				if (!(bm0->cmesh->selfCollide())) return true;
				if (selfCollisions & CT_VERTEX_FACE) intersectNodes(bm0->cmesh->getVertexTree().root, bm0->cmesh->getFaceTree().root, vfCollide, tolerance*cspring, fastSelfCollide);
				if (selfCollisions & CT_EDGE_EDGE) bm0->cmesh->getEdgeTree().selfIntersect(eeCollide, tolerance*cspring, fastSelfCollide);
				return true;
			}

			if (meshMeshCollisions & CT_VERTEX_FACE) {
				intersectNodes(bm0->cmesh->getVertexTree().root, bm1->cmesh->getFaceTree().root, vfCollide, tolerance*cspring);
				intersectNodes(bm1->cmesh->getVertexTree().root, bm0->cmesh->getFaceTree().root, vfCollide, tolerance*cspring);
			}
			if ((meshMeshCollisions & CT_EDGE_EDGE)) intersectNodes(bm0->cmesh->getEdgeTree().root, bm1->cmesh->getEdgeTree().root, eeCollide, tolerance*cspring);

			return true;
		}
};

template<class Mesh>
void CollisionDetector<Mesh>::collide(real timestep, real tolerance, int meshMeshCollisions, int selfCollisions, int fastSelfCollide) {
	// Detect new collisions
	clist->clear();
	MeshMeshCollide<Mesh> mmCollide(timestep, tolerance, clist, meshMeshCollisions, selfCollisions, fastSelfCollide);
	cmeshTree.selfIntersect(mmCollide, tolerance, false);

	// Initialize the detected collisions
/*	clist.resetCounter();
	Vlado::Collision<DynamicObject> *c;
	while (c=clist.nextElement()) c->initParticles();*/
}

template<class Mesh>
float CollisionDetector<Mesh>::getMinCollision(void) {
	clist->resetCounter();
	Collision<DynamicObject> *c;
	float minCollision=1e18f;
	while (c=clist->nextElement()) {
		minCollision=Min(minCollision, c->getDistRel());
	}
	return minCollision;
}

template<class Mesh>
class EdgeFaceCollide: public BBCallback {
	public:
		int *isect;

		void init(int *intersection) {
			isect=intersection;
		}

		bool intersect(BVolume *b0, BVolume *b1) {
			if (*isect) return false;
			BEdge<Mesh> *be=(BEdge<Mesh>*) b0;
			BFace<Mesh> *bf=(BFace<Mesh>*) b1;

			if (!bf->valid) return true;

			if (be->cmesh==bf->cmesh) {
				if (!((Mesh*) (be->cmesh))->selfCollide()) return true;
				if (!be->selfCollide || !bf->selfCollide) return true;
			}

			if (be->fixed && bf->fixed) return true;
			if (!be->checkIntersections || !bf->checkIntersections) return true;
			if (!be->collide || !bf->collide) return true;

			Vector &p=be->p;
			Vector &dir=be->dir;
			Vector q=p+dir;

			Transform &tm=bf->tm;

			real s0=p.x*(tm.m.f[0].z)+p.y*(tm.m.f[1].z)+p.z*(tm.m.f[2].z)+(tm.offs.z);
			real s1=q.x*(tm.m.f[0].z)+q.y*(tm.m.f[1].z)+q.z*(tm.m.f[2].z)+(tm.offs.z);

			if (s0*s1<-EPS) {
				real s=s0/(s0-s1);
				if (s<EPS || s>1.0f-EPS) return true;
				Vector m=p+dir*s;

				real u=m.x*(tm.m.f[0].x)+m.y*(tm.m.f[1].x)+m.z*(tm.m.f[2].x)+(tm.offs.x);
				if (u<1e-12f) return true;

				real v=m.x*(tm.m.f[0].y)+m.y*(tm.m.f[1].y)+m.z*(tm.m.f[2].y)+(tm.offs.y);
				if (v<1e-12f) return true;
				if (u+v>1.0f-1e-12f) return true;

				*isect=true;
				return false;
			}
			return true;
		}
};

template<class Mesh>
class MeshMeshIntersect: public BBCallback {
	public:
		int *isect;
		EdgeFaceCollide<Mesh> efIntersect;

		MeshMeshIntersect(int *intersection) {
			isect=intersection;
			efIntersect.init(isect);
		}

		bool intersect(BVolume *b0, BVolume *b1) {
			if (*isect) return false;
			BMesh<Mesh> *bm0=(BMesh<Mesh>*) b0;
			BMesh<Mesh> *bm1=(BMesh<Mesh>*) b1;

			if (bm0->cmesh==bm1->cmesh) {
				if (!(bm0->cmesh->selfCollide())) return true;

				intersectNodes(bm0->cmesh->edgeTree.root, bm0->cmesh->faceTree.root, efIntersect, 0.0f);
				if (*isect) return false;
				else return true;
			}

			if (bm0->cmesh->objType==obj_deflector && bm1->cmesh->objType==obj_deflector) return true;

			intersectNodes(bm0->cmesh->edgeTree.root, bm1->cmesh->faceTree.root, efIntersect, 0.0f);
			intersectNodes(bm1->cmesh->edgeTree.root, bm0->cmesh->faceTree.root, efIntersect, 0.0f);
			if (*isect) return false;
			else return true;
		}
};

template<class Mesh>
int CollisionDetector<Mesh>::checkIntersection(void) {
	// Detect new collisions
	int intersection=false;
	MeshMeshIntersect<Mesh> mmIntersect(&intersection);
	cmeshTree.selfIntersect(mmIntersect, 0.0f, false);

	return intersection;
}

static const int numDirs=6;
static Vector dirNrm[numDirs]={
	Vector(1,0,0),
	Vector(0,1,0),
	Vector(0,0,1),
	Vector(-1,0,0),
	Vector(0,-1,0),
	Vector(0,0,-1),
};

inline int normalFlags(Vector &v) {
	int maxd=0;
	float maxk=0.0f;
	for (int dir=0; dir<numDirs; dir++) {
		float k=v*dirNrm[dir];
		if (k>maxk) { maxd=dir; maxk=k; }
	}
	return (1<<maxd);
}

template<class Mesh>
void CollisionDetector<Mesh>::updateTrees(Mesh *cmesh, int rebuildAll) {
	cmesh->vertexTree.bvolumes.resetCounter();
	for (int i=0; i<cmesh->vertexTree.bvolumes.count(); i++) {
		BVertex<Mesh> *v=cmesh->vertexTree.bvolumes.nextElement();

		Vector p=cmesh->vertexPos(v->idx);
		v->bv.init(p);

		v->p=p;
		v->bv+=(v->cmesh)->vertexPosPrev(v->idx);

		v->normal=(v->cmesh)->vertexNormal(v->idx);
		v->flags=normalFlags(v->normal);

		v->currentCollidePriority=-1;
		v->deflectorCollision=NULL;

		v->collide=(cmesh->objType==obj_rigid)? true : cmesh->getParticle(v->idx)->collide;
		v->selfCollide=(cmesh->objType==obj_rigid)? false : cmesh->getParticle(v->idx)->selfCollide;
		v->checkIntersections=(cmesh->objType==obj_rigid)? true: cmesh->getParticle(v->idx)->checkIntersections;

		cmesh->vertexFlags[v->idx]=0;
	}
	if (rebuildAll) cmesh->vertexTree.build();
	else cmesh->vertexTree.refit();

	cmesh->edgeTree.bvolumes.resetCounter();
	for (i=0; i<cmesh->edgeTree.bvolumes.count(); i++) {
		BEdge<Mesh> *be=cmesh->edgeTree.bvolumes.nextElement();

		Vector p0=cmesh->vertexPos(be->idx[0]);
		Vector p1=cmesh->vertexPos(be->idx[1]);

		be->bv.init(p0);
		be->bv+=p1;
		be->bv+=(be->cmesh)->vertexPosPrev(be->idx[0]);
		be->bv+=(be->cmesh)->vertexPosPrev(be->idx[1]);
		be->invisible=be->cmesh->invisibleEdge(be->edgeIdx);

		be->p=p0;
		be->dir=p1-p0;

		be->collide=(cmesh->objType==obj_rigid)? true :
			(cmesh->getParticle(be->idx[0])->collide &&
			 cmesh->getParticle(be->idx[1])->collide);
		be->selfCollide=(cmesh->objType==obj_rigid)? false :
			(cmesh->getParticle(be->idx[0])->selfCollide &&
			 cmesh->getParticle(be->idx[1])->selfCollide);
		be->checkIntersections=(cmesh->objType==obj_rigid)? true :
			(cmesh->getParticle(be->idx[0])->checkIntersections &&
			 cmesh->getParticle(be->idx[1])->checkIntersections);

		be->flags=normalFlags((be->cmesh)->edgeNormal(be->edgeIdx));
	}
	if (rebuildAll) cmesh->edgeTree.build();
	else cmesh->edgeTree.refit();

	cmesh->faceTree.bvolumes.resetCounter();
	for (i=0; i<cmesh->faceTree.bvolumes.count(); i++) {
		BFace<Mesh> *bf=cmesh->faceTree.bvolumes.nextElement();

		Vector p0=cmesh->vertexPos(bf->idx[0]);
		Vector p1=cmesh->vertexPos(bf->idx[1]);
		Vector p2=cmesh->vertexPos(bf->idx[2]);

		bf->bv.init(p0);
		bf->bv+=p1;
		bf->bv+=p2;

		bf->bv+=(bf->cmesh)->vertexPosPrev(bf->idx[0]);
		bf->bv+=(bf->cmesh)->vertexPosPrev(bf->idx[1]);
		bf->bv+=(bf->cmesh)->vertexPosPrev(bf->idx[2]);

		Vector n=(p1-p0)^(p2-p0);
		float len=length(n);
		if (len<1e-8f) {
			bf->valid=false;
			bf->flags=0;
			continue;
		}

		bf->normal=n/len;
		Transform tm;
		tm.m.f[0]=p1-p0;
		tm.m.f[1]=p2-p0;
		tm.m.f[2]=bf->normal;
		tm.offs=p0;

		bf->tm=inverse(tm);
		bf->valid=true;

		bf->collide=(cmesh->objType==obj_rigid)? true :
			(cmesh->getParticle(bf->idx[0])->collide &&
			 cmesh->getParticle(bf->idx[1])->collide &&
			 cmesh->getParticle(bf->idx[2])->collide);
		bf->selfCollide=(cmesh->objType==obj_rigid)? false :
			(cmesh->getParticle(bf->idx[0])->selfCollide &&
			 cmesh->getParticle(bf->idx[1])->selfCollide &&
			 cmesh->getParticle(bf->idx[2])->selfCollide);
		bf->checkIntersections=(cmesh->objType==obj_rigid)? true :
			(cmesh->getParticle(bf->idx[0])->checkIntersections &&
			 cmesh->getParticle(bf->idx[1])->checkIntersections &&
			 cmesh->getParticle(bf->idx[2])->checkIntersections);

		bf->flags=normalFlags(bf->normal);
	}
	if (rebuildAll) cmesh->faceTree.build();
	else cmesh->faceTree.refit();
}


END_VLADO

#endif