/*****************************************************
	FILE: box.h
	DESCRIPTION: Axis-aligned box in 3D
	Copyright (C) 2001 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __BOX_H__
#define __BOX_H__

BEGIN_VLADO

class Box {
	public:
		Vector pmin, pmax;

		Box(void) {}
		Box(Vector &p) { pmin=pmax=p; }

		void init(void) {
			pmin.set(1e20f, 1e20f, 1e20f);
			pmax.set(-1e20f, -1e20f, -1e20f);
		}

		void makeInfinite(void) {
			pmin.set(-1e20f, -1e20f, -1e20f);
			pmax.set(1e20f, 1e20f, 1e20f);
		}

		void init(Vector &p) { pmin=pmax=p; }

		real dx() const {
			return pmax.x-pmin.x;
		}
		
		real dy() const {
			return pmax.y-pmin.y;
		}
		
		real dz() const {
			return pmax.z-pmin.z;
		}

		// Add a point to the box
		void operator +=(const Vector &p) {
			if (p[0]<pmin[0]) pmin[0]=p[0];
			if (p[0]>pmax[0]) pmax[0]=p[0];

			if (p[1]<pmin[1]) pmin[1]=p[1];
			if (p[1]>pmax[1]) pmax[1]=p[1];

			if (p[2]<pmin[2]) pmin[2]=p[2];
			if (p[2]>pmax[2]) pmax[2]=p[2];
		}

		// Add a box to the box
		void operator +=(const Box &b) {
			if (b.pmin[0]<pmin[0]) pmin[0]=b.pmin[0];
			if (b.pmax[0]>pmax[0]) pmax[0]=b.pmax[0];

			if (b.pmin[1]<pmin[1]) pmin[1]=b.pmin[1];
			if (b.pmax[1]>pmax[1]) pmax[1]=b.pmax[1];

			if (b.pmin[2]<pmin[2]) pmin[2]=b.pmin[2];
			if (b.pmax[2]>pmax[2]) pmax[2]=b.pmax[2];
		}

		// Intersect the box with a box
		void operator &=(const Box &b) {
			for (int i=0; i<3; i++) {
				if (b.pmin[i]>pmin[i]) pmin[i]=b.pmin[i];
				if (b.pmax[i]<pmax[i]) pmax[i]=b.pmax[i];
			}
		}

		// Returns true if the given point is in the box
		bool isInside(const Vector &p, real tolerance=0.0f) const {
			if (pmin.x-tolerance<=p.x && p.x<=pmax.x+tolerance && 
				pmin.y-tolerance<=p.y && p.y<=pmax.y+tolerance &&
				pmin.z-tolerance<=p.z && p.z<=pmax.z+tolerance) return true;
			return false;
		}

		// Returns true if the given box is inside this one
		bool isInside(const Box &b) const {
			return
				(b.pmin[0]>=pmin[0] && b.pmax[0]<=pmax[0]) &&
				(b.pmin[1]>=pmin[1] && b.pmax[1]<=pmax[1]) &&
				(b.pmin[2]>=pmin[2] && b.pmax[2]<=pmax[2]);
		}

		// Returns true if the given box intersects this one
		bool intersect(const Box &b) const {
			if (pmax.x<b.pmin.x || pmin.x>b.pmax.x) return false;
			if (pmax.y<b.pmin.y || pmin.y>b.pmax.y) return false;
			if (pmax.z<b.pmin.z || pmin.z>b.pmax.z) return false;
			return true;
		}

		bool intersectTolerance(const Box &b, real tolerance) const {
			if (pmax.x+tolerance<b.pmin.x || pmin.x-tolerance>b.pmax.x) return false;
			if (pmax.y+tolerance<b.pmin.y || pmin.y-tolerance>b.pmax.y) return false;
			if (pmax.z+tolerance<b.pmin.z || pmin.z-tolerance>b.pmax.z) return false;
			return true;
		}

		Vector width(void) const { return pmax-pmin; }
		Vector center(void) const { return (pmin+pmax)*0.5f; }

		void expand(real dist) {
			pmin.x-=dist;
			pmin.y-=dist;
			pmin.z-=dist;

			pmax.x+=dist;
			pmax.y+=dist;
			pmax.z+=dist;
		}

		void _scale(real f) {
			for (int i=0; i<3; i++) {
				pmin[i]-=(1.0f+fabsf(pmin[i]))*f;
				pmax[i]+=(1.0f+fabsf(pmax[i]))*f;
			}
		}

		real volume(void) const {
			return (pmax[0]-pmin[0])*(pmax[1]-pmin[1])*(pmax[2]-pmin[2]);
		}

		int maxDimension(void) const {
			return maxComponent(width());
		}

		bool empty(real tolerance=0.0f) const {
			if (pmin[0]>pmax[0]+tolerance) return true;
			if (pmin[1]>pmax[1]+tolerance) return true;
			if (pmin[2]>pmax[2]+tolerance) return true;
			return false;
		}

		bool empty(int dim, real tolerance=0.0f) const {
			return pmin[dim]>pmax[dim]+tolerance;
		}

		Vector operator[](int corner) const {
			switch (corner) {
				case 0: return pmin;
				case 1: return Vector(pmax.x, pmin.y, pmin.z);
				case 2: return Vector(pmin.x, pmax.y, pmin.z);
				case 3: return Vector(pmax.x, pmax.y, pmin.z);
				case 4: return Vector(pmin.x, pmin.y, pmax.z);
				case 5: return Vector(pmax.x, pmin.y, pmax.z);
				case 6: return Vector(pmin.x, pmax.y, pmax.z);
				case 7: return Vector(pmax.x, pmax.y, pmax.z);
			}
			return pmin;
		}

		Vector clamp(const Vector &p) const {
			return Vector(
				Vlado::clamp(p.x, pmin.x, pmax.x),
				Vlado::clamp(p.y, pmin.y, pmax.y),
				Vlado::clamp(p.z, pmin.z, pmax.z));
		}

		int compare(const Box &b) const {
			int res=0;
			if (b.pmin[0]!=pmin[0]) res|=1<<16;
			if (b.pmax[0]!=pmax[0]) res|=1<<(16+4);
			if (b.pmin[1]!=pmin[1]) res|=1<<17;
			if (b.pmax[1]!=pmax[1]) res|=1<<(17+4);
			if (b.pmin[2]!=pmin[2]) res|=1<<18;
			if (b.pmax[2]!=pmax[2]) res|=1<<(18+4);
			return res;
		}

		INLINE Vector &c(int i) { return (&pmin)[i]; }
};

INLINE Box transform(const Box &b, const Vlado::Transform &tm) {
	Box r;
	r.init();
	for (int i=0; i<8; i++) r+=tm*b[i];
	return r;
}

INLINE int compareBBoxes(const Box &b0, const Box &b1) {
	return b0.compare(b1);
}

END_VLADO

#endif
