#include "Headers.h"
#include "resource.h"
#include "Utils.h"
#include "ToMax.h"
#include "SimClothModifier.h"
#include "SimClothToMax.h"
#include "VertexGroup.h"

VertexGroupClassDesc vertexGroupClassDesc;
ClassDesc* GetVertexGroupDesc() { return &vertexGroupClassDesc;}

//**************************************************************
// Group parameter block
ParamBlockDesc2 group_param_blk (sc3_group, _T("VertexGroupParams"),  IDS_PARAMS_GROUP, &vertexGroupClassDesc, 
	P_AUTO_CONSTRUCT,
	0, // Reference ID

	sc3_group_name, _T("name"), TYPE_STRING, 0, ids_group_name,
	end,
	sc3_group_selfCollide, _T("selfCollisions"), TYPE_BOOL, P_ANIMATABLE, ids_group_selfCollide,
		p_default, TRUE,
		p_ui, TYPE_SINGLECHEKBOX, cb_group_selfCollide,
	end,
	sc3_group_collide, _T("collisions"), TYPE_BOOL, P_ANIMATABLE, ids_group_collide,
		p_default, TRUE,
		p_ui, TYPE_SINGLECHEKBOX, cb_group_collide,
	end,
	sc3_group_attached, _T("attached"), TYPE_BOOL, P_ANIMATABLE, ids_group_attached,
		p_default, FALSE,
		p_ui, TYPE_SINGLECHEKBOX, cb_group_attached,
		p_enable_ctrls, 1, sc3_group_attachNode,
	end,
	sc3_group_attachNode, _T("attachNode"), TYPE_INODE, 0, ids_group_attachNode,
		p_ui, TYPE_PICKNODEBUTTON, bn_group_pickNode,
	end,
	sc3_group_checkIntersections, _T("checkIntersections"), TYPE_BOOL, P_ANIMATABLE, ids_group_checkIntersections,
		p_default, TRUE,
		p_ui, TYPE_SINGLECHEKBOX, cb_group_checkIntersections,
	end,
	sc3_group_scale, _T("scale"), TYPE_FLOAT, P_ANIMATABLE, ids_group_scale,
		p_default, 1.0f,
		p_range, 0.000001f, 1000000.0f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_group_scale, sp_group_scale, 0.1f,
	end,
end
);

//**************************************************************
// VertexGroup
VertexGroup::VertexGroup(void) {
	numPts=0;
	selected=NULL;
	pblock=NULL;
	partCoords=NULL;

	vertexGroupClassDesc.MakeAutoParamBlocks(this);
}

VertexGroup::~VertexGroup(void) {
	freeData();
}

//******************************************************
// Cloning
RefTargetHandle VertexGroup::Clone(RemapDir& remap) {
	VertexGroup* newgroup=new VertexGroup();
	newgroup->ReplaceReference(0, pblock->Clone(remap));
	return newgroup;
}

//******************************************************
// Change notification
RefResult VertexGroup::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message) {
	switch (message) {
		case REFMSG_CHANGE:
			break;
		case REFMSG_TARGET_DELETED:
			if (hTarget==pblock) pblock=NULL;
			break;
	}
	return REF_SUCCEED;
}

//**************************************************************
// Sub-anims
int VertexGroup::NumSubs() { return 1; }
Animatable* VertexGroup::SubAnim(int i) { return pblock; }
TSTR VertexGroup::SubAnimName(int i) { return pblock->GetStr(sc3_group_name); }

//**************************************************************
// Loading and saving
#define SELECTION_CHUNK 0x100

IOResult VertexGroup::Load(ILoad *iload) {
	IOResult res;
	ULONG nbytes;

	numPts=0;
	if (selected) delete[] selected;
	selected=NULL;

	while (IO_OK==(res=iload->OpenChunk())) {
		switch(iload->CurChunkID())  {
			case SELECTION_CHUNK: {
				res=iload->Read(&numPts, sizeof(numPts), &nbytes);
				if (selected) delete[] selected;
				selected=new unsigned char[numPts];
				res=iload->Read(selected, sizeof(selected[0])*numPts, &nbytes);
				break;
			}
		}
		iload->CloseChunk();
		if (res!=IO_OK) return res;
	}

	return IO_OK;
}

IOResult VertexGroup::Save(ISave *isave) {
	IOResult res=IO_OK;
	ULONG nbytes;

	isave->BeginChunk(SELECTION_CHUNK);
	res=isave->Write(&numPts, sizeof(numPts), &nbytes);
	res=isave->Write(selected, sizeof(selected[0])*numPts, &nbytes);
	isave->EndChunk();

	return IO_OK;
}

//**************************************************************
// Other methods
void VertexGroup::beginSimulation(TimeValue t, SimClothObject *scObj) {
	attachNode=pblock->GetINode(sc3_group_attachNode);
	if (attachNode) {
		Matrix3 tm=attachNode->GetObjTMBeforeWSM(t);
		Vlado::Transform itm=Vlado::inverse(toTransform(tm));
		// Save the particle coordinates relative to the node
		partCoords=new Vlado::Vector[scObj->numParts];
		for (int i=0; i<scObj->numParts; i++) {
			partCoords[i]=itm*(scObj->parts[i].getPos());
		}
	}
	prevAttached=true;
}

void VertexGroup::endSimulation(void) {
	if (partCoords) delete[] partCoords;
	partCoords=NULL;
}

void VertexGroup::updateSimulation(TimeValue t, SimClothObject *scObj) {
	collide=pblock->GetInt(sc3_group_collide, t);
	selfCollide=pblock->GetInt(sc3_group_selfCollide, t);
	prevAttached=attached;
	attached=pblock->GetInt(sc3_group_attached, t);
	checkIntersections=pblock->GetInt(sc3_group_checkIntersections, t);
	scale=pblock->GetFloat(sc3_group_scale, t);

	if (attachNode) {
		tm=toTransform(attachNode->GetObjTMBeforeWSM(t));
		if (attached && !prevAttached) {
			Vlado::Transform itm=Vlado::inverse(tm);
			// Save the particle coordinates relative to the node
			for (int i=0; i<scObj->numParts; i++) {
				partCoords[i]=itm*(scObj->parts[i].getPos());
			}
		}
	}
}

void VertexGroup::setSelected(int numPoints, unsigned char *selPts) {
	if (selected && numPoints!=numPts) {
		delete[] selected;
		selected=NULL;
	}

	if (!selected) selected=new unsigned char[numPoints];
	numPts=numPoints;
	memcpy(selected, selPts, numPoints*(sizeof(selPts[0])));
}

void VertexGroup::setName(TCHAR *newName) {
	pblock->SetValue(sc3_group_name, 0, newName);
}

TCHAR *VertexGroup::getName(void) {
	return pblock->GetStr(sc3_group_name);
}

void VertexGroup::freeData(void) {
	if (selected) delete[] selected;
	selected=NULL;
	numPts=0;
	if (partCoords) delete[] partCoords;
	partCoords=NULL;
}

int VertexGroup::getAttachPos(int index, Vlado::Vector &p) {
	if (!attached || !attachNode || !selected[index]) return false;
	p=tm*partCoords[index];
	return true;
}
