/******************************************************
	FILE: SimClothToMax.h
	DESCRIPTION: This is header file for the connection between the SimCloth engine and the Max modifier
	CREATED: 1st December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) 2000-2002 by Vladimir Koylazov

	HISTORY:
		December 1st: file created

******************************************************/

#ifndef __SIMCLOTHTOMAX_H__
#define __SIMCLOTHTOMAX_H__

#include "ClothForces.h"
#include "EdgeLister.h"
#include "resource.h"

class SpringsArray {
	float *ks;
	float *kd;
	float *restLen;

	int numSprings;
	int idx;

	void doSprings4(int i);
	void doSprings1(int i);
public:
	float *scale;
	Vlado::Particle<DynamicObject> **p0, **p1;

	SpringsArray(void) {
		ks=kd=restLen=scale=NULL;
		p0=p1=NULL;
		idx=numSprings=0;
	}
	~SpringsArray(void) {
		freeMem();
	}

	void freeMem(void) {
		if (ks) _mm_free(ks);
		if (kd) _mm_free(kd);
		if (restLen) _mm_free(restLen);
		if (scale) _mm_free(scale);
		if (p0) delete[] p0;
		if (p1) delete[] p1;

		ks=kd=restLen=scale=NULL;
		p0=p1=NULL;
		idx=numSprings=0;
	}

	void setCount(int count) {
		freeMem();
		numSprings=count;
		ks=(float*) _mm_malloc(numSprings*sizeof(float), 16);
		kd=(float*) _mm_malloc(numSprings*sizeof(float), 16);
		restLen=(float*) _mm_malloc(numSprings*sizeof(float), 16);
		scale=(float*) _mm_malloc(numSprings*sizeof(float), 16);
		p0=new Vlado::Particle<DynamicObject>*[numSprings];
		p1=new Vlado::Particle<DynamicObject>*[numSprings];
		idx=0;
	}

	void addSpring(Vlado::Particle<DynamicObject> &p0, Vlado::Particle<DynamicObject> &p1, float ks, float kd) {
		float len=Vlado::length(p0.getPos()-p1.getPos());
		if (len<1e-6f) return;

		restLen[idx]=len;
		this->p0[idx]=&p0;
		this->p1[idx]=&p1;
		this->ks[idx]=ks;
		this->kd[idx]=kd;
		scale[idx]=1.0f;
		idx++;
	}

	void applyForces(int useSSE=false);

	int count(void) { return idx; }
};

class SimClothObject: public DynamicObject {
	public:
		// Some Max-specific data
		INode *node;
		SimCloth3 *mod;
		SimClothLocalData *localData;

		// Properties of the object
		int stretchType;
		float springsStiffness;
		float springsDamping;

		float stretch_stiffness;
		float shear_stiffness;
		float stretchShear_damping;

		int bendEnabled;
		int bendType;
		int skipSelectedEdges;
		float bendSpringStiffness;
		float bendSpringDamping;

		float bendAngleStiffness;
		float bendAngleDamping;

		float friction;
		float particleMass;

		Matrix3 tm;
		int deflector;
		int rigid;
		int selfCollideOn;

		float airDrag;
		int useSSE;

		TimeValue worldTime;

		// The internal forces of the object
		// int numSprings;
		// Vlado::SpringForce<Vlado::Particle<DynamicObject> > *springs;
		SpringsArray springsArray;

		int numBends;
		Vlado::BendForce<Vlado::Particle<DynamicObject> > *bends;

		int numStretchShears;
		Vlado::StretchShearForce<Vlado::Particle<DynamicObject> > *stretchShears;

		Vlado::Table<ForceField*> forceFields;

		// The geometry of the object
		int numVerts;
		Vlado::Vector *objVerts; // Vertices of the mesh in object space
		Vlado::Vector *vertexNormals;

		int numEdges;
		Vlado::EdgeData *edges;
		Vlado::Vector *edgeNormals;

		int numFaces;
		Vlado::FaceData *faces;
		Vlado::ForceTriplet *triplets;

		SimClothObject(void);
		~SimClothObject(void);

		void init(INode *node, SimCloth3 *mod, SimClothLocalData *localData);
		void freeData(void);

		void beginSimulation(TimeValue t);
		void updateSimulation(TimeValue t);
		void endSimulation(void);
		void saveCache(TimeValue t);

		// From DynamicObject
		void applyForces(float dt, int firstCall);
		float getFriction(void) { return friction; }

		// Support for EdgeLister
		Mesh *theMesh;
		void setNumEdges(int numEdges) {
			this->numEdges=numEdges;
			this->edges=new Vlado::EdgeData[numEdges];
		}
		void setEdge(int i, Vlado::EdgeData &edge) {
			edges[i]=edge;
		}
		void setFaceEdge(int face, int edgeFace, int edgeIdx) {}

		int getNumVerts(void) { return numVerts; }
		int getNumEdges(void) { return numEdges; }
		int getNumFaces(void) { return numFaces; }
		int getEdgeVertex(int edge, int vertex) { return edges[edge].v[vertex]; }
		int getFaceVertex(int face, int vertex) { return faces[face].v[vertex]; }
		Vlado::Vector getVertexObjPos(int vertIdx) { return objVerts[vertIdx]; }
		int invisibleEdge(int edgeIdx) {
			if (!deflector && !rigid) return false;
			if (edges[edgeIdx].numFaces<2) return false;
			Vlado::Vector &n0=triplets[edges[edgeIdx].f[0]].normal;
			Vlado::Vector &n1=triplets[edges[edgeIdx].f[1]].normal;
			return (n0*n1)>0.99f;
		}
		Vlado::Vector vertexNormal(int vertexIndex) {
			return vertexNormals[vertexIndex];
		}
		Vlado::Vector edgeNormal(int edgeIndex) {
			return edgeNormals[edgeIndex];
		}

		void buildNormals(void);

		// Other methods
		void prepareMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
		void updateMesh(TriObject *tri, Mesh &mesh, Matrix3 &tm);
		void updateForceScale(void);

		int getParticleSelfCollide(int index);
		int getParticleCollide(int index);
		int getParticleCheckIntersections(int index);
};

class ClothObject: public SimClothObject {
	public:
		ClothObject(INode *node, SimCloth3 *mod, SimClothLocalData *localData) { init(node, mod, localData); }

		Vlado::Vector vertexPos(int i) { return parts[i].p; }
		Vlado::Vector vertexVel(int i) { return parts[i].v; }
		Vlado::Vector vertexPosPrev(int i) { return parts[i].pp; }
		int isVertexFixed(int i) { return parts[i].invMass<1e-6f; }
		int selfCollide(void) { return selfCollideOn; }

		void prepareCollisionDetection(void) { buildNormals(); }
		int isDeflector(void) { return false; }
		int isRigidBody(void) { return false; }
};

class DeflectorObject: public SimClothObject {
	public:
		DeflectorObject(INode *node, SimCloth3 *mod, SimClothLocalData *localData) { init(node, mod, localData); }

		Vlado::Vector vertexPos(int i) { return parts[i].p; }
		Vlado::Vector vertexVel(int i) { return parts[i].v; }
		Vlado::Vector vertexPosPrev(int i) { return parts[i].pp; }
		int isVertexFixed(int i) { return true; }
		int selfCollide(void) { return false; }

		void prepareCollisionDetection(void) {}
		int isDeflector(void) { return true; }
		int isRigidBody(void) { return false; }
};

class RigidObject: public SimClothObject {
	public:
		Vlado::Transform posTm, velTm, prevPosTm;

		RigidObject(INode *node, SimCloth3 *mod, SimClothLocalData *localData) { init(node, mod, localData); }

		Vlado::Vector vertexPos(int i) { return posTm*getVertexObjPos(i); }
		Vlado::Vector vertexVel(int i) { return velTm*getVertexObjPos(i); }
		Vlado::Vector vertexPosPrev(int i) { return prevPosTm*getVertexObjPos(i); }
		int isVertexFixed(int i) { return false; }
		int selfCollide(void) { return false; }

		void prepareCollisionDetection(void) {
			posTm=Vlado::Transform(Vlado::Matrix(parts[1].p-parts[0].p, parts[2].p-parts[0].p, parts[3].p-parts[0].p), parts[0].p);
			velTm=Vlado::Transform(Vlado::Matrix(parts[1].v-parts[0].v, parts[2].v-parts[0].v, parts[3].v-parts[0].v), parts[0].v);
			prevPosTm=Vlado::Transform(Vlado::Matrix(parts[1].pp-parts[0].pp, parts[2].pp-parts[0].pp, parts[3].pp-parts[0].pp), parts[0].pp);
			// buildNormals();
		}
		int isDeflector(void) { return false; }
		int isRigidBody(void) { return true; }
};

class SimClothToMax: public SimClothConnection {
	private:
		Vlado::Table<SimClothObject*> simClothObjects;
		HWND hWnd;
		int substeps;
	public:
		HFONT hFont;
		int abort;

		// SimClothEngine connection
		int numDynamicObjects(void) { return simClothObjects.count(); }
		DynamicObject* getDynamicObject(int i) { return simClothObjects[i]; }
		void getSimClothEngineParams(SimClothEngineParams &params);

		void displayMessage(TCHAR *str) { SetWindowText(GetDlgItem(hWnd, st_stats), str); }
		void setFrameProgressRange(int minRange, int maxRange) {
			SendMessage(GetDlgItem(hWnd, pb_frame), PBM_SETRANGE32, minRange, maxRange);
			substeps=maxRange;
		}
		void setFrameProgressPos(int progressPos) {
			SendMessage(GetDlgItem(hWnd, pb_frame), PBM_SETPOS, progressPos, 0);
			TCHAR str[512];
			_stprintf(str, "Current frame (substep %i of %i)", progressPos, substeps);
			SetWindowText(GetDlgItem(hWnd, st_frame), str);
		}
		void setNumCollisions(int collisions) {
			TCHAR str[512];
			_stprintf(str, "%i", collisions);
			SetWindowText(GetDlgItem(hWnd, st_collisions), str);
		}
		void setNumCGIterations(int iterations) {
			TCHAR str[512];
			_stprintf(str, "%i", iterations);
			SetWindowText(GetDlgItem(hWnd, st_cgIterations), str);
		}
		int getAbort(void) {
			if (GetAsyncKeyState(VK_ESCAPE)) return true;
			if (abort) return true;
			if (!GetCOREInterface()->CheckMAXMessages()) return true;
			return false;
		}


		// Other methods
		int simulate(SimClothEngine *scEngine);
		void init(void);
		void freeData(void);

		void listNodes(INode *root);
};

#endif