/********************************************************
	FILE: SimClothModifier.cpp
	DESCRIPTION: The Max modifier part of simcloth3
	CREATED: 1 December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) by Vladimir Koylazov (Vlado)

	HISTORY:
		December 1st: file created

*********************************************************/

#include "Headers.h"
#include "resource.h"
#include "Utils.h"
#include "VertexGroup.h"
#include "SimClothModifier.h"
#include "SimClothToMax.h"

extern TCHAR *GetString(int id);

IObjParam* SimCloth3::ip=NULL;

SimCloth3ClassDesc simCloth3ClassDesc;
ClassDesc* GetSimCloth3Desc() { return &simCloth3ClassDesc;}

static SimCloth3DlgProc globalsDlgProc; // The dialog procedure for the rollups to handle buttons
static SimClothEngine simClothEngine; // The SimCloth engine
static SimClothToMax simClothConnection; // The connection between Max and the SimCloth engine

// Spacewarp validator
class WarpValidator: public PBValidator {
	public:
		BOOL Validate(PB2Value& v) {
			INode *node=(INode*) v.r;
			Object *obj=node->GetObjectRef()->FindBaseObject();
			if (obj->SuperClassID()==WSM_OBJECT_CLASS_ID) return TRUE;
			
			return FALSE;
		}
} warpValidator;

// Subobject levels
const TCHAR *subObjNames[]={ "Vertex" };
int numSubObjLevels=1;

//******************************************************
// Function interface
#if GET_MAX_RELEASE(VERSION_3DSMAX) >= 4000

static FPInterfaceDesc sc3_mixininterface(
	SC3_INTERFACE, // The unique ID of the interface
	_T("SimCloth3"), // Name of the interface
	0, // Resource description
	&simCloth3ClassDesc, // ClassDesc
	FP_MIXIN, // flags
	sc3_calculate, _T("calculate"), 0, TYPE_VOID, 0, 0, 
end 
); 

FPInterfaceDesc* SC3Interface::GetDesc() {
	return &sc3_mixininterface;
}

#endif

//******************************************************
// Parameter blocks

// The "General" parameter block
ParamBlockDesc2 sc3_general_pblock(sc3_general, _T("generalParams"),  0, &simCloth3ClassDesc, P_AUTO_CONSTRUCT + P_AUTO_UI, REFNO_PBLOCK_GENERAL, 
	// Rollout
	IDD_GENERAL, IDS_GENERAL, 0, 0, NULL,
	// Params
	sc3_general_type, _T("type"), TYPE_INT, 0, ids_general_type,
		p_default, 0,
		p_range, 0, 1,
		p_ui, TYPE_RADIO, 3, rb_general_type_deflector, rb_general_type_cloth, rb_general_type_rigid,
	end,
	sc3_general_smoothResult, _T("smoothResult"), TYPE_BOOL, 0, ids_general_type,
		p_default, FALSE,
		p_ui, TYPE_SINGLECHEKBOX, cb_general_smoothResult,
	end,
	sc3_general_smoothIterations, _T("smoothIterations"), TYPE_INT, 0, ids_general_type,
		p_default, 1,
		p_range, 1, 10,
		p_ui, TYPE_SPINNER, EDITTYPE_INT, ed_general_iterations, sp_general_iterations, 1.0f,
	end,
	sc3_general_particleMass, _T("particleMass"), TYPE_FLOAT, 0, ids_tobedone,
		p_default, 1.0f,
		p_range, 1e-6f, 1e6f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_general_particleMass, sp_general_particleMass, SPIN_AUTOSCALE,
	end,
end
);

// The "Global" parameter block; this is a class-level parameter block, that is, the parameters
// are shared between all instances of the plugin class. It's not automatically saved with the Max file
// so explicit saving is necessary
ParamBlockDesc2 sc3_global_pblock(sc3_general, _T("globalParams"),  0, &simCloth3ClassDesc, P_CLASS_PARAMS+P_AUTO_UI,
	// Rollout
	IDD_GLOBAL, IDS_GLOBAL, 0, 0, &globalsDlgProc,
	// Params

	// Simulation time extents
	sc3_global_useActiveTimeSegment, _T("useActiveTimeSegment"), TYPE_BOOL, 0, ids_tobedone,
		p_default, TRUE,
		p_ui, TYPE_SINGLECHEKBOX, cb_global_activeTimeSegment,
	end,

	sc3_global_startFrame, _T("startFrame"), TYPE_INT, 0, ids_global_startFrame,
		p_default, 0,
		p_range, TIME_NegInfinity, TIME_PosInfinity,
		p_ui, TYPE_SPINNER, EDITTYPE_INT, ed_global_startFrame, sp_global_startFrame, 1.0f,
	end,

	sc3_global_endFrame, _T("endFrame"), TYPE_INT, 0, ids_global_endFrame,
		p_default, 100,
		p_range, TIME_NegInfinity, TIME_PosInfinity,
		p_ui, TYPE_SPINNER, EDITTYPE_INT, ed_global_endFrame, sp_global_endFrame, 1.0f,
	end,

	// Timestep control
	sc3_global_substeps, _T("substeps"), TYPE_INT, 0, ids_global_substeps,
		p_default, 10,
		p_range, 1, 1000,
		p_ui, TYPE_SPINNER, EDITTYPE_INT, ed_global_substeps, sp_global_substeps, 1.0f,
	end,

	sc3_global_adaptiveSubdivs, _T("adaptiveSubdivs"), TYPE_INT, 0, ids_global_adaptiveSubdivs,
		p_default, 3,
		p_range, 0, 100,
		p_ui, TYPE_SPINNER, EDITTYPE_INT, ed_global_adaptiveSubdivs, sp_global_adaptiveSubdivs, 1.0f,
	end,

	// Gravity
	sc3_global_gravity, _T("gravity"), TYPE_FLOAT, P_ANIMATABLE, ids_global_gravity,
		p_default, 500.0f,
		p_range, -1e9f, 1e9f,
		p_dim, stdWorldDim,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_global_gravity, sp_global_gravity, SPIN_AUTOSCALE,
	end,

	// Collisions
	sc3_global_collisionTolerance, _T("collisionTolerance"), TYPE_FLOAT, 0, ids_global_collisionTolerance,
		p_default, 3.0f,
		p_range, 1e-6f, 1e6f,
		p_dim, stdWorldDim,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_global_collisionTolerance, sp_global_collisionTolerance, SPIN_AUTOSCALE,
	end,

	sc3_global_checkIntersections, _T("checkForIntersection"), TYPE_BOOL, 0, ids_checkIntersections,
		p_default, FALSE,
		p_ui, TYPE_SINGLECHEKBOX, cb_global_checkIntersections,
	end,

	// Solver
	sc3_global_solverPrecision, _T("solverPrecision"), TYPE_FLOAT, 0, ids_global_solverPrecision,
		p_default, 0.01f,
		p_range, 1e-6f, 1e6f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_global_solverPrecision, sp_global_solverPrecision, SPIN_AUTOSCALE,
	end,

	sc3_global_useSSE, _T("useSSEinstructions"), TYPE_BOOL, 0, ids_global_useSSE,
		p_default, FALSE,
		p_ui, TYPE_SINGLECHEKBOX, cb_global_useSSE,
	end,
end
);

// The "Integrity" parameter block
ParamBlockDesc2 sc3_integrity_pblock(sc3_integrity, _T("Integrity"),  0, &simCloth3ClassDesc, P_AUTO_CONSTRUCT + P_AUTO_UI, REFNO_PBLOCK_INTEGRITY, 
	// Rollout
	IDD_FORCES, IDS_INTEGRITY, 0, 0, NULL,
	// Params

	// Type of stretch forces
	sc3_integrity_stretchType, _T("stretchType"), TYPE_INT, 0, ids_stretch_type,
		p_default, 0,
		p_range, 0, 1,
		p_ui, TYPE_RADIO, 2, rb_springs, rb_stretchShear,
	end,

	// Stretch spring forces
	sc3_integrity_springsStiffness, _T("springsStiffness"), TYPE_FLOAT, 0, ids_springs_stiffness,
		p_default, 100.0f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_springs_stiffness, sp_springs_stiffness, SPIN_AUTOSCALE,
	end,
	sc3_integrity_springsDamping, _T("springsDamping"), TYPE_FLOAT, 0, ids_springs_stiffness,
		p_default, 0.1f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_springs_damping, sp_springs_damping, SPIN_AUTOSCALE,
	end,

	// Stretch/shear forces
	sc3_integrity_stretch_stiffness, _T("stretch_stiffness"), TYPE_FLOAT, 0, ids_springs_stiffness,
		p_default, 100.0f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_stretch_stiffness, sp_stretch_stiffness, SPIN_AUTOSCALE,
	end,
	sc3_integrity_shear_stiffness, _T("shear_stiffness"), TYPE_FLOAT, 0, ids_springs_stiffness,
		p_default, 50.0f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_shear_stiffness, sp_shear_stiffness, SPIN_AUTOSCALE,
	end,
	sc3_integrity_stretchShear_damping, _T("stretchShear_damping"), TYPE_FLOAT, 0, ids_springs_stiffness,
		p_default, 0.1f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_stretchShear_damping, sp_stretchShear_damping, SPIN_AUTOSCALE,
	end,

	// Bend forces
	sc3_integrity_bendEnabled, _T("bendEnabled"), TYPE_BOOL, 0, ids_stretch_type,
		p_default, FALSE,
		p_ui, TYPE_SINGLECHEKBOX, cb_integrity_bendEnabled,
	end,
	sc3_integrity_bendType, _T("bendType"), TYPE_INT, 0, ids_stretch_type,
		p_default, 1,
		p_range, 0, 1,
		p_ui, TYPE_RADIO, 2, rb_bend_springs, rb_bend_angle,
	end,
	sc3_integrity_excludeSelectedEdges, _T("excludeSelectedEdges"), TYPE_BOOL, 0, ids_excludeSelectedEdges,
		p_default, FALSE,
		p_ui, TYPE_SINGLECHEKBOX, cb_integrity_bendExcludeSelected,
	end,

	// Bend springs
	sc3_integrity_bendSpring_stiffness, _T("bendSpringsStiffness"), TYPE_FLOAT, 0, ids_springs_stiffness,
		p_default, 10.0f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_bendSprings_stiffness, sp_bendSprings_stiffness, SPIN_AUTOSCALE,
	end,
	sc3_integrity_bendSpring_damping, _T("bendSpringsdamping"), TYPE_FLOAT, 0, ids_tobedone,
		p_default, 0.1f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_bendSprings_damping, sp_bendSprings_damping, SPIN_AUTOSCALE,
	end,

	// Bend angle
	sc3_integrity_bendAngle_stiffness, _T("bendAngleStiffness"), TYPE_FLOAT, 0, ids_tobedone,
		p_default, 10.0f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_bendAngle_stiffness, sp_bendAngle_stiffness, SPIN_AUTOSCALE,
	end,
	sc3_integrity_bendAngle_damping, _T("bendAngledamping"), TYPE_FLOAT, 0, ids_tobedone,
		p_default, 0.1f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_bendAngle_damping, sp_bendAngle_damping, SPIN_AUTOSCALE,
	end,

	// Space warps
	sc3_integrity_spacewarps, _T("spacewarps"), TYPE_INODE_TAB, 0, 0, ids_tobedone,
		p_validator, &warpValidator,
		p_ui, TYPE_NODELISTBOX, lb_spacewarps, bn_spacewarps_pick, 0, bn_spacewarps_remove,
	end,

	// Air drag
	sc3_integrity_airDrag, _T("airDrag"), TYPE_FLOAT, P_ANIMATABLE, ids_tobedone,
		p_default, 0.001f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_integrity_airDrag, sp_integrity_airDrag, SPIN_AUTOSCALE,
	end,
end
);

// The "Collisions" pblock
ParamBlockDesc2 sc3_collisions_pblock(sc3_collisions, _T("Collisions"),  0, &simCloth3ClassDesc, P_AUTO_CONSTRUCT + P_AUTO_UI, REFNO_PBLOCK_COLLISIONS, 
	// Rollout
	IDD_COLLISIONS, IDS_COLLISIONS, 0, 0, NULL,
	// Params
	sc3_collisions_selfCollide, _T("selfCollide"), TYPE_BOOL, 0, ids_stretch_type,
		p_default, TRUE,
		p_ui, TYPE_SINGLECHEKBOX, cb_collisions_selfCollide,
	end,
	sc3_collisions_friction, _T("friction"), TYPE_FLOAT, 0, ids_tobedone,
		p_default, 250.0f,
		p_range, 0.0f, 1e9f,
		p_ui, TYPE_SPINNER, EDITTYPE_FLOAT, ed_collisions_friction, sp_collisions_friction, SPIN_AUTOSCALE,
	end,
end
);

//******************************************************
// The dialog box handler implementation
BOOL SimCloth3DlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	int id=LOWORD(wParam);
	switch (msg) {
		case WM_INITDIALOG:
			// Dialog initialization
			return TRUE;
		case WM_DESTROY:
			// Dialog destruction
			return TRUE;
		case WM_COMMAND:
			// Commands (from buttons mostly)
			int notifyCode=HIWORD(wParam);
			int controlID=LOWORD(wParam);
			HWND controlHWnd=(HWND) lParam;

			switch (controlID) {
				case bn_calculate: {
					mod->calculate();
					break;
				}
				case bn_clear: {
					mod->clearCaches();
					break;
				}
			}

			return TRUE;
	}
	return FALSE;
}

//******************************************************
// Constructor
SimCloth3::SimCloth3() {
	pblockGeneral=NULL;
	pblockIntegrity=NULL;
	pblockCollisions=NULL;
	simCloth3ClassDesc.MakeAutoParamBlocks(this);
	dontModify=false;

	editMod=NULL;

	aboutWnd=NULL;
	subobjWnd=NULL;

	numPts=0;
	selPts=NULL;
}

//******************************************************
// Desctructor
SimCloth3::~SimCloth3() {
	if (selPts) delete[] selPts;
	selPts=NULL;
	numPts=0;
}

//******************************************************
// Plugin identification
void SimCloth3::GetClassName(TSTR& s) {
	s=TSTR(GetString(IDS_CLASSNAME));
}

Class_ID SimCloth3::ClassID() {
	return SIMCLOTH3_CLASS_ID;
}

TCHAR *SimCloth3::GetObjectName() {
	return GetString(IDS_CLASSNAME);
}

//******************************************************
// Invalidate our UI (or the recently changed parameter)
void SimCloth3::InvalidateUI() {
	sc3_general_pblock.InvalidateUI(pblockGeneral->LastNotifyParamID());
	sc3_integrity_pblock.InvalidateUI(pblockIntegrity->LastNotifyParamID());
	sc3_collisions_pblock.InvalidateUI(pblockCollisions->LastNotifyParamID());
}

//******************************************************
// Open and Close dialog UIs
// We ask the ClassDesc2 to handle Beginning and Ending EditParams for us
INT_PTR CALLBACK aboutDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	switch (message) {
		case WM_INITDIALOG:
			return TRUE;
		default: return FALSE;
	}
	return FALSE;
}

void SimCloth3::BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev) {
	this->ip=ip;
	editMod=this;
	subobjWnd=NULL;
	globalsDlgProc.SetThing(this);
	simCloth3ClassDesc.BeginEditParams(ip, this, flags, prev);

#if GET_MAX_RELEASE(VERSION_3DSMAX) < 4000
	ip->RegisterSubObjectTypes(subObjNames, numSubObjLevels);
#endif

	selectMode=new SelectModBoxCMode(this, ip);

	aboutWnd=ip->AddRollupPage(hInstance, MAKEINTRESOURCE(IDD_ABOUT), aboutDlgProc, GetString(IDS_PARAMS_ABOUT), 0, APPENDROLL_CLOSED);

	TimeValue t=ip->GetTime();
	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_BEGIN_EDIT);
	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_MOD_DISPLAY_ON);
	SetAFlag(A_MOD_BEING_EDITED);
}
	
void SimCloth3::EndEditParams( IObjParam *ip, ULONG flags, Animatable *next) {
	if (aboutWnd) ip->DeleteRollupPage(aboutWnd);
	aboutWnd=NULL;

	if (subobjWnd) {
		ip->DeleteRollupPage(subobjWnd);
		subobjWnd=NULL;
	} else simCloth3ClassDesc.EndEditParams(ip, this, flags, next);

	if (selectMode) {
		ip->DeleteMode(selectMode);
		delete selectMode;
		selectMode=NULL;
	}

	editMod=NULL;

	TimeValue t=ip->GetTime();
	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_END_EDIT);
	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_MOD_DISPLAY_OFF);
	ClearAFlag(A_MOD_BEING_EDITED);

	this->ip=NULL;
	globalsDlgProc.SetThing(NULL);
}

void SimCloth3::grayGlobal(IParamMap2 *map) {
	int enableStartEndFrame=!sc3_global_pblock.GetInt(sc3_global_useActiveTimeSegment, 0);
	map->Enable(sc3_global_startFrame, enableStartEndFrame);
	map->Enable(sc3_global_endFrame, enableStartEndFrame);
}

//******************************************************
// Sub-anims
int SimCloth3::NumSubs() {
	return 3+groups.Count();
}

Animatable* SimCloth3::SubAnim(int i) 	{
	switch (i) {
		case 0: return pblockGeneral;
		case 1: return pblockIntegrity;
		case 2: return pblockCollisions;
		default: {
			i-=3;
			if (i>=0 && i<groups.Count()) return groups[i];
			break;
		}
	}
	return NULL;
}

TSTR SimCloth3::SubAnimName(int i) {
	switch (i) {
		case 0: return GetString(IDS_GENERAL);
		case 1: return GetString(IDS_INTEGRITY);
		case 2: return GetString(IDS_COLLISIONS);
		default: {
			i-=3;
			if (i>=0 && i<groups.Count()) return groups[i]->getName();
			break;
		}
	}
	return _T("");
}

//******************************************************
// References
int SimCloth3::NumRefs() {
	return REFNO_GROUPS+groups.Count();
}

RefTargetHandle SimCloth3::GetReference(int i) {
	switch (i) {
		case REFNO_PBLOCK_GENERAL: return pblockGeneral;
		case REFNO_PBLOCK_INTEGRITY: return pblockIntegrity;
		case REFNO_PBLOCK_COLLISIONS: return pblockCollisions;
		default: {
			i-=REFNO_GROUPS;
			if (i>=0 && i<groups.Count()) return groups[i];
			break;
		}
	}
	return NULL;
}

void SimCloth3::SetReference(int i, RefTargetHandle rtarg) {
	switch (i) {
		case REFNO_PBLOCK_GENERAL: pblockGeneral=(IParamBlock2*) rtarg; break;
		case REFNO_PBLOCK_INTEGRITY: pblockIntegrity=(IParamBlock2*) rtarg; break;
		case REFNO_PBLOCK_COLLISIONS: pblockCollisions=(IParamBlock2*) rtarg; break;
		default: {
			i-=REFNO_GROUPS;
			if (i>=0 && i<groups.Count()) groups[i]=(VertexGroup*) rtarg;
			break;
		}
	}
}

//******************************************************
// Cloning
RefTargetHandle SimCloth3::Clone(RemapDir& remap) {	
	SimCloth3* newmod=new SimCloth3();
	newmod->ReplaceReference(REFNO_PBLOCK_GENERAL, pblockGeneral->Clone(remap));
	newmod->ReplaceReference(REFNO_PBLOCK_INTEGRITY, pblockIntegrity->Clone(remap));
	newmod->ReplaceReference(REFNO_PBLOCK_COLLISIONS, pblockCollisions->Clone(remap));

	newmod->groups.SetCount(groups.Count());
	for (int i=0; i<groups.Count(); i++) newmod->ReplaceReference(REFNO_GROUPS+i, groups[i]->Clone(remap));

	return newmod;
}

//******************************************************
// Change notification
RefResult SimCloth3::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID,  RefMessage message) {
	switch (message) {
		case REFMSG_CHANGE:
			break;
	}
	return REF_SUCCEED;
}

//******************************************************
// Parameter blocks access
int	SimCloth3::NumParamBlocks() {
	return 3;
}

IParamBlock2* SimCloth3::GetParamBlock(int i) {
	switch (i) {
		case 0: return pblockGeneral;
		case 1: return pblockIntegrity;
		case 2: return pblockCollisions;
		default: return NULL;
	}
}

IParamBlock2* SimCloth3::GetParamBlockByID(BlockID id) {
	for (int i=NumParamBlocks()-1; i>=0; i--) {
		IParamBlock2 *pblock=GetParamBlock(i);
		if (pblock->ID()==id) return pblock;
	}
	return NULL;
}

int SimCloth3::GetParamBlockIndex(int id) {
	for (int i=NumParamBlocks()-1; i>=0; i--) {
		IParamBlock2 *pblock=GetParamBlock(i);
		if (pblock->ID()==id) return i;
	}
	return -1;
}

//******************************************************
// Validity
Interval SimCloth3::GetValidity(TimeValue t) {
	return Interval(t,t);
	int objectType;
	Interval valid = FOREVER;
	pblockGeneral->GetValue(sc3_general_type, t, objectType, valid);
	if (objectType==0) return FOREVER;

	return valid;
}

Interval SimCloth3::LocalValidity(TimeValue t) {
	return GetValidity(t);
}

Interval SimCloth3::ObjectValidity(TimeValue t) {
	return GetValidity(t);
}

//******************************************************
// File loading and saving
#define KEYS_CHUNK 0x100
#define GLOBALS_CHUNK 0x200
#define GLOBALS1_CHUNK 0x201
#define GLOBALS2_CHUNK 0x202
#define NUM_GROUP_PBLOCKS_CHUNK 0x300

// Load the local data (the point cache) for an object
IOResult SimCloth3::LoadLocalData(ILoad *iload, LocalModData **pld) {
	SimClothLocalData *ld=new SimClothLocalData;

	IOResult res=IO_OK;

	while (IO_OK==(res=iload->OpenChunk())) {
		switch(iload->CurChunkID())  {
			case KEYS_CHUNK: {
				ld->cache.load(iload);
				break;
			}
		}
		iload->CloseChunk();
		if (res!=IO_OK) return res;
	}

	*pld=ld;
	return IO_OK;
}

// Save the local data (the point cache) for an object
IOResult SimCloth3::SaveLocalData(ISave *isave, LocalModData *modData) {
	SimClothLocalData *ld=(SimClothLocalData*) modData;

	IOResult res=IO_OK;

	// The geometry cache
	isave->BeginChunk(KEYS_CHUNK);
	res=ld->cache.save(isave);
	isave->EndChunk();

	return res;
}

IOResult SimCloth3::Save(ISave *isave) {
	// Save the global data
	IOResult res=IO_OK;
	ULONG nbytes;

	// The global parameters - actually these are saved by every modifier in the scene - bit of a waste
	isave->BeginChunk(GLOBALS_CHUNK);

	int startFrame=sc3_global_pblock.GetInt(sc3_global_startFrame);
	int endFrame=sc3_global_pblock.GetInt(sc3_global_endFrame);
	int substeps=sc3_global_pblock.GetInt(sc3_global_substeps);
	int adaptiveSubdivs=sc3_global_pblock.GetInt(sc3_global_adaptiveSubdivs);
	float gravity=sc3_global_pblock.GetFloat(sc3_global_gravity);
	float tolerance=sc3_global_pblock.GetFloat(sc3_global_collisionTolerance);
	float precision=sc3_global_pblock.GetFloat(sc3_global_solverPrecision);
	int activeSegment=sc3_global_pblock.GetInt(sc3_global_useActiveTimeSegment);

	res=isave->Write(&startFrame, sizeof(startFrame), &nbytes);
	res=isave->Write(&endFrame, sizeof(startFrame), &nbytes);
	res=isave->Write(&substeps, sizeof(substeps), &nbytes);
	res=isave->Write(&adaptiveSubdivs, sizeof(adaptiveSubdivs), &nbytes);
	res=isave->Write(&gravity, sizeof(gravity), &nbytes);
	res=isave->Write(&tolerance, sizeof(tolerance), &nbytes);
	res=isave->Write(&precision, sizeof(precision), &nbytes);
	res=isave->Write(&activeSegment, sizeof(activeSegment), &nbytes);

	isave->EndChunk();

	isave->BeginChunk(GLOBALS1_CHUNK);
	int checkIntersections=sc3_global_pblock.GetInt(sc3_global_checkIntersections);
	res=isave->Write(&checkIntersections, sizeof(checkIntersections), &nbytes);
	isave->EndChunk();

	isave->BeginChunk(GLOBALS2_CHUNK);
	int useSSE=sc3_global_pblock.GetInt(sc3_global_useSSE);
	res=isave->Write(&useSSE, sizeof(useSSE), &nbytes);
	isave->EndChunk();

	// Save the number of group parameter blocks
	isave->BeginChunk(NUM_GROUP_PBLOCKS_CHUNK);
	int numGroups=groups.Count();
	res=isave->Write(&numGroups, sizeof(numGroups), &nbytes);
	isave->EndChunk();

	return IO_OK;
}

IOResult SimCloth3::Load(ILoad *iload) {
	// Load the global data
	IOResult res;
	ULONG nbytes;
	while (IO_OK==(res=iload->OpenChunk())) {
		switch(iload->CurChunkID())  {
			case GLOBALS_CHUNK: {
				int startFrame;
				int endFrame;
				int substeps;
				int adaptiveSubdivs;
				float gravity;
				float tolerance;
				float precision;
				int activeSegment;

				res=iload->Read(&startFrame, sizeof(startFrame), &nbytes);
				res=iload->Read(&endFrame, sizeof(startFrame), &nbytes);
				res=iload->Read(&substeps, sizeof(substeps), &nbytes);
				res=iload->Read(&adaptiveSubdivs, sizeof(adaptiveSubdivs), &nbytes);
				res=iload->Read(&gravity, sizeof(gravity), &nbytes);
				res=iload->Read(&tolerance, sizeof(tolerance), &nbytes);
				res=iload->Read(&precision, sizeof(precision), &nbytes);
				res=iload->Read(&activeSegment, sizeof(activeSegment), &nbytes);

				sc3_global_pblock.SetValue(sc3_global_startFrame, 0, startFrame);
				sc3_global_pblock.SetValue(sc3_global_endFrame, 0, endFrame);
				sc3_global_pblock.SetValue(sc3_global_substeps, 0, substeps);
				sc3_global_pblock.SetValue(sc3_global_adaptiveSubdivs, 0, adaptiveSubdivs);
				sc3_global_pblock.SetValue(sc3_global_gravity, 0, gravity);
				sc3_global_pblock.SetValue(sc3_global_collisionTolerance, 0, tolerance);
				sc3_global_pblock.SetValue(sc3_global_solverPrecision, 0, precision);
				sc3_global_pblock.SetValue(sc3_global_useActiveTimeSegment, 0, activeSegment);

				break;
			}
			case GLOBALS1_CHUNK: {
				int checkIntersections;
				res=iload->Read(&checkIntersections, sizeof(checkIntersections), &nbytes);
				sc3_global_pblock.SetValue(sc3_global_checkIntersections, 0, checkIntersections);
				break;
			}
			case GLOBALS2_CHUNK: {
				int useSSE;
				res=iload->Read(&useSSE, sizeof(useSSE), &nbytes);
				sc3_global_pblock.SetValue(sc3_global_useSSE, 0, useSSE);
				break;
			}
			case NUM_GROUP_PBLOCKS_CHUNK: {
				int numGroups;
				res=iload->Read(&numGroups, sizeof(numGroups), &nbytes);
				groups.SetCount(numGroups);
				for (int i=0; i<numGroups; i++) groups[i]=NULL;
				break;
			}
		}
		iload->CloseChunk();
		if (res!=IO_OK) return res;
	}

	return IO_OK;
}

//******************************************************
// Other functions
void SimCloth3::clearNodeCache(INode *root) {
	for (int i=0; i<root->NumberOfChildren(); i++) {
		INode *node=root->GetChildNode(i);
		clearNodeCache(node);

		Object *obj=node->GetObjectRef();
		if(!obj) continue;
		while (obj->SuperClassID()==GEN_DERIVOB_CLASS_ID) {
			IDerivedObject *derivObj=(IDerivedObject*) obj;

			for (int i=0; i<derivObj->NumModifiers(); i++) {
				Modifier *mod=derivObj->GetModifier(i);

				if (mod->ClassID()==SIMCLOTH3_CLASS_ID && (SimCloth3*) mod==this) {
					ModContext *mc=derivObj->GetModContext(i);
					if (mc->localData) ((SimClothLocalData*) (mc->localData))->cache.freeData();
				}
			}
			obj=derivObj->GetObjRef();
		}
	}
}

void SimCloth3::clearCaches(void) {
	Interface *ip=GetCOREInterface();
	int res=MessageBox(ip->GetMAXHWnd(), _T("This will clear the cache for all objects with this modifier. Are you sure?"), _T("SimCloth3"), MB_YESNO | MB_DEFBUTTON2);
	if (res==IDYES) {
		clearNodeCache(ip->GetRootNode());
		NotifyDependents(FOREVER, PART_ALL, REFMSG_CHANGE);
	}
}

int SimCloth3::calculate(void) {
	simClothConnection.init();
	int res=simClothConnection.simulate(&simClothEngine);
	simClothConnection.freeData();

	return res;
}

//******************************************************
// SimClothLocalData
SimClothLocalData::SimClothLocalData(void) {
	InitializeCriticalSection(&csect);
	cachedMesh=NULL;
}

SimClothLocalData::~SimClothLocalData(void) {
	DeleteCriticalSection(&csect);
	freeData();
}

void SimClothLocalData::freeData(void) {
	cache.freeData();

	if (cachedMesh) delete cachedMesh;
	cachedMesh=NULL;
}

void SimClothLocalData::saveMesh(Mesh *mesh) {
	EnterCriticalSection(&csect);

	if (cachedMesh) delete cachedMesh;
	cachedMesh=new Mesh(*mesh);

	LeaveCriticalSection(&csect);
}
