/********************************************************
	FILE: SimClothEdit.cpp
	DESCRIPTION: The object modification part of the simcloth3 modifier
	CREATED: 1 December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) by Vladimir Koylazov (Vlado)
	
	HISTORY:
		
*********************************************************/
		 
#include "Headers.h"
#include "resource.h"
#include "SimClothModifier.h"
#include "TriMesh.h"
#include "toMax.h"
#include "LoopSubdivide.h"

//******************************************************
// Parameter blocks
void fromMaxMesh(Vlado::TriMesh &vmesh, Mesh &maxMesh) {
	vmesh.freeData();

	vmesh.numVerts=maxMesh.numVerts;
	vmesh.verts=new Vlado::Vector[vmesh.numVerts];
	for (int i=0; i<vmesh.numVerts; i++) vmesh.verts[i]=toVector(maxMesh.verts[i]);

	vmesh.numFaces=maxMesh.numFaces;
	vmesh.faces=new Vlado::TriMeshFace[vmesh.numFaces];
	for (i=0; i<vmesh.numFaces; i++) {
		Face &f=maxMesh.faces[i];
		vmesh.faces[i].set(f.v[0], f.v[1], f.v[2]);
	}
	vmesh.buildEdges();
}

void toMaxMesh(Mesh &maxMesh, Vlado::TriMesh &vmesh) {
	maxMesh.setNumVerts(vmesh.numVerts);
	maxMesh.setNumFaces(vmesh.numFaces);

	maxMesh.setNumMapVerts(1, vmesh.numVerts);
	maxMesh.setNumMapFaces(1, vmesh.numFaces);

	UVVert *mapVerts=maxMesh.mapVerts(1);
	TVFace *mapFaces=maxMesh.mapFaces(1);

	for (int i=0; i<vmesh.numVerts; i++) {
		maxMesh.verts[i]=toPoint3(vmesh.verts[i]);
		mapVerts[i]=maxMesh.verts[i];
	}

	for (i=0; i<vmesh.numFaces; i++) {
		Vlado::TriMeshFace &f=vmesh.faces[i];
		maxMesh.faces[i].setVerts(f.v[0], f.v[1], f.v[2]);
		maxMesh.faces[i].setSmGroup(1);
		maxMesh.faces[i].setEdgeVisFlags(EDGE_VIS, EDGE_VIS, EDGE_VIS);
		mapFaces[i].setTVerts(f.v[0], f.v[1], f.v[2]);
	}

	maxMesh.InvalidateTopologyCache();
}

class MeshWrapper {
	public:
		Mesh &mesh;

		MeshWrapper(Mesh &m):mesh(m) {
		}

		int getNumVerts(void) const { return mesh.numVerts; }
		int getNumFaces(void) const { return mesh.numFaces; }

		void setNumVerts(int n) {
			mesh.setNumVerts(n);
			mesh.setNumMapVerts(1, n);
		}
		void setNumFaces(int n) {
			mesh.setNumFaces(n);
			mesh.setNumMapFaces(1, n);
		}

		int getFaceVertex(int face, int vertex) const { return mesh.faces[face].v[vertex]; }

		Vlado::Vector getVert(int i) const { return toVector(mesh.verts[i]); }

		void setAndCopyVert(int i, Vlado::Vector &p, MeshWrapper &fromMesh, int fromIdx) {
			mesh.verts[i]=toPoint3(p);
			bool sup=mesh.mapSupport(1) && fromMesh.mesh.mapSupport(1);
			if (sup) mesh.mapVerts(1)[i]=fromMesh.mesh.mapVerts(1)[fromIdx];
		}

		void setAndSplitVert(int i, Vlado::Vector &p, MeshWrapper &fromMesh, int fromIdx0, int fromIdx1) {
			mesh.verts[i]=toPoint3(p);
			bool sup=mesh.mapSupport(1) && fromMesh.mesh.mapSupport(1);
			if (sup) mesh.mapVerts(1)[i]=0.5f*(fromMesh.mesh.mapVerts(1)[fromIdx0]+fromMesh.mesh.mapVerts(1)[fromIdx1]);
		}

		void setFace(int i, int v0, int v1, int v2) {
			mesh.faces[i].setVerts(v0, v1, v2);
			mesh.faces[i].setSmGroup(1);
			mesh.faces[i].setEdgeVisFlags(EDGE_VIS, EDGE_VIS, EDGE_VIS);
			if (mesh.mapSupport(1)) mesh.mapFaces(1)[i].setTVerts(v0, v1, v2);
		}
};

void loopSubdivide(Mesh &inMesh, Mesh &outMesh, int steps) {
	if (steps==0) { outMesh.DeepCopy(&inMesh, ALL_CHANNELS); return; }
	if (steps==1) {
		if (inMesh.mapSupport(1)) outMesh.setMapSupport(1, TRUE);
		MeshWrapper wIn(inMesh), wOut(outMesh);
		Vlado::_loopSubdivideStep(wIn, wOut);
		return;
	}

	Mesh temp;
	if (inMesh.mapSupport(1)) {
		temp.setMapSupport(1, TRUE);
		outMesh.setMapSupport(1, TRUE);
	}

	MeshWrapper wIn(inMesh), wOut(outMesh);
	Vlado::_loopSubdivideStep(wIn, wOut);

	Mesh *t0=&outMesh, *t1=&temp;
	for (int i=1; i<steps; i++) {
		MeshWrapper wIn(*t0), wOut(*t1);
		Vlado::_loopSubdivideStep(wIn, wOut);
		Vlado::swap(t0, t1);
	}
	if (t0!=&outMesh) outMesh.DeepCopy(&temp, ALL_CHANNELS);
}

void SimCloth3::ModifyObject(TimeValue time, ModContext &mc, ObjectState *os, INode *node) {
	int objectType=pblockGeneral->GetInt(sc3_general_type, time);
	if (objectType==0) return;

	os->obj->UpdateValidity(GEOM_CHAN_NUM, Interval(time, time));

	if (!mc.localData) mc.localData=new SimClothLocalData;
	SimClothLocalData *ld=(SimClothLocalData*) mc.localData;
	if (!ld) return;

	if (dontModify) {
		if (editMod==this) {
			TriObject *tri=(TriObject*) os->obj;
			ld->saveMesh(&tri->mesh);
			if (numPts!=tri->mesh.numVerts) { delete[] selPts; selPts=NULL; }
			numPts=tri->mesh.numVerts;
			if (!selPts) { selPts=new unsigned char[numPts]; memset(selPts, 0, numPts*sizeof(selPts[0])); }
		}
		return;
	}

	SimClothCache &cache=ld->cache;
	if (cache.keys.count()>1 && time>=cache.keys[0].time && time<=cache.keys[cache.keys.count()-1].time) {
		int nextKey=1;
		while (nextKey<cache.keys.count() && time>cache.keys[nextKey].time) nextKey++;

		int keyIdx=nextKey-1;

		SimClothKey &k0=cache.keys[keyIdx], &k1=cache.keys[nextKey];

		float t=0.0f;
		float dt=float(k1.time-k0.time);
		if (fabs(dt)>1e-5) t=float(time-k0.time)/dt;

		if (k0.numPoints==k1.numPoints && k0.numPoints==os->obj->NumPoints()) {
			float t1=1.0f-t;

			Object *obj=os->obj;
			for (int i=0; i<obj->NumPoints(); i++) obj->SetPoint(i, k0.points[i]*t1+k1.points[i]*t);
			obj->PointsWereChanged();
		}
	}

	if (editMod==this) {
		TriObject *tri=(TriObject*) os->obj;
		ld->saveMesh(&tri->mesh);
		if (numPts!=tri->mesh.numVerts) { delete[] selPts; selPts=NULL; }
		numPts=tri->mesh.numVerts;
		if (!selPts) { selPts=new unsigned char[numPts]; memset(selPts, 0, numPts*sizeof(selPts[0])); }
	}

	int smoothResult=pblockGeneral->GetInt(sc3_general_smoothResult, time);
	int smoothIterations=pblockGeneral->GetInt(sc3_general_smoothIterations, time);

	if (smoothResult && smoothIterations>0) {
		Mesh &inMesh=((TriObject*) (os->obj))->GetMesh();
		Mesh outMesh;
		loopSubdivide(inMesh, outMesh, smoothIterations);
		inMesh.DeepCopy(&outMesh, ALL_CHANNELS);
		inMesh.InvalidateTopologyCache();
	}
}

//******************************************************
// NotifyInputChanged is called each time the input object is changed in some way
// We can find out how it was changed by checking partID and message
void SimCloth3::NotifyInputChanged(Interval changeInt, PartID partID, RefMessage message, ModContext *mc) {
}

