#ifndef __VECTORBASE_H__
#define __VECTORBASE_H__

#ifndef _USE_SSE

BEGIN_VLADO

class Vector {
	public:
		real x, y, z;

		INLINE Vector() {};

		INLINE Vector(const Vector &a) { x=a.x; y=a.y; z=a.z; }
		INLINE Vector& operator=(const Vector &a) { x=a.x; y=a.y; z=a.z; return *this; }

		INLINE Vector(int ix, int iy, int iz) { x=real(ix); y=real(iy); z=real(iz); }
		INLINE Vector(float ix, float iy, float iz) { x=real(ix); y=real(iy); z=real(iz); }
		INLINE Vector(double ix, double iy, double iz) { x=real(ix); y=real(iy); z=real(iz); }

		INLINE void makeZero(void) { x=y=z=0.0f; }
		INLINE void set(real ix, real iy, real iz) { x=ix; y=iy; z=iz; }

		INLINE void operator +=(const Vector &a) { x+=a.x; y+=a.y; z+=a.z; }
		INLINE void operator -=(const Vector &a) { x-=a.x; y-=a.y; z-=a.z; }
		INLINE void operator *=(real f) { x*=f; y*=f; z*=f; }
		INLINE void operator /=(real f) { x/=f; y/=f; z/=f; }

		INLINE Vector operator-(void) const { return(Vector(-x,-y,-z)); } 

		INLINE real& operator [](const int index) { return (&x)[index]; }
		INLINE const real& operator [](const int index) const { return (&x)[index]; }

		INLINE real length(void) const { return real(sqrt(x*x+y*y+z*z)); }
		INLINE real lengthSqr(void) const { return x*x+y*y+z*z; }
		INLINE Vector normalize(void) const {
			real len=length();
			return Vector(x/len, y/len, z/len);
		}
		INLINE void makeNormalized(void) {
			real len=length();
			x/=len;
			y/=len;
			z/=len;
		}
		INLINE void makeNormalized0(void) {
			real len=length();
			if (len>1e-12f) {
				x/=len;
				y/=len;
				z/=len;
			}
		}

		INLINE void rotate(const Vector &axis);
};

INLINE Vector operator *(const Vector &a, real f) {
	Vector res(a);
	res*=f;
	return res;
}

INLINE Vector operator *(real f, const Vector &a) {
	Vector res(a);
	res*=f;
	return res;
}

INLINE Vector operator /(const Vector &a, real f) {
	Vector res(a);
	res/=f;
	return res;
}

INLINE real operator *(const Vector &a, const Vector &b) {
	return a.x*b.x+a.y*b.y+a.z*b.z;
}

INLINE Vector operator +(const Vector &a, const Vector &b) {
	Vector res(a);
	res+=b;
	return res;
}

INLINE Vector operator -(const Vector &a, const Vector &b) {
	Vector res(a);
	res-=b;
	return res;
}

INLINE Vector operator /(const Vector &a, const Vector &b) {
	return Vector(a.x/b.x, a.y/b.y, a.z/b.z);
}

INLINE Vector mul(const Vector &a, const Vector &b) {
	return Vector(a.x*b.x, a.y*b.y, a.z*b.z);
}

END_VLADO

#else

#include <fvec.h>

BEGIN_VLADO

#pragma pack(push,16)

__declspec(align(16)) class Vector {
	public:
		union {
			struct { real x, y, z; };
			float f[4];
			__m128 m;
		};

		operator __m128(void) const { return m; }
		Vector (__m128 im) { m=im; }
		
		INLINE Vector() {};

		INLINE Vector(const Vector &a) { x=a.x; y=a.y; z=a.z; }
		INLINE Vector& operator=(const Vector &a) { x=a.x; y=a.y; z=a.z; return *this; }


		INLINE Vector(int ix, int iy, int iz) { x=real(ix); y=real(iy); z=real(iz); }
		INLINE Vector(float ix, float iy, float iz) { x=real(ix); y=real(iy); z=real(iz); }
		INLINE Vector(double ix, double iy, double iz) { x=real(ix); y=real(iy); z=real(iz); }

		explicit Vector(float f) { m=_mm_set_ps1(f); }	

		INLINE void makeZero(void) { x=y=z=0.0f; }
		INLINE void set(real ix, real iy, real iz) { x=ix; y=iy; z=iz; }

		INLINE Vector& operator +=(const Vector &a) { return *this = _mm_add_ps(m,a); }
		INLINE Vector& operator -=(const Vector &a) { return *this = _mm_sub_ps(m,a); }   
		// INLINE Vector& operator *=(float k) { return *this = _mm_mul_ps(m,Vector(k)); } 
		// INLINE Vector& operator /=(float k) { return *this = _mm_div_ps(m,Vector(k)); }
		INLINE Vector& operator *=(float k) { x*=k; y*=k; z*=k; return *this; }
		INLINE Vector& operator /=(float k) { x/=k; y/=k; z/=k; return *this; }
/*
		INLINE Vector& operator *=(float k) {
#pragma vector aligned
			for (int i=0; i<4; i++) f[i]*=k;
			return *this;
		}
		INLINE Vector& operator /=(float k) {
#pragma vector aligned
			for (int i=0; i<4; i++) f[i]/=k;
			return *this;
		}

		INLINE Vector operator-(void) const {
			Vector res;
#pragma vector aligned
			for (int i=0; i<4; i++) res.f[i]=-f[i];
			return res;
		}*/
		INLINE Vector operator-(void) const { return Vector(-x,-y,-z); }

		INLINE real& operator [](const int index) { return (&x)[index]; }
		INLINE const real& operator [](const int index) const { return (&x)[index]; }

		INLINE real lengthSqr(void) const { return x*x+y*y+z*z; }
		INLINE real length(void) const { return sqrtf(lengthSqr()); }
		INLINE Vector normalize(void) const { Vector r(*this); r/=length(); return r; }
		INLINE void makeNormalized(void) { *this/=length(); }
		INLINE void makeNormalized0(void) {
			real len=length();
			if (len>1e-12f) *this/=len;
		}

		INLINE void rotate(const Vector &axis);

};

INLINE Vector operator +(const Vector &a, const Vector &b) { return _mm_add_ps(a.m, b.m); }
INLINE Vector operator -(const Vector &a, const Vector &b) { return _mm_sub_ps(a,b); } 
/*
INLINE Vector operator *(const Vector &a, float k) { return _mm_mul_ps(a,Vector(k)); } 
INLINE Vector operator *(float k, const Vector &a) { return _mm_mul_ps(a,Vector(k)); } 
INLINE Vector operator /(const Vector &a, float k) { return _mm_div_ps(a,Vector(k)); }
*/
INLINE Vector operator /(const Vector &a, const Vector &b) { return _mm_div_ps(a,b); }
INLINE Vector mul(const Vector &a, const Vector &b) { return _mm_mul_ps(a,b); }

INLINE Vector operator *(const Vector &a, float k) { Vector r(a); r*=k; return r; }
INLINE Vector operator *(float k, const Vector &a) { Vector r(a); r*=k; return r; }
INLINE Vector operator /(const Vector &a, float k) { Vector r(a); r/=k; return r; }

INLINE real operator *(const Vector &a, const Vector &b) { return a.x*b.x+a.y*b.y+a.z*b.z; }

/*
INLINE Vector operator *(const Vector &a, float k) {
	Vector res;
#pragma vector aligned
	for (int i=0; i<4; i++) res.f[i]=a.f[i]*k;
	return res;
}

INLINE Vector operator *(float k, const Vector &a) {
	Vector res;
#pragma vector aligned
	for (int i=0; i<4; i++) res.f[i]=a.f[i]*k;
	return res;
}

INLINE Vector operator /(const Vector &a, float k) {
	Vector res;
#pragma vector aligned
	for (int i=0; i<4; i++) res.f[i]=a.f[i]/k;
	return res;
}

INLINE real operator *(const Vector &a, const Vector &b) {
	real res=0.0f;
#pragma vector aligned
	for (int i=0; i<3; i++) res+=a.f[i]*b.f[i];
	return res;
}
*/

END_VLADO

#pragma pack(pop)

#endif

#endif
