/*****************************************************
	FILE: vector.h
	DESCRIPTION: Vector in 3D
	Copyright (C) 2001-2003 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __VECTOR_H__
#define __VECTOR_H__

#include "vectorbase.h"

BEGIN_VLADO

INLINE bool operator!=(const Vector &a, const Vector &b) {
	return a.x!=b.x || a.y!=b.y || a.z!=b.z;
}

INLINE int maxComponent(const Vector &a) {
	int maxDim=0;
	if (fabs(a[1])>fabs(a[0])) maxDim=1;
	if (fabs(a[2])>fabs(a[maxDim])) maxDim=2;
	return maxDim;
}

INLINE int minComponent(const Vector &a) {
	int minDim=0;
	if (fabs(a[1])<fabs(a[0])) minDim=1;
	if (fabs(a[2])<fabs(a[minDim])) minDim=2;
	return minDim;
}

INLINE Vector operator ^(const Vector &a, const Vector &b) {
	return Vector(
		a.y*b.z-b.y*a.z,
		a.z*b.x-b.z*a.x,
		a.x*b.y-b.x*a.y);
}

INLINE real lengthSqr(const Vector &a) { return a.lengthSqr(); }
INLINE real length(const Vector &a) { return a.length(); }

INLINE Vector normalize(const Vector &a) { return a.normalize(); }

INLINE Vector vabs(const Vector &a) {
	return Vector(fabsf(a.x), fabsf(a.y), fabsf(a.z));
}

INLINE Vector vector_min(const Vector &a, const Vector &b) {
	return 0.5f*(a+b-vabs(a-b));
}

INLINE Vector vector_max(const Vector &a, const Vector &b) {
	return 0.5f*(a+b+vabs(a-b));
}

INLINE void vector_minMax(Vector &a, Vector &b) {
	Vector s=a+b;
	Vector d=vabs(a-b);
	a=0.5f*(s-d);
	b=0.5f*(s+d);
}

INLINE real maxValue(const Vector &a) {
	if (a.x>a.y) {
		if (a.x>a.z) return a.x;
		else return a.z;
	} else {
		if (a.y>a.z) return a.y;
		else return a.z;
	}
}

INLINE real minValue(const Vector &a) {
	if (a.x<a.y) {
		if (a.x<a.z) return a.x;
		else return a.z;
	} else {
		if (a.y<a.z) return a.y;
		else return a.z;
	}
}

INLINE Vector rotate(const Vector &v, const Vector &axis) {
	real oldLen=length(v);

	Vector d=v^axis;
	d+=v;

	real newLen=length(d);
	if (newLen<almostZero) return Vector(0,0,0);

	return d*oldLen/newLen;
}

INLINE void Vector::rotate(const Vector &axis) {
	real oldLen=length();

	Vector d=(*this)^axis;
	d+=(*this);

	real newLen=d.length();
	if (newLen>almostZero) (*this)=d*oldLen/newLen;
}

// Some macros
#define _add3(a,b,r) { (r)[0]=(a)[0]+(b)[0]; (r)[1]=(a)[1]+(b)[1]; (r)[2]=(a)[2]+(b)[2]; }
#define _sub3(a,b,r) { (r)[0]=(a)[0]-(b)[0]; (r)[1]=(a)[1]-(b)[1]; (r)[2]=(a)[2]-(b)[2]; }
#define _dot3(a,b) ((a)[0]*(b)[0]+(a)[1]*(b)[1]+(a)[2]*(b)[2])
#define _cross3(a,b,r) { (r)[0]=(a)[1]*(b)[2]-(b)[1]*(a)[2]; (r)[1]=(a)[2]*(b)[0]-(b)[2]*(a)[0]; (r)[2]=(a)[0]*(b)[1]-(b)[0]*(a)[1]; }

class Vector2 {
	public:
		real x,y;

		Vector2() {}
		Vector2(real ix, real iy) { x=ix; y=iy; }

		void makeZero(void) { x=y=0.0f; }
};

END_VLADO

#endif
