#include "utils.h"

#include "trimesh.h"
#include "misc_ray.h"
#include "fileutils.h"

USE_VLADO

void TriMesh::init(TriMesh &mesh) {
	freeData();

	numVerts=mesh.numVerts;
	verts=new Vector[numVerts];
	memcpy(verts, mesh.verts, numVerts*sizeof(Vector));

	numFaces=mesh.numFaces;
	faces=new TriMeshFace[numFaces];
	memcpy(faces, mesh.faces, numFaces*sizeof(TriMeshFace));

	if (mesh.tverts) {
		tverts=new Vector[numVerts];
		memcpy(tverts, mesh.tverts, numVerts*sizeof(Vector));
		buildTNormals();
	}

	buildEdges();
}

void TriMesh::buildVNormals(bool buildTransformed) {
	Vector *&v=(buildTransformed)? tverts : verts;
	Vector *&fnrm=(buildTransformed)? tfnormals: fnormals;
	float *&farea=(buildTransformed)? this->tfarea2: this->farea2;
	Vector *&vnrm=(buildTransformed)? tvnormals: vnormals;

	if (!fnrm) fnrm=new Vector[numFaces];
	if (!farea) farea=new float[numFaces];
	if (!vnrm) vnrm=new Vector[numVerts];

	int i;
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];

		Vector &p0=v[face.v[0]];
		Vector &p1=v[face.v[1]];
		Vector &p2=v[face.v[2]];

		Vector n=(p1-p0)^(p2-p0);
		farea[i]=n.length();
		fnrm[i]=n/farea[i];
	}

	for (i=0; i<numVerts; i++) vnrm[i]=Vector(0,0,0);
	for (i=0; i<numFaces; i++) {
		for (int j=0; j<3; j++) vnrm[faces[i].v[j]]+=fnrm[i];
	}
	for (i=0; i<numVerts; i++) vnrm[i]=normalize(vnrm[i]);
}

inline int clip3a(int a) {
	if (a>=3) return a-3;
	return a;
}

inline int clip3b(int a) {
	if (a<0) return a+3;
	return a;
}

void TriMesh::buildEdges(bool deleteOld) {
	numEdges=0;

	if (deleteOld && edges) delete[] edges;
	if (strips) { delete[] strips; strips=NULL; stripLength=0; }
	if (deleteOld) edges=new TriMeshEdge[numFaces*3];

	int *ngbPtrs=new int[numVerts];
	memset(ngbPtrs, 0xFF, sizeof(int)*numVerts);

	int *ngbs=new int[numFaces*3];
	int *nextNgb=new int[numFaces*3];
	int *edgeIdx=new int[numFaces*3];
	int freeNgb=0;

	buildNormals();
	int i;
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		for (int j=0; j<3; j++) {
			int v0=face.v[j], v1=face.v[clip3a(j+1)];
			int r0=v0, r1=v1;

			if (v0>v1) { int t=v0; v0=v1; v1=t; }
			bool found=false;
			int idx=ngbPtrs[v0];
			while (idx>=0) {
				if (ngbs[idx]==v1) { found=true; break; }
				idx=nextNgb[idx];
			}
			if (!found) {
				TriMeshEdge &edge=edges[numEdges];
				face.e[j]=numEdges;
				edge.fe[0]=j;
				edge.v[0]=r0;
				edge.v[1]=r1;
				edge.color=face.color;
				edge.numFaces=1;
				edge.f[0]=i;
				edge.tv[0]=face.v[clip3a(j+2)];
				edge.visible=true;
				numEdges++;

				nextNgb[freeNgb]=ngbPtrs[v0];
				ngbs[freeNgb]=v1;
				edgeIdx[freeNgb]=numEdges-1;
				ngbPtrs[v0]=freeNgb;
				freeNgb++;
			} else {
				TriMeshEdge &edge=edges[edgeIdx[idx]];
				face.e[j]=edgeIdx[idx];
				edge.color+=face.color;
				edge.numFaces++;
				if (edge.numFaces==2) {
					edge.fe[1]=j;
					edge.f[1]=i;
					edge.tv[1]=face.v[clip3a(j+2)];

					Vector &nrm0=fnormals[edge.f[0]];
					Vector &nrm1=fnormals[edge.f[1]];
					if (length(nrm0^nrm1)<1e-3f) edge.visible=false;
				}
			}
		}
	}
	for (i=0; i<numEdges; i++) {
		edges[i].color/=float(edges[i].numFaces);
	}

	if (ngbPtrs) delete[] ngbPtrs;
	if (ngbs) delete[] ngbs;
	if (nextNgb) delete[] nextNgb;
	if (edgeIdx) delete[] edgeIdx;
}

void TriMesh::randomVerts(int n, float size, Vector center) {
	freeData();

	numVerts=n;
	verts=new Vector[numVerts];
	for (int i=0; i<numVerts; i++) {
		Vector dir(
			-size*0.5f+size*float(rand())/float(RAND_MAX),
			-size*0.5f+size*float(rand())/float(RAND_MAX),
			-size*0.5f+size*float(rand())/float(RAND_MAX));
		dir=normalize(dir);
		// dir.x*=2.0f;
		// dir.z=lengthSquared(Vector(dir.x, dir.y, 0.0f));
		/*float cs=float(cos(45.0f*3.1415f/180.0f));
		float sn=float(cos(45.0f*3.1415f/180.0f));

		float y=dir.y*cs-dir.z*sn;
		float z=dir.y*sn+dir.z*cs;

		dir.y=y; dir.z=z;*/
		verts[i]=dir*(size*0.5f*float(rand())/float(RAND_MAX))+center;
	}

	numFaces=0;
	numEdges=0;
}

void TriMesh::createTetra(Color col) {
	// find two different vertices
	Vector p0=verts[0], p1=verts[1];

	float s=lengthSqr(p0-p1);
	int lvert=2;
	while (s<1e-6f && lvert<numVerts) {
		verts[1]=verts[lvert];
		verts[lvert]=p1;
		p1=verts[1];
		s=lengthSqr(p0-p1);
		lvert++;
	}

	// find a third vertex not on the same line
	Vector p2=verts[2];
	s=lengthSqr((p1-p0)^(p2-p0));
	lvert=3;
	while (fabs(s)<1e-6f && lvert<numVerts) {
		verts[2]=verts[lvert];
		verts[lvert]=p2;
		p2=verts[2];
		s=lengthSqr((p1-p0)^(p2-p0));
		lvert++;
	}

	// find a fourth vertex not in the same plane
	Vector p3=verts[3];
	s=((p1-p0)^(p2-p0))*(p3-p0);
	lvert=4;
	while (fabs(s)<1e-1f && lvert<numVerts) {
		verts[3]=verts[lvert];
		verts[lvert]=p3;
		p3=verts[3];
		s=((p1-p0)^(p2-p0))*(p3-p0);
		lvert++;
	}

	// flip if orientation is wrong
	if (s>0.0f) {
		Vector t=verts[2];
		verts[2]=verts[3];
		verts[3]=t;
	}

	// create four faces
	numFaces=4;
	faces[0].set(0, 1, 2, col);
	faces[1].set(0, 2, 3, col);
	faces[2].set(1, 0, 3, col);
	faces[3].set(2, 1, 3, col);

	// build the edges
	buildEdges(false);
}

void TriMesh::convexHull(void) {
	if (edges) delete[] edges;
	edges=NULL;
	numEdges=0;

	if (faces) delete[] faces;
	faces=NULL;
	numFaces=0;

	if (fnormals) delete[] fnormals;
	fnormals=NULL;

	if (numVerts<4) return;

	int maxFaces=numVerts*10;
	int maxEdges=numVerts*10;

	edges=new TriMeshEdge[maxEdges];
	faces=new TriMeshFace[maxFaces];
	fnormals=new Vector[maxFaces];

	int *visFaces=new int[maxFaces];

	// set up a tetrahedron
	Color col(0.4f, 0.3f, 0.0f);
	createTetra(col);

	int *nextFace=new int[maxFaces];
	int *nextEdge=new int[maxEdges];
	int *horVerts=new int[numVerts*2];

	int j;
	for (j=0; j<numVerts; j++) horVerts[j*2]=-1;

	for (j=0; j<maxFaces; j++) nextFace[j]=j+1;
	nextFace[numFaces-1]=-1;
	nextFace[maxFaces-1]=-1;

	for (j=0; j<maxEdges; j++) nextEdge[j]=j+1;
	nextEdge[numEdges-1]=-1;
	nextEdge[maxEdges-1]=-1;

	int freeFace=numFaces, freeEdge=numEdges;
	int firstFace=0, firstEdge=0;

	Vector vsum=verts[0]+verts[1]+verts[2]+verts[3];
	int nverts=4;

	// add vertices one by one and maintain the convex hull
	for (int i=4; i<numVerts; i++) {
		Vector &p=verts[i];

		// check if vertex is inside current hull by checking the visibility of the faces
		bool inside=true;
		int fidx=firstFace;
		int fprev=-1;

		while (fidx>=0) {
			TriMeshFace &face=faces[fidx];
			if (isFaceVisible(p, fidx)) {
				visFaces[fidx]=1; // this face is visible from the outside
				inside=false; // the point is outside the hull

				// delete the face
				numFaces--;
				int nxt=nextFace[fidx];

				if (fprev>=0) nextFace[fprev]=nxt;
				else firstFace=nxt;

				nextFace[fidx]=freeFace;
				freeFace=fidx;

				fidx=nxt;
			} else {
				visFaces[fidx]=0;
				fprev=fidx;
				fidx=nextFace[fidx];
			}
		}

		// if vertex is inside - all is ok
		if (inside) continue;

		// update the center of mass
		vsum+=p;
		nverts++;

		// delete visible edges and expand horizon edges
		int eidx=firstEdge;
		int eprev=-1;

		while (eidx>=0) {
			TriMeshEdge &edge=edges[eidx];

			int code=visFaces[edge.f[0]]+(visFaces[edge.f[1]]<<1);
			if (code==3) {
				// both faces visible - delete the edge
				int nxt=nextEdge[eidx];

				if (eprev>=0) nextEdge[eprev]=nxt;
				else firstEdge=nxt;

				nextEdge[eidx]=freeEdge;
				freeEdge=eidx;

				eidx=nxt;
				continue;
			}

			if (code>0) {
				// only one face is visible - this is a horizon edge
				// add a new face
				int fidx=freeFace;
				freeFace=nextFace[freeFace];
				nextFace[fidx]=firstFace;
				firstFace=fidx;

				// check direction of the face
				int v0=edge.v[0], v1=edge.v[1];
				Vector n=(verts[v0]-p)^(verts[v1]-p);
				if ((vsum/float(nverts)-p)*n>0.0f) { int t=v0; v0=v1; v1=t; n=-n; }

				faces[fidx].set(i, v0, v1, col);
				fnormals[fidx]=n;

				numFaces++;

				if (code==1) edge.f[0]=fidx; else edge.f[1]=fidx;

				// create new edges for the new face
				if (horVerts[v0*2]!=i) {
					horVerts[v0*2]=i;
					horVerts[v0*2+1]=fidx;
				} else {
					// create a new edge
					int eidx=freeEdge;
					freeEdge=nextEdge[freeEdge];
					nextEdge[eidx]=firstEdge;
					firstEdge=eidx;

					TriMeshEdge &edge=edges[eidx];
					edge.color=col;
					edge.v[0]=i;
					edge.v[1]=v0;
					edge.f[0]=horVerts[v0*2+1];
					edge.f[1]=fidx;
					edge.numFaces=2;
					edge.visible=true;
				}

				if (horVerts[v1*2]!=i) {
					horVerts[v1*2]=i;
					horVerts[v1*2+1]=fidx;
				} else {
					// create a new edge
					int eidx=freeEdge;
					freeEdge=nextEdge[freeEdge];
					nextEdge[eidx]=firstEdge;
					firstEdge=eidx;

					TriMeshEdge &edge=edges[eidx];
					edge.color=col;
					edge.v[0]=i;
					edge.v[1]=v1;
					edge.f[0]=horVerts[v1*2+1];
					edge.f[1]=fidx;
					edge.numFaces=2;
					edge.visible=true;
				}
			}
			
			eprev=eidx;
			eidx=nextEdge[eidx];
		}
	}

	// collect all the faces in one place
	TriMeshFace *newFaces=NULL;
	if (numFaces>0) {
		newFaces=new TriMeshFace[numFaces];
		int fidx=firstFace;
		for (int i=0; i<numFaces; i++) {
			newFaces[i]=faces[fidx];
			fidx=nextFace[fidx];
		}
	}

	delete[] visFaces;
	delete[] nextFace;
	delete[] nextEdge;
	delete[] horVerts;
	delete[] faces;

	faces=newFaces;
	buildEdges();
}

void TriMesh::collapseVerts(void) {
	int *usedIdx=new int[numVerts];
	memset(usedIdx, 0, sizeof(bool)*numVerts);

	// mark used vertices
	int i;
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		usedIdx[face.v[0]]=1;
		usedIdx[face.v[1]]=1;
		usedIdx[face.v[2]]=1;
	}

	// count the used vertices
	int n=0;
	for (i=0; i<numVerts; i++) if (usedIdx[i]) usedIdx[i]=n++; else usedIdx[i]=-1;

	// list the used vertices
	Vector *newVerts=new Vector[n];
	int idx=0;
	for (i=0; i<numVerts; i++) if (usedIdx[i]>=0) newVerts[idx++]=verts[i];

	// delete old vertices
	delete[] verts;
	verts=newVerts;
	numVerts=n;

	// fix faces
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		face.v[0]=usedIdx[face.v[0]];
		face.v[1]=usedIdx[face.v[1]];
		face.v[2]=usedIdx[face.v[2]];
	}

	delete[] usedIdx;

	buildEdges();
}

void TriMesh::removeZeroFaces(real tolerance) {
	buildNormals();
	for (int i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		if (farea2[i]<tolerance) {
			faces[i]=faces[numFaces-1];
			fnormals[i]=fnormals[numFaces-1];
			farea2[i]=farea2[numFaces-1];
			numFaces--;
			i--;
		}
	}
	buildEdges();
}

void TriMesh::saveToFile(FILE *fp) {
	if (!fp) return;

	fprintf(fp, "%i\n", numVerts);
	int i;
	for (i=0; i<numVerts; i++) {
		fprintf(fp, "(%g %g %g)\n", verts[i].x, verts[i].y, verts[i].z);
	}

	fprintf(fp, "%i\n", numFaces);
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];

		fprintf(fp, "(%i %i %i) (%g %g %g)\n", face.v[0], face.v[1], face.v[2], face.color.r, face.color.g, face.color.b);
	}
}

void TriMesh::saveToFile(char *fname) {
	FILE *fp=fopen(fname, "wt");
	saveToFile(fp);
	fclose(fp);
}

void TriMesh::loadFromFile(FILE *fp, real scale) {
	if (!fp) return;
	freeData();

	fscanf(fp, "%i", &numVerts);
	verts=new Vector[numVerts];

	int i;
	for (i=0; i<numVerts; i++) {
		fscanf(fp, " (%g %g %g) ", &verts[i].x, &verts[i].y, &verts[i].z);
		verts[i]*=scale;
	}

	fscanf(fp, "%i", &numFaces);
	faces=new TriMeshFace[numFaces];

	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		fscanf(fp, " (%i %i %i) (%g %g %g) ", &face.v[0], &face.v[1], &face.v[2], &face.color.r, &face.color.g, &face.color.b);
	}

	buildEdges();
}

// Binary .ply file
uint32 invertInt(uint32 x) {
	return ((x>>24) & 0xFF) | ((x>>8) & 0xFF00) | ((x<<8) & 0xFF0000) | ((x<<24) & 0xFF0000);
}

void TriMesh::loadFromFilePly(FILE *fp, real scale) {
	if (!fp) return;
	freeData();

	char tag[512];
	while (readWord(fp, tag)) {
		if (strcmp(tag, "end_header")==0) break;
		else if (strcmp(tag, "format")==0) {
			readWord(fp, tag);
			if (strcmp(tag, "binary_big_endian")!=0) return;
			float f=0.0f;
			readFloat(fp, f);
			if (f!=1.0f) return;
		} else if (strcmp(tag, "vertex")==0) {
			readInt(fp, numVerts);
		} else if (strcmp(tag, "face")==0) {
			readInt(fp, numFaces);
		}
	}

	skipSpaces(fp);

	verts=new Vector[numVerts];
	faces=new TriMeshFace[numFaces];

	int i;
	for (i=0; i<numVerts; i++) {
		uint32 x;
		float f;
		for (int j=0; j<3; j++) {
			fread(&x, 4, 1, fp);
			x=invertInt(x);
			f=*((float*) (&x));
			verts[i][j]=f*scale;
		}
	}

	for (i=0; i<numFaces; i++) {
		unsigned char nbytes;
		uint32 x;
		fread(&nbytes, 1, 1, fp);
		for (int j=0; j<3; j++) {
			fread(&x, 4, 1, fp);
			x=invertInt(x);
			faces[i].v[j]=x;
		}
	}
}

void TriMesh::loadFromFile(char *fname, real scale) {
	FILE *fp=fopen(fname, "rb");
	if (!fp) return;

	char buf[4];
	buf[0]=fgetc(fp);
	buf[1]=fgetc(fp);
	buf[2]=fgetc(fp);
	buf[3]=fgetc(fp);

	if (buf[0]=='p' && buf[1]=='l' && buf[2]=='y' && buf[3]=='\n') {
		loadFromFilePly(fp, scale);
		fclose(fp);
	}

	fclose(fp);
	fp=fopen(fname, "rt");
	loadFromFile(fp, scale);
	fclose(fp);
}

void TriMesh::transform(const Transform &tm) {
	if (!tverts) {
		tverts=new Vector[numVerts];
	}
	for (int i=0; i<numVerts; i++) tverts[i]=tm*verts[i];
}

void TriMesh::makeCube(real a, real b, real c) {
	freeData();

	numVerts=8;
	verts=new Vector[8];

	verts[0].set(-a, -b, -c);
	verts[1].set(a, -b, -c);
	verts[2].set(-a, -b, c);
	verts[3].set(a, -b, c);

	verts[4].set(-a, b, -c);
	verts[5].set(a, b, -c);
	verts[6].set(-a, b, c);
	verts[7].set(a, b, c);

	numFaces=12;
	faces=new TriMeshFace[12];

	faces[0].set(0, 1, 2, Color(0.0f,-1.0f,0.0f));
	faces[1].set(1, 3, 2, Color(0.0f,0.0f,-1.0f));

	faces[2].set(4, 6, 5, Color(0.0f,-1.0f,0.0f));
	faces[3].set(5, 6, 7, Color(-1.0f,0.0f,0.0f));

	faces[4].set(1, 5, 3, Color(0.0f,-1.0f,0.0f));
	faces[5].set(5, 7, 3, Color(0.0f,0.0f,-1.0f));

	faces[6].set(0, 2, 4, Color(0.0f,-1.0f,0.0f));
	faces[7].set(4, 2, 6, Color(-1.0f,0.0f,0.0f));

	faces[8].set(2, 3, 6, Color(0.0f,-1.0f,0.0f));
	faces[9].set(3, 7, 6, Color(0.0f,0.0f,-1.0f));

	faces[10].set(0, 4, 1, Color(0.0f,-1.0f,0.0f));
	faces[11].set(1, 4, 5, Color(-1.0f,0.0f,0.0f));

	buildEdges();
}

void TriMesh::makeRegularPlane(real a, real b, int subdivsx, int subdivsy, real delone, bool alternate) {
	freeData();

	numVerts=(subdivsx+1)*(subdivsy+1);
	numFaces=subdivsx*subdivsy*2;

	verts=new Vector[numVerts];
	faces=new TriMeshFace[numFaces];

	Random rnd(1);
	for (int i=0; i<=subdivsy; i++) {
		for (int j=0; j<=subdivsx; j++) {
			float fx=0.0f, fy=0.0f;
			if (i>0 && i<subdivsy && j>0 && j<subdivsx) {
				fx=(rnd.rnd()-0.5f)*delone;
				fy=(rnd.rnd()-0.5f)*delone;
			}
			verts[i*(subdivsx+1)+j].set(-a*0.5f+a*(float(j)+fx)/float(subdivsx), b*0.5f-b*(float(i)+fy)/float(subdivsy), 0.0f);
		}
	}

	if (!delone) {
		for (int i=0; i<subdivsy; i++) {
			for (int j=0; j<subdivsx; j++) {
				int fidx=i*subdivsx+j;
				int vidx=i*(subdivsx+1)+j;

				if (((i+j)&1)==1 && alternate) {
					faces[fidx*2].set(vidx, vidx+subdivsx+1, vidx+subdivsx+2);
					faces[fidx*2+1].set(vidx, vidx+subdivsx+2, vidx+1);
				} else {
					faces[fidx*2].set(vidx, vidx+subdivsx+1, vidx+1);
					faces[fidx*2+1].set(vidx+1, vidx+subdivsx+1, vidx+subdivsx+2);
				}
			}
		}
	} else {
		int i;
		for (i=0; i<numVerts; i++) verts[i].z=sqr(verts[i].x)+sqr(verts[i].y);
		convexHull();
		for (i=0; i<numVerts; i++) verts[i].z=0.0f;

		buildNormals();
		for (i=0; i<numFaces; i++) {
			TriMeshFace &face=faces[i];
			if (lengthSqr(fnormals[i])<1e-6f || fnormals[i].z>-1e-6f) {
				faces[i]=faces[numFaces-1];
				fnormals[i]=fnormals[numFaces-1];
				numFaces--;
				i--;
			}
		}
	}

	buildEdges();
}

void TriMesh::makeDelonePlane(real a, real b, int edgeSubdivs, int numPoints) {
	freeData();

	numVerts=(edgeSubdivs*4)+numPoints;
	verts=new Vector[numVerts];

	int idx=0;
	edgeSubdivs--;
	int i;
	for (i=0; i<edgeSubdivs+1; i++) {
		float x=-a*0.5f+a*float(i)/float(edgeSubdivs);
		float y=b*0.5f;
		verts[idx++].set(x, y, x*x+y*y);
	}
	for (i=1; i<edgeSubdivs; i++) {
		float x=-a*0.5f;
		float y=-b*0.5f+b*float(i)/float(edgeSubdivs);
		verts[idx++].set(x, y, x*x+y*y);
	}

	for (i=0; i<edgeSubdivs+1; i++) {
		float x=a*0.5f-a*float(i)/float(edgeSubdivs);
		float y=-b*0.5f;
		verts[idx++].set(x, y, x*x+y*y);
	}
	for (i=1; i<edgeSubdivs; i++) {
		float x=a*0.5f;
		float y=b*0.5f-b*float(i)/float(edgeSubdivs);
		verts[idx++].set(x, y, x*x+y*y);
	}

	Random rnd(1);
	for (i=idx; i<numVerts; i++) {
		float x=-a*0.45f+a*rnd.rnd()*0.9f;
		float y=-b*0.45f+b*rnd.rnd()*0.9f;
		// float x=-a*0.5f+a*rnd.rnd();
		// float y=-b*0.5f+b*rnd.rnd();
		verts[idx++].set(x, y, x*x+y*y);
	}

	convexHull();

	for (i=0; i<numVerts; i++) verts[i].z=0.0f;

	buildNormals();
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		if (lengthSqr(fnormals[i])<1e-6f || fnormals[i].z>-1e-6f) {
			faces[i]=faces[numFaces-1];
			fnormals[i]=fnormals[numFaces-1];
			numFaces--;
			i--;
		}
	}
	buildEdges();
}

void TriMesh::makeBox(Box &box) {
	freeData();

	numVerts=8;
	verts=new Vector[8];

	Vector &p=box.pmin;
	Vector &q=box.pmax;

	verts[0].set(p.x, p.y, p.z);
	verts[1].set(q.x, p.y, p.z);
	verts[2].set(p.x, p.y, q.z);
	verts[3].set(q.x, p.y, q.z);

	verts[4].set(p.x, q.y, p.z);
	verts[5].set(q.x, q.y, p.z);
	verts[6].set(p.x, q.y, q.z);
	verts[7].set(q.x, q.y, q.z);

	numFaces=12;
	faces=new TriMeshFace[12];

	faces[0].set(0, 1, 2);
	faces[1].set(1, 3, 2);

	faces[2].set(4, 6, 5);
	faces[3].set(5, 6, 7);

	faces[4].set(1, 5, 3);
	faces[5].set(5, 7, 3);

	faces[6].set(0, 2, 4);
	faces[7].set(4, 2, 6);

	faces[8].set(2, 3, 6);
	faces[9].set(3, 7, 6);

	faces[10].set(0, 4, 1);
	faces[11].set(1, 4, 5);

	buildEdges();
}

void TriMesh::makeSphere(float radius, int subdivs) {
	freeData();
	numVerts=sqr(subdivs+1);
	verts=new Vector[numVerts];

	numFaces=sqr(subdivs)*2;
	faces=new TriMeshFace[numFaces];

	int i;
	for (i=0; i<=subdivs; i++) {
		for (int j=0; j<=subdivs; j++) {
			float fx=-1.0f+2.0f*float(i)/float(subdivs);
			float fy=float(j)/float(subdivs);
			verts[i*(subdivs+1)+j]=getSphereCoords(fx*pi*0.5f, fy*pi*2.0f)*radius;
		}
	}

	for (i=0; i<subdivs; i++) {
		for (int j=0; j<subdivs; j++) {
			int fidx=i*subdivs+j;
			int vidx=i*(subdivs+1)+j;

			if (((i+j)&1)==1) {
				faces[fidx*2].set(vidx, vidx+subdivs+1, vidx+subdivs+2);
				faces[fidx*2+1].set(vidx, vidx+subdivs+2, vidx+1);
			} else {
				faces[fidx*2].set(vidx, vidx+subdivs+1, vidx+1);
				faces[fidx*2+1].set(vidx+1, vidx+subdivs+1, vidx+subdivs+2);
			}
		}
	}

	removeZeroFaces(1e-6f);
	buildEdges();
}

void TriMesh::makeCylinder(float radius, float height, int subdivs) {
	freeData();
	numVerts=(subdivs+1)*2+2;
	verts=new Vector[numVerts];

	numFaces=subdivs*4;
	faces=new TriMeshFace[numFaces];

	verts[0].set(0.0f, 0.0f, -height*0.5f);
	verts[1].set(0.0f, 0.0f, height*0.5f);
	int i;
	for (i=0; i<=subdivs; i++) {
		float f=float(i)/float(subdivs);
		float sn=sinf(f*2.0f*pi), cs=cosf(f*2.0f*pi);
		verts[2+i*2].set(radius*cs, radius*sn, -height*0.5f);
		verts[2+i*2+1].set(radius*cs, radius*sn, height*0.5f);
	}

	for (i=0; i<subdivs; i++) {
		int v0=2+i*2;
		int v1=2+(i+1)*2;
		int v2=2+(i+1)*2+1;
		int v3=2+i*2+1;
		faces[i*4].set(0, v1, v0);
		faces[i*4+1].set(v0, v1, v2);
		faces[i*4+2].set(v2, v3, v0);
		faces[i*4+3].set(1, v3, v2);
	}

	removeZeroFaces();
}

void TriMesh::buildStrips(void) {
	unsigned char *fdone=new unsigned char[numFaces];
	memset(fdone, 0, sizeof(unsigned char)*numFaces);

	int *idx=new int[numFaces*4];
	int numIdx=0;

	int *single=new int[numFaces*4];

	for (int i=0; i<numFaces; i++) {
		if (fdone[i]) continue;

		// fdone[i]=true;

		// choose an edge
		int nextFace=-1;
		int nextEdge=-1;
		int prevEdge;
		int v0, v1;

		TriMeshFace &face=faces[i];
		for (int j=0; j<3; j++) {
			TriMeshEdge &edge=edges[face.e[j]];

			if (edge.numFaces<2) continue;
			if (i!=edge.f[0] && !fdone[edge.f[0]]) {
				nextEdge=face.e[j];
				prevEdge=face.e[clip3a(j+1)];
				v0=edge.v[0]; v1=edge.v[1];
				break;
			}
			if (i!=edge.f[1] && !fdone[edge.f[1]]) {
				nextEdge=face.e[j];
				prevEdge=face.e[clip3b(j-1)];
				v0=edge.v[1]; v1=edge.v[0];
				break;
			}
		}

		if (nextEdge==-1) {
			idx[numIdx++]=face.v[0];
			idx[numIdx++]=face.v[1];
			idx[numIdx++]=face.v[2];
			idx[numIdx++]=-1;
			continue;
		}

		int sm=numFaces/2+1;
		int sb=sm;
		int sa=sm;

		if (face.v[0]!=v0 && face.v[0]!=v1) single[sb++]=face.v[0];
		if (face.v[1]!=v0 && face.v[1]!=v1) single[sb++]=face.v[1];
		if (face.v[2]!=v0 && face.v[2]!=v1) single[sb++]=face.v[2];

		single[sb++]=v1;
		single[sb++]=v0;

		nextFace=i;
		fdone[i]=1;

		// extend the strip forward
		while (1) {
			TriMeshFace &face=faces[nextFace];
			TriMeshEdge &edge=edges[nextEdge];

			if (nextFace==edge.f[0]) {
				if (edge.numFaces>1) {
					nextFace=edge.f[1];
					if (fdone[nextFace]) break;

					single[sb++]=edge.tv[1];

					fdone[nextFace]=1;

					if (single[sb-2]==edge.v[0]) nextEdge=faces[nextFace].e[clip3a(edge.fe[1]+1)];
					else nextEdge=faces[nextFace].e[clip3b(edge.fe[1]-1)];
				} else break;
			} else if (edge.numFaces>1 && nextFace==edge.f[1]) {
				nextFace=edge.f[0];
				if (fdone[nextFace]) break;

				single[sb++]=edge.tv[0];

				fdone[nextFace]=1;

				if (single[sb-2]==edge.v[1]) nextEdge=faces[nextFace].e[clip3a(edge.fe[0]+1)];
				else nextEdge=faces[nextFace].e[clip3b(edge.fe[0]-1)];
			} else break;
		}

		fdone[nextFace]=1;

		// extend the strip backward
		nextEdge=prevEdge;
		nextFace=i;

		while (1) {
			TriMeshFace &face=faces[nextFace];
			TriMeshEdge &edge=edges[nextEdge];

			if (nextFace==edge.f[0]) {
				if (edge.numFaces>1) {
					nextFace=edge.f[1];
					if (fdone[nextFace]) break;

					single[--sa]=edge.tv[1];

					fdone[nextFace]=1;

					if (single[sa+1]==edge.v[0]) nextEdge=faces[nextFace].e[clip3a(edge.fe[1]+1)];
					else nextEdge=faces[nextFace].e[clip3b(edge.fe[1]-1)];
				} else break;
			} else if (edge.numFaces>1 && nextFace==edge.f[1]) {
				nextFace=edge.f[0];
				if (fdone[nextFace]) break;

				single[--sa]=edge.tv[0];

				fdone[nextFace]=1;

				if (single[sa+1]==edge.v[1]) nextEdge=faces[nextFace].e[clip3a(edge.fe[0]+1)];
				else nextEdge=faces[nextFace].e[clip3b(edge.fe[0]-1)];
			} else break;
		}

		// check if strip needs to be flipped
		if (((sm-sa)&1)==1) {
			single[sa-1]=single[sa];
			sa--;
		}

		// add the end marker
		single[sb++]=-1;

		// copy the strip
		memcpy(idx+numIdx, single+sa, (sb-sa)*sizeof(int));
		numIdx+=(sb-sa);
	}

	if (strips) delete[] strips;
	strips=new int[numIdx];
	memcpy(strips, idx, sizeof(int)*numIdx);
	stripLength=numIdx;

	delete[] idx;
	delete[] single;
}

void TriMesh::boundingBox(Box &box) {
	box.init();
	for (int i=0; i<numVerts; i++) box+=verts[i];
}

void TriMesh::merge(TriMesh &mesh) {
	Vector *nverts=new Vector[numVerts+mesh.numVerts];

	int i;
	for (i=0; i<numVerts; i++) nverts[i]=verts[i];
	for (i=0; i<mesh.numVerts; i++) nverts[i+numVerts]=mesh.verts[i];

	TriMeshFace *nfaces=new TriMeshFace[numFaces+mesh.numFaces];
	for (i=0; i<numFaces; i++) nfaces[i]=faces[i];
	for (i=0; i<mesh.numFaces; i++) {
		for (int j=0; j<3; j++) {
			nfaces[i+numFaces].v[j]=mesh.faces[i].v[j]+numVerts;
		}
	}

	delete[] verts;
	verts=nverts;
	numVerts+=mesh.numVerts;

	delete[] faces;
	faces=nfaces;
	numFaces+=mesh.numFaces;

	buildEdges();
}

void TriMesh::splitQuads(void) {
	bool *diag=new bool[numEdges];
	memset(diag, 0, sizeof(bool)*numEdges);

	// Mark the diagonal edges and count normal faces
	int numNormal=0;
	int i;
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		bool quad=false;
		for (int j=0; j<3; j++) {
			if (face.color[j]<0.0f) { diag[face.e[j]]=true; quad=true; break; }
		}
		if (!quad) numNormal++;
	}

	// Count the diagonal edges
	int numDiags=0;
	for (i=0; i<numEdges; i++) if (diag[i]) numDiags++;

	Vector *nverts=new Vector[numVerts+numDiags];
	memcpy(nverts, verts, sizeof(Vector)*numVerts);
	int numNewVerts=numVerts;

	TriMeshFace *nfaces=new TriMeshFace[numNormal+numDiags*4];
	int numNewFaces=0;

	// Copy the normal faces
	for (i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		bool quad=false;
		for (int j=0; j<3; j++) {
			if (face.color[j]<0.0f) { quad=true; break; }
		}
		if (!quad) { nfaces[numNewFaces++]=face; continue; }
	}

	// Create the new faces and vertices
	for (i=0; i<numEdges; i++) {
		TriMeshEdge &edge=edges[i];
		if (diag[i]) {
			Vector p=0.25f*(verts[edge.v[0]]+verts[edge.v[1]]+verts[edge.tv[0]]+verts[edge.tv[1]]);
			nverts[numNewVerts]=p;

			nfaces[numNewFaces++].set(edge.v[0], numNewVerts, edge.tv[0]);
			nfaces[numNewFaces++].set(numNewVerts, edge.v[1], edge.tv[0]);
			nfaces[numNewFaces++].set(edge.tv[1], numNewVerts, edge.v[0]);
			nfaces[numNewFaces++].set(edge.tv[1], edge.v[1], numNewVerts);

			numNewVerts++;
		}
	}

	delete[] verts;
	verts=nverts;
	numVerts=numNewVerts;

	delete[] faces;
	faces=nfaces;
	numFaces=numNewFaces;

	buildEdges();
}

//********************************************
// smoothing

void loopSubdivideStep(TriMesh &mesh, TriMesh &smesh) {
	smesh.freeData();

	smesh.numVerts=mesh.numVerts+mesh.numEdges;
	smesh.verts=new Vector[smesh.numVerts];

	smesh.numFaces=mesh.numFaces*4;
	smesh.faces=new TriMeshFace[smesh.numFaces];

	// mark the boundary vertices
	int *boundary=new int[mesh.numVerts];
	memset(boundary, 0, sizeof(int)*mesh.numVerts);

	int i;
	for (i=0; i<mesh.numEdges; i++) {
		TriMeshEdge &edge=mesh.edges[i];
		if (edge.numFaces<2) {
			boundary[edge.v[0]]=1;
			boundary[edge.v[1]]=1;
		}
	}

	// find and add the neighboring vertices
	int *numNgbs=new int[mesh.numVerts];
	memset(numNgbs, 0, sizeof(int)*mesh.numVerts);

	Vector *psum=new Vector[mesh.numVerts];
	memset(psum, 0, sizeof(Vector)*mesh.numVerts);

	for (i=0; i<mesh.numEdges; i++) {
		TriMeshEdge &edge=mesh.edges[i];
		if (!boundary[edge.v[0]] || boundary[edge.v[1]]) {
			psum[edge.v[0]]+=mesh.verts[edge.v[1]];
			numNgbs[edge.v[0]]++;
		}
		if (!boundary[edge.v[1]] || boundary[edge.v[0]]) {
			psum[edge.v[1]]+=mesh.verts[edge.v[0]];
			numNgbs[edge.v[1]]++;
		}
	}

	// set the vertices of the new mesh
	for (i=0; i<mesh.numVerts; i++) {
		Vector p=psum[i];

		float n=float(numNgbs[i]);
		if (!boundary[i]) {
			float a=5.0f/8.0f-sqr(3.0f+2.0f*cosf(2.0f*pi/n))/64.0f;
			float alpha=n*(1.0f-a)/a;

			p+=alpha*mesh.verts[i];
			p/=alpha+n;
		} else {
			p+=6.0f*mesh.verts[i];
			p/=(6.0f+n);
		}

		smesh.verts[i]=p;
	}

	for (i=0; i<mesh.numEdges; i++) {
		TriMeshEdge &edge=mesh.edges[i];

		Vector p=3.0f*mesh.verts[edge.v[0]]+3.0f*mesh.verts[edge.v[1]];

		if (edge.numFaces==1) p+=p/3.0f;
		else p+=mesh.verts[edge.tv[0]]+mesh.verts[edge.tv[1]];

		p/=8.0f;

		smesh.verts[mesh.numVerts+i]=p;
	}

	// set the faces of the new mesh
	for (i=0; i<mesh.numFaces; i++) {
		TriMeshFace &face=mesh.faces[i];

		int nv0=mesh.numVerts+face.e[0];
		int nv1=mesh.numVerts+face.e[1];
		int nv2=mesh.numVerts+face.e[2];

		smesh.faces[i*4].set(nv0, nv2, face.v[0]);
		smesh.faces[i*4+1].set(nv1, nv0, face.v[1]);
		smesh.faces[i*4+2].set(nv2, nv1, face.v[2]);
		smesh.faces[i*4+3].set(nv0, nv1, nv2);
	}

	smesh.buildEdges();

	delete[] psum;
	delete[] numNgbs;
	delete[] boundary;
}

void Vlado::loopSubdivide(TriMesh &mesh, TriMesh &smesh, int numSteps) {
	if (numSteps==0) {
		smesh.init(mesh);
		return;
	}

	if (numSteps==1) {
		loopSubdivideStep(mesh, smesh);
		return;
	}

	TriMesh temp;
	loopSubdivideStep(mesh, temp);

	if (numSteps==2) {
		loopSubdivideStep(temp, smesh);
		return;
	}

	TriMesh temp1;

	TriMesh *m0=&temp, *m1=&temp1;
	for (int i=2; i<numSteps; i++) {
		loopSubdivideStep(*m0, *m1);
		m0->freeData();
		m0->init();
		swap(m0, m1);
	}

	loopSubdivideStep(*m0, smesh);
}

void TriMesh::drawWire(Rasterizer3D &rs, const Color &wireColor, bool drawTransformed) {
	if (!tverts && drawTransformed) return;
	for (int i=0; i<numEdges; i++) {
		TriMeshEdge &edge=edges[i];
		if (!drawTransformed) rs.drawLine(verts[edge.v[0]], verts[edge.v[1]], wireColor);
		else rs.drawLine(tverts[edge.v[0]], tverts[edge.v[1]], wireColor);
	}
}

void TriMesh::drawShadedWire(Rasterizer3D &rs, const Color &wireColor, bool drawTransformed) {
	if (!tverts && drawTransformed) return;
	for (int i=0; i<numEdges; i++) {
		TriMeshEdge &edge=edges[i];
		int v0=edge.v[0];
		int v1=edge.v[1];
		if (!drawTransformed) rs.drawShadedLine(verts[v0], verts[v1], vnormals[v0], vnormals[v1], wireColor);
		else rs.drawShadedLine(tverts[v0], tverts[v1], tvnormals[v0], tvnormals[v1], wireColor);
	}
}

void TriMesh::drawShaded(Rasterizer3D &rs, const Color &faceColor, bool drawTransformed) {
	if (!tverts && drawTransformed) return;
	if (drawTransformed && !tvnormals) buildTNormals();
	for (int i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		int v0=face.v[0];
		int v1=face.v[1];
		int v2=face.v[2];

		if (!drawTransformed) rs.drawShadedTriangle(verts[v0], verts[v1], verts[v2], vnormals[v0], vnormals[v1], vnormals[v2], faceColor);
		else rs.drawShadedTriangle(tverts[v0], tverts[v1], tverts[v2], tvnormals[v0], tvnormals[v1], tvnormals[v2], faceColor);
	}
}

void TriMesh::drawFaceted(Rasterizer3D &rs, const Color &faceColor, bool drawTransformed) {
	if (!tverts && drawTransformed) return;
	if (drawTransformed && !tvnormals) buildTNormals();
	for (int i=0; i<numFaces; i++) {
		TriMeshFace &face=faces[i];
		int v0=face.v[0];
		int v1=face.v[1];
		int v2=face.v[2];

		if (!drawTransformed) rs.drawShadedTriangle(verts[v0], verts[v1], verts[v2], fnormals[i], fnormals[i], fnormals[i], faceColor);
		else rs.drawShadedTriangle(tverts[v0], tverts[v1], tverts[v2], tfnormals[i], tfnormals[i], tfnormals[i], faceColor);
	}
}
