#ifndef __VGL_CAMERAS_H__
#define __VGL_CAMERAS_H__

#include "misc_ray.h"

BEGIN_VLADO

struct RayDeriv {
	Vector PdX;
	Vector PdY;
	Vector DdX;
	Vector DdY;
	void Init()	{ PdX = PdY = DdX = DdY = Vector(0,0,0); }
};

class Camera {
	public:
		// Returns false when there are no rays for the associated pixel
		virtual bool GetScreenRay(float u, float v, float dof_uc, float dof_vc, Ray& ray, RayDeriv* p=NULL)=0;
		virtual void MapToScreen(const Vector &p, float &xs, float &ys)=0;
		virtual void update(float time) {}

		virtual void LookAt(const Vector &from, const Vector &to, const Vector &upDir) {
			mPos	 = from;
			mForward = normalize(to-from);
			mRight	 = normalize(mForward^upDir);
			mUp		 = normalize(mRight^mForward);
		}

		virtual void SetPos(const Vector &pos, const Vector &forward, const Vector &up, const Vector &right) {
			mPos=pos;
			mRight=right;
			mUp=up;
			mForward=forward;
		}

		virtual void SetClippingPlanes(bool usePlanes, float nearPlane, float farPlane) {
			useClippingPlanes=usePlanes;
			clipNear=nearPlane;
			clipFar=farPlane;
		}

		void ClipRay(const Ray &ray, float &mint, float &maxt) {
			if (!useClippingPlanes) { mint=0.0f, maxt=1e18f; return; }

			float zslope=ray.dir*mForward;
			if (fabs(zslope)<1e-12f) return;

			float zval=GetZValue(ray.p);

			float t0=(clipNear-zval)/zslope;
			float t1=(clipFar-zval)/zslope;

			if (t0<0.0f) t0=0.0f;

			mint=t0;
			maxt=t1;
		}

		virtual float GetZValue(const Vector &p) {
			return (p-mPos)*mForward;
		}

		virtual Vector mapToCamSpace(const Vector &p) { return p; }

		// Camera position & orientation
		Vector	mPos;
		Vector	mRight;
		Vector	mUp;
		Vector	mForward;
		float width, height;

		bool useClippingPlanes;
		float clipNear, clipFar;

	protected:
		Vector GetNormDeriv(const Vector& deriv, const Vector& dir, float dirLen)
		{ return ((deriv*dirLen)-(dir*((dir/dirLen)*deriv)))/(dirLen*dirLen); }
};

class PinholeCamera: public Camera {
	public:
		float fov;
		float xscale, yscale;
		float width, height;
		float dofShutter, dofDist;

		PinholeCamera(void) { Init(); }
		PinholeCamera(float w, float h, float fov, float aspect=1.0f) { Init(w, h, fov, aspect); }

		void Init(void) {
			width=320.0f;
			height=400.0f;
			mPos.makeZero();
			mForward.set(0.0f, 1.0f, 0.0f);
			mUp.set(0.0f, 0.0f, 1.0f);
			mRight.set(1.0f, 0.0f, 0.0f);
			fov=45*pi/180.0f;
		}

		void Init(float w, float h, float fieldOfView, float aspect=1.0f, float shutter=0.0f, float focalDist=1.0f) {
			width=w;
			height=h;
			dofShutter=shutter;
			dofDist=focalDist;
			fov=fieldOfView; // fieldOfView*pi/180.0f;
			xscale=tanf(fov*0.5f)*2.0f/width;
			yscale=-xscale/aspect;
		}

		// From Camera
		bool GetScreenRay(float xs, float ys, float dof_uc, float dof_vc, Ray& ray, RayDeriv* pDeriv) {
			float fx, fy;
			Vlado::getDiscPoint(dof_uc, dof_vc, fx, fy);
			fx*=dofShutter;
			fy*=dofShutter;

			ray.p=mPos+mRight*fx+mUp*fy;

			float u=(xs-width*0.5f)*xscale;
			float v=(ys-height*0.5f)*yscale;

			// The line below replaces these two lines (keeps precision better)
			// --> Vector fpt=mPos+(mRight*u - mUp*v + mForward)*dofDist;
			// --> ray.dir=fpt-ray.p;

			ray.dir=(mRight*(u*dofDist-fx) + mUp*(v*dofDist-fy) + mForward*dofDist);
			float dirLen = length(ray.dir);

			// if Ray derivatives are needed, then compute them
			if (pDeriv)
			{
				pDeriv->PdX = Vector(0,0,0);
				pDeriv->PdY = Vector(0,0,0);

				pDeriv->DdX = mRight*dofDist*xscale;
				pDeriv->DdY = mUp*dofDist*yscale;
				pDeriv->DdX = GetNormDeriv(pDeriv->DdX, ray.dir, dirLen);
				pDeriv->DdY = GetNormDeriv(pDeriv->DdY, ray.dir, dirLen);
			}

			ray.dir/=dirLen;

			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			Vector offs=p-mPos;

			Vector sp(offs*mRight, offs*mUp, offs*mForward);
			sp/=sp.z;

			xs=sp.x/xscale+width*0.5f;
			ys=sp.y/yscale+height*0.5f;
		}

		void update(float time){}

		Vector mapToCamSpace(const Vector &p) {
			Vector offs=p-mPos;

			Vector sp(offs*mRight, offs*mUp, offs*mForward);

			sp.x=sp.x/(sp.z*xscale)+width*0.5f;
			sp.y=sp.y/(sp.z*yscale)+height*0.5f;
			sp.z=1.0f/sp.z;

			return sp;
		}
};

class OrthoCamera: public Camera
{
	public:
		OrthoCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 100.0f, 100.0f);
		}

		void Init(float w, float h, float CamWidth, float CamHeight)
		{
			width		= w;
			height		= h;
			mCamWidth	= CamWidth*2.0f;
			mCamHeight  = CamHeight*2.0f*(h/w);
			mXScale     = 1.0f/w;
			mYScale		= 1.0f/h;
		}

		virtual bool GetScreenRay(float xs, float ys, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{

			CamRay.p	= mPos + (xs*mXScale-0.5f)*mCamWidth*mRight + (0.5f-ys*mYScale)*mCamHeight*mUp;

//			CamRay.p	= mPos + (u-0.5f)*mWidth*mRight + (0.5f-v)*mHeight*mUp;
			CamRay.dir	= mForward;
			if(pDeriv)
			{
				pDeriv->PdX = (mCamWidth*mXScale)*mRight*1.0f;
				pDeriv->PdY = (-mCamHeight*mYScale)*mUp*1.0f;
				pDeriv->DdX = Vector(0,0,0);
				pDeriv->DdY = Vector(0,0,0);
			}
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float mCamWidth;
		float mCamHeight;
		float mXScale;
		float mYScale;
};

class BoxCamera: public Camera
{
	public:
		BoxCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 1.0f, 1.0f, 1.0f, 0.0f);
		}


		void Init(float w, float h, float x, float y, float z, float borderSize) 
		{ Init(w, h, Vector(x, y, z), borderSize); }

		void Init(float w, float h, const Vector& box, float borderSize)
		{
			width		= w;
			height		= h;
			mBox		= box;
			mHalfBox	= box*0.5f;
			mScaleU		= (2.0f*mBox.z + mBox.x)/width;
			mScaleV		= 2.0f*(mBox.z + mBox.y)/height;
			mX1 = mBox.z;
			mX2 = mX1 + mBox.x;
			mY1 = mBox.y;
			mY2 = mY1 + mBox.z;
			mY3 = mY1 + mY2;
			mBorderSize=borderSize;
		}

		virtual bool GetScreenRay(float sx, float sy, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			Vector dir;	
			if(!BoxIsec(sx*mScaleU, sy*mScaleV, dir, pDeriv))
				return false;
			float dirLen = length(dir);

			if(pDeriv)
			{
				pDeriv->PdX = Vector(0,0,0);
				pDeriv->PdY = Vector(0,0,0);
				pDeriv->DdX = GetNormDeriv(pDeriv->DdX, dir, dirLen);
				pDeriv->DdY = GetNormDeriv(pDeriv->DdY, dir, dirLen);
				pDeriv->DdX = mRight*pDeriv->DdX.x + mForward*pDeriv->DdX.y + mUp*pDeriv->DdX.z;
				pDeriv->DdY = mRight*pDeriv->DdY.x + mForward*pDeriv->DdY.y + mUp*pDeriv->DdY.z;
			}
			CamRay.dir	= normalize( mRight*dir.x + mForward*dir.y + mUp*dir.z);
			CamRay.p	= mPos;
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		Vector	mHalfBox;
		Vector	mBox;
		float	mScaleU;
		float	mScaleV;
		float   mX1;
		float   mX2;
		float   mY1;
		float   mY2;
		float   mY3;
		float		mBorderSize;

		/*	 0  1  0
		 *   2  3  4
		 *	 0  5  0
		 *	 0  6  0
		 */

		bool BoxIsec(float u, float v, Vector& vec,RayDeriv* pDeriv)
		{
			if (u <= mX1 && v <= mY2+mBorderSize*mScaleV && v >= mY1-mBorderSize*mScaleV) {
				// region 2
				vec.x = 0;
				vec.y = mBox.y + u - mX1;
				vec.z = mBox.z - v + mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->DdX = Vector(0.0f, mScaleU,     0.0f);
					pDeriv->DdY = Vector(0.0f,    0.0f, -mScaleV);
				}
				return true;
			}

			if (u >= mX2 && v <= mY2+mBorderSize*mScaleV && v >= mY1-mBorderSize*mScaleV) {
				// region 4
				vec.x = mBox.x;
				vec.y = mBox.y + mX2 - u;
				vec.z = mBox.z - v + mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->DdX = Vector(0.0f, -mScaleU,     0.0f);
					pDeriv->DdY = Vector(0.0f,     0.0f, -mScaleV);
				}
				return true;
			}

			if (u >= mX1-mBorderSize*mScaleU && u <= mX2+mBorderSize*mScaleU && v <= mY1) {
				// region 1
				vec.y = v;
				vec.z = mBox.z;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->DdY = Vector(0.0f, mScaleV, 0.0f);
					pDeriv->DdX = Vector(mScaleU, 0.0f, 0.0f);
				}
				return true;
			}

			if (u >= mX1-mBorderSize*mScaleU && u <= mX2+mBorderSize*mScaleU && v >= mY3) {
				// region 6
				vec.y = 0.0f;
				vec.z = v - mY3;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->DdY = Vector(0.0f, 0.0f, +mScaleV);
					pDeriv->DdX = Vector(mScaleU, 0.0f, 0.0f);
				}
				return true;
			}

			if (u >= mX1-mBorderSize*mScaleU && u <= mX2+mBorderSize*mScaleU && v <= mY3 && v>mY2) {
				// region 5
				vec.y = mBox.y + mY2 - v;
				vec.z = 0.0f;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if (pDeriv) {
					pDeriv->DdY = Vector(0.0f, -mScaleV, 0.0f);
					pDeriv->DdX = Vector(mScaleU, 0.0f, 0.0f);
				}
				return true;
			}

			if (u >= mX1 && u <= mX2 && v <= mY2 && v >= mY1) {
				// region 3
				vec.y = mBox.y;
				vec.z = mBox.z - v + mY1;
 				vec.x = u - mX1;
				vec -= mHalfBox;

				// ray derivatives:
				if(pDeriv) {
					pDeriv->DdX = Vector(mScaleU, 0.0f, 0.0f);
					pDeriv->DdY = Vector(0.0f, 0.0f, -mScaleV);
				}
				return true;
			}

			return false;
		}
};


/*
 *  class SphereCamera
 */
class SphereCamera: public Camera
{
	public:
		SphereCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f);
		}

  		void Init(float w, float h, float FovX, float FovY)
		{
			width=w;
			height=h;
			float aspect=(height/width)*pi;

			xscale=FovX*0.5f/(width*pi);
			yscale=Vlado::Min(FovY*aspect*0.5f, pi)/(height*pi);
		}

		virtual bool GetScreenRay(float xs, float ys, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			float u=(xs-0.5f*width)*xscale+0.25f;
			float v=(0.5f*height-ys)*yscale*2.0f;
			if (v<-1.0f) v+=2.0f;
			if (v>1.0f) v-=2.0f;

			float CosPhi = cosf(u*2.0f*pi);
			float SinPhi = sinf(u*2.0f*pi);
			float CosTheta = cosf(pi*0.5f*v) ;// sqrtf(1.0f - v*v);
			float SinTheta = sinf(pi*0.5f*v); // -v;

			// now calculate the direction:
			Vector dir		= -CosPhi*CosTheta*mRight + SinPhi*CosTheta*mForward +SinTheta*mUp;
			float dirLen	= length( dir );
			if(pDeriv)
			{
				pDeriv->PdX = Vector(0,0,0);
				pDeriv->PdY = Vector(0,0,0);
				// we just need the direction of the derivatives, so looking at the following exact
				// formulas we can omit the unnecessary operations
				
				// pDeriv->DdX = xscale*2.0f*pi*CosTheta*(SinPhi*mRight + CosPhi*mForward); // derivative dx, exact formula
				// if (fabs(CosTheta)<1e-12f) pDeriv->DdY.makeZero();
				// else pDeriv->DdY = 2.0f*v*yscale/CosTheta*(CosPhi*mRight-SinPhi*mForward)-2.0f*yscale*mUp;
				float du_dx=xscale;
				float dv_dy=-2.0f*yscale;

				float dphi_dx=pi*2.0f*du_dx;
				float dtheta_dy=pi*0.5f*dv_dy;

				pDeriv->DdX=(SinPhi*dphi_dx*CosTheta)*mRight+(CosPhi*dphi_dx*CosTheta)*mForward;
				pDeriv->DdY=(CosPhi*SinTheta*dtheta_dy)*mRight-(SinPhi*SinTheta*dtheta_dy)*mForward+(CosTheta*dtheta_dy)*mUp;

				// now normalize the derivatives:
				pDeriv->DdX = GetNormDeriv(pDeriv->DdX, dir, dirLen);
				pDeriv->DdY = GetNormDeriv(pDeriv->DdY, dir, dirLen);
			}
			CamRay.p	= mPos;
			CamRay.dir 	= dir/dirLen;
			return true;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float xscale, yscale;
		float mFovX;
		float mFovY;
};


/*
 *  class FishEyeCamera - implements the Fisheye efect. This is acheived by intersecting
 *  camera rays with a virtual sphere. Then the sphere normal at the intersection point
 *  is returned as the Screen Ray (the sphere reflects the scene into camera)
 *
 */
class FishEyeCamera: public Camera
{
	public:
		FishEyeCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f, false, 2.0f, 1.0f);
		}

		void Init(float w, float h, float FovX, float FovY, int AutoFit, float Delta=2.0f, float Curv=1.0f, float shutter=0.0f, float focalDist=1.0f)
		{
			Delta		= Max(Delta, 1.0f);
			dofShutter=shutter;
			dofDist=focalDist;

			width		= w;
			height		= h;
			float aspect=(height/width);

			mOffsetX	= width*0.5f;
			mOffsetY	= height*0.5f;
			mFovX		= FovX;
			mFovY		= FovY;

			if(AutoFit)
			{
	//			float SphSin	= 1.0f / mDelta;
	//			float MaxAngle  = asin(SphSin);
				Delta  = 1.0f /	(float)sin( Min( mFovX, mFovY ));
				if(Delta < 1.0f) Delta = 1.0f;
			}

			mFovX = (float)tan(mFovX)*2.0f/width;
			mFovY = (float)tan(mFovY)*2.0f*aspect/height;

			mSphCenter	= Vector(0.0f, Delta, 0.0f);
			mCurvFactor = Curv;
		}

		virtual bool GetScreenRay(float sx, float sy, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv)
		{
			float fx, fy;
			Vlado::getDiscPoint(dof_uc, dof_vc, fx, fy);
			fx*=dofShutter;
			fy*=dofShutter;

			CamRay.p=Vector(fx, 0.0f, fy); // mPos+mRight*fx+mUp*fy;
			// CamRay.p = Vector(0,0,0);

			float u = (sx - mOffsetX)*mFovX;
			float v = (sy - mOffsetY)*mFovY;

			CamRay.dir=Vector(-u, 1.0f, -v);

			float time;
			if(intersectSphere(CamRay, mSphCenter, 1.0f, time))
			{
				Vector point	= CamRay.p + CamRay.dir*time;	// intersection point
				Vector normal	= point - mSphCenter;			// sphere normal at intersection point
				// mCurvFactor==1.0f is equivalent to dir = getReflectDir( CamRay.dir, normal );
				Vector dir = CamRay.dir*mCurvFactor - 2.0f*normal*(normal*CamRay.dir)*(2.0f - mCurvFactor);
				if(pDeriv)
				{
					// derivatives of starting ray:
					Vector DirDX  = Vector(-mFovX, 0.0f,   0.0f); 
					Vector DirDY  = Vector(  0.0f, 0.0f, -mFovY); 
	//				DirDX = GetNormDeriv(DirDX, CamRay.dir, length(CamRay.dir));
	//				DirDY = GetNormDeriv(DirDY, CamRay.dir, length(CamRay.dir));

					// the transfer equation (the ray hits the spehre):
					pDeriv->PdX = time*DirDX;
					pDeriv->PdY = time*DirDY;

					float  dTdX =  -(pDeriv->PdX*normal)/(CamRay.dir*normal);
					float  dTdY =  -(pDeriv->PdY*normal)/(CamRay.dir*normal);

					pDeriv->PdX += dTdX*CamRay.dir;
					pDeriv->PdY += dTdY*CamRay.dir;

					// the reflection equation (the ray is reflected by the sphere):
					Vector NormDX = pDeriv->PdX;	// derivative of the normal
					Vector NormDY = pDeriv->PdY;	// derivative of the normal

					// compute derivatives according to the reflection formula:
					pDeriv->DdX   = DirDX*mCurvFactor - 2.0f*(2.0f - mCurvFactor)*
									( NormDX*(normal*CamRay.dir) + (NormDX*CamRay.dir + DirDX*normal)*normal );

					pDeriv->DdY   = DirDY*mCurvFactor - 2.0f*(2.0f - mCurvFactor)*
									( NormDY*(normal*CamRay.dir) + (NormDY*CamRay.dir + DirDY*normal)*normal );

					// normalize the derivatives:
					float dirLen = length(dir);
					pDeriv->DdX   = GetNormDeriv(pDeriv->DdX, dir, dirLen);
					pDeriv->DdY   = GetNormDeriv(pDeriv->DdY, dir, dirLen);

					// now orientate derivatives in the correct position:
					pDeriv->DdX	  = (pDeriv->DdX.x*mRight + pDeriv->DdX.y*mForward + pDeriv->DdX.z*mUp);
					pDeriv->DdY	  = (pDeriv->DdY.x*mRight + pDeriv->DdY.y*mForward + pDeriv->DdY.z*mUp);

	//				pDeriv->PdX = Vector(0,0,0);
	//				pDeriv->PdY = Vector(0,0,0);
				}

				CamRay.dir	= normalize( mRight*dir.x + mForward*dir.y + mUp*dir.z );
				point		-= mSphCenter;
				CamRay.p	= point.x*mRight+point.y*mForward+point.z*mUp + mPos;
	//			CamRay.p	= mPos;
				return true;
			}
			else
			{
				//	CamRay.p   = mPos;
				//	CamRay.dir = normalize( mRight*CamRay.dir.x + mForward*CamRay.dir.y + mUp*CamRay.dir.z );
				return false;
			}
		}

		// camera is actually looking in the opposite direction (mForward = normalize(from-to))
		// (at a virtual sphere that reflects the scene)
		virtual void LookAt(const Vector &from, const Vector &to, const Vector &upDir) 
		{
			mPos	 = from;
			mForward = normalize(from-to);
			mRight	 = normalize(mForward^upDir);
			mUp		 = normalize(mRight^mForward);
		}
		virtual void SetPos(const Vector &pos, const Vector &forward, const Vector &up, const Vector &right) {
			mPos=pos;
			mRight=-right;
			mUp=up;
			mForward=-forward;
		}


		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float	mFovX;
		float	mFovY;
		float	mOffsetX;
		float	mOffsetY;
		float   mCurvFactor;
		Vector	mSphCenter;
		float dofShutter, dofDist;
};


/*
 *  class CylindricCamera
 */
class CylindricCamera: public Camera
{
	public:
		CylindricCamera()
		{
			LookAt(Vector(0,0,0), Vector(0,1,0), Vector(0,0,1));
			Init(320.0f, 240.0f, 60.0f, 60.0f, 1, 100.0f);
		}

		virtual bool GetScreenRay(float sx, float sy, float dof_uc, float dof_vc, Ray& CamRay, RayDeriv* pDeriv=NULL)
		{
			Vector		dir;

			float u		= (0.5f-sx*mXScale)*mFovX;
			float v		= sy*mYScale;
			float CosA = (float)cos(u);
			float SinA = (float)sin(u);


			// if the rays are beamd from the cylinder's center:
			if(mbFromCenter)
			{
				float z  = (0.5f - v)*mFovY;
//				float y = 1.0f;

				// rotate dir around z-axis. dir is normalized, so is the result
				dir	= -SinA*mRight +CosA*mForward + z*mUp;
				float dirLen = length(dir);

				if(pDeriv)
				{
					pDeriv->PdX = Vector(0,0,0);
					pDeriv->PdY = Vector(0,0,0);

					pDeriv->DdX = (CosA*mFovX*mRight + SinA*mFovX*mForward)*mXScale;
					pDeriv->DdY = -mFovY*mUp*mYScale;

					pDeriv->DdX = GetNormDeriv(pDeriv->DdX, dir, dirLen);
					pDeriv->DdY = GetNormDeriv(pDeriv->DdY, dir, dirLen);
				}
				
				CamRay.p	= mPos;
				CamRay.dir	= dir / dirLen;
			}
			else
			{
				// rays are parallel with xy-plane
				dir = -SinA*mRight + CosA*mForward;
				float dirLen = length(dir);

				if(pDeriv)
				{
					pDeriv->PdX = Vector(0,0,0);
					pDeriv->PdY = -mHeight*mUp*mYScale;
					pDeriv->DdX = (CosA*mRight + SinA*mForward)*mXScale*mFovX;
					pDeriv->DdY = Vector(0,0,0);
					pDeriv->DdX = GetNormDeriv(pDeriv->DdX, dir, dirLen);
				}

				CamRay.p	= mPos + (0.5f-v)*mHeight*mUp;
				CamRay.dir	= dir / dirLen;
			}
			return true;
		}

		void Init(float w, float h, float FovX, float FovY, int FromCenter=1, float Height=100.0f) {
//			if(mFovX < 0.0f	 ) mFovX =   0.0f;
//			if(mFovX > 360.0f) mFovX = 360.0f;
//			if(mFovY < 0.0f	 ) mFovY =   0.0f;
//			if(mFovY > 180.0f) mFovY = 180.0f;

//			mFovX = FovX*pi/180.0f;									// FOVX [0..2pi]
//			mFovY = (float)tan(FovY*pi/360.0f)*2.0f;

			width			= w;
			height			= h;

			float aspect=(height/width);

			mXScale			= 1.0f/width;
			mYScale			= 1.0f/height;
			mFovX			= FovX;
			mFovY			= tanf(FovY*0.5f)*2.0f*aspect;
			mHeight			= Height;
			mbFromCenter	= FromCenter;
		}

		void MapToScreen(const Vector &p, float &xs, float &ys) {
			xs=ys=0.0f;
		}

	protected:
		float	mFovX;			// X viewing angle
		float	mFovY;			// FovY, only used when mbFromCenter is false (0)
		float   mHeight;		// cylinder height, only used when mbFromCenter is true (1)
		int		mbFromCenter;	// wheter to beam the rays from cylinder's center
		float	mXScale;
		float	mYScale;
};

END_VLADO

#endif
