/*****************************************************
	FILE: ray.h
	DESCRIPTION: A ray in 3D space; also ray-box intersection
	Copyright (C) 2001-2003 by Vladimir Koylazov
	vkoylazov@hotmail.com

	OS SUPPORT: Win32, Linux
*****************************************************/
#ifndef __RAY_H__
#define __RAY_H__

BEGIN_VLADO

class Ray {
	public:
		Vector p, dir;

		Ray(void) {}
		Ray(const Vector &pt, const Vector &d) { p=pt; dir=d; }

		void set(const Vector &pt, const Vector &d) { p=pt; dir=d; }
};

INLINE Ray operator*(const Matrix &m, const Ray &ray) {
	return Ray(m*ray.p, m*ray.dir);
}

INLINE Ray operator*(const Transform &tm, const Ray &ray) {
	return Ray(tm*ray.p, tm.transformVec(ray.dir));
}

typedef double Ireal;

class IRay {
	public:
		Vector p, dir;
		Ireal rdir[3];
		int dirFlags[3];
		Ireal cmint, cmaxt;

		IRay(void) {}
		IRay(const Vector &pt, const Vector &d, Ireal mint=0.0f, Ireal maxt=1e18f) { init(pt, dir, mint, maxt); }
		IRay(const Ray &ray, Ireal mint=0.0f, Ireal maxt=1e18f) { init(ray.p, ray.dir, mint, maxt); }

		void init(const Vector &pt, const Vector &d, Ireal mint, Ireal maxt) {
			p=pt;
			dir=d;
			cmint=mint;
			cmaxt=maxt;

			for (int i=2; i>=0; i--) {
				Ireal a=dir[i];
				if (fabs(a)<1e-12f) {
					if (a>0.0f) { dirFlags[i]=0; rdir[i]=1e18f; }
					else { dirFlags[i]=1; rdir[i]=-1e18f; }
				} else {
					rdir[i]=1.0f/a;
					if (a>0.0f) dirFlags[i]=0;
					else dirFlags[i]=1;
				}
			}
		}

		INLINE int intersectBoxDim(int dim, int dirFlag, Box &b) {
			Ireal r;
			r=(b.c(dirFlag)[dim]-p[dim])*rdir[dim];
			if (r>cmint) cmint=r;

			r=(b.c(1-dirFlag)[dim]-p[dim])*rdir[dim];
			if (r<cmaxt) cmaxt=r;

			return cmint<=cmaxt*1.000001f;
		}

		INLINE int intersectBox(Box &b, int bboxFlags) {
			if ((bboxFlags & 0x110000) && !intersectBoxDim(0, dirFlags[0], b)) return false;
			if ((bboxFlags & 0x220000) && !intersectBoxDim(1, dirFlags[1], b)) return false;
			if ((bboxFlags & 0x440000) && !intersectBoxDim(2, dirFlags[2], b)) return false;
			return true;
		}
};

END_VLADO

#endif
