#ifndef __MISC_RAY_H__
#define __MISC_RAY_H__

BEGIN_VLADO

// A color mapping class
class ColorMapFunc {
public:
	virtual Color mapColor(Color &color)=0;
};

//*******************************************************************************
// Returns a point in the unit disc, uniformly distributed

// Polar mapping
inline void getDiscPoint(real u, real v, real &fx, real &fy) {
	real phi=2.0f*pi*v;
	real rad=sqrtf(1.0f-u);

	fx=cosf(phi)*rad;
	fy=sinf(phi)*rad;
}

// Concentric mapping
inline void getDiscPoint1(real u, real v, real &fx, real &fy) {
	real uc=u*2.0f-1.0f, vc=v*2.0f-1.0f;
	int code=((uc<-vc)<<1) | (uc<vc);

	real phi, rad;
	switch (code) {
		case 0:
			// uc>-vc && uc>vc
			rad=uc;
			phi=vc/uc;
			break;
		case 1:
			// uc>-vc && uc<vc
			rad=vc;
			phi=2.0f-uc/vc;
			break;
		case 2:
			// uc<-vc && uc>vc
			rad=-vc;
			phi=6.0f-uc/vc;
			break;
		case 3:
			// uc<-vc && uc<vc
			rad=-uc;
			phi=4.0f+vc/uc;
			break;
	}

	phi*=pi*0.25f;
	fx=cosf(phi)*rad;
	fy=sinf(phi)*rad;
}

//*******************************************************************************
// Returns a direction on a sphere, given two spherical coordinates
inline Vector getSphereCoords(real theta, real phi) {
	real thetaCos=cosf(theta);
	real thetaSin=sinf(theta);
	return Vector(cosf(phi)*thetaCos, sinf(phi)*thetaCos, real(thetaSin));
}

//*******************************************************************************
// Returns a direction on a hemisphere, directions are uniformly distributed
inline Vector getSphereDir(real u, real v) {
	real thetaSin=u;
	real thetaCos=sqrtf(1.0f-thetaSin*thetaSin);
	real phi=2.0f*pi*v;
	return Vector(cosf(phi)*thetaCos, sinf(phi)*thetaCos, real(thetaSin));
}

//*******************************************************************************
// Returns the inverse probability of a direction to be picked by getSphereDir
inline float getSphereDirProbInverse(Vector &dir, Vector &nrm) {
	return 2.0f*pi;
}

//*******************************************************************************
// Returns a direction on a hemisphere for a diffuse reflection
static float pi2=pi*2.0f;

// Polar mapping
inline Vector getDiffuseDir(real u, real v) {
	real thetaSin=sqrtf(u);
	real thetaCos=sqrtf(1.0f-u);
	real phi=2.0f*pi*v;

	return Vector(cosf(phi)*thetaCos, sinf(phi)*thetaCos, real(thetaSin));
}

// Given a direction, return the polar coordinates
inline void invertDiffuseDir(Vector &dir, real &u, real &v) {
	u=sqr(dir.z);
	v=atan2f(dir.y, dir.x)*0.5f/pi;
	if (v<0.0f) v=1.0f+v;
}

// Concentric mapping - generate point on a disc and then map onto hemisphere
inline Vector getDiffuseDir1(real u, real v) {
	real uc=u*2.0f-1.0f, vc=v*2.0f-1.0f;
	int code=((uc<-vc)<<1) | (uc<vc);

	real phi, rad;
	switch (code) {
		case 0:
			// uc>-vc && uc>vc
			rad=uc;
			phi=vc/uc;
			break;
		case 1:
			// uc>-vc && uc<vc
			rad=vc;
			phi=2.0f-uc/vc;
			break;
		case 2:
			// uc<-vc && uc>vc
			rad=-vc;
			phi=6.0f-uc/vc;
			break;
		case 3:
			// uc<-vc && uc<vc
			rad=-uc;
			phi=4.0f+vc/uc;
			break;
	}

	phi*=pi*0.25f;
	return Vector(cosf(phi)*rad, sinf(phi)*rad, sqrtf(1.0f-rad*rad));
}

inline void invertDiffuseDir1(Vector &dir, real &u, real &v) {
	int code=((dir.x<-dir.y)<<1) | (dir.x<dir.y);
	real rad=sqrtf(dir.x*dir.x+dir.y*dir.y);
	real phi=atan2f(dir.y, dir.x)*4.0f/pi;
	if (phi<-1.0f) phi=8.0f+phi;
	switch (code) {
		case 0:
			// uc>-vc && uc>vc
			u=rad;
			v=phi*u;
			break;
		case 1:
			// uc>-vc && uc<vc
			v=rad;
			u=-(phi-2.0f)*v;
			break;
		case 2:
			// uc<-vc && uc>vc
			v=-rad;
			u=-(phi-6.0f)*v;
			break;
		case 3:
			// uc<-vc && uc<vc
			u=-rad;
			v=(phi-4.0f)*u;
			break;
	}
	u=u*0.5f+0.5f;
	v=v*0.5f+0.5f;
}

//*******************************************************************************
// Returns the inverse probability of picking the given direction with getDiffuseDir
inline float getDiffuseDirProbInverse(Vector &dir, Vector &nrm) {
	float cs=dir*nrm;
	if (cs<0.0f) return 0.0f;
	return pi/cs;
}

//*******************************************************************************
// Returns a direction on a hemisphere for a specular reflection
inline Vector getSpecularDir(real u, real v, real n) {
	real thetaSin=powf(u, 1.0f/(n+1.0f));
	real thetaCos=sqrtf(1.0f-thetaSin*thetaSin);
	real phi=2.0f*pi*v;
	return Vector(cosf(phi)*thetaCos, sinf(phi)*thetaCos, real(thetaSin));
}

//*******************************************************************************
// Returns the inverse probability of a direction being picked by getSpecularDir
inline float getSpecularDirProbInverse(Vector &dir, Vector &nrm, real n) {
	float cs=dir*nrm;
	if (cs<0.0f) return 0.0f;
	return 2.0f*pi/((n+1.0f)*powf(cs, n));
}

//*******************************************************************************
// Returns a NORMAL on the hemisphere based on Schlick's BRDF model
// r is roughness (0==specular, 1==diffuse), p is isotropy (0==anisotropic, 1==isotropic)
inline Vector getSchlickNormal(float u, float v, float r, float p=1.0f) {
	if (r<1e-6f) return Vector(0.0f, 0.0f, 1.0f);

	int q=FLOOR(v*4.0f);
	float f=v*4.0f-float(q);
	float phi=float(q)*pi*0.5f+pi*0.5f*sqrtf(sqr(p*f)/(1.0f-sqr(f)+sqr(p*f)));

	float cosAlpha=clamp(sqrtf(u/(r-u*r+u)), 0.0f, 1.0f);
	float sinAlpha=sqrtf(1.0f-sqr(cosAlpha));

	return Vector(cosf(phi)*sinAlpha, sinf(phi)*sinAlpha, cosAlpha);
}

//*******************************************************************************
// Returns the reflected direction
inline Vector getReflectDir(Vector &viewDir, Vector &normal) {
	return viewDir-2.0f*normal*(normal*viewDir);
}

//*******************************************************************************
// Returns the refracted direction
inline Vector getRefractDir(Vector &viewDir, Vector &normal, real ior, bool &internalReflection) {
	Vector h=normal^viewDir;
	if (lengthSqr(h)<1e-6f) return viewDir;
	else {
		int outToIn=((normal*viewDir)<0.0f);

		float sina=length(h);
		h=(h^normal)/sina;

		float sinb=outToIn? sina/ior : sina*ior;
		if (sinb>1.0) {
			if (outToIn) sinb=0.95f;
			else {
				internalReflection=true;
				return getReflectDir(viewDir, normal);
			}
		}

		internalReflection=false;

		float cosb=sqrtf(1.0f-sinb*sinb);
		if (outToIn) cosb=-cosb;
		return normalize(cosb*normal+sinb*h);
	}
}

//*******************************************************************************
// Returns the Fresnel coefficient (reflected/refracted intensity ratio)
inline float getFresnelCoeff(Vector &viewDir, Vector &normal, Vector &refractDir) {
	float cosIn=-(viewDir*normal);
	float sinIn=sqrtf(1.0f-cosIn*cosIn);

	float cosR=-(refractDir*normal);
	float sinR=sqrtf(1.0f-cosR*cosR);

	float sinDiff=sinIn*cosR-cosIn*sinR;
	float cosDiff=cosIn*cosR+sinIn*sinR;

	float sinSum=sinIn*cosR+cosIn*sinR;
	float cosSum=cosIn*cosR-sinIn*sinR;

	return 0.5f*sqr(sinDiff/sinSum)*(1.0f+sqr(cosSum/cosDiff));
}

//*******************************************************************************
// Returns the Fresnel coefficient (reflected/refracted intensity ratio), more precise than the above
inline float getFresnelCoeff(Vector &viewDir, Vector &normal, Vector &refractDir, float ior) {
	float cosIn=-(viewDir*normal);
	float cosR=-(refractDir*normal);

	if (cosIn>1.0f-1e-12f || cosR>1.0f-1e-12f) {
		// View direction is perpendicular to the surface
		return sqr((ior-1.0f)/(ior+1.0f));
	}

	float ks=(cosR/cosIn)*ior;
	float Fs=sqr((ks-1.0f)/(ks+1.0f));

	float kp=(cosIn/cosR)*ior;
	float Fp=sqr((kp-1.0f)/(kp+1.0f));

	return 0.5f*(Fs+Fp);
}

//*******************************************************************************
// Given a normal vector, computes two orthogonal vectors in the tangent plane
inline void computeTangentVectors(const Vector &n, Vector &u, Vector &v) {
	u=n^Vector(0.643782f, 0.98432f, 0.324632f);
	if (lengthSqr(u)<1e-12f) u=n^Vector(0.432902f, 0.43223f, 0.908953f);
	/*u=n^Vector(0,0,1);
	if (lengthSqr(u)<1e-12f) u=n^Vector(0,1,0);*/
	u=normalize(u);
	v=normalize(n^u);
}

//*******************************************************************************
// Given a normal vector, computes a matrix for transforming from spherical coords to world coords
inline void makeNormalMatrix(const Vector &n, Matrix &m) {
	computeTangentVectors(n, m.f[0], m.f[1]);
	m.f[2]=n;
}

//*******************************************************************************
// Given a normal vector, computes a transformation from spherical coords to world coords
inline void makeNormalTransform(const Vector &p, const Vector &n, Transform &tm) {
	makeNormalMatrix(n, tm.m);
	tm.offs=p;
}

//*******************************************************************************
// Computes an intersection of a ray with a sphere, assume ray origin is outside the sphere
inline bool intersectSphere(Ray &ray, Vector &center, float radius, float &t) {
	Vector offs=ray.p-center;

	float ac=ray.dir*ray.dir;
	float bc=offs*ray.dir;
	float cc=offs*offs-sqr(radius);

	if (cc<0.0f) return false; // If ray origin is inside the sphere, return no intersection

	float D=bc*bc-ac*cc;
	if (D<1e-6f) return false;

	t=(-bc-sqrtf(D))/ac;
	return true;
}

//*******************************************************************************
// Computes a point on a surface passing through 3 given points and the normals at those points,
// and given the barycentric coordinates
inline Vector hermite(Vector &p0, Vector &n0, Vector &p1, Vector &n1, float u, Vector *tangent=NULL) {
	float A=((p0-p1)*n0)/(n0*n0);
	float B=((p0-p1)*n1)/(n1*n1);

	float a=A+B;
	float b=-B-2.0f*A;
	float c=A;

	float offset=((a*u+b)*u+c)*u;

	Vector n=n0*(1.0f-u)+n1*u;
	Vector p=p0*(1.0f-u)+p1*u+offset*n;

	if (tangent) *tangent=(p1-p0)+(n1-n0)*offset+n*((3.0f*a*u+2.0f*b)*u+c);

	return p;
}

inline Vector hermite2(Vector &p0, Vector &n0, Vector &p1, Vector &n1, Vector &p2, Vector &n2, float u, float v) {
	Vector tg0, tg1;
	Vector q0=hermite(p0, n0, p1, n1, u+v, &tg0);
	Vector q1=hermite(p0, n0, p2, n2, u+v, &tg1);

	Vector t0=(tg0^(n0+(n1-n0)*(u+v)))^tg0;
	Vector t1=(tg1^(n0+(n2-n0)*(u+v)))^tg1;
	return hermite(q0, t0, q1, t1, v/(u+v));
}

inline float hermitef(float t) { return (3.0f-2.0f*t)*t*t; }

inline Vector interpolateNormalSurface(Vector &p0, Vector &p1, Vector &p2, Vector &n0, Vector &n1, Vector &n2, Vector &bary) {
	Vector pv[3];

	pv[0]=hermite2(p0, n0, p1, n1, p2, n2, bary.y, bary.z);
	pv[1]=hermite2(p1, n1, p2, n2, p0, n0, bary.z, bary.x);
	pv[2]=hermite2(p2, n2, p0, n0, p1, n1, bary.x, bary.y);

	float wt[3]={ hermitef(bary.x), hermitef(bary.y), hermitef(bary.z) };

	return (pv[0]*wt[0]+pv[1]*wt[1]+pv[2]*wt[2])/(wt[0]+wt[1]+wt[2]);
}

END_VLADO

#endif
