/******************************************************
	FILE: SimClothModifier.h
	DESCRIPTION: This is header file for the modifier plugin in Max
	CREATED: 1st December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) 2000-2002 by Vladimir Koylazov

	HISTORY:
		December 1st: file created

******************************************************/

#ifndef __SIMCLOTHMODIFIER_H__
#define __SIMCLOTHMODIFIER_H__

// Disable the "function xxx is hidden by yyy - virtual override intended?" warning of ICL
// Which is issued all over the place in the MaxSDK
#ifdef __INTEL_COMPILER
#pragma warning(disable: 1125)
#endif

#include "SimClothEngine.h"
#include "SimClothCache.h"

// The simcloth3 plugin ClassID
#define	SIMCLOTH3_CLASS_ID Class_ID(0x7f8e55b4, 0x2acd6712)

// Reference IDs
#define REFNO_PBLOCK_GENERAL 0
#define REFNO_PBLOCK_INTEGRITY 1
#define REFNO_PBLOCK_COLLISIONS 2
#define REFNO_GROUPS 10

// Access to the class descriptor
extern ClassDesc* GetSimCloth3Desc();
extern TCHAR *GetString(int id);

// Parameter block IDs
enum {
	sc3_general, // General parameters
	sc3_global, // Global simulation parameters (gravity, air resistance etc)
	sc3_integrity, // Surface properties (stretch/bend etc)
	sc3_collisions, // Collision properties
}; 

// Parameters for the "General" pblock
enum {
	sc3_general_type,
	sc3_general_smoothResult,
	sc3_general_smoothIterations,
	sc3_general_particleMass,
};

// Parameters for the "Global" pblock
enum {
	sc3_global_startFrame,
	sc3_global_endFrame,
	sc3_global_substeps,
	sc3_global_adaptiveSubdivs,
	sc3_global_gravity,
	sc3_global_collisionTolerance,
	sc3_global_solverPrecision,
	sc3_global_useActiveTimeSegment,
	sc3_global_checkIntersections,
	sc3_global_useSSE,
};

// Parameters for the "Integrity" pblock
enum {
	sc3_integrity_stretchType,
	sc3_integrity_springsStiffness,
	sc3_integrity_springsDamping,

	sc3_integrity_stretch_stiffness,
	sc3_integrity_shear_stiffness,
	sc3_integrity_stretchShear_damping,

	sc3_integrity_bendType,
	sc3_integrity_bendSpring_stiffness,
	sc3_integrity_bendSpring_damping,

	sc3_integrity_bendAngle_stiffness,
	sc3_integrity_bendAngle_damping,
	sc3_integrity_bendEnabled,

	sc3_integrity_spacewarps,
	sc3_integrity_airDrag,

	sc3_integrity_excludeSelectedEdges,
};

// Parameters for the "Collisions" pblock
enum {
	sc3_collisions_selfCollide,
	sc3_collisions_friction,
};

// MaxScript function access
#if GET_MAX_RELEASE(VERSION_3DSMAX) >= 4000

#include "iFnPub.h"

// Interface ID
#define SC3_INTERFACE Interface_ID(0x37227d2d, 0x5869464c)
#define GetSCInterface(obj) ((SC3Interface*) obj->GetInterface(SC3_INTERFACE)) 

// Function IDs 
enum { sc3_calculate, }; 

// Mixin interface 
class SC3Interface: public FPMixinInterface { 
	BEGIN_FUNCTION_MAP 
		FN_0(sc3_calculate, TYPE_INT, Calculate); 
	END_FUNCTION_MAP 

	FPInterfaceDesc* GetDesc();
	virtual int Calculate(void)=0; 
};

#endif

//******************************************************
// SimCloth3 modifier definition

class VertexGroup;
class SimClothObject;

class SimCloth3: public Modifier
#if GET_MAX_RELEASE(VERSION_3DSMAX) >= 4000
, public SC3Interface
#endif
{
	public:
		// Interface pointer passed by Max in BeginEditParams()
		static IObjParam *ip;

		// Parameter blocks
		IParamBlock2 *pblockGeneral, *pblockIntegrity, *pblockCollisions;
		Tab<VertexGroup*> groups; // The vertex groups

		// A flag to indicate whether the modifier should actually modify the underlying object, or just
		// pass it though unchanged. This is used when the original shape of the object is needed (for example
		// at the start of the simulation to get the initial cloth state)
		int dontModify;

		// About and subobject rollup windows
		HWND aboutWnd, subobjWnd;

		// Used for subobject selection
		SelectModBoxCMode *selectMode;

		// The modifier currently being edited
		SimCloth3 *editMod;

		// The currently selected subobject points
		int numPts;
		unsigned char *selPts;

		// Constructor/destructor/deletion
		SimCloth3();
		~SimCloth3();
		void DeleteThis() { delete this; }

		// Plugin identification
		void GetClassName(TSTR& s);
		Class_ID ClassID();
		TCHAR *GetObjectName();

		// Defines the behavior for this modifier
		ChannelMask ChannelsUsed()  { return ALL_CHANNELS; }
		ChannelMask ChannelsChanged() { return ALL_CHANNELS; }
		Class_ID InputType() { return triObjectClassID; }
		BOOL ChangeTopology() { return FALSE; }

		// Calculate the local validity from the parameters
		Interval LocalValidity(TimeValue t);
		Interval GetValidity(TimeValue t);
		Interval ObjectValidity(TimeValue t);

		// Object modification and notification of change
		void ModifyObject(TimeValue t, ModContext &mc, ObjectState *os, INode *node);
		void NotifyInputChanged(Interval changeInt, PartID partID, RefMessage message, ModContext *mc);

		// Reference support
		int NumRefs();
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		RefResult NotifyRefChanged( Interval changeInt,RefTargetHandle hTarget, PartID& partID, RefMessage message);

		// SubAnim support
		int NumSubs();
		Animatable* SubAnim(int i);
		TSTR SubAnimName(int i);

		// Direct paramblock access
		int	NumParamBlocks();
		IParamBlock2* GetParamBlock(int i);
		IParamBlock2* GetParamBlockByID(BlockID id);
		int GetParamBlockIndex(int id);

		// Does not use createmouse callbacks
		CreateMouseCallBack* GetCreateMouseCallBack() { return NULL; }

		// Load and unload the UI
		void BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev);
		void EndEditParams(IObjParam *ip, ULONG flags, Animatable *next);
		void InvalidateUI();

		// File loading and saving
		IOResult Save(ISave *isave);
		IOResult Load(ILoad *iload);

		IOResult LoadLocalData(ILoad *iload, LocalModData **pld);
		IOResult SaveLocalData(ISave *isave, LocalModData *ld);

		// Subobjects
		void ActivateSubobjSel(int level, XFormModes& modes);
		void GetWorldBoundBox(TimeValue t, INode* inode, ViewExp *vpt, Box3& box, ModContext *mc);
		int Display(TimeValue t, INode* inode, ViewExp *vpt, int flagst, ModContext *mc);
		int HitTest(TimeValue t, INode* inode, int type, int crossing, int flags, IPoint2 *p, ViewExp *vpt, ModContext* mc);
		void SelectSubComponent(HitRecord *hitRec, BOOL selected, BOOL all, BOOL invert);
		void GetSubObjectCenters(SubObjAxisCallback *cb,TimeValue t,INode *node,ModContext *mc);
		void GetSubObjectTMs(SubObjAxisCallback *cb,TimeValue t,INode *node,ModContext *mc);
		void ClearSelection(int selLevel);
		void SelectAll(int selLevel);
		void InvertSelection(int selLevel);
		int SubObjectIndex(HitRecord *hitRec);
#if GET_MAX_RELEASE(VERSION_3DSMAX) >= 4000
		int NumSubObjTypes(void) { return 1; }
		ISubObjType *GetSubObjType(int i) {
			if (i==0) return new GenSubObjType(_T("Vertex"), _T("SubObjectIcons"), 0);
			return NULL;
		}

		// From SC3Interface
		BaseInterface* GetInterface(Interface_ID id) {
			if (id==SC3_INTERFACE) return (SC3Interface*) this;
			else return Modifier::GetInterface(id);
		}
		int Calculate(void) { return calculate(); }
#endif

		// Other methods
		void grayGlobal(IParamMap2 *map); // Gray the controls in the "Global" parameter pblock
		void clearCaches(void); // Clears the caches of all objects with this modifier
		void clearNodeCache(INode *root); // Clears the cache of the node and its children
		int calculate(void);

		// Groups methods
		void listGroups(HWND lbox); // Find all groups for this modifier and stuff them into the list box
		void setGroup(TCHAR *name); // Copy the currently selected points into the given group or create a new one
		VertexGroup *selectGroup(TCHAR *name); // Copy the selection from the given group to the current selection
		void delGroup(TCHAR *name); // Delete the group with the given name
		void setSelected(int numPts, unsigned char *selPts); // Copy the given selection into the current selection

		int getVertexCollide(int index); // Returns true if the vertex is allowed to participate in cloth/deflector collisions
		int getVertexSelfCollide(int index); // Returns true if the vertex is allowed to participate in cloth/cloth collisions within the same object
		int getVertexCheckIntersections(int index); // Returns true if the vertex is allowed to participate in intersection checks
		int getAttachPos(int index, Vlado::Vector &p);

		void beginSimulation(TimeValue t, SimClothObject *scObj);
		void updateSimulation(TimeValue t, SimClothObject *scObj);
		void endSimulation(void);
};

//******************************************************
// Class SimCloth3DlgProc
// The dialog box processor for simcloth3 interface
class SimCloth3DlgProc: public ParamMap2UserDlgProc {
	public:
		// The SimCloth3 modifier
		SimCloth3 *mod;

		// Constructor/destructor/deletion
		SimCloth3DlgProc() {}
		void DeleteThis() {}

		BOOL DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
		void SetThing(ReferenceTarget *m) { mod=(SimCloth3*) m; }
};

//******************************************************
// Class SimClothLocalData
// Stores the point cache for an object (see class SimClothCache)
class SimClothLocalData: public LocalModData {
	public:
		// The point cache for the object
		SimClothCache cache;

		// The modified mesh, cached for handy access during subobject operations
		Mesh *cachedMesh;

		CRITICAL_SECTION csect;

		SimClothLocalData(void);
		~SimClothLocalData(void);
		LocalModData *Clone(void) {
			SimClothLocalData *newData=new SimClothLocalData;
			newData->cache.init(cache);
			return newData;
		}

		void freeData(void);
		void saveMesh(Mesh *mesh); // Saves the given mesh to cachedMesh
};

//******************************************************
// class SimCloth3ClassDesc
// The description of the plugin class, an instance is returned to Max in LibClassDesc()
class SimCloth3ClassDesc:public ClassDesc2 {
	public:
		int IsPublic() { return TRUE; }
		void* Create(BOOL loading ) { return new SimCloth3; }
		const TCHAR* ClassName() { return GetString(IDS_CLASSNAME); }
		SClass_ID SuperClassID() { return OSM_CLASS_ID; }
		Class_ID ClassID() { return SIMCLOTH3_CLASS_ID; }
		const TCHAR* Category() { return _T("");  }

		// Hardwired name, used by MAX Script as unique identifier
		const TCHAR* InternalName() { return _T("SimCloth3"); }
		HINSTANCE HInstance() { return hInstance; }
};

#endif
