/********************************************************
	FILE: SimClothEngine.h
	DESCRIPTION: Header file for the simcloth engine interface
	CREATED: 1 December 2002
	AUTHOR: Vladimir Koylazov
	Copyright (C) by Vladimir Koylazov (Vlado)

	HISTORY:

*********************************************************/

#ifndef __SIMCLOTHENGINE_H__
#define __SIMCLOTHENGINE_H__

#include "Utils.h"
#include "Table.h"
#include "CGSolver.h"
#include "CollisionMesh.h"
#include "Collision.h"
#include "CollisionDetector.h"

// A structure with the parameters for the SimCloth engine
struct SimClothEngineParams {
	int substeps;
	int adaptiveSubdivs;
	Vlado::Vector gravity;
	float collisionTolerance;
	float solverPrecision;
	int checkIntersections;
	int useSSE;
};

// A class that prepares the data for the SimCloth engine
class DynamicObject;
class SimClothConnection {
	public:
		virtual int numDynamicObjects(void)=0;
		virtual DynamicObject* getDynamicObject(int i)=0;
		virtual void getSimClothEngineParams(SimClothEngineParams &params)=0;

		virtual void displayMessage(TCHAR *str)=0;
		virtual void setFrameProgressRange(int minRange, int maxRange)=0;
		virtual void setFrameProgressPos(int progressPos)=0;
		virtual void setNumCollisions(int collisions)=0;
		virtual void setNumCGIterations(int iterations)=0;
		virtual int getAbort(void)=0;
};

enum DynamicObjectType { obj_deflector, obj_cloth, obj_rigid };

// A dynamic object (a collection of particles)
class DynamicObject {
	public:
		DynamicObjectType objType;
		int is_selfCollide;
		float is_friction;
		int collisionPriority;

		// Support for the collision detector template class
		typedef Vlado::BVTree<Vlado::BVertex<DynamicObject> > BVertexTree;
		typedef Vlado::BVTree<Vlado::BEdge<DynamicObject> > BEdgeTree;
		typedef Vlado::BVTree<Vlado::BFace<DynamicObject> > BFaceTree;

		Vlado::Vector *origp;
		BVertexTree vertexTree;
		BEdgeTree edgeTree;
		BFaceTree faceTree;
		int *vertexFlags;

		void allocOrigp(int n) {
			if (origp) delete[] origp;
			if (vertexFlags) delete[] vertexFlags;
			origp=new Vlado::Vector[n];
			vertexFlags=new int[n];
		}
		void freeOrigp(void) {
			if (origp) delete[] origp; origp=NULL;
			if (vertexFlags) delete[] vertexFlags; vertexFlags=NULL;
		}
		Vlado::Vector &getOrigp(int i) { return origp[i]; }
		BVertexTree &getVertexTree(void) { return vertexTree; }
		BEdgeTree &getEdgeTree(void) { return edgeTree; }
		BFaceTree &getFaceTree(void) { return faceTree; }

		// The following methods are used during collision detection
		virtual Vlado::Vector vertexPos(int i)=0;
		virtual Vlado::Vector vertexVel(int i)=0;
		virtual Vlado::Vector vertexPosPrev(int i)=0;
		virtual int isVertexFixed(int i)=0;
		virtual int selfCollide(void)=0;

		Vlado::real getDynamicFriction(void) { return getFriction(); }
		Vlado::real getStaticFriction(void) { return 0.0f; }
		Vlado::real getBounce(void) { return 0.0f; }

		// The following methods are used during collision response
		virtual void prepareCollisionDetection(void)=0;
		virtual int getNumVerts(void)=0;
		virtual int getNumEdges(void)=0;
		virtual int getNumFaces(void)=0;
		virtual int getEdgeVertex(int edgeIdx, int vertexIdx)=0;
		virtual int getFaceVertex(int faceIdx, int vertexIdx)=0;
		virtual int invisibleEdge(int edgeIdx)=0;
		virtual Vlado::Vector getVertexObjPos(int vertIdx)=0;

		virtual Vlado::Vector vertexNormal(int vertexIndex)=0;
		virtual Vlado::Vector edgeNormal(int vertexIndex)=0;

		// The particles in the object
		int numParts;
		Vlado::Particle<DynamicObject> *parts;

		Vlado::Transform rigidBodyPosTm; // The transform for a rigid body; this is used just
			// for convenience and is equal to Transform(Matrix(parts[1].np-parts[0].np, parts[2].np-parts[0].np, parts[3].np-parts[0].np), parts[0].np)
		Vlado::Transform rigidBodyVelTm; // The transform for a rigid body; this is used just
			// for convenience and is equal to Transform(Matrix(parts[1].nv-parts[0].nv, parts[2].nv-parts[0].nv, parts[3].nv-parts[0].nv), parts[0].nv)
		Vlado::Transform rigidBodyPrevPosTm; // The transform for a rigid body; this is used just
			// for convenience and is equal to Transform(Matrix(parts[1].pp-parts[0].pp, parts[2].pp-parts[0].pp, parts[3].pp-parts[0].pp), parts[0].pp)

		virtual void applyForces(float dt, int firstCall)=0; // This is called to apply the various forces to the particles
			// As a result, each particle should have its f member set to the sum of the forces acting on the particle
		int isDeflector(void) { return objType==obj_deflector; } // Returns true if the object is a deflector (all particles are fixed)
		int isSelfCollideOn(void) { return is_selfCollide; } // Returns true if self-collisions should be detected
		float getFriction(void) { return is_friction; } // Returns the friction for the object, 0.0 means no friction,
			// higher values increase the friction
		int isRigidBody(void) { return objType==obj_rigid; }

		void init(void) {
			numParts=0;
			parts=NULL;
			origp=NULL;
			vertexFlags=NULL;
		}

		void freeData(void) {
			numParts=0;
			if (parts) delete[] parts;
			parts=NULL;
			if (origp) delete[] origp;
			origp=NULL;
			vertexTree.freeData();
			edgeTree.freeData();
			faceTree.freeData();
		}

		void setNumParts(int n) {
			freeData();
			numParts=n;
			parts=new Vlado::Particle<DynamicObject>[numParts];
		}

		void applyGravity(const Vlado::Vector &g) {
			for (int i=0; i<numParts; i++) {
				if (parts[i].invMass) parts[i].f+=g/parts[i].invMass;
			}
		}

		void clearForces(void) {
			for (int i=0; i<numParts; i++) {
				parts[i].f.makeZero();
				parts[i].posImp[0].makeZero();
				parts[i].posImp[1].makeZero();
				parts[i].velImp[0].makeZero();
				parts[i].velImp[1].makeZero();
			}
		}

		// Support for the cloth force template classes
		Vlado::Vector getPos(int i) { return parts[i].np; }
		Vlado::Vector getVel(int i) { return parts[i].nv; }
		Vlado::Vector getForce(int i) { return parts[i].f; }
		float getInvMass(int i) { return parts[i].invMass; }
		void applyForce(int i, const Vlado::Vector &f) { parts[i].f+=f; }
		
		Vlado::Particle<DynamicObject> *getParticle(int index) { return parts+index; }
		int getCollidePriority(void) { return collisionPriority; }
};

// The SimCloth engine itself
class SimClothEngine {
	private:
		SimClothConnection *connection; // The SimCloth connection
		Vlado::Table<DynamicObject*> objects; // The objects in the simulation
		Vlado::CollisionDetector<DynamicObject> cdetector; // The collision detector
		SimClothEngineParams params; // The SimCloth engine parameters
		Vlado::Vector *r, *p, *r1; // These are passed to cgSolveVectorTarget() as working variables

		int totalParticles; // The total number of particles in the simulation

		float dt, dt_inv; // Time step and the inverse of the time step
		int integrationSteps;
		int fastSelfCollide;
		int collisionsOnly;

		int substeps;
		int incStep;
		int sum, numSum;
		int substepWait;
		int subdivLevel;
		int savePos;
		int okSteps;

		// Make a single substep of the simulation
		int makeSubstep(float dt, int first, float substepOffset, float substepSize);

		// Compute the forces acting on the particles
		void applyForces(float dt, int firstCall);

		// Compute the collision response (the position and velocity impulses for the particles)
		void applyCollisions(float dt);

		// Integration - computes the position and velocity of the particles at the next moment of time
		int implicitEuler(float dt, float substepOffset, float substepSize);
	public:
		// Set the simulation environment
		void setConnection(SimClothConnection *connection);

		// Prepares the simulation
		void beginSimulation(void);

		// Make one simulation step
		int step(float timeStep);

		// Cleanes up after simulation
		void endSimulation(void);

		// Support for cgSolveVectorTarget
		Vlado::Vector *get_r(void) { return r; } // Returns some temporary storage
		Vlado::Vector *get_r1(void) { return r1; } // Returns some temporary storage
		Vlado::Vector *get_p(void) { return p; } // Returns some temporary storage
		int getNumVars(void); // Returns the number of simulated variables
		void addChanges(Vlado::Vector *x, Vlado::real scale); // Adds the given vectors to the simulated variables
		void addChangesEven(Vlado::Vector *x, Vlado::real scale); // Adds the given vectors to the simulated variables
		void addChangesOdd(Vlado::Vector *x, Vlado::real scale); // Adds the given vectors to the simulated variables
		void computeChanges(Vlado::Vector *x, int firstCall); // Computes the difference between the current variables and the target state
		void reset(float frameTime, float substepSize);
};

#endif