//Blender reading
#include "bParse/bBlenderFile.h"
#include "bParse/bMain.h"
#include "bParse/autogenerated/blender.h"

//Scene generation
#include <vector>
#include "geom.h"
#include "camera.h"


//Class
#include "BlenderReader.h"

using namespace bParse;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @typedef	std::vector<Blender::Object*> ObjectsVector
///
/// @brief	Defines an alias representing the blender objects vector .
////////////////////////////////////////////////////////////////////////////////////////////////////
typedef std::vector<Blender::Object*> ObjectsVector;


///////////////////Implementation///////////////////////
CBlenderReader::CBlenderReader(void)
{
}

CBlenderReader::~CBlenderReader(void)
{
	delete m_File;
}

int CBlenderReader::ReadFile(const char* FileName) {
	m_LastError = kOk;
	m_File = new bBlenderFile(FileName);
	m_File->parse(false);
	if(!m_File->ok()) {
		delete m_File;
		m_File = NULL;
		m_LastError = kParsingFailed;
		return m_LastError;
	}
		
	return m_LastError;
}

Sphere* CBlenderReader::GenerateScene(unsigned int &GeneratedSpheres, Camera &SceneCamera) {
	m_LastError = kOk;
	GeneratedSpheres = NULL;

	bMain* main = m_File->getMain();

	if(main == 0) {
		m_LastError = kNoMain;
		return NULL;
	}

	//Objects to add to scene
	std::vector<Blender::Object*> objs_to_scene;
	std::vector<Blender::Lamp*> lamps_to_scene;
	//First camera from scene
	Blender::Object* camera = NULL;

	//Find all meshes,lamps and spheres
	bListBasePtr* objs = main->getObject();
	for (size_t i=0; i<objs->size(); i++) {
		Blender::Object *ob= (Blender::Object*)objs->at(i);
		  printf("Object type=%d, (%s) \r\n",ob->type,ob->id.name);
		  switch(ob->type) {
			case ObjectMesh:
			case ObjectLight:
				objs_to_scene.push_back(ob);
			break;
			case ObjectCamera:
				if(!camera)
					camera = ob;
			break;
		  }
	}

	if(objs_to_scene.size() == 0) {
		m_LastError = kEmptyScene;
		return NULL;
	}

	#pragma pack(1)
	//Generate scene
	GeneratedSpheres = objs_to_scene.size();
	Sphere* scene = (Sphere*)malloc(sizeof(Sphere)*GeneratedSpheres);
	
	for(int i=0;i<GeneratedSpheres;i++) {
		Blender::Object* ob = objs_to_scene[i];

		//Location
		scene[i].p.x = ob->loc.x; scene[i].p.y = ob->loc.y; scene[i].p.z = ob->loc.z;
		
		//Radius
		if(ob->type == ObjectMesh) {
			scene[i].rad = (ob->size.x);//Spheres must have cube as BoundingBox
			if(scene[i].rad < 0) scene[i].rad = - scene[i].rad;
		} else if(ob->type == ObjectLight){
			scene[i].rad = ((Blender::Lamp*)ob->data)->area_size;
		} else {
			scene[i].rad = 1.f;
		}

		//Type
		scene[i].refl = DIFF;

		//Color
		if(ob->type == ObjectLight) {
			Blender::Lamp* lamp = (Blender::Lamp*)ob->data;
			scene[i].e.x=lamp->r*lamp->energy;	scene[i].e.y=lamp->g*lamp->energy; scene[i].e.z=lamp->b*lamp->energy;
			scene[i].c.x = scene[i].c.y = scene[i].c.z = 0.0f;
		} else {
			Blender::Mesh* mesh = (Blender::Mesh*)ob->data;
			if(mesh->mat && mesh->mat[0]) {
				scene[i].c.x = mesh->mat[0]->r; scene[i].c.y = mesh->mat[0]->g; scene[i].c.z = mesh->mat[0]->b; scene[i].c.w = mesh->mat[0]->alpha;
			} else {
				scene[i].c.x = ob->col.x; scene[i].c.y = ob->col.y; scene[i].c.z = ob->col.z; scene[i].c.w = ob->col.w;;
			}
			scene[i].e.x = scene[i].e.y = scene[i].e.z = 0.0f;
	
			//Type for mesh
			if(mesh->mat && (mesh->mat[0]->mode&RAY_MIRROR)!=0)
				scene[i].refl = SPEC;
			else if(mesh->mat && (mesh->mat[0]->mode&RAY_TRANSP)!=0)
				scene[i].refl = REFR;
				
		}
		
		
		
	}

	//Setup camera
	if(camera) {
		SceneCamera.orig.x = camera->loc.x;
		SceneCamera.orig.y = camera->loc.y;
		SceneCamera.orig.z = camera->loc.z;
		
		Vec t = {0.0f,0.0f,-10.0f};
		rotateVec(t,-camera->rot.x,camera->rot.y,camera->rot.z);
		vadd(t,t,SceneCamera.orig);
		SceneCamera.target = t;
		SceneCamera.roll.x = -camera->rot.x;
		SceneCamera.roll.y = camera->rot.y;
		SceneCamera.roll.z = camera->rot.z;
		
	}

		return scene;
}
